
/**************************************************************************
 *                                                                        *
 *  BTools - Miscellaneous Java utility classes                           *
 *                                                                        *
 *  Copyright (c) 1998-2001, Ben Burton                                   *
 *  For further details contact Ben Burton (benb@acm.org).                *
 *                                                                        *
 *  This program is free software; you can redistribute it and/or         *
 *  modify it under the terms of the GNU General Public License as        *
 *  published by the Free Software Foundation; either version 2 of the    *
 *  License, or (at your option) any later version.                       *
 *                                                                        *
 *  This program is distributed in the hope that it will be useful, but   *
 *  WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *  General Public License for more details.                              *
 *                                                                        *
 *  You should have received a copy of the GNU General Public             *
 *  License along with this program; if not, write to the Free            *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,        *
 *  MA 02111-1307, USA.                                                   *
 *                                                                        *
 **************************************************************************/

/* end stub */

package org.gjt.btools.gui.dialog;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import org.gjt.btools.gui.*;

/**
 * Represents a dialog that asks the user for a simple text string.
 * Note that pressing <i>Return</i> in the dialog will have the same
 * effect as pressing the <i>OK</i> button.
 * <p>
 * Generally you will never need to create a new dialog; the static
 * member function <tt>getInput()</tt> will do all the work for you.
 */
public class InputBox extends JDialog implements ActionListener {
    /**
     * Message on the <i>OK</i> button.
     */
    private static final String msgOK = "OK";

    /**
     * Message on the <i>Cancel</i> button.
     */
    private static final String msgCANCEL = "Cancel";

    /**
     * Panel in which the buttons reside.
     * @serial
     */
    private JPanel buttonPanel;

    /**
     * The field in which the user can enter the requested text.
     * @serial
     */
    private JTextField text;

    /**
     * The text entered by the user; this is <tt>null</tt> until
     * the dialog has been closed using <i>OK</i>.
     * @serial
     */
    private String result = null;

    /**
     * Create a new input box.
     *
     * @param parent the window to use as parent for the new dialog.
     * @param text the prompt to present to the user.
     */
    public InputBox(Frame parent, String title) {
        super(parent, title, true);
        init(null);
    }

    /**
     * Create a new input box.
     *
     * @param parent the window to use as parent for the new dialog.
     * @param title the title to give the new dialog.
     * @param text the prompt to present to the user.
     */
    public InputBox(Frame parent, String title, String text) {
        super(parent, title, true);
        init(text);
    }

    /**
     * Create a new input box.
     *
     * @param parent the window to use as parent for the new dialog.
     * @param text the prompt to present to the user.
     */
    public InputBox(Dialog parent, String title) {
        super(parent, title, true);
        init(null);
    }

    /**
     * Create a new input box.
     *
     * @param parent the window to use as parent for the new dialog.
     * @param title the title to give the new dialog.
     * @param text the prompt to present to the user.
     */
    public InputBox(Dialog parent, String title, String text) {
        super(parent, title, true);
        init(text);
    }

    /**
     * Called when a button is pressed.
     *
     * @param e information on the button press event.
     */
    public void actionPerformed(ActionEvent e) {
        if (e.getSource() instanceof JButton) {
            // Find out which button was pressed.
            String cmd = e.getActionCommand();
            if (cmd.equals(msgOK))
                result = text.getText();
            else if (cmd.equals(msgCANCEL))
                result = null;

            // Close the message box.
            dispose();
        } else if (e.getSource() == text) {
            result = text.getText();
            dispose();
        }
    }

    /**
     * Initialises the input box, creating and inserting all
     * necessary components.
     *
     * @param initText the default to be offered in the input text field.
     */
    private void init(String initText) {
        // Initialise the layout manager.
        GridBagLayout gb = new GridBagLayout();
        GridBagConstraints c = new GridBagConstraints();
        c.fill = c.BOTH;
        c.gridwidth = c.REMAINDER;
        getContentPane().setLayout(gb);

        // Initialise the text region and insert the message.
        text = new JTextField(20);
        if (initText != null)
            text.setText(initText);
        text.addActionListener(this);

        // Insert the text region into the message box.
        c.weightx = 1;
        c.weighty = 1;
        c.insets = new Insets(10, 10, 0, 10);
        c.ipadx = 5;
        c.ipady = 5;
        gb.setConstraints(text, c);
        getContentPane().add(text);

        // Create and initialise the button panel.
        buttonPanel = new JPanel();
        insertButtons();

        // Insert the button panel into the message box.
        c.weightx = 0;
        c.weighty = 0;
        c.ipadx = 0;
        c.ipady = 0;
        c.insets = new Insets(5, 5, 5, 5);
        gb.setConstraints(buttonPanel, c);
        getContentPane().add(buttonPanel);

        // Tidy up.
        setDefaultCloseOperation(DO_NOTHING_ON_CLOSE);
        pack();
        Positioner.centerOnScreen(this);
    }

    /**
     * Inserts a specified button into the button panel and
     * registers the input box as an action listener for
     * that button.
     *
     * @param b the button to insert.
     */
    private void insertButton(JButton b) {
        buttonPanel.add(b);
        b.addActionListener(this);
    }

    /**
     * Inserts the button set into the button panel and
     * registers the input box as an action listener for those
     * buttons.
     */
    private void insertButtons() {
        JButton ok = new JButton(msgOK);
        insertButton(ok);
        insertButton(new JButton(msgCANCEL));

        getRootPane().setDefaultButton(ok);
    }

    /**
     * Displays the input box and returns the information collected from
     * the user.
     *
     * @return the information collected from the user if <i>OK</i>
     * was pressed to close the dialog, or <tt>null</tt> otherwise.
     */
    public String runBox() {
        show();
        return result;
    }

    /**
     * Creates and runs a new input box and returns the information
     * collected from the user.
     *
     * @param parent the window to use as parent for the new dialog.
     * @param text the prompt to present to the user.
     * @return the information collected from the user if <i>OK</i>
     * was pressed to close the dialog, or <tt>null</tt> otherwise.
     */
    public static String getInput(Frame parent, String text) {
        return new InputBox(parent, "Request Input", text).runBox();
    }

    /**
     * Creates and runs a new input box and returns the information
     * collected from the user.
     *
     * @param parent the window to use as parent for the new dialog.
     * @param title the title to give the new dialog.
     * @param text the prompt to present to the user.
     * @return the information collected from the user if <i>OK</i>
     * was pressed to close the dialog, or <tt>null</tt> otherwise.
     */
    public static String getInput(Frame parent, String title, String text) {
        return new InputBox(parent, title, text).runBox();
    }

    /**
     * Creates and runs a new input box and returns the information
     * collected from the user.
     *
     * @param parent the window to use as parent for the new dialog.
     * @param text the prompt to present to the user.
     * @return the information collected from the user if <i>OK</i>
     * was pressed to close the dialog, or <tt>null</tt> otherwise.
     */
    public static String getInput(Dialog parent, String text) {
        return new InputBox(parent, "Request Input", text).runBox();
    }

    /**
     * Creates and runs a new input box and returns the information
     * collected from the user.
     *
     * @param parent the window to use as parent for the new dialog.
     * @param title the title to give the new dialog.
     * @param text the prompt to present to the user.
     * @return the information collected from the user if <i>OK</i>
     * was pressed to close the dialog, or <tt>null</tt> otherwise.
     */
    public static String getInput(Dialog parent, String title, String text) {
        return new InputBox(parent, title, text).runBox();
    }
}

