/*
 * $Header: /home/cvspublic/jakarta-commons/httpclient/src/test/org/apache/commons/httpclient/TestCookie.java,v 1.10 2002/05/01 07:07:50 rwaldhoff Exp $
 * $Revision: 1.10 $
 * $Date: 2002/05/01 07:07:50 $
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "Tomcat", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.commons.httpclient;

import junit.framework.Test;
import junit.framework.TestCase;
import junit.framework.TestSuite;
import java.util.Date;
import java.util.Vector;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.Iterator;


/**
 * Test cases for Cookie
 *
 * @author BC Holmes
 * @author Rod Waldhoff
 * @author dIon Gillard
 * @author <a href="mailto:JEvans@Cyveillance.com">John Evans</a>
 * @author Marc A. Saegesser
 * @version $Revision: 1.10 $
 */
public class TestCookie extends TestCase {


    // -------------------------------------------------------------- Constants

    private static final String DOMAIN_NAME = "www.apache.org";
    private static final String TEST_COOKIE = "cookie-name=cookie-value";
    private static final String OLD_EXPIRY = "Expires=Thu, 01-Jan-1970 00:00:10 GMT";
    private static final String SEP = ";";
    private static final String ROOT_PATH = "/";

    private String[] testName = { "custno", "name", "name" };
    private String[] testValue = { "12345", "John", "Doe, John" };
    private String[] testDomain = { "www.apache.org", ".apache.org",
        ".apache.org" };

    // ------------------------------------------------------------ Constructor


    public TestCookie(String name) {
        super(name);
    }


    // ------------------------------------------------------- TestCase Methods


    public static Test suite() {
        return new TestSuite(TestCookie.class);
    }


    // ------------------------------------------------------------ Parse1 Test


    /**
     * Test basic parse (with various spacings
     */
    public void testParse1() throws Exception {
        String headerValue = "custno = 12345; comment=test; version=1," +
            " name=John; version=1; max-age=600; secure; domain=.apache.org";
        Cookie[] cookies = Cookie.parse(DOMAIN_NAME,"/", true, new Header(
            "set-cookie", headerValue));
        checkResultsOfParse(cookies, 2, 0);
    }


    protected void checkResultsOfParse(
        Cookie[] cookies, int length, int offset) throws Exception {

        assertTrue("number of cookies should be " + length + ", but is " +
               cookies.length + " instead.", cookies.length == length);

        for (int i = 0; i < cookies.length; i++) {

            assertTrue("Name of cookie " + i + " should be \"" +
                   testName[i+offset] + "\", but is " + cookies[i].getName() +
                   " instead.",
                   testName[i+offset].equals(cookies[i].getName()));
            assertTrue("Value of cookie " + i + " should be \"" +
                   testValue[i+offset] + "\", but is " +
                   cookies[i].getValue() + " instead.",
                   testValue[i+offset].equals(cookies[i].getValue()));
            assertTrue("Domain of cookie " + i + " should be \"" +
                   testDomain[i+offset] + "\", but is " +
                   cookies[i].getDomain() + " instead.",
                   testDomain[i+offset].equalsIgnoreCase(
                       cookies[i].getDomain()));
        }
    }


    // ------------------------------------------------------------ Parse2 Test


    /**
     * Test no spaces
     */
    public void testParse2() throws Exception {
        String headerValue = "custno=12345;comment=test; version=1," +
            "name=John;version=1;max-age=600;secure;domain=.apache.org";
        Cookie[] cookies = Cookie.parse(DOMAIN_NAME, "/", true, new Header(
            "set-cookie", headerValue));
        checkResultsOfParse(cookies, 2, 0);
    }


    // ------------------------------------------------------------ Parse3 Test


    /**
     * Test parse with quoted text
     */
    public void testParse3() throws Exception {
        String headerValue =
            "name=\"Doe, John\";version=1;max-age=600;secure;domain=.apache.org";
        Cookie[] cookies = Cookie.parse(DOMAIN_NAME,"/", true, new Header(
            "set-cookie", headerValue));
        checkResultsOfParse(cookies, 1, 2);
    }

    // ------------------------------------------------------------- More Tests

    // see http://nagoya.apache.org/bugzilla/show_bug.cgi?id=5279
    public void testQuotedExpiresAttribute() throws Exception {
        String headerValue = "custno=12345;Expires='Thu, 01-Jan-2070 00:00:10 GMT'";
        Cookie[] cookies = Cookie.parse(DOMAIN_NAME,"/",true,new Header(
            "set-cookie", headerValue));
        assertNotNull("Expected some cookies",cookies);
        assertEquals("Expected 1 cookie",1,cookies.length);
        assertNotNull("Expected cookie to have getExpiryDate",cookies[0].getExpiryDate());
    }

    public void testSecurityError() throws Exception {
        String headerValue = "custno=12345;comment=test; version=1," +
            "name=John;version=1;max-age=600;secure;domain=jakarta.apache.org";
        try {
            Cookie[] cookies = Cookie.parse(DOMAIN_NAME, "/", new Header(
                "set-cookie", headerValue));
            fail("HttpException exception should have been thrown");
        } catch (HttpException e) {
            // expected
        }
    }

    public void testParseSimple() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/path/path",setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertEquals("Value","cookie-value",parsed[0].getValue());
        assertTrue("Comment",null == parsed[0].getComment());
        assertTrue("ExpiryDate",null == parsed[0].getExpiryDate());
        //assertTrue("isToBeDiscarded",parsed[0].isToBeDiscarded());
        assertTrue("isPersistent",!parsed[0].isPersistent());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/path",parsed[0].getPath());
        assertTrue("Secure",!parsed[0].getSecure());
        assertEquals("Version",0,parsed[0].getVersion());
    }

    public void testParseNoName() throws Exception {
        Header setCookie = new Header("Set-Cookie","=cookie-value");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/",setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","",parsed[0].getName());
        assertEquals("Value","cookie-value",parsed[0].getValue());
        assertTrue("Comment",null == parsed[0].getComment());
        assertTrue("ExpiryDate",null == parsed[0].getExpiryDate());
        //assertTrue("isToBeDiscarded",parsed[0].isToBeDiscarded());
        assertTrue("isPersistent",!parsed[0].isPersistent());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/",parsed[0].getPath());
        assertTrue("Secure",!parsed[0].getSecure());
        assertEquals("Version",0,parsed[0].getVersion());
    }

    public void testParseNoValue() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/",setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertTrue("Value",null == parsed[0].getValue());
        assertTrue("Comment",null == parsed[0].getComment());
        assertTrue("ExpiryDate",null == parsed[0].getExpiryDate());
        //assertTrue("isToBeDiscarded",parsed[0].isToBeDiscarded());
        assertTrue("isPersistent",!parsed[0].isPersistent());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/",parsed[0].getPath());
        assertTrue("Secure",!parsed[0].getSecure());
        assertEquals("Version",0,parsed[0].getVersion());
    }

    public void testParseWithWhiteSpace() throws Exception {
        Header setCookie = new Header("Set-Cookie"," cookie-name  =    cookie-value  ");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/",setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertEquals("Value","cookie-value",parsed[0].getValue());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/",parsed[0].getPath());
        assertTrue("Secure",!parsed[0].getSecure());
        assertTrue("ExpiryDate",null == parsed[0].getExpiryDate());
        assertTrue("Comment",null == parsed[0].getComment());
    }

    public void testParseWithQuotes() throws Exception {
        Header setCookie = new Header("Set-Cookie"," cookie-name  =  \" cookie-value \" ;path=/");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/",setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertEquals("Value"," cookie-value ",parsed[0].getValue());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/",parsed[0].getPath());
        assertTrue("Secure",!parsed[0].getSecure());
        assertTrue("ExpiryDate",null == parsed[0].getExpiryDate());
        assertTrue("Comment",null == parsed[0].getComment());
    }

    public void testParseWithPath() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; Path=/path/");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/path/path",setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertEquals("Value","cookie-value",parsed[0].getValue());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/path/",parsed[0].getPath());
        assertTrue("Secure",!parsed[0].getSecure());
        assertTrue("ExpiryDate",null == parsed[0].getExpiryDate());
        assertTrue("Comment",null == parsed[0].getComment());
    }

    public void testParseWithDomain() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; Domain=127.0.0.1");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/",setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertEquals("Value","cookie-value",parsed[0].getValue());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/",parsed[0].getPath());
        assertTrue("Secure",!parsed[0].getSecure());
        assertTrue("ExpiryDate",null == parsed[0].getExpiryDate());
        assertTrue("Comment",null == parsed[0].getComment());
    }

    public void testParseWithSecure() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; secure");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/",true,setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertEquals("Value","cookie-value",parsed[0].getValue());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/",parsed[0].getPath());
        assertTrue("Secure",parsed[0].getSecure());
        assertTrue("ExpiryDate",null == parsed[0].getExpiryDate());
        assertTrue("Comment",null == parsed[0].getComment());
    }

    public void testParseWithComment() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; comment=\"This is a comment.\"");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/",true,setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertEquals("Value","cookie-value",parsed[0].getValue());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/",parsed[0].getPath());
        assertTrue("Secure",!parsed[0].getSecure());
        assertTrue("ExpiryDate",null == parsed[0].getExpiryDate());
        assertEquals("Comment","This is a comment.",parsed[0].getComment());
    }

    public void testParseWithExpires() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value;Expires=Thu, 01-Jan-1970 00:00:10 GMT");
        Cookie[] parsed = Cookie.parse("127.0.0.1","/",true,setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertEquals("Value","cookie-value",parsed[0].getValue());
        assertEquals("Domain","127.0.0.1",parsed[0].getDomain());
        assertEquals("Path","/",parsed[0].getPath());
        assertTrue("Secure",!parsed[0].getSecure());
        assertEquals(new Date(10000L),parsed[0].getExpiryDate());
        assertTrue("Comment",null == parsed[0].getComment());
    }

    public void testParseWithAll() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value;Version=1;Path=/commons;Domain=.apache.org;Comment=This is a comment.;secure;Expires=Thu, 01-Jan-1970 00:00:10 GMT");
        Cookie[] parsed = Cookie.parse(".apache.org","/commons/httpclient",true,setCookie);
        assertEquals("Found 1 cookie.",1,parsed.length);
        assertEquals("Name","cookie-name",parsed[0].getName());
        assertEquals("Value","cookie-value",parsed[0].getValue());
        assertEquals("Domain",".apache.org",parsed[0].getDomain());
        assertEquals("Path","/commons",parsed[0].getPath());
        assertTrue("Secure",parsed[0].getSecure());
        assertEquals(new Date(10000L),parsed[0].getExpiryDate());
        assertEquals("Comment","This is a comment.",parsed[0].getComment());
        assertEquals("Version",1,parsed[0].getVersion());
    }

    public void testParseMultipleDifferentPaths() throws Exception {
        Header setCookie = new Header("Set-Cookie","name1=value1;Version=1;Path=/commons,name1=value2;Version=1;Path=/commons/httpclient;Version=1");
        Cookie[] parsed = Cookie.parse(".apache.org","/commons/httpclient",true,setCookie);
        HttpState state = new HttpState();
        state.addCookies(parsed);
        Cookie[] cookies = state.getCookies();
        assertEquals("Wrong number of cookies.",2,cookies.length);
        assertEquals("Name","name1",cookies[0].getName());
        assertEquals("Value","value1",cookies[0].getValue());
        assertEquals("Name","name1",cookies[1].getName());
        assertEquals("Value","value2",cookies[1].getValue());
    }

    public void testParseMultipleSamePaths() throws Exception {
        Header setCookie = new Header("Set-Cookie","name1=value1;Version=1;Path=/commons,name1=value2;Version=1;Path=/commons");
        Cookie[] parsed = Cookie.parse(".apache.org","/commons/httpclient",true,setCookie);
        HttpState state = new HttpState();
        state.addCookies(parsed);
        Cookie[] cookies = state.getCookies();
        assertEquals("Found 1 cookies.",1,cookies.length);
        assertEquals("Name","name1",cookies[0].getName());
        assertEquals("Value","value2",cookies[0].getValue());
    }

    public void testParseWithWrongDomain() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=127.0.0.1; version=1");
        try {
            Cookie[] parsed = Cookie.parse("127.0.0.2","/",setCookie);
            fail("HttpException exception should have been thrown");
        } catch (HttpException e) {
            // expected
        }
    }

    public void testParseWithWrongDomain2() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=.c.com; version=1");
        try {
            Cookie[] parsed = Cookie.parse("a.b.c.com","/",setCookie);
            fail("HttpException exception should have been thrown");
        } catch (HttpException e) {
            // expected
        }
    }

    /**
     * Domain has no embedded dots
     */
    public void testParseWithIllegalDomain() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=.com; version=1");
        try {
            Cookie[] parsed = Cookie.parse("b.com","/",setCookie);
            fail("HttpException exception should have been thrown");
        } catch (HttpException e) {
            // expected
        }
    }

    /**
     * Domain has no embedded dots again
     */
    public void testParseWithIllegalDomain2() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=.com.; version=1");
        try {
            Cookie[] parsed = Cookie.parse("b.com","/",setCookie);
            fail("HttpException exception should have been thrown");
        } catch (HttpException e) {
            // expected
        }
    }

    public void testParseWithIllegalNetscapeDomain1() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=.com");
        try {
            Cookie[] parsed = Cookie.parse("a.com","/",setCookie);
            fail("HttpException exception should have been thrown");
        } catch (HttpException e) {
            // expected
        }
    }

    public void testParseWithWrongNetscapeDomain2() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=.y.z");
        try {
            Cookie[] parsed = Cookie.parse("x.y.z","/",setCookie);
            fail("HttpException exception should have been thrown");
        } catch (HttpException e) {
            // expected
        }
    }

    public void testParseWithWrongPath() throws Exception {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=127.0.0.1; path=/not/just/root");
        try {
            Cookie[] parsed = Cookie.parse("127.0.0.1","/",setCookie);
            fail("HttpException exception should have been thrown");
        } catch (HttpException e) {
            // expected
        }
    }

    public void testParseWithNullDomain() {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=127.0.0.1; path=/; secure");
        try {
            Cookie[] parsed = Cookie.parse(null,"/",false,setCookie);
            fail("IllegalArgumentException should have been thrown");
        } catch (IllegalArgumentException e) {
            // expected
        } catch (Exception e){
            fail("Should have thrown IllegalArgumentException.");
        }
    }

    public void testParseWithNullPath() {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=127.0.0.1; path=/; secure");
        try {
            Cookie[] parsed = Cookie.parse("127.0.0.1",null,false,setCookie);
            fail("IllegalArgumentException should have been thrown");
        } catch (IllegalArgumentException e) {
            // expected
        } catch (Exception e){
            fail("Should have thrown IllegalArgumentException.");
        }
    }

    public void testParseWithNullDomainAndPath() {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; domain=127.0.0.1; path=/; secure");
        try {
            Cookie[] parsed = Cookie.parse(null,null,false,setCookie);
            fail("IllegalArgumentException should have been thrown");
        } catch (IllegalArgumentException e) {
            // expected
        } catch (Exception e){
            fail("Should have thrown IllegalArgumentException.");
        }
    }
    
    public void testParseWithPathMismatch() {
        Header setCookie = new Header("Set-Cookie","cookie-name=cookie-value; path=/path/path/path");
        try {
            Cookie[] parsed = Cookie.parse("127.0.0.1","/path",false,setCookie);
            fail("HttpException should have been thrown.");
        } catch (HttpException e) {
            // expected
        } catch (Exception e){
            fail("Should have thrown HttpException.");
        }
    }
    
    public void testComparator() throws Exception {
        Header setCookie = null;
        Cookie[] parsed = null;
        Vector cookies = new Vector();
        // Cookie 0
        setCookie = new Header("Set-Cookie","cookie-name=cookie-value;Path=/commons;Domain=.apache.org;Expires=Thu, 01-Jan-1970 00:00:10 GMT");
        parsed = Cookie.parse(".apache.org", "/commons/httpclient", true,
                              setCookie);
        cookies.add(parsed[0]);
        // Cookie 1
        setCookie = new Header("Set-Cookie","cookie-name=cookie-value;Path=/commons/bif;Domain=.apache.org;Expires=Thu, 01-Jan-1970 00:00:10 GMT");
        parsed = Cookie.parse(".apache.org","/commons/bif/httpclient",true,setCookie);
        cookies.add(parsed[0]);
        // Cookie 2
        setCookie = new Header("Set-Cookie","cookie-name=cookie-value;Path=/commons;Domain=.baz.org;Expires=Thu, 01-Jan-1970 00:00:10 GMT");
        parsed = Cookie.parse(".baz.org","/commons/httpclient",true,setCookie);
        cookies.add(parsed[0]);
        // Cookie 3
        setCookie = new Header("Set-Cookie","cookie-name=cookie-value;Path=/commons/bif;Domain=.baz.org;Expires=Thu, 01-Jan-1970 00:00:10 GMT");
        parsed = Cookie.parse(".baz.org","/commons/bif/httpclient",true,setCookie);
        cookies.add(parsed[0]);
        // Cookie 4
        setCookie = new Header("Set-Cookie","cookie-name=cookie-value;Path=/commons;Domain=.baz.com;Expires=Thu, 01-Jan-1970 00:00:10 GMT");
        parsed = Cookie.parse(".baz.com","/commons/httpclient",true,setCookie);
        cookies.add(parsed[0]);
        // The order should be:
        // 1, 0, 3, 2, 4
        parsed = (Cookie[])cookies.toArray(new Cookie[0]);
        SortedSet set = new TreeSet(parsed[0]);
        int pass = 0;
        for (Iterator itr = set.iterator(); itr.hasNext();) {
            Cookie cookie = (Cookie)itr.next();
            switch (pass) {
                case 0:
                    assertTrue("0th cookie should be cookie[1]", cookie == parsed[1]);
                    break;
                case 1:
                    assertTrue("1st cookie should be cookie[0]", cookie == parsed[0]);
                    break;
                case 2:
                    assertTrue("2nd cookie should be cookie[3]", cookie == parsed[3]);
                    break;
                case 3:
                    assertTrue("3rd cookie should be cookie[2]", cookie == parsed[2]);
                    break;
                case 4:
                    assertTrue("4th cookie should be cookie[4]", cookie == parsed[4]);
                    break;
                default:
                    fail("This should never happen.");
            }
            pass++;
        }
        try {
            parsed[0].compare("foo", "bar");
            fail("Should have thrown an exception trying to compare non-cookies");
        }
        catch (ClassCastException ex) {
            // expected
        }
    }
    
    /** Call Cookie.createCookieHeader providing null for domain to match on
     */
    public void testCreateCookieHeaderWithNullDomain() throws Exception {
        Header setCookie = new Header("Set-Cookie",
                                      TEST_COOKIE + SEP + OLD_EXPIRY);
        Cookie[] parsed = Cookie.parse(DOMAIN_NAME, ROOT_PATH, true, setCookie);

        try{
            Header header = Cookie.createCookieHeader(null, ROOT_PATH, false, parsed);
            fail("IllegalArgumentException should have been thrown.");
        }catch(IllegalArgumentException e){
            // Expected
        }catch(Exception e){
            fail("Threw wrong type of exception.  Expected IllegalArgumentException.");
        }
    }
    
    /** Call Cookie.createCookieHeader providing null for path to match on
     */
    public void testCreateCookieHeaderWithNullPath() throws Exception{
        Header setCookie = new Header("Set-Cookie",
                                      TEST_COOKIE + SEP + OLD_EXPIRY);
        Cookie[] parsed = Cookie.parse(DOMAIN_NAME, ROOT_PATH, false, setCookie);

        try{
            Header header = Cookie.createCookieHeader(DOMAIN_NAME, null, false, parsed);
            fail("IllegalArgumentException should have been thrown.");
        }catch(IllegalArgumentException e){
            // Expected
        }catch(Exception e){
            fail("Threw wrong type of exception.  Expected IllegalArgumentException.");
        }
    }

    /**
     * Verify that cookies with no domain or path don't get added to a cookie header.
     */
    public void testCreateCookieHeaderWithUninitializedCookies() throws Exception {
        Cookie cookies[] = new Cookie[2];
        cookies[0] = new Cookie(null, "name0", "value0");
        cookies[1] = new Cookie(null, "name1", "value1", null, null, false);

        Header header = Cookie.createCookieHeader(DOMAIN_NAME, ROOT_PATH, false, cookies);
        assertEquals("createCookieHeader added cookies with null domains or paths", null, header);
    }

    /** Call Cookie.createCookieHeader providing null for domain and path to
     * match on
     */
    public void testCreateCookieHeaderWithNullDomainAndPath() throws Exception {
        Header setCookie = new Header("Set-Cookie",
                                      TEST_COOKIE + SEP + OLD_EXPIRY);
        Cookie[] parsed = Cookie.parse(DOMAIN_NAME, ROOT_PATH, true, setCookie);

        try{
            Header header = Cookie.createCookieHeader(null, null, false, parsed);
            fail("IllegalArgumentException should have been thrown.");
        }catch(IllegalArgumentException e){
            // Expected
        }catch(Exception e){
            fail("Threw wrong type of exception.  Expected IllegalArgumentException.");
        }
    }
}

