/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.vise;

import java.util.ResourceBundle;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.event.*;

import com.redhat.rhdb.vise.resources.ActionResources;

/**
 * Password dialog for Explain. 
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 */
public class PasswordDialog extends JDialog implements ActionListener {
	private Frame owner;
	private JPasswordField passfield;
	private int result;
	private JLabel dblabel;
    private ResourceBundle res;

	/** User has closed window with intention of accepting changes. */
	public static final int OK_OPTION = 0;

	/** User has closed window with intention of discarding changes. */
	public static final int CANCEL_OPTION = 1;

	/** User has closed window with intention of discarding changes. */
	public static final int CLOSED_OPTION = 2;
	
	/**
	 * Creates a new <code>PasswordDialog</code> instance that is to
	 * be owned by the Frame <i>owner</i>.
	 *
	 * @param owner a <code>Frame</code> value
	 */
	public PasswordDialog(Frame owner)
	{
		super(owner, ExplainResources.getString(ExplainResources.PASSWORD_NEEDED_TITLE), true);

		this.owner = owner;

		addWindowListener(new WindowAdapter() {
			public void windowClosing(WindowEvent e)
			{
				close();
			}
		});

        res = ResourceBundle.getBundle("com.redhat.rhdb.vise.resources.ActionResources");

		// initialize components
		initComponents();
	}

	/**
	 * Set the database model that the password dialog should prompt a
	 * password for. The model is not modified in any way.
	 *
	 * @param d a <code>DatabaseModel</code> value
	 */
	public void setDatabaseModel(DatabaseModel d)
	{
		dblabel.setText(ExplainResources.getString(ExplainResources.PASSWORD_NEEDED_MSG,
												   d.getName()));
	}
	
	/**
	 * Show the dialog, first setting the location relative to
	 * owner's. Password entry field is cleared.
	 *
	 */
	public void show()
	{
		pack();
		setLocationRelativeTo(owner);
		passfield.setText("");
		passfield.requestFocus();

		super.show();
	}

	/**
	 * Get how/why dialog was closed.
	 *
	 * @return an <code>int</code> value
	 */
	public int getResult()
	{
		return result;
	}

	/**
	 * Get the entered password if the window closing result was
	 * OK_OPTION.
	 *
	 * @return a <code>String</code> value
	 */
	public String getPassword()
	{
		if (result == OK_OPTION)
			return new String(passfield.getPassword());
		return null;
	}
	
	//
	// ActionListener
	//

	/**
	 * Action listener.
	 *
	 * @param e an <code>ActionEvent</code> value
	 */
	public void actionPerformed(ActionEvent e)
	{
		String command = e.getActionCommand();

		if (command.equals(res.getString(ActionResources.OK_COMMAND)))
		{
			ok();
		}
		else if (command.equals(res.getString(ActionResources.CANCEL_COMMAND)))
		{
			cancel();
		}
		else
		{
			System.err.println("PasswordDialog: unknown action command" + command);
		}
	}

	//
	// private methods
	//

	private void cancel()
	{
		result = CANCEL_OPTION;
		hide();
	}

	private void close()
	{
		result = CLOSED_OPTION;
		hide();
	}

	private void ok()
	{
		result = OK_OPTION;
		hide();
	}
	
	/*
	 * create all components
	 */
	private void initComponents()
	{
		JPanel content;
		JPanel buttonPanel;
		JPanel passpanel;

		content = new JPanel();
		content.setLayout(new BorderLayout());
		content.setBorder(BorderFactory.createEmptyBorder(12, 12, 11, 11));
		setContentPane(content);

		passpanel = new JPanel();
		passpanel.setLayout(new BoxLayout(passpanel, BoxLayout.Y_AXIS));
		passpanel.add(createPasswordPanel());
		passpanel.add(Box.createRigidArea(new Dimension(0, 17)));
		content.add(passpanel, BorderLayout.CENTER);

		buttonPanel = new ButtonPanel(this);
		content.add(buttonPanel, BorderLayout.SOUTH);
	}

	/*
	 * create the panel for getting password
	 */
	private JPanel createPasswordPanel()
	{
		JPanel panel;
		JLabel label;
		
		GridBagLayout gbl = new GridBagLayout();
		GridBagConstraints gbc = new GridBagConstraints();

		gbc.fill = GridBagConstraints.HORIZONTAL;
		gbc.anchor = GridBagConstraints.WEST;
		gbc.weighty = 1.0;

		panel = new JPanel();
		panel.setLayout(gbl);

		// password
		dblabel = new JLabel();
		gbc.gridx = 0;
		gbc.gridy = 0;
		gbc.weightx = 0;
		gbc.gridwidth = 2;
		gbc.insets = new Insets(0, 0, 11, 0);
		gbl.setConstraints(dblabel, gbc);
		panel.add(dblabel);

		// entry
		label = new JLabel(ExplainResources.getString(ExplainResources.PASSWORD_NEEDED_LABEL));
		gbc.gridx = 0;
		gbc.gridy = 1;
		gbc.weightx = 0;
		gbc.gridwidth = 1;
		gbc.insets = new Insets(0, 0, 0, 0);
		gbl.setConstraints(label, gbc);
		panel.add(label);

		passfield = new JPasswordField(20);
		gbc.gridx = 1;
		gbc.weightx = 1;
		gbc.insets = new Insets(0, 11, 0, 0);
		gbl.setConstraints(passfield, gbc);
		panel.add(passfield);
		label.setLabelFor(passfield);
		label.setDisplayedMnemonic(((Integer)ExplainResources.getObject(ExplainResources.PASSWORD_NEEDED_MNEMONIC)).intValue());

		return panel;
	}

}// PasswordDialog
