/*
 * snes9express
 * video.cc
 * Copyright (C) 1998-2004  David Nordlund
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * For further details, please read the included COPYING file,
 * or go to http://www.gnu.org/copyleft/gpl.html
 */

#include <stdlib.h>
#include "video.h"

struct _TV_modes
{
        char*   name;
        char*   args[2];
        char*   tip;
};

struct _TV_modes tv_mode[] =
{
  { "Normal",       {"-y",  "-interpolate" }, "Normal TV mode"},
  { "Super 2xSAI",  {"-y2", "-interpolate2"}, "Kreed's Super 2xSAI image processing mode"},
  { "Super Eagle",  {"-y3", "-interpolate3"}, "Kreed's Super Eagle image processing mode"},
  { "2xSAI",        {"-y4", "-interpolate4"}, "Kreed's 2xSAI image processing mode"},
  { "Filter/Scale", {"-y5", "-interpolate5"}, "Kreed's software bi-linear filtering & scaling mode"},
};

s9x_Video::s9x_Video(fr_Notebook*parent):
s9x_Notepage(parent, "Video"),
Scale(this, "Scale", false),
HiRes(this, "Hi-Res", false),
FullScreen(this, "Full Screen", false),
VideoSystem(this, "Video System"),
TVmode(this, "TV-mode"),
UseFrameSkip(this, "Frame-skip:", false),
UseFrameTime(this, "Frame-time:", false),
DisplayFrameRate(this, "Display Frame Rate", false),
FrameSkip(this, "", 0, 0, 30, 0),
FrameTime(this, "", 16.7, 0.0, 1000.0, 1),
FrameSkipLabel(this, "auto"),
FrameTimeLabel(this, "auto")
{
   fr_MenuItem *MI;

   SetGridSize(4, 3, true);

   Scale.Args << fr_CaseInsensitive << "-scale" << "-sc";
   Scale.SetTooltip("In full-screen mode, resize the window to the actual screen size. (CPU intensive)");
   Pack(Scale);

   HiRes.Args << "-hires";
   HiRes.SetTooltip("This will double the screen size, but will run a little slower.");
   Pack(HiRes);

   FullScreen.Args << fr_CaseInsensitive << "-fullscreen";
   FullScreen.SetKeyStroke("Alt+F or Alt+Enter");
   FullScreen.SetTooltip("Put snes9x in Full Screen mode.  (requires root"
			 " privileges)");
   Pack(FullScreen);

   MI = new fr_MenuItem(&TVmode, "off");
   MI->SetTooltip("do not enable TV-mode");
   for(int i=0; i<5; i++)
   {
        TVmodes[i] = new fr_MenuItem(&TVmode, tv_mode[i].name);
        TVmodes[i]->Args << fr_CaseInsensitive
                         << tv_mode[i].args[0] << tv_mode[i].args[1];
        TVmodes[i]->SetTooltip(tv_mode[i].tip);
   }
   TVmode.SetTooltip("TV-modes\nuse double resolution with pixel blended scan lines"
		     " inserted in video to mimic the look of a TV screen.");
   Pack(TVmode);

   MI = new fr_MenuItem(&VideoSystem, "auto");
   NTSC = new fr_MenuItem(&VideoSystem, "NTSC");
   NTSC->Args << fr_CaseInsensitive << "-ntsc";
   NTSC->SetTooltip("Convince ROMs that they are on a NTSC system");
   NTSC->AddListener(this);
        
   PAL = new fr_MenuItem(&VideoSystem, "PAL");
   PAL->Args << fr_CaseInsensitive << "-pal";
   PAL->SetTooltip("Convince ROMs that they are on a PAL system");
   PAL->AddListener(this);
   Pack(VideoSystem, 2, 1);
	
   DisplayFrameRate.Args << fr_CaseInsensitive << "-dfr" << "-displayframerate";
   DisplayFrameRate.SetTooltip("During game play, display the frames"
			       " per second that snes9x achieves.");
   Pack(DisplayFrameRate, 2, 1);

   UseFrameSkip.SetTooltip("Use auto-frame skip, or select a frame-skip constant");
   UseFrameSkip.AddListener(this);
   Pack(UseFrameSkip, 0, 2, 1, 3);

   FrameSkip.Args << fr_CaseInsensitive << "-frameskip" << "-f";
   FrameSkip.SetKeyStroke("the + and - keys");
   FrameSkip.SetTooltip("Render 1 out of every n frames");
   FrameSkip.AddListener(this);
   Pack(FrameSkip, 1, 2, 2, 3);
        
   Pack(FrameSkipLabel, 1, 2, 2, 3);

   UseFrameTime.SetTooltip("Use auto-frame time, or specify a frame-time constant in milliseconds");
   UseFrameTime.AddListener(this);
   Pack(UseFrameTime, 2, 2, 3, 3);
        
   FrameTime.Args << fr_CaseInsensitive << "-ft" << "-frametime";
   FrameTime.SetKeyStroke("Shift '+' / Shift '-'");
   FrameTime.SetTooltip("time (in milliseconds) in which a frame should be rendered in.  Higher value slows things down, lower value speeds up");
   Pack(FrameTime, 3, 2, 4, 3);

   Pack(FrameTimeLabel, 3, 2, 4, 3);
}

s9x_Video::~s9x_Video()
{
}

void s9x_Video::Set9xVersion(float version) {
   HiRes.EnableIf(version >= 1.0);
   TVmode.EnableIf(version >= 1.08);
   Scale.EnableIf(version >= 1.08);
   FrameTime.EnableIf(version > 1.08);
   UseFrameTime.EnableIf(version > 1.08);
   DisplayFrameRate.EnableIf(version > 1.18);
   FullScreen.EnableIf(version > 1.18);
   for(int i=1; i<5; i++)
        TVmodes[i]->EnableIf(version > 1.36);
}

void s9x_Video::SetToDefaults() {
   VideoSystem.SetToDefault();
   FrameSkip.SetToDefault();
   Scale.SetToDefault();
   HiRes.SetToDefault();
   TVmode.SetToDefault();
   FullScreen.SetToDefault();
   UseFrameTime.SetToDefault();
   FrameTime.SetToDefault();
   DisplayFrameRate.SetToDefault();
   fr_Event(&UseFrameSkip, fr_Click);
};

void s9x_Video::EventOccurred(fr_Event*e) {
   bool b, s;
   if(e->Is(*PAL) && PAL->GetState())
     FrameTime.ResetDefault(20.0);
   else if(e->Is(*NTSC) && NTSC->GetState())
     FrameTime.ResetDefault(16.7);
   else if(e->Is(UseFrameSkip)||e->Is(FrameSkip)) {
      b = (FrameSkip.GetValue() != 0);
      s = UseFrameSkip.GetState();
      if(e->Is(FrameSkip)) {
	 UseFrameSkip.SetState(b);
	 if(!b)
	   s = false;
      } else
	FrameSkip.SetValue(2*s);
      FrameSkip.SetVisibility(s);
      FrameSkipLabel.SetVisibility(!s);

      UseFrameTime.SetState(false);
      UseFrameTime.SetEditable(!s);
      FrameTime.SetVisibility(false);
      FrameTimeLabel.SetVisibility(true);
      FrameTimeLabel.SetEditable(!s);
   };

   if(e->Is(UseFrameTime)) {
      s = UseFrameTime.GetState();
      UseFrameSkip.SetEditable(!s);
      FrameSkipLabel.SetEditable(!s);
      FrameTime.SetVisibility(s);
      FrameTimeLabel.SetVisibility(!s);
   };
}

void s9x_Video::SiftArgs(fr_ArgList& L) {
   VideoSystem.SiftArgs(L);
   Scale.SiftArgs(L);
   HiRes.SiftArgs(L);
   TVmode.SiftArgs(L);
   FullScreen.SiftArgs(L);
   FrameSkip.SiftArgs(L);
   if(!FrameSkip.IsDefault())
     UseFrameSkip.SetState(true);
   FrameTime.SiftArgs(L);
   if(!FrameTime.IsDefault())
     UseFrameTime.SetState(true);
   DisplayFrameRate.SiftArgs(L);
}

void s9x_Video::CompileArgs(fr_ArgList& L) {
   VideoSystem.CompileArgs(L);
   Scale.CompileArgs(L);
   HiRes.CompileArgs(L);
   TVmode.CompileArgs(L);
   FullScreen.CompileArgs(L);
   if(UseFrameSkip.GetState())
     FrameSkip.CompileArgs(L);

   else if(UseFrameTime.GetState())
     FrameTime.CompileArgs(L);
   DisplayFrameRate.CompileArgs(L);
}

