;;; x-pgp-sig.el -- Yet another PGP authenticating utility for Emacsen.

;; Copyright (C) 1996-1999 Katsumi Yamaoka
;; Copyright (C) 1996-1999 OKUNISHI Fujikazu
;; Copyright (C) 1996-1999 KOSEKI Yoshinori
;; Copyright (C) 1996-1999 Fumitoyo ONO
;; Copyright (C) 1996-1999 Hidekazu Nakamura
;; Author: Katsumi Yamaoka   <yamaoka@jpl.org>
;;         OKUNISHI Fujikazu <fuji0924@mbox.kyoto-inet.or.jp>
;;         KOSEKI Yoshinori  <kose@yk.netlaputa.or.jp>
;;         Fumitoyo ONO      <ono@sa.osk.sumikin.co.jp>
;;         Hidekazu Nakamura <u90121@uis-inf.co.jp>
;; Maintainer: Katsumi Yamaoka <yamaoka@jpl.org>
;; Created: 1996/11/12
;; Revised: 1999/05/31
;; References: /anonymous@ftp.isc.org:/pub/pgpcontrol/README
;;             ftp://ftp.isc.org/pub/pgpcontrol/README.html
;; Keywords:   semi-gnus, gnus, mh-e, mew, cmail, vm, wl,
;;             pgp, pgpverify, signcontrol, x-pgp-sig

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to the
;; Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;; Commands:
;;  x-pgp-sig-set-signer
;;     "Set PGP signer and passphrase."
;;  x-pgp-sig-set-signer-maybe
;;     "Set PGP signer and passphrase if necessary."
;;  x-pgp-sig-switch-signing
;;     "Judging whether to add X-PGP signature."
;;  x-pgp-sig-verify
;;     "Verify X-PGP signature in current message."

;; Functions:
;;  x-pgp-sig-sign
;;     "Add X-PGP signature."
;;  x-pgp-sig-mew-encode-coding-and-sign
;;     "Encode the message to the network code and sign."
;;  x-pgp-sig-reserve-sign
;;     "Reserve sign or not sign before sending messages."
;;  x-pgp-sig-cancel-reserve-sign
;	"Cancel reserve sign flag after sending messages."
;--------------------------------------------------------------------

;;; Code:

(defconst x-pgp-sig-version-number "1.3.5.1"
  "Version number for this version of x-pgp-sig.")
(defconst x-pgp-sig-version
  (format "Yet another PGP authenticating utility v%s"
	  x-pgp-sig-version-number)
  "Version string for this version of x-pgp-sig.")
(defconst x-pgp-sig-codename
;;  "Paperback Writer"
  "Rain"
;;  "Sun King"
;;  "A Taste Of Honey"
;;  "When I'm Sixty-Four"
;;  "Your Mother Should Know"
;;  "All You Need Is Love"
;;  "Blue Jay Way"
;;  "Cry Baby Cry"
;;  "Do You Want To Know A Secret"
;;  "Eight Days A Week"
;;  "For You Blue"
;;  "Glass Onion"
;;  "Helter Skelter"
;;  "I've Got A Feeling"
;;  "Hey Jude" ; "Jude Hey"
;;  "Kome Together"
;;  "Long Tall Sally"
;;  "Mother Nature's Son"
;;  "Nowhere Man"
;;  "Old Brown Shoe"
;;  "Piggies"
;;  "Revolution"
;;  "Sgt.Pepper's"
;;  "Twist & Shout"
;;  "With A Little Help From My Friends"
;;  "You Never Give Me Your Money"
;;  "Dizzy Miss Lizzy"
  "Codename of this version of x-pgp-sig (a string).")

(if (fboundp 'eval-when-compile)
    (eval-when-compile (require 'cl))
  (require 'cl))

(unless (and (condition-case ()
		 (require 'custom)
	       (file-error nil))
	     (fboundp 'defgroup)
	     (fboundp 'defcustom))
  (defmacro defgroup (&rest args))
  (defmacro defcustom (symbol value &optional doc &rest args)
    (let ((doc (concat "*" (or doc ""))))
      (` (defvar (, symbol) (, value) (, doc)))))
  )

;; Silence the byte compiler.
(and
 (fboundp 'eval-when-compile)
 (eval-when-compile
   (save-excursion
     (beginning-of-defun)
     (eval-region (point-min) (point)))
   (let (case-fold-search)
     (if (string-match "alpha\\|beta" x-pgp-sig-version-number)
	 (mapcar
	  (function
	   (lambda (symbol)
	     (unless (boundp symbol)
	       (make-local-variable symbol)
	       (eval (list 'setq symbol nil)))))
	  '(:group
	    :prefix :type
	    cmail-current-folder
	    gnus-Article-buffer gnus-article-buffer gnus-article-display-hook
	    gnus-original-article-buffer
	    gnus-current-article gnus-local-domain gnus-newsgroup-name
	    gnus-show-mime gnus-summary-buffer
	    mail-host-address
	    mc-pgp-always-fetch
	    mew-cs-mime-trans mew-header-separator mew-summary-message-regex
	    mh-folder-filename
	    mime-echo-buffer-name
	    mime-editor/pgp-processing
	    mime-edit-pgp-processing
	    mime-preview-original-major-mode
	    mime-raw-buffer
	    mime-view-original-major-mode
	    mime::preview/original-major-mode
	    to-ascii-display to-ascii-fileio to-ascii-process
	    to-kanji-display to-kanji-fileio to-kanji-process
	    vm-mail-buffer vm-message-pointer
	    x-pgp-sig-start-position
	    ))
       (make-local-variable 'byte-compile-warnings)
       (setq byte-compile-warnings nil)
       ))))

(defgroup x-pgp-sig nil
  "Yet another PGP authenticating utility."
  :prefix "x-pgp-sig-"
  :group 'applications)

(defcustom x-pgp-sig-from
  (or
   (and (boundp 'user-mail-address)
	(stringp user-mail-address)
	user-mail-address)
   (and (boundp 'mail-host-address)
	(stringp mail-host-address)
	(concat (user-login-name) "@" mail-host-address))
   (and (boundp 'gnus-local-domain)
	(stringp gnus-local-domain)
	(concat (user-login-name) "@" gnus-local-domain))
   (concat (user-login-name) "@" (system-name)))
  "Mail address of the signer."
  :group 'x-pgp-sig
  :type 'string)

(defcustom x-pgp-sig-default-signer
  (format "%s <%s>" (user-full-name) x-pgp-sig-from)
  "Default PGP user ID."
  :group 'x-pgp-sig
  :type 'string)

(defcustom x-pgp-sig-default-signer-list nil
  "List of PGP user IDs for PGP2 and PGP5."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-pgp-v5-p nil
  "Non-nil means PGP version 5 is used."
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-pgp-program "pgp"
  "Name of PGP executable."
  :group 'x-pgp-sig
  :type 'string)

(defcustom x-pgp-sig-process-asynchronous-p t
  "Don't use start-process() if it is nil."
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-shell-program "/bin/sh"
  "File name to load inferior shells from."
  :group 'x-pgp-sig
  :type 'file)
(defcustom x-pgp-sig-shell-program-arg "-c"
  "Arg used to have the shell execute its command line argument."
  :group 'x-pgp-sig
  :type 'string)

(defcustom x-pgp-sig-pipe-command "/bin/cat"
  "File name to be used for pipe command."
  :group 'x-pgp-sig
  :type 'file)

(defcustom x-pgp-sig-toMime-program "toMime"
  "Name of toMime executable."
  :group 'x-pgp-sig
  :type 'string)
(defcustom x-pgp-sig-toMime-options nil
  "List of options used to have toMime its command line argument."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-mew-encode-message-header-function
  'x-pgp-sig-mew-encode-message-header-with-toMime
  "Function to call to encode the message of Mew. Legal values include
`x-pgp-sig-mew-encode-message-header-with-toMime' (the default),
`x-pgp-sig-mew-encode-message-header-with-semi' and
`x-pgp-sig-mew-encode-message-header-with-tm'."
  :group 'x-pgp-sig
  :type '(radio (function-item x-pgp-sig-mew-encode-message-header-with-toMime)
		(function-item x-pgp-sig-mew-encode-message-header-with-semi)
		(function-item x-pgp-sig-mew-encode-message-header-with-tm)
		function))

(defcustom x-pgp-sig-process-coding-system-for-output nil
  "Coding system for PGP process (output from Mule/XEmacs)."
  :group 'x-pgp-sig
  :type 'symbol)
(defcustom x-pgp-sig-process-kanji-code 2
  "Kanji code for PGP process (NEmacs).
0:No-conversion might not work well."
  :group 'x-pgp-sig
  :type 'integer)

(defcustom x-pgp-sig-pgppath
  (or (let ((pp (getenv "PGPPATH")))
	(when pp
	  (setq pp (expand-file-name
		    (if (string-match "/+$" pp)
			(substring pp 0 (match-beginning 0))
		      pp)))
	  (and (file-directory-p pp)
	       pp)))
      (expand-file-name "~/.pgp"))
  "Environment variable PGPPATH."
  :group 'x-pgp-sig
  :type 'directory)

(defcustom x-pgp-sig-pubring
  (or (let ((pr (getenv "PUBRING"))
	    pr2)
	(when pr
	  (setq pr (expand-file-name
		    (if (string-match "/+$" pr)
			(substring pr 0 (match-beginning 0))
		      pr)
		    x-pgp-sig-pgppath))
	  (if (and (string-match "\\.pkr$" pr)
		   (setq pr2
			 (concat (substring pr 0 (match-beginning 0)) ".pgp"))
		   (file-exists-p pr2)
		   (not (file-directory-p pr2)))
	      pr2
	    (and (file-exists-p pr)
		 (not (file-directory-p pr))
		 pr))))
      (expand-file-name "pubring.pgp" x-pgp-sig-pgppath))
  "Environment variable PUBRING."
  :group 'x-pgp-sig
  :type 'file)

(defcustom x-pgp-sig-pubring-v5
  (or (let ((pr (getenv "PUBRING"))
	    pr5)
	(when pr
	  (setq pr (expand-file-name
		    (if (string-match "/+$" pr)
			(substring pr 0 (match-beginning 0))
		      pr)
		    x-pgp-sig-pgppath))
	  (if (and (string-match "\\.pgp$" pr)
		   (setq pr5
			 (concat (substring pr 0 (match-beginning 0)) ".pkr"))
		   (file-exists-p pr5)
		   (not (file-directory-p pr5)))
	      pr5
	    (and (file-exists-p pr)
		 (not (file-directory-p pr))
		 pr))))
      (expand-file-name "pubring.pkr" x-pgp-sig-pgppath))
  "Environment variable PUBRING for PGP v5."
  :group 'x-pgp-sig
  :type 'file)

(defcustom x-pgp-sig-secring
  (or (let ((sr (getenv "SECRING"))
	    sr2)
	(when sr
	  (setq sr (expand-file-name
		    (if (string-match "/+$" sr)
			(substring sr 0 (match-beginning 0))
		      sr)
		    x-pgp-sig-pgppath))
	  (if (and (string-match "\\.skr$" sr)
		   (setq sr2
			 (concat (substring sr 0 (match-beginning 0)) ".pgp"))
		   (file-exists-p sr2)
		   (not (file-directory-p sr2)))
	      sr2
	    (and (file-exists-p sr)
		 (not (file-directory-p sr))
		 sr))))
      (expand-file-name "secring.pgp" x-pgp-sig-pgppath))
  "Environment variable SECRING."
  :group 'x-pgp-sig
  :type 'file)

(defcustom x-pgp-sig-secring-v5
  (or (let ((sr (getenv "SECRING"))
	    sr5)
	(when sr
	  (setq sr (expand-file-name
		    (if (string-match "/+$" sr)
			(substring sr 0 (match-beginning 0))
		      sr)
		    x-pgp-sig-pgppath))
	  (if (and (string-match "\\.pgp$" sr)
		   (setq sr5
			 (concat (substring sr 0 (match-beginning 0)) ".skr"))
		   (file-exists-p sr5)
		   (not (file-directory-p sr5)))
	      sr5
	    (and (file-exists-p sr)
		 (not (file-directory-p sr))
		 sr))))
      (expand-file-name "secring.skr" x-pgp-sig-pgppath))
  "Environment variable SECRING for PGP v5."
  :group 'x-pgp-sig
  :type 'file)

(defcustom x-pgp-sig-delete-last-empty-line-while-verifying t
  "If non-nil, delete trailing empty lines of message while verifying."
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-approved-field-body x-pgp-sig-from
  "Mail address for Approved field."
  :group 'x-pgp-sig
  :type 'string)

(defcustom x-pgp-sig-x-info-field-body nil
  "String for X-Info field body."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-add-version-header nil
  "Non-nil forces 'x-pgp-sig-sign to add `Version' field. :-p"
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-version-field-name "X-PGP-Sig-Version"
  "Name of the `Version' field."
  :group 'x-pgp-sig
  :type 'string)

(defcustom x-pgp-sig-version-field-body
  (format "v%s - \"%s\"
                   (Yet another PGP authenticating utility)"
	  x-pgp-sig-version-number x-pgp-sig-codename)
  "String for `Version' field body."
  :group 'x-pgp-sig
  :type 'string)

(defcustom x-pgp-sig-add-key-info-field nil
  "If non-nil, `Key-Info' field will be added."
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-key-info-field-name-alist
  '(("RSA" . "X-PGP-Key-Info") ("DSS" . "X-PGP5-Key-Info"))
  "Name of the `Key-Info' field."
  :group 'x-pgp-sig
  :type '(repeat (cons :format "%v"
		       (choice (choice-item :tag "RSA" "RSA")
			       (choice-item :tag "DSS" "DSS"))
		       (string :tag "Field name"))))

(defcustom x-pgp-sig-add-fingerprint-field nil
  "If non-nil, `Fingerprint' field will be added."
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-fingerprint-field-name-alist
  '(("RSA" . "X-PGP-Fingerprint") ("DSS" . "X-PGP-Fingerprint20"))
  "Name of the `Fingerprint' field."
  :group 'x-pgp-sig
  :type '(repeat (cons :format "%v"
		       (choice (choice-item :tag "RSA" "RSA")
			       (choice-item :tag "DSS" "DSS"))
		       (string :tag "Field name"))))

(defcustom x-pgp-sig-sign-fields
  '("Subject" "Control" "Message-ID" "Date" "From" "Sender" "User-Agent"
    "X-Mailer" "X-Newsreader" "X-Face" "X-Face-Type")
  "List of fields to be signed."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-adjust-sign-fields-alist nil
  "Adjust `x-pgp-sig-sign-fields' if matched text exists in narrowed header.

\((REGEXP1 (sub \"FIELD1\" \"FIELD2\" ...)
\          (add \"FIELD3\" \"FIELD4\" ...))
\ (REGEXP2 (sub \"FIELD5\" \"FIELD6\" ...)
\          (add \"FIELD7\" \"FIELD8\" ...)))

REGEXP can also be a function."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-last-fields
  '(
    "Content-Type" "Content-Transfer-Encoding"
    "X-Face" "X-Face-Version" "X-Face-Type"
    )
  "PGP signed field will be added above these fields."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-always-sign nil
  "If non-nil, always add X-PGP signature."
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-inhibit-repeated-sign t
  "If non-nil, inhibit sign if the signature has been already exists."
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-enter-keyid-first nil
  "If non-nil, x-pgp-sig-set-signer(-maybe) requires KeyID first."
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-add-signature-judging-function
  'x-pgp-sig-add-signature-y-or-n-p
  "Function whether to add X-PGP signature."
  :group 'x-pgp-sig
  :type 'function)

(defcustom x-pgp-sig-load-hook nil
  "Hook to be run after the x-pgp-sig package has been loaded."
  :group 'x-pgp-sig
  :type 'hook)

(defcustom x-pgp-sig-reload-hook nil
  "Hook to be run after the x-pgp-sig package has been *re*loaded."
  :group 'x-pgp-sig
  :type 'hook)

(defcustom x-pgp-sig-sign-hook nil
  "Hook run after adding X-PGP-Sig field."
  :group 'x-pgp-sig
  :type 'hook)

(defcustom x-pgp-sig-prepare-sign-hook nil
  "Hook run before signing pgp process."
  :group 'x-pgp-sig
  :type 'hook)

(defcustom x-pgp-sig-post-sign-hook nil
  "Hook run after signing pgp process."
  :group 'x-pgp-sig
  :type 'hook)

(defcustom x-pgp-sig-good-signature-regexps
  '(
    "^Good signature from user.*$"	;; v2.6
    "^Good signature made .* by key:$"	;; v5
    )
  "Regular expression matching a PGP signature validation message."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-bad-signature-regexps
  (list
   (concat "\\("			;; v2.6
	   "^Bad signature from user.*$"
	   "\\|"
	   "Unsupported packet format - you need a newer version of PGP"
	   "\\)")
   "^BAD signature made .* by key:$"	;; v5
   )
  "Regular expression not matching a PGP signature validation message."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-expected-regexps
  '(
    "Key ID \\(\\S +\\) not found"			;; v2.6
    "^Signature by unknown keyid: 0x\\(\\S +\\)$"	;; v5
    )
  "Regular expression that no keys found."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-pgp-v5-result-regexp
  (format "%s\\|%s\\|%s"
	  (cadr x-pgp-sig-good-signature-regexps)
	  (cadr x-pgp-sig-bad-signature-regexps)
	  (cadr x-pgp-sig-expected-regexps))
  "Regexp search pattern for the result of PGP v5."
  :group 'x-pgp-sig
  :type 'regexp)

(defcustom x-pgp-sig-good-passphrase-regexp-for-decryption
  '(
    "\\.+Pass phrase appears good\\. \\."			;; V2.6
    "^Message is encrypted\\.\nOpening file .+ type text\\.\n"	;; v5
    )
  "Regular expression matching PGP decryption \"good\" message."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-verify-method-alist
  '(
    (gnus-summary-mode
     . (x-pgp-sig-gnus-get-article
	x-pgp-sig-find-gnus-article-buffer
	x-pgp-sig-gnus-unwind-after-verify))
    (gnus-Subject-mode
     . (x-pgp-sig-gnus-get-article
	x-pgp-sig-find-gnus-article-buffer
	x-pgp-sig-gnus-unwind-after-verify))
    (mh-folder-mode
     . (x-pgp-sig-mh-get-article
	x-pgp-sig-msg-to-verify-buffer))
    (mew-summary-mode
     . (x-pgp-sig-mew-get-article
	x-pgp-sig-msg-to-verify-buffer))
    (mew-virtual-mode
     . (x-pgp-sig-mew-get-article
	x-pgp-sig-msg-to-verify-buffer))
    (cmail-summary-mode
     . (x-pgp-sig-cmail-get-article
	x-pgp-sig-msg-to-verify-buffer))
    (vm-mode
     . (x-pgp-sig-vm-get-article
	x-pgp-sig-msg-to-verify-buffer))
    (vm-presentation-mode
     . (x-pgp-sig-vm-get-article
	x-pgp-sig-msg-to-verify-buffer))
    (vm-summary-mode
     . (x-pgp-sig-vm-get-article
	x-pgp-sig-msg-to-verify-buffer))
    (wl-summary-mode
     . (x-pgp-sig-wl-get-article
	x-pgp-sig-msg-to-verify-buffer))
    )
  "Alist of the MAJOR-MODE and the list of PREPARE_FUNCTION,
BUFFER_or_FUNCTION and POST_FUNCTION.  POST_FUNCTION ban be omitted."
  :group 'x-pgp-sig
  :type 'sexp)

(defcustom x-pgp-sig-use-mime-pgp-fetch-key nil
  "Use `mime-pgp-fetch-key' for fetching a key if it is non-nil."
  :group 'x-pgp-sig
  :type 'boolean)

(defcustom x-pgp-sig-debug nil
  "Turn on the debug mode if it is non-nil."
  :group 'x-pgp-sig
  :type 'boolean)


;;; Macros
(defmacro x-pgp-sig-default-signer ()
  '(let (signer)
     (or (and (eq 2 (length x-pgp-sig-default-signer-list))
	      (setq signer (if x-pgp-sig-pgp-v5-p
			       (cadr x-pgp-sig-default-signer-list)
			     (car x-pgp-sig-default-signer-list)))
	      (or (stringp signer) (consp signer))
	      signer)
	 x-pgp-sig-default-signer)))

(defmacro x-pgp-sig-read-char-exclusive ()
  (cond ((featurep 'xemacs)
	 '(let ((table (quote ((backspace . ?\C-h) (delete . ?\C-?)
			       (left . ?\C-h))))
		event key)
	    (while (not
		    (and
		     (key-press-event-p (setq event (next-command-event)))
		     (setq key (or (event-to-character event)
				   (cdr (assq (event-key event) table)))))))
	    key))
	((fboundp 'read-char-exclusive)
	 '(read-char-exclusive))
	(t
	 '(read-char))))


;;; Internal variables
(defvar x-pgp-sig-no-conversion
  (cond (; XEmacs
	 (featurep 'xemacs) (coding-system-name (get-coding-system nil)))
	(; Mule 2.3
	 (boundp 'MULE) '*noconv*)
	(; Emacs 20.x, MULE 3.0
	 (fboundp 'check-coding-system)
	 (or (condition-case ()
		 (check-coding-system 'binary)
	       (coding-system-error nil))
	     (condition-case ()
		 (check-coding-system 'no-conversion)
	       (coding-system-error nil))
	     ))))

(defconst x-pgp-sig-field-name "X-PGP-Sig"
  "Field name of PGP signature.")

(defconst x-pgp-sig-signed-headers-field-name "X-Signed-Headers"
  "Field name of PGP signed fields.")

(defconst x-pgp-sig-msg-begin-line "-----BEGIN PGP MESSAGE-----"
  "Text for start of PGP message delimiter.")

(defconst x-pgp-sig-msg-end-line "-----END PGP MESSAGE-----"
  "Text for end of PGP message delimiter.")

(defconst x-pgp-sig-signed-msg-begin-line "-----BEGIN PGP SIGNED MESSAGE-----"
  "Text for start of PGP signed messages.")

(defconst x-pgp-sig-signed-begin-line "-----BEGIN PGP SIGNATURE-----"
  "Text for start of PGP signature.")

(defconst x-pgp-sig-signed-end-line "-----END PGP SIGNATURE-----"
  "Text for end of PGP signature.")

(defvar x-pgp-sig-working-buffer " *x-pgp-sig-working*")

(defvar x-pgp-sig-msg-to-verify-buffer nil)

(defvar x-pgp-sig-enable-sign)
(setq x-pgp-sig-enable-sign x-pgp-sig-always-sign)

(or (stringp x-pgp-sig-default-signer)
    (consp x-pgp-sig-default-signer)
    (setq x-pgp-sig-default-signer
	  (format "%s <%s>" (user-full-name) x-pgp-sig-from)))

(and (boundp 'x-pgp-sig-pgp-program)
     (stringp x-pgp-sig-pgp-program)
     (string-match "pgp[eksv]$" x-pgp-sig-pgp-program)
     (setq x-pgp-sig-pgp-program
	   (substring x-pgp-sig-pgp-program 0 (1- (match-end 0)))))

(defvar x-pgp-sig-current-signer)
(setq x-pgp-sig-current-signer (x-pgp-sig-default-signer))

(defvar x-pgp-sig-passphrase-alist)
(setq x-pgp-sig-passphrase-alist nil)

(defvar x-pgp-sig-secret-key-alist)
(setq x-pgp-sig-secret-key-alist nil)

(defvar x-pgp-sig-secret-key-data-list)
(setq x-pgp-sig-secret-key-data-list nil)

(defconst x-pgp-sig-idea-passphrase
  (let ((i 254)
	p r)
    (while (> i 0)
      (decf i)
      (setq r (abs (random))
	    p
	    (concat
	     p
	     (char-to-string
	      (+ (/ (* 95 (- r (* (/ r 100) 100))) 100) 32)))))
    p))

(defvar x-pgp-sig-reserve-sign-buffer nil)
(defvar x-pgp-sig-reserve-sign-flag nil)

(defvar x-pgp-sig-debug-buffer-num 1)
(defvar x-pgp-sig-debug-buffer-list nil)
(defvar x-pgp-sig-debug-buffer-max 16)
(defvar x-pgp-sig-debug-log-file nil)


;;; Emulations for each Emacs variants.

(and (boundp 'to-kanji-display)
     (setq to-ascii-display ?B
	   to-ascii-fileio ?B
	   to-ascii-process ?B
	   to-kanji-display ?B
	   to-kanji-fileio ?B
	   to-kanji-process ?B))

(mapcar
 (function
  (lambda (elt)
    (let ((func (car elt)))
      (or (fboundp func)
	  (fset func (symbol-function (cdr elt)))))))
 '(
   (buffer-substring-no-properties . buffer-substring)
   (defsubst . defun)
   ))

(or (fboundp 'delete)
    ; Imported from emu-18.el
    (defun delete (elt list)
      "Delete by side effect any occurrences of ELT as a member of LIST.
The modified LIST is returned.  Comparison is done with `equal'.
If the first member of LIST is ELT, deleting it is not a side effect;
it is simply using a different list.
Therefore, write `(setq foo (delete element foo))'
to be sure of changing the value of `foo'."
      (if (equal elt (car list))
	  (cdr list)
	(let ((rest list)
	      (rrest (cdr list))
	      )
	  (while (and rrest (not (equal elt (car rrest))))
	    (setq rest rrest
		  rrest (cdr rrest))
	    )
	  (rplacd rest (cdr rrest))
	  list))))

(or (fboundp 'match-string)
    (defun match-string (num &optional string)
      "Return string of text matched by last search.
NUM specifies which parenthesized expression in the last regexp.
Value is nil if NUMth pair didn't match, or there were less than NUM pairs.
Zero means the entire text matched by the whole regexp or whole string.
STRING should be given if the last search was by `string-match' on STRING."
      (if (match-beginning num)
	  (if string
	      (substring string (match-beginning num) (match-end num))
	    (buffer-substring (match-beginning num) (match-end num))))))

(or (fboundp 'member)
    (defun member (elt list)
      "Return non-nil if ELT is an element of LIST. Comparison done with EQUAL.
The value is actually the tail of LIST whose car is ELT."
      (while (and list (not (equal elt (car list))))
	(setq list (cdr list)))
      list))

(or (fboundp 'put-alist)
    ; Imported from tl-list.el.
    (defun put-alist (item value alist)
      "Modify ALIST to set VALUE to ITEM.
If there is a pair whose car is ITEM, replace its cdr by VALUE.
If there is not such pair, create new pair (ITEM . VALUE) and
return new alist whose car is the new pair and cdr is ALIST."
      (let ((pair (assoc item alist)))
	(if pair
	    (progn
	      (setcdr pair value)
	      alist)
	  (cons (cons item value) alist)
	  ))))

(if (and (condition-case ()
	     (require 'env)
	   (error nil))
	 (fboundp 'setenv))
    (fset 'x-pgp-sig-setenv (symbol-function 'setenv))
  (defun x-pgp-sig-setenv (variable &optional value)
    (let ((re (concat "^" (regexp-quote variable) "="))
	  (envs process-environment)
	  env
	  case-fold-search)
      (while (setq env (pop envs))
	(and (string-match re env)
	     (setq process-environment (delete env process-environment))))
      (and value
	   (setq process-environment
		 (cons (concat variable "=" value) process-environment)))))
  )

(or (fboundp 'rassoc)
    (defun rassoc (key list)
      "Return non-nil if KEY is `equal' to the cdr of an element of LIST.
The value is actually the element of LIST whose cdr equals KEY."
      (let ((elt (assoc key (mapcar
			     (function (lambda (elt)
					 (cons (cdr elt) (car elt))))
			     list))))
	(and elt (cons (cdr elt) (car elt)))))
    )


;;; Function for DOSish OSes (OS/2,M$Windoze-95|NT).

(defsubst x-pgp-sig-delcr ()
  (goto-char (point-min))
  (while (search-forward "\r\n" nil t)
    (replace-match "\n")))


;;; Functions.

(defsubst x-pgp-sig-buffer-live-p (buffer)
  "Say whether BUFFER is alive or not."
  (and buffer
       (get-buffer buffer)
       (buffer-name (get-buffer buffer))))

(defun x-pgp-sig-read-noecho (prompt &optional stars)
  "Read a single line of text from user without echoing, and return it."
  (let ((ans "")
	(c 0)
	(echo-keystrokes 0)
	(cursor-in-echo-area t)
	(log-message-max-size 0)
	message-log-max done msg truncate)
    (while (not done)
      (if (or (not stars) (string-equal "" ans))
	  (setq msg prompt)
	(setq msg (concat prompt (make-string (length ans) ?*)))
	(setq truncate
	      (1+ (- (length msg) (window-width (minibuffer-window)))))
	(and (> truncate 0)
	     (setq msg (concat "$" (substring msg (1+ truncate))))))
      (message msg)
      (setq c (x-pgp-sig-read-char-exclusive))
      (cond ((eq ?\C-g c)
	     (setq quit-flag t
		   done t))
	    ((memq c '(?\r ?\n ?\e))
	     (setq done t))
	    ((eq ?\C-u c)
	     (setq ans ""))
	    ((and (/= ?\b c) (/= ?\177 c))
	     (setq ans (concat ans (char-to-string c))))
	    ((> (length ans) 0)
	     (setq ans (substring ans 0 -1)))))
    (if quit-flag
	(prog1
	    (setq quit-flag nil)
	  (message "Quit")
	  (beep t))
      (message "")
      ans)))

(defsubst x-pgp-sig-insert-file-contents-no-conversion (filename &rest args)
  (let (; NEmacs
	kanji-flag
	; Mule
	(input-coding-system x-pgp-sig-no-conversion)
	; XEmacs (>=20.1b6)
	(coding-system-for-read x-pgp-sig-no-conversion))
    (apply 'insert-file-contents filename args)))

(defun x-pgp-sig-error (&rest args)
  "(x-pgp-sig-error FORMAT ARG ARG ... BUFFER)."
  (let* ((len (length args))
	 (buffer (nth (1- len) args)))
    (cond ((zerop len) (error ""))
	  ((and (> len 1) (x-pgp-sig-buffer-live-p buffer))
	   (and x-pgp-sig-debug
		(let ((errbuf (generate-new-buffer
			       (concat "*x-pgp-sig-error-("
				       (buffer-name buffer)
				       ")*")))
		      content point)
		  (save-excursion
		    (set-buffer buffer)
		    (setq content (buffer-substring-no-properties
				   (point-min)
				   (point-max))
			  point (point))
		    (set-buffer errbuf)
		    (insert content)
		    (goto-char point)
		    (set-buffer-modified-p nil)
		    (setq buffer-read-only t)
		    (cond ((fboundp 'set-buffer-file-coding-system)
			   (set-buffer-file-coding-system
			    x-pgp-sig-no-conversion))
			  ((fboundp 'set-file-coding-system)
			   (set-file-coding-system x-pgp-sig-no-conversion))
			  ((fboundp 'set-kanji-fileio-code)
			   (set-kanji-fileio-code nil))))
		  (display-buffer errbuf)))
	   (apply 'error (reverse (cdr (reverse args)))))
	  (t (apply 'error args)))))

(defsubst x-pgp-sig-narrow-to-header (&optional include-separator)
  (widen)
  (goto-char (point-min))
  (narrow-to-region
   (point)
   (or (and (re-search-forward
	     (format "^\\(%s\\)?\n"
		     (regexp-quote mail-header-separator))
	     nil t)
	    (if include-separator
		(match-end 0)
	      (match-beginning 0)))
       (point-max)))
  (goto-char (point-min)))

(defun x-pgp-sig-debug-make-history (&optional out additional-buffer)
  (let ((cur (current-buffer))
	(buffer (generate-new-buffer
		 (format "**X-PGP-Sig-DEBUG-%04d-/%s"
			 x-pgp-sig-debug-buffer-num
			 (if out "O" "I")))))
    (save-excursion
      (set-buffer buffer)
      (when additional-buffer
	(insert-buffer-substring additional-buffer)
	(kill-buffer additional-buffer)
	(insert-char ?- 70) (insert "\n"))
      (insert-buffer-substring cur))
    (setq x-pgp-sig-debug-buffer-list
	  (append x-pgp-sig-debug-buffer-list (list buffer))))
  (when out
    (incf x-pgp-sig-debug-buffer-num)
    (while (> (length x-pgp-sig-debug-buffer-list)
	      x-pgp-sig-debug-buffer-max)
      (kill-buffer (pop x-pgp-sig-debug-buffer-list)))))

(defun x-pgp-sig-debug-list-buffers ()
  (interactive)
  (list-buffers)
  (save-excursion
    (set-buffer (get-buffer "*Buffer List*"))
    (let (buffer-read-only pt)
      (goto-char (point-min))
      (forward-line 2)
      (while (not (eobp))
	(setq pt (point))
	(if (looking-at "^..  \\*\\*X-PGP-Sig-DEBUG-[0-9]+-/[IO]")
	    (forward-line 1)
	  (forward-line 1)
	  (delete-region pt (point))
	  ))
      (goto-char (point-min)))))

(defun x-pgp-sig-process-buffer (args &optional passphrase buffer tail)
  (let ((obuf (or (and (x-pgp-sig-buffer-live-p buffer) buffer)
		  (current-buffer)))
	(pgpbuf (get-buffer-create " *PGP*"))
	case-fold-search str process err)
    (mapcar
     (function (lambda (a) (or (member a args) (setq args (cons a args)))))
     (list "+verbose=1"
	   (if x-pgp-sig-pgp-v5-p "+language=us" "+language=en")
	   "+batchmode"))
    (save-excursion
      (set-buffer obuf)
      (setq str (buffer-string))
      (set-buffer pgpbuf)
      (setq buffer-read-only nil)
      (erase-buffer)
      (setq case-fold-search nil)
      (when (or x-pgp-sig-debug (not x-pgp-sig-process-asynchronous-p))
	(and passphrase (insert passphrase "\n"))
	(insert str))
      (when x-pgp-sig-debug
	(x-pgp-sig-debug-make-history)
	(and x-pgp-sig-process-asynchronous-p (erase-buffer)))
      (let* (; NEmacs
	     (default-kanji-process-code
	       (or x-pgp-sig-process-kanji-code 0))
	     program-kanji-code-alist
	     ;
	     (outcode (or x-pgp-sig-process-coding-system-for-output
			  x-pgp-sig-no-conversion))
	     ; Mule
	     (input-coding-system x-pgp-sig-no-conversion)
	     (output-coding-system x-pgp-sig-no-conversion)
	     (default-process-coding-system
	       (cons x-pgp-sig-no-conversion outcode))
	     call-process-hook
	     ; XEmacs (<= 20.1b6)
	     (file-coding-system outcode)
	     ; XEmacs (>= 20.1b7)
	     (buffer-file-coding-system outcode)
	     ; XEmacs (>= 20.2b1)
	     (coding-system-for-read x-pgp-sig-no-conversion)
	     (coding-system-for-write outcode)
	     (process-environment process-environment)
	     process-connection-type
	     (lines 16)
	     ; for debugging
	     debug-buffer debug-mark
	     )
	(x-pgp-sig-setenv "PGPPATH" x-pgp-sig-pgppath)
	(x-pgp-sig-setenv "PGPPASSFD" (and passphrase "0"))
	(if x-pgp-sig-process-asynchronous-p
	    (progn
	      (and x-pgp-sig-debug
		   (setq debug-buffer (get-buffer-create
				       " *X-PGP-Sig-DEBUG-PROCESS*"))
		   (save-excursion
		     (set-buffer debug-buffer)
		     (setq buffer-read-only nil)
		     (erase-buffer)
		     (insert (format
			      "******* X-PGP-Sig-DEBUG-PROCESS-%04d *******\n"
			      x-pgp-sig-debug-buffer-num))
		     (and x-pgp-sig-debug-log-file
			  (write-region
			   (point-min) (setq debug-mark (point))
			   x-pgp-sig-debug-log-file 'append 'silence))))
	      (setq process (apply 'start-process "*PGP*" pgpbuf
				   (concat x-pgp-sig-pgp-program tail) args))
	      (and x-pgp-sig-debug
		   (save-excursion
		     (set-buffer debug-buffer)
		     (insert "("
			     (prin1-to-string (process-status process))
			     ")\n")
		     (and x-pgp-sig-debug-log-file
			  (write-region
			   debug-mark (setq debug-mark (point))
			   x-pgp-sig-debug-log-file 'append 'silence))))
	      (and (fboundp 'set-process-coding-system)
		   (set-process-coding-system
		    process x-pgp-sig-no-conversion outcode))
	      (process-kill-without-query process)
	      (when passphrase
		(process-send-string process passphrase)
		(process-send-string process "\n"))
	      (or (and tail (string-match "[sv]" tail))
		  (accept-process-output process 1))
	      (and x-pgp-sig-debug
		   passphrase
		   (save-excursion
		     (set-buffer debug-buffer)
		     (insert
		      (format "%-8s>Turtle Power!\\n\n"
			      (concat
			       "("
			       (prin1-to-string (process-status process))
			       ")")))
		     (and x-pgp-sig-debug-log-file
			  (write-region
			   debug-mark (setq debug-mark (point))
			   x-pgp-sig-debug-log-file 'append 'silence))))
	      (set-buffer obuf)
	      (goto-char (point-min))
	      (while (not (eobp))
		(setq str (buffer-substring
			   (point) (progn (end-of-line) (point))))
		(process-send-string process str)
		(unless (eobp)
		  (process-send-string process "\n")
		  (decf lines)
		  (and (or (and (zerop lines)
				(setq lines 16)
				tail
				(string-equal "s" tail))
			   (not (and tail (string-match "[sv]" tail))))
		       (accept-process-output process 1)))
		(and x-pgp-sig-debug
		     (save-excursion
		       (or (eobp) (setq str (concat str "\\n")))
		       (set-buffer debug-buffer)
		       (insert
			(format "%-8s>%s\n"
				(concat
				 "("
				 (prin1-to-string (process-status process))
				 ")")
				(if (string-match "\\\\n$" str)
				    str
				  "Turtle Power!")))
		       (and x-pgp-sig-debug-log-file
			    (write-region
			     debug-mark (setq debug-mark (point))
			     x-pgp-sig-debug-log-file 'append 'silence))))
		(forward-line 1))
	      (set-buffer pgpbuf)
	      (process-send-eof process)
	      (and x-pgp-sig-debug
		   (save-excursion
		     (set-buffer debug-buffer)
		     (insert
		      (format "%-8s>EOF\n"
			      (concat
			       "("
			       (prin1-to-string (process-status process))
			       ")")))
		     (and x-pgp-sig-debug-log-file
			  (write-region
			   debug-mark (setq debug-mark (point))
			   x-pgp-sig-debug-log-file 'append 'silence))))
	      (while (eq 'run (process-status process))
		(and x-pgp-sig-debug
		     (save-excursion
		       (set-buffer debug-buffer)
		       (insert "("
			       (prin1-to-string (process-status process))
			       ")\n")
		       (and x-pgp-sig-debug-log-file
			    (write-region
			     debug-mark (setq debug-mark (point))
			     x-pgp-sig-debug-log-file 'append 'silence))))
		(accept-process-output process 1))
	      (and x-pgp-sig-debug
		   (save-excursion
		     (set-buffer debug-buffer)
		     (insert "("
			     (prin1-to-string (process-status process))
			     ")\n")
		     (and x-pgp-sig-debug-log-file
			  (write-region
			   debug-mark (setq debug-mark (point))
			   x-pgp-sig-debug-log-file 'append 'silence))))
	      )
	  (if (and tail (string-equal "s" tail))
	      (call-process-region
	       (point-min) (point-max)
	       x-pgp-sig-shell-program
	       t pgpbuf nil
	       x-pgp-sig-shell-program-arg
	       (concat x-pgp-sig-pipe-command
		       "|" x-pgp-sig-pgp-program tail " '"
		       (mapconcat 'identity args "' '") "'"))
	    (apply 'call-process-region (point-min) (point-max)
		   (concat x-pgp-sig-pgp-program tail) t pgpbuf nil args)
	    ))
	(and x-pgp-sig-debug (x-pgp-sig-debug-make-history 'out debug-buffer))
	)
      ; for 'OS/2 & 'windoze-98 convert CRLF to LF
      (x-pgp-sig-delcr)
      ;
      (goto-char (point-min))
      (setq err
	    (and (re-search-forward "\n\n+" nil t)
		 (if x-pgp-sig-pgp-v5-p
		     (or (and (looking-at "^A private key is required")
			      "Passphrase incorrect")
			 (and (looking-at "^Cannot find a private key")
			      (match-string 0)))
		   (and (looking-at "^\C-gError:[\t ]*\\(.*\\)$")
			(match-string 1)))))
      (setq str (buffer-string))
      (set-buffer obuf)
      (setq buffer-read-only nil)
      (erase-buffer)
      (insert str))
    (if err
	(list (format "\"%s\"" err) pgpbuf)
      (kill-buffer pgpbuf)
      nil)))

(defun x-pgp-sig-idea-encrypt-string (str)
  "Encrypt string using IDEA method."
  (and (stringp str)
       (let ((buffer (get-buffer-create x-pgp-sig-working-buffer))
	     beg result)
	 (unwind-protect
	     (save-excursion
	       (set-buffer buffer)
	       (setq buffer-read-only nil)
	       (erase-buffer)
	       (insert str)
	       (setq str nil)
	       (setq result
		     (if x-pgp-sig-pgp-v5-p
			 (x-pgp-sig-process-buffer
			  '("-fact") x-pgp-sig-idea-passphrase nil "e")
		       (x-pgp-sig-process-buffer
			'("-fact") x-pgp-sig-idea-passphrase)))
	       (and result (apply 'x-pgp-sig-error "!! %s" result))
	       (goto-char (point-min))
	       (or (re-search-forward
		    (format "%s$" (regexp-quote x-pgp-sig-msg-begin-line))
		    nil t)
		   (x-pgp-sig-error
		    "!! Regexp \"%s\$\" not found"
		    (regexp-quote x-pgp-sig-msg-begin-line)
		    buffer))
	       (or (search-forward "\n\n" nil t)
		   (x-pgp-sig-error "!! String \"\\n\\n\" not found"
				    buffer))
	       (setq beg (point))
	       (or (re-search-forward
		    (format "%s$" (regexp-quote x-pgp-sig-msg-end-line))
		    nil t)
		   (x-pgp-sig-error
		    "!! Regexp \"%s\$\" not found"
		    (regexp-quote x-pgp-sig-msg-end-line)
		    buffer))
	       (setq result (buffer-substring beg (match-beginning 0))))
	   (kill-buffer buffer))
	 result)))

(defun x-pgp-sig-idea-decrypt-string (str)
  "Decrypt string using IDEA method."
  (and (stringp str)
       (let ((buffer (get-buffer-create x-pgp-sig-working-buffer))
	     (re (concat
		  (if x-pgp-sig-pgp-v5-p
		      (cadr x-pgp-sig-good-passphrase-regexp-for-decryption)
		    (car x-pgp-sig-good-passphrase-regexp-for-decryption))
		  (if x-pgp-sig-process-asynchronous-p
		      ""
		    (concat "\\("
			    (regexp-quote x-pgp-sig-idea-passphrase)
			    "\n\\)?"))))
	     result)
	 (unwind-protect
	     (save-excursion
	       (set-buffer buffer)
	       (setq buffer-read-only nil)
	       (erase-buffer)
	       (insert x-pgp-sig-msg-begin-line "\n\n")
	       (and x-pgp-sig-pgp-v5-p (insert "\n"))
	       (insert  str)
	       (setq str nil)
	       (or (eolp) (insert "\n"))
	       (insert x-pgp-sig-msg-end-line "\n")
	       (setq result
		     (if x-pgp-sig-pgp-v5-p
			 (x-pgp-sig-process-buffer
			  '("-f") x-pgp-sig-idea-passphrase nil "v")
		       (x-pgp-sig-process-buffer
			'("-f") x-pgp-sig-idea-passphrase)))
	       (and result (apply 'x-pgp-sig-error "!! %s" result))
	       (goto-char (point-min))
	       (or (re-search-forward re nil t)
		   (x-pgp-sig-error
		    "!! Regexp \"%s\" not found" (regexp-quote re) buffer))
	       (setq result (buffer-substring
			     (point)
			     (if (and (featurep 'xemacs)
				      x-pgp-sig-process-asynchronous-p)
				 (progn
				   (goto-char (point-max))
				   (forward-line -1)
				   (1- (point)))
			       (point-max)))))
	   (kill-buffer buffer))
	 result)))

(defun x-pgp-sig-lookup-keys-5 (&optional pub)
  (let ((case-fold-search t)
	(keyring (expand-file-name
		  (if pub x-pgp-sig-pubring-v5 x-pgp-sig-secring-v5)))
	(re (concat
	     "^\\(" (if pub "pub\\|sec.?" "sec.?") "\\)"	;;1
	     "[\t ]+"
	     "\\([0-9]+\\)"	;;2 bits
	     "[\t ]+0x"
	     "\\([0-9a-f]+\\)"	;;3 keyid
	     "[\t ]+"
	     "\\([0-9/-]+\\)"	;;4 date
	     "[\t ]+[^\n\t ]+[\t ]+"
	     "\\([^\n\t ]+\\)"	;;5 algorithm
	     ))
	(process-environment process-environment)
	process-connection-type
	buffer id data result)
    (or (file-exists-p keyring)
	(error "!! File \"%s\" not found" keyring))
    (unwind-protect
	(save-excursion
	  (setq buffer (set-buffer
			(get-buffer-create x-pgp-sig-working-buffer)))
	  (setq buffer-read-only nil)
	  (erase-buffer)
	  (x-pgp-sig-setenv "PGPPATH" x-pgp-sig-pgppath)
	  (call-process (concat x-pgp-sig-pgp-program "k") nil t nil
			"+verbose=1" "+language=us" "-ll")
	  ; for 'windoze-98
	  (x-pgp-sig-delcr)
	  ;
	  (goto-char (point-min))
	  (when (looking-at "^Type[\t ]+Bits[\t ]+KeyID[\t ]+")
	    (or pub (setq x-pgp-sig-secret-key-data-list nil))
	    (while (re-search-forward re nil t)
	      (setq id (concat "0x" (match-string 3)))
	      (or pub
		  (setq data
			(list id
			      (match-string 2)	;; bits
			      (match-string 4)	;; date
			      (match-string 5)	;; algorithm
			      )))
	      (forward-line 1)
	      (save-restriction
		(narrow-to-region
		 (point)
		 (or (and (re-search-forward "^$" nil t)
			  (match-beginning 0))
		     (point-max)))
		(goto-char (point-min))
		(re-search-forward "=[\t ]+\\(.+\\)$")
		(or pub
		    (setq x-pgp-sig-secret-key-data-list
			  (append x-pgp-sig-secret-key-data-list
				  (list (append data
						(list (match-string 1)))))))
		(while (re-search-forward "^uid[\t ]+\\(.+\\)$" nil t)
		  (setq result
			(append result (list (cons (match-string 1) id)))))
		(goto-char (point-max)))
	      )))
      (kill-buffer buffer))
    result))

(defun x-pgp-sig-lookup-keys-2 (&optional pub)
  (let ((case-fold-search t)
	(keyring (expand-file-name
		  (if pub x-pgp-sig-pubring x-pgp-sig-secring)))
	(re (concat
	     "^" (if pub "pub" "sec") "[\t ]+"
	     "\\([0-9]+\\)"	;;1 bits
	     "/"
	     "\\([0-9a-f]+\\)"	;;2 keyid
	     "[\t ]+"
	     "\\([0-9/-]+\\)"	;;3 date
	     "[\t ]+"
	     "\\(.+\\)$"	;;4 userid
	     ))
	(process-environment process-environment)
	process-connection-type
	buffer id data result)
    (or (file-exists-p keyring)
	(error "!! File \"%s\" not found" keyring))
    (unwind-protect
	(save-excursion
	  (setq buffer (set-buffer
			(get-buffer-create x-pgp-sig-working-buffer)))
	  (setq buffer-read-only nil)
	  (erase-buffer)
	  (x-pgp-sig-setenv "PGPPATH" x-pgp-sig-pgppath)
	  (call-process x-pgp-sig-pgp-program nil t nil
			"+verbose=1" "+language=en" "-kvc" keyring)
	  ; for 'windoze-98
	  (x-pgp-sig-delcr)
	  ;
	  (goto-char (point-min))
	  (when (re-search-forward
		 (format "^Key ring: '%s'\nType.+\n" (regexp-quote keyring))
		 nil t)
	    (or pub (setq x-pgp-sig-secret-key-data-list nil))
	    (while (re-search-forward re nil t)
	      (setq id (concat "0x" (match-string 2))
		    result (append result (list (cons (match-string 4) id))))
	      (or pub
		  (setq data
			(list id
			      (match-string 1)	;; bits
			      (match-string 3)	;; date
			      "RSA"		;; algorithm
			      )))
	      (forward-line 1)
	      (re-search-forward "=[\t ]+\\(.+\\)$")
	      (setq x-pgp-sig-secret-key-data-list
		    (append x-pgp-sig-secret-key-data-list
			    (list (append data
					  (list (match-string 1))))))
	      (forward-line 1)
	      (while (not (or (eobp) (looking-at "[^\n\t ]")))
		(re-search-forward "[^\n\t ].+$")
		(setq result
		      (append result (list (cons (match-string 0) id))))
		(forward-line 1))
	      )))
      (kill-buffer buffer))
    result))

(defun x-pgp-sig-lookup-keys (&optional regexp case pub)
  "Extract data from the keyring."
  (let ((result (if x-pgp-sig-pgp-v5-p
		    (x-pgp-sig-lookup-keys-5 pub)
		  (x-pgp-sig-lookup-keys-2 pub)))
	(case-fold-search case))
    (if (stringp regexp)
	(delq nil (mapcar
		   (function (lambda (elt)
			       (and (string-match regexp (car elt)) elt)))
		   result))
      result)))

(defun x-pgp-sig-search-secret-key (id type)
  (let ((keys (cdr (if (and x-pgp-sig-secret-key-alist
			    (if x-pgp-sig-pgp-v5-p
				(eq 5 (car x-pgp-sig-secret-key-alist))
			      (not (eq 5 (car x-pgp-sig-secret-key-alist)))))
		       x-pgp-sig-secret-key-alist
		     (setq x-pgp-sig-secret-key-alist
			   (cons (if x-pgp-sig-pgp-v5-p 5 2)
				 (x-pgp-sig-lookup-keys))))))
	(case-fold-search t)
	key result)
    (or keys (error "!! No keys in \"%s\"" (if x-pgp-sig-pgp-v5-p
					       x-pgp-sig-secring-v5
					     x-pgp-sig-secring)))
    (cond ((eq 'keyid type)
	   (while (and (not result) (setq key (pop keys)))
	     (and (string-equal (downcase id) (downcase (cdr key)))
		  (setq result key))))
	  ((eq 'uid type)
	   (while (and (not result) (setq key (pop keys)))
	     (and (string-match (regexp-quote id) (car key))
		  (setq result key)))))
    result))

(defun x-pgp-sig-build-message-to-sign ()
  "Extract fields and body to sign in current buffer."
  (let ((case-fold-search t)
	(sign-fields (copy-sequence x-pgp-sig-sign-fields))
	(adjust x-pgp-sig-adjust-sign-fields-alist)
	(fields "")
	adj adj1 sub add existing-fields field re body buffer)
    (save-excursion
      (save-restriction
	(x-pgp-sig-narrow-to-header t)
	(while (setq adj (pop adjust))
	  (setq adj1 (pop adj))
	  (goto-char (point-min))
	  (when (cond ((stringp adj1) (re-search-forward adj1 nil t))
		      ((fboundp adj1) (funcall adj1)))
	    (setq sub (cdr (assq 'sub adj))
		  add (cdr (assq 'add adj)))
	    (while sub
	      (setq sign-fields (delete (car sub) sign-fields))
	      (setq sub (cdr sub)))
	    (while add
	      (setq sign-fields (append sign-fields (list (car add)))
		    add (cdr add)))))
	(while (setq field (pop sign-fields))
	  (setq re (format "^%s:" (regexp-quote field)))
	  (goto-char (point-min))
	  (and (re-search-forward re nil t)
	       (or (re-search-forward re nil t)
		   (setq fields (concat fields
					field ":"
					(buffer-substring-no-properties
					 (point)
					 (or (and (re-search-forward
						   "^[^\n\t ]\\|^$" nil t)
						  (match-beginning 0))
					     (point-max))))
			 existing-fields (append existing-fields
						 (list field))))))
	(or existing-fields
	    (error "!! No fields to sign"))
	(goto-char (point-max))
	(widen)
	(setq body (buffer-substring-no-properties (point) (point-max)))
	(setq buffer
	      (set-buffer (get-buffer-create " *x-pgp-sig-msg-to-sign*")))
	(setq buffer-read-only nil)
	(erase-buffer)
	(insert body)
	(or (bolp) (insert "\n"))
	(goto-char (point-min))
	(insert x-pgp-sig-signed-headers-field-name ": "
		(mapconcat 'identity existing-fields ",") "\n"
		fields "\n")
	(cons existing-fields buffer)))))

(defun x-pgp-sig-create-x-pgp-signature ()
  "To sign the message and create field body."
  (let ((case-fold-search t)
	(msgs (x-pgp-sig-build-message-to-sign))
	(passphrase (and (consp x-pgp-sig-current-signer)
			 (x-pgp-sig-idea-decrypt-string
			  (cdr (assoc (cdr x-pgp-sig-current-signer)
				      x-pgp-sig-passphrase-alist)))))
	buffer result beg version)
    (when (or passphrase
	      (setq passphrase (x-pgp-sig-set-signer)))
      (unwind-protect
	  (save-excursion
	    (set-buffer (setq buffer (cdr msgs)))
	    (setq case-fold-search t)
	    (run-hooks 'x-pgp-sig-prepare-sign-hook)
	    (setq result
		  (if x-pgp-sig-pgp-v5-p
		      (x-pgp-sig-process-buffer
		       (list "-fatu" (cdr x-pgp-sig-current-signer))
		       passphrase nil "s")
		    (x-pgp-sig-process-buffer
		     (list "-fastu" (cdr x-pgp-sig-current-signer))
		     passphrase)))
	    (and result (apply 'x-pgp-sig-error "!! %s" result))
	    (run-hooks 'x-pgp-sig-post-sign-hook)
	    (goto-char (point-min))
	    (or (re-search-forward
		 (format "^%s\n" (regexp-quote x-pgp-sig-signed-begin-line))
		 nil t)
		(x-pgp-sig-error
		 "!! Regexp \"%s\\n\" not found"
		 (regexp-quote x-pgp-sig-signed-begin-line)
		 buffer))
	    (setq beg (point))
	    (or (re-search-forward
		 (format "^%s$" (regexp-quote x-pgp-sig-signed-end-line))
		 nil t)
		(x-pgp-sig-error
		 "!! Regexp \"%s\$\" not found"
		 (regexp-quote x-pgp-sig-signed-end-line)
		 buffer))
	    (narrow-to-region beg (match-beginning 0))
	    (goto-char (point-min))
	    (setq version
		  (or
		   (and (re-search-forward
			 (format "^version:.*[\t ]\\(%s\\.[^\n\t ]+\\)"
				 (if x-pgp-sig-pgp-v5-p
				     "5"
				   "[0-4]"))
			 nil t)
			(match-string 1))
		   "0.0"))
	    (or (re-search-forward "\n\n\\([^\n\t ]\\)" nil t)
		(x-pgp-sig-error
		 "!! Regexp \"\\n\\n\\\\(\[\^\\n\\t \]\\\\)\" not found"
		 buffer))
	    (delete-region (point-min) (match-beginning 1))
	    (goto-char (point-min))
	    (insert version " "
		    (mapconcat 'identity (car msgs) ",")
		    "\n")
	    (while (not (eobp))
	      (insert "\t")
	      (end-of-line)
	      (forward-char 1))
	    (setq msgs (buffer-substring (point-min) (1- (point-max)))))
	(kill-buffer buffer))
      msgs)))

(defun x-pgp-sig-build-message-to-verify (&optional buffer silent)
  "Extract header and body to verify in specified buffer."
  (if (and buffer (not (x-pgp-sig-buffer-live-p buffer)))
      (format "!! Buffer %s does not exist"
	      (prin1-to-string buffer))
    (and buffer (set-buffer (get-buffer buffer)))
    (let ((case-fold-search t)
	  lines version beg end fields fieldsx field
	  pgpsig pgphead body clines ldiff vbuff)
      (save-restriction
	(x-pgp-sig-narrow-to-header t)
	(and (re-search-forward "^lines:[\t ]+\\([0-9]+\\)" nil t)
	     (setq lines (string-to-int (match-string 1))))
	(goto-char (point-min))
	(if (re-search-forward
	     (format "^%s:%s%s"
		     (regexp-quote x-pgp-sig-field-name)
		     "[\t ]+\\([^\n\t ,]+\\)[\t ]+\\([^\n\t ]+\\)"
		     "\\(\n\\([\t ]+.+\n\\)+\\|\\([\t ]+[^\n\t ]+\\)+\n\\)"
		     ) nil t)
	    (progn
	      (setq version (match-string 1)
		    beg (match-end 2)
		    end (match-end 0))
	      (goto-char (match-beginning 2))
	      (while (re-search-forward "[^\n\t ,]+" beg t)
		(setq fields (append fields (list (match-string 0)))))
	      (setq fieldsx fields)
	      (goto-char beg)
	      (while (re-search-forward "[\t ]+\\([^\n\t ]+\\)" end t)
		(setq pgpsig (concat pgpsig (match-string 1) "\n")))
	      (while (setq field (pop fields))
		(goto-char (point-min))
		(setq pgphead
		      (concat pgphead field
			      (or (and (re-search-forward
					(format "^\\(%s\\):\\([\t ]+.+\n\\)+"
						(regexp-quote field))
					nil t)
				       (buffer-substring-no-properties
					(match-end 1) (match-end 0)))
				  ": \n"))))
	      (goto-char (point-max))
	      (widen)
	      (setq body (buffer-substring-no-properties (point) (point-max))
		    clines (count-lines (point) (point-max))
		    ldiff (or (and lines (- lines clines)) 0)
		    vbuff (set-buffer
			   (get-buffer-create " *x-pgp-sig-msg-to-verify*")))
	      (setq buffer-read-only nil)
	      (erase-buffer)
	      (insert body)
	      (or (eolp) (insert "\n"))
	      (goto-char (point-min))
	      (while (re-search-forward "^-" nil t)
		(replace-match "- -"))
	      (goto-char (point-min))
	      (insert x-pgp-sig-signed-msg-begin-line "\n"
		      "\n"
		      x-pgp-sig-signed-headers-field-name ": "
		      (mapconcat 'identity fieldsx ",") "\n"
		      pgphead
		      "\n")
	      (goto-char (point-max))
	      (insert "\n"
		      x-pgp-sig-signed-begin-line "\n"
		      "Version: " version "\n"
		      "\n"
		      pgpsig
		      x-pgp-sig-signed-end-line "\n")
	      (cons ldiff vbuff))
	  (if silent
	      ""
	    (format "!! Field \"%s\" not found"
		    x-pgp-sig-field-name)))))))

(defun x-pgp-sig-put-field (field-name field-body)
  "Insert field without dragging faces."
  (let ((case-fold-search t)
	(re (format "^%s:" (regexp-quote field-name)))
	(last (format "^\\(%s\\):"
		      (mapconcat 'identity x-pgp-sig-last-fields "\\|"))))
    (save-excursion
      (goto-char (point-min))
      (save-restriction
	(narrow-to-region
	 (point)
	 (or (and (re-search-forward
		   (format "^\\(%s\\)?\n" (regexp-quote mail-header-separator))
		   nil t)
		  (match-beginning 0))
	     (point-max)))
	(goto-char (point-min))
	(while (re-search-forward re nil t)
	  (delete-region
	   (match-beginning 0)
	   (or (and (re-search-forward "^[^\n\t ]\\|^$" nil t)
		    (match-beginning 0))
	       (point-max))))
	(goto-char (point-min))
	(if (re-search-forward last nil t)
	    (goto-char (match-beginning 0))
	  (goto-char (point-max)))
	(unless (boundp 'x-pgp-sig-start-position)
	  (make-local-variable 'x-pgp-sig-start-position)
	  (setq x-pgp-sig-start-position (point)))
	(if (bobp)
	    (insert field-name ": " field-body "\n")
	  (backward-char 1)
	  (insert "\n" field-name ": " field-body))
	))))

(defun x-pgp-sig-add-signature-y-or-n-p ()
  (y-or-n-p "Do you want to add X-PGP-Sig field? "))

(defun x-pgp-sig-enter-keyid (init)
  (let (kid kids)
    (completing-read
     "Enter the key's KeyID: "
     (delq nil
	   (mapcar
	    (function (lambda (elt)
			(setq kid (cdr elt))
			(unless (member kid kids)
			  (setq kids (cons kid kids))
			  (cons kid (car elt)))))
	    (cdr x-pgp-sig-secret-key-alist)))
     nil t (cdr init))))

(defun x-pgp-sig-enter-uid (init)
  (completing-read "Enter the key's UserID: "
		   (cdr x-pgp-sig-secret-key-alist) nil t (car init)))

(defun x-pgp-sig-enter-uid-or-keyid (init)
  (let ((keys (cdr x-pgp-sig-secret-key-alist))
	(ret (if x-pgp-sig-enter-keyid-first
		 (x-pgp-sig-enter-keyid init)
	       (x-pgp-sig-enter-uid init))))
    (if (string-match "^[\t ]*$" ret)
	(if x-pgp-sig-enter-keyid-first
	    (assoc (x-pgp-sig-enter-uid init) keys)
	  (rassoc (x-pgp-sig-enter-keyid init) keys))
      (if x-pgp-sig-enter-keyid-first
	  (rassoc ret keys)
	(assoc ret keys)))))


;;; Commands and functions to be autoloaded.

;;;###autoload
(defun x-pgp-sig-reserve-sign (&optional force)
  "Reserve sign or not sign before sending messages."
  (make-local-variable 'x-pgp-sig-reserve-sign-flag)
  (setq x-pgp-sig-reserve-sign-buffer (current-buffer)
	x-pgp-sig-reserve-sign-flag
	(or force
	    x-pgp-sig-always-sign
	    x-pgp-sig-enable-sign
	    (and (fboundp x-pgp-sig-add-signature-judging-function)
		 (funcall x-pgp-sig-add-signature-judging-function)))))

;;;###autoload
(defun x-pgp-sig-cancel-reserve-sign ()
  "Cancel reserve sign flag after sending messages."
  (and (x-pgp-sig-buffer-live-p x-pgp-sig-reserve-sign-buffer)
       (save-excursion
	 (set-buffer x-pgp-sig-reserve-sign-buffer)
	 (setq x-pgp-sig-reserve-sign-flag nil
	       x-pgp-sig-reserve-sign-buffer nil))))

;;;###autoload
(defun x-pgp-sig-sign (&optional force)
  "Add X-PGP signature."
  (let ((case-fold-search t)
	(inhibit-read-only t)
	end clength signature keyinfo algorithm infofield fpfield)
    (save-excursion
      (save-restriction
	(widen)
	(unless (and x-pgp-sig-inhibit-repeated-sign
		     (goto-char (point-min))
		     (search-forward "\n\n" nil t)
		     (re-search-backward
		      (format "^%s:" x-pgp-sig-field-name) nil t))
	  (goto-char (point-min))
	  (when (re-search-forward
		 (format "^\\(%s\\)?\n" (regexp-quote mail-header-separator))
		 nil t)
	    (setq end (match-beginning 0)
		  clength (- (point-max) (match-end 0)))
	    (goto-char (point-min))
	    (and
	     (not (zerop clength))
	     (if (x-pgp-sig-buffer-live-p x-pgp-sig-reserve-sign-buffer)
		 (save-excursion
		   (set-buffer x-pgp-sig-reserve-sign-buffer)
		   x-pgp-sig-reserve-sign-flag)
	       (or force
		   x-pgp-sig-always-sign
		   (and
		    (not (re-search-forward
			  "^control:[\t ]*cancel\\|^resent-\\(to\\|cc\\):"
			  end t))
		    (or x-pgp-sig-enable-sign
			(and (fboundp x-pgp-sig-add-signature-judging-function)
			     (funcall x-pgp-sig-add-signature-judging-function)
			     )))))
	     (when
		 (setq signature
		       (condition-case err
			   (progn
			     (message "Signing the message...")
			     (let ((sig (x-pgp-sig-create-x-pgp-signature)))
			       (and sig (message "Signing the message...done"))
			       sig))
			 (error
			  (message (if (and (listp err)
					    (stringp (car (cdr err))))
				       (car (cdr err))
				     (prin1-to-string err)))
			  nil)))
	       (and x-pgp-sig-approved-field-body
		    (re-search-forward "^control:" end t)
		    (x-pgp-sig-put-field "Approved"
					 x-pgp-sig-approved-field-body))
	       (and x-pgp-sig-x-info-field-body
		    (x-pgp-sig-put-field "X-Info"
					 x-pgp-sig-x-info-field-body))
	       (and x-pgp-sig-add-version-header
		    (x-pgp-sig-put-field x-pgp-sig-version-field-name
					 x-pgp-sig-version-field-body))
	       (setq keyinfo (assoc (cdr x-pgp-sig-current-signer)
				    x-pgp-sig-secret-key-data-list)
		     algorithm (nth 3 keyinfo)
		     infofield (cdr (assoc
				     algorithm
				     x-pgp-sig-key-info-field-name-alist))
		     fpfield (cdr (assoc
				   algorithm
				   x-pgp-sig-fingerprint-field-name-alist)))
	       (and x-pgp-sig-add-key-info-field
		    infofield
		    (x-pgp-sig-put-field
		     infofield
		     (format "%sbits, KeyID %s, Created %s, Algorithm %s"
			     (nth 1 keyinfo) (car keyinfo)
			     (nth 2 keyinfo) algorithm)))
	       (and x-pgp-sig-add-fingerprint-field
		    fpfield
		    (x-pgp-sig-put-field fpfield (nth 4 keyinfo)))
	       (x-pgp-sig-put-field x-pgp-sig-field-name signature)
	       (goto-char x-pgp-sig-start-position)
	       (run-hooks 'x-pgp-sig-sign-hook)))))))
    (setq x-pgp-sig-enable-sign nil)))

;;;###autoload
(defun x-pgp-sig-set-signer (&optional signer passphrase)
  "Set PGP signer and passphrase."
  (interactive)
  (let ((uk (cond
	     ((stringp signer)
	      (or (x-pgp-sig-search-secret-key signer 'keyid)
		  (x-pgp-sig-search-secret-key signer 'uid)))
	     ((consp x-pgp-sig-current-signer)
	      x-pgp-sig-current-signer)
	     ((stringp x-pgp-sig-current-signer)
	      (x-pgp-sig-search-secret-key x-pgp-sig-current-signer 'uid))
	     (t
	      (x-pgp-sig-search-secret-key (user-login-name) 'uid))))
	(interactive (interactive-p))
	(case-fold-search t)
	(success t)
	result)
    (and (or interactive (not uk))
	 (setq uk (x-pgp-sig-enter-uid-or-keyid uk)
	       interactive t))
    (or interactive
	uk
	(error "!! Secret key for %s not found" (prin1-to-string signer)))
    (and (or interactive (not passphrase))
	 (setq passphrase (x-pgp-sig-read-noecho
			   (format "PGP passphrase for %s (%s): "
				   (car uk) (cdr uk))
			   t)
	       interactive t))
    (if (zerop (length passphrase))
	(progn
	  (message "!! Empty passphrase")
	  (sit-for 1)
	  nil)
      (and interactive
	   (message "Confirming passphrase...")
	   (let ((buffer (get-buffer-create x-pgp-sig-working-buffer)))
	     (unwind-protect
		 (save-excursion
		   (set-buffer buffer)
		   (setq buffer-read-only nil)
		   (erase-buffer)
		   (insert "**Confirmation**\n")
		   (setq result
			 (if x-pgp-sig-pgp-v5-p
			     (x-pgp-sig-process-buffer
			      (list "-fatu" (cdr uk)) passphrase nil "s")
			   (x-pgp-sig-process-buffer
			    (list "-fastu" (cdr uk)) passphrase)))
		   (if result
		       (if x-pgp-sig-debug
			   (apply 'x-pgp-sig-error "!! %s" result)
			 (message "!! %s" result)
			 (sit-for 1)
			 (setq success nil))
		     (message "Confirming passphrase...done")))
	       (kill-buffer buffer))))
      (when success
	(setq x-pgp-sig-passphrase-alist
	      (put-alist (cdr uk)
			 (progn (message "Encrypting passphrase...")
				(x-pgp-sig-idea-encrypt-string passphrase))
			 x-pgp-sig-passphrase-alist))
	(setq x-pgp-sig-current-signer uk)
	(message "The signer and the passphrase have been registered.")
	passphrase))))

;;;###autoload
(defun x-pgp-sig-set-signer-maybe ()
  "Set PGP signer and passphrase if necessary."
  (interactive)
  (or (stringp x-pgp-sig-default-signer)
      (consp x-pgp-sig-default-signer)
      (setq x-pgp-sig-default-signer
	    (format "%s <%s>" (user-full-name) x-pgp-sig-from)))
  (let ((uk (cond
	     ((consp x-pgp-sig-current-signer)
	      x-pgp-sig-current-signer)
	     ((stringp x-pgp-sig-current-signer)
	      (x-pgp-sig-search-secret-key x-pgp-sig-current-signer 'uid))
	     (t
	      (x-pgp-sig-search-secret-key (user-login-name) 'uid))))
	(default (x-pgp-sig-default-signer))
	(minibuffer-setup-hook (and (boundp 'minibuffer-setup-hook)
				    minibuffer-setup-hook))
	default-key)
    (if uk
	(progn
	  (or (if (stringp default)
		  (progn
		    (or (string-equal default (car uk))
			(and (setq default-key
				   (cdr (assoc default
					       x-pgp-sig-secret-key-alist)))
			     (string-equal (downcase default-key)
					   (downcase (cdr uk))))))
		(and (string-equal (car default) (car uk))
		     (string-equal (downcase (cdr default))
				   (downcase (cdr uk)))))
	      (if (y-or-n-p (format "X-PGP signer %s (%s) is OK? "
				    (car uk) (cdr uk)))
		  (message "")
		(setq minibuffer-setup-hook (append minibuffer-setup-hook
						    '(beginning-of-line))
		      uk (x-pgp-sig-enter-uid-or-keyid uk))))
	  (and uk
	       (if (assoc (cdr uk) x-pgp-sig-passphrase-alist)
		   (setq x-pgp-sig-current-signer uk)
		 (x-pgp-sig-set-signer (cdr uk)))))
      (call-interactively 'x-pgp-sig-set-signer))))

;;;###autoload
(defun x-pgp-sig-switch-signing (&optional switch)
  "Judging whether to add X-PGP signature."
  (interactive
   (list (y-or-n-p
	  (format "Do you want to add %s field? " x-pgp-sig-field-name))))
  (let ((msg "This message will %sbe signed."))
    (if (setq x-pgp-sig-enable-sign (not (not switch)))
	(progn
	  (and (boundp 'mime-edit-pgp-processing)
	       (eq mime-edit-pgp-processing 'sign)
	       (setq mime-edit-pgp-processing nil))
	  (and (boundp 'mime-editor/pgp-processing)
	       (eq mime-editor/pgp-processing 'sign)
	       (setq mime-editor/pgp-processing nil))
	  (setq msg (format msg "")))
      (setq msg (format msg "not ")))
    (and (interactive-p) (message msg))))

(defun x-pgp-sig-fetch-key (pgp-id)
  (cond ((and x-pgp-sig-use-mime-pgp-fetch-key
	      (or (featurep 'mime-pgp)
		  (load "mime-pgp" t))
	      (fboundp 'mime-pgp-fetch-key))
	 (let ((mime-preview-buffer
		(if (and (get-buffer mime-echo-buffer-name)
			 (get-buffer-window mime-echo-buffer-name))
		    mime-preview-buffer
		  (window-buffer (get-largest-window))))
	       (pgp-version (if x-pgp-sig-pgp-v5-p
				'pgp50
			      'pgp)))
	   (mime-pgp-fetch-key (cons pgp-id nil))))
	(x-pgp-sig-pgp-v5-p
	 (mc-pgp50-fetch-key (cons pgp-id nil)))
	(t
	 (mc-pgp-fetch-key (cons pgp-id nil)))))

;;;###autoload
(defun x-pgp-sig-verify ()
  "Verify X-PGP signature in current message."
  (interactive)
  (let ((method (assq (or (and (boundp 'mime-raw-buffer)
			       mime-raw-buffer
			       (save-excursion
				 (set-buffer mime-raw-buffer)
				 major-mode))
			  (and (boundp 'mime-preview-original-major-mode)
			       mime-preview-original-major-mode)
			  (and (boundp 'mime-view-original-major-mode)
			       mime-view-original-major-mode)
			  (and (boundp 'mime::preview/original-major-mode)
			       mime::preview/original-major-mode)
			  major-mode)
		      x-pgp-sig-verify-method-alist))
	(delete x-pgp-sig-delete-last-empty-line-while-verifying)
	(re (format "\\(\n+\\)%s" (regexp-quote x-pgp-sig-signed-begin-line)))
	(count 0)
	prepare buffer post vbuff vbuf2 ldiff plus newlines
	nokey good bad pgp-id)
    (and method
	 (setq method (cdr method)
	       prepare (car method)
	       buffer (car (cdr method))
	       post (car (cdr (cdr method)))))
    (and (interactive-p) (message "Verifying signature..."))
    (and prepare (funcall prepare))
    (setq buffer (condition-case ()
		     (if (fboundp buffer)
			 (funcall buffer)
		       (symbol-value buffer))
		   (error nil)))
    (or (x-pgp-sig-buffer-live-p buffer)
	(setq buffer (current-buffer)))
    (save-excursion
      (setq vbuff (x-pgp-sig-build-message-to-verify
		   buffer (not (interactive-p))))
      (unless (stringp vbuff)
	(setq ldiff (car vbuff)
	      plus (if (>= ldiff 0) "+" "")
	      vbuff (set-buffer (cdr vbuff)))
	(if (zerop ldiff)
	    (message "Verifying signature...")
	  (message "(%s%d/0) Verifying signature..." plus ldiff))
	(goto-char (point-min))
	(setq newlines
	      (or (and (re-search-forward re nil t)
		       (length (match-string 1)))
		  0))
	(setq vbuf2
	      (set-buffer
	       (get-buffer-create " *x-pgp-sig-msg-to-verify-2*")))
	(setq buffer-read-only nil)
	(while (and (or delete (not (or good bad))) (not nokey))
	  (erase-buffer)
	  (insert-buffer-substring vbuff)
	  (and delete
	       (> count 0)
	       (> newlines (1+ count))
	       (goto-char (point-min))
	       (re-search-forward re nil t)
	       (goto-char (match-beginning 0))
	       (message "(%s%d/-%d) Verifying signature..."
			plus ldiff count)
	       (delete-char count))
	  (or (> newlines (+ 2 count))
	      (setq delete nil))
	  (if x-pgp-sig-pgp-v5-p
	      (x-pgp-sig-process-buffer '("-f") nil vbuf2 "v")
	    (x-pgp-sig-process-buffer '("-f") nil vbuf2))
	  (incf count)
	  (if x-pgp-sig-pgp-v5-p
	      (progn
		(goto-char (point-max))
		(and (re-search-backward
		      x-pgp-sig-pgp-v5-result-regexp nil t)
		     (looking-at (cadr x-pgp-sig-good-signature-regexps))
		     (progn
		       (forward-line 2)
		       (skip-chars-forward " ")
		       (setq good
			     (format "Good signature from user %s."
				     (buffer-substring-no-properties
				      (point)
				      (progn (end-of-line) (point))))
			     delete nil))))
	    (goto-char (point-min))
	    (and (re-search-forward
		  (car x-pgp-sig-good-signature-regexps) nil t)
		 (setq good (match-string 0)
		       delete nil)))
	  (if (if x-pgp-sig-pgp-v5-p
		  (progn
		    (goto-char (point-max))
		    (forward-line (if (featurep 'xemacs) -3 -1))
		    (looking-at (cadr x-pgp-sig-expected-regexps)))
		(goto-char (point-min))
		(re-search-forward (car x-pgp-sig-expected-regexps) nil t))
	      (progn
		(setq pgp-id (concat "0x"
				     (buffer-substring-no-properties
				      (match-beginning 1) (match-end 1))))
		(or
		 (and (or (boundp 'mc-pgp-always-fetch)
			  (condition-case ()
			      (load-library "mc-pgp"))
			  (error nil))
		      (not (eq mc-pgp-always-fetch 'never))
		      (or mc-pgp-always-fetch
			  (or (y-or-n-p
			       (format "Key %s not found; attempt to fetch? "
				       pgp-id))
			      (progn
				(setq good "Aborted")
				nil)))
		      (x-pgp-sig-fetch-key pgp-id)
		      (setq count 0))
		 (setq bad (format "!! Key %s not found" pgp-id)
		       nokey t)))
	    (if x-pgp-sig-pgp-v5-p
		(progn
		  (goto-char (point-max))
		  (setq bad
			(format
			 "!! Bad signature%s"
			 (or (and (re-search-backward
				   x-pgp-sig-pgp-v5-result-regexp nil t)
				  (looking-at
				   (cadr x-pgp-sig-bad-signature-regexps))
				  (progn
				    (forward-line 2)
				    (skip-chars-forward " ")
				    (format " from user %s."
					    (buffer-substring-no-properties
					     (point)
					     (progn (end-of-line) (point))))))
			     ""))))
	      (goto-char (point-min))
	      (setq bad
		    (concat
		     "!! "
		     (or (and (re-search-forward
			       (car x-pgp-sig-bad-signature-regexps) nil t)
			      (match-string 1))
			 "Bad signature"))))
	    ))))
    (if (stringp vbuff)
	(message vbuff)
      (and vbuff (kill-buffer vbuff))
      (and vbuf2
	   (if (and x-pgp-sig-debug
		    (not good)
		    bad)
	       (progn
		 (pop-to-buffer vbuf2)
		 (and x-pgp-sig-pgp-v5-p
		      (goto-char (point-max))
		      (recenter (1- (window-height)))))
	     (kill-buffer vbuf2)))
      (and post (funcall post))
      (and (x-pgp-sig-buffer-live-p x-pgp-sig-msg-to-verify-buffer)
	   (kill-buffer x-pgp-sig-msg-to-verify-buffer))
      (decf count)
      (message "%s%s"
	       (if (and (zerop ldiff) (zerop count))
		   ""
		 (format "(%s%d/-%d) " plus ldiff count))
	       (or good bad)))))


;;; GNUS, Gnus, Semi-Gnus
(eval-when-compile
  (autoload 'mime-find-field-decoder "eword-decode")
  ;; Pickup the macro `gnus-summary-article-number'.
  (load "gnus-sum" t)
  )

(defun x-pgp-sig-gnus-get-article ()
  (and (boundp 'gnus-article-display-hook)
       (memq 'x-pgp-sig-verify-for-gnus gnus-article-display-hook)
       (setq gnus-article-display-hook
	     (append (delq 'x-pgp-sig-verify-for-gnus
			   gnus-article-display-hook)
		     '(x-pgp-sig-verify-for-gnus))))
  (or (and (boundp 'gnus-original-article-buffer)
	   (x-pgp-sig-buffer-live-p gnus-original-article-buffer)
	   (eq gnus-current-article (gnus-summary-article-number)))
      (cond ((eq 'gnus-summary-mode major-mode)
	     (if (boundp 'gnus-original-article-buffer)
		 ; >= v5.2
		 (gnus-summary-show-article)
	       ; <= v5.1
	       (let ((gnus-show-all-headers t)
		     gnus-show-mime)
		 (gnus-summary-show-article))))
	    ((eq 'gnus-Subject-mode major-mode)
	     ; <= 3.14.4
	     (let ((gnus-show-all-headers t))
	       (gnus-Subject-show-article))))))

(defun x-pgp-sig-find-gnus-article-buffer ()
  "Find article buffer may includes X-PGP-Sig field."
  (or (and (boundp 'gnus-original-article-buffer)
	   (x-pgp-sig-buffer-live-p gnus-original-article-buffer))
      (and (boundp 'gnus-article-buffer)
	   (x-pgp-sig-buffer-live-p gnus-article-buffer))
      (and (boundp 'gnus-Article-buffer)
	   (x-pgp-sig-buffer-live-p gnus-Article-buffer))))

(defun x-pgp-sig-gnus-unwind-after-verify ()
  (or (boundp 'gnus-original-article-buffer)
      (cond ((eq 'gnus-summary-mode major-mode)
	     (gnus-summary-show-article))
	    ((eq 'gnus-Subject-mode major-mode)
	     (gnus-Subject-show-article)))))

(defvar x-pgp-sig-verify-for-gnus-unread-articles nil)
(setq-default x-pgp-sig-verify-for-gnus-unread-articles nil)

(defun x-pgp-sig-verify-for-gnus ()
  (unless (car x-pgp-sig-verify-for-gnus-unread-articles)
    (make-local-variable 'x-pgp-sig-verify-for-gnus-unread-articles)
    (setq x-pgp-sig-verify-for-gnus-unread-articles
	  (cons t (gnus-list-of-unread-articles gnus-newsgroup-name))))
  (and gnus-show-mime
       gnus-current-article
       (not (zerop gnus-current-article))
       (when (memq gnus-current-article
		   x-pgp-sig-verify-for-gnus-unread-articles)
	 (setq x-pgp-sig-verify-for-gnus-unread-articles
	       (delq gnus-current-article
		     x-pgp-sig-verify-for-gnus-unread-articles))
	 (gnus-summary-goto-subject gnus-current-article)
	 (gnus-summary-recenter)
	 (x-pgp-sig-verify))))

;;; MH-E
(defun x-pgp-sig-mh-get-article ()
  "Get current message in article buffer."
  (let ((file (format "%s%d" mh-folder-filename (mh-get-msg-num t))))
    (save-excursion
      (setq x-pgp-sig-msg-to-verify-buffer
	    (set-buffer (get-buffer-create " *x-pgp-sig-tmp-article*")))
      (setq buffer-read-only nil)
      (erase-buffer)
      (x-pgp-sig-insert-file-contents-no-conversion file)
      (x-pgp-sig-delcr))))

;;; Mew
(eval-when-compile
  (if (or (featurep 'use-mew-1.94b20-or-later)
	  (and (fboundp 'function-max-args)
	       (or (fboundp 'mew-summary-display)
		   (load "mew-summary" t))
	       (eq 2 (function-max-args 'mew-summary-display))))
      (progn
	(defmacro x-pgp-sig-mew-summary-display ()
	  '(mew-summary-display t))
	(message "Use mew-1.94b20 or later."))
    (defmacro x-pgp-sig-mew-summary-display ()
      '(condition-case nil
	   (mew-summary-display)
	 (wrong-number-of-arguments
	  (mew-summary-display t))))
    ))

(defun x-pgp-sig-mew-get-article ()
  "Get current message in article buffer."
  (and (if (fboundp 'mew-summary-goto-message)
	   (mew-summary-goto-message)	;; >= 1.93b19
	 (mew-summary-part-number)	;; <= 1.93b17
	 (re-search-backward mew-summary-message-regex nil t))
       (x-pgp-sig-mew-summary-display))
  (let ((file
	 (if (fboundp 'mew-expand-folder)
	     (mew-expand-folder (mew-summary-folder-name)     ;; Mew 1.78 -
				(mew-summary-message-number))
	   (mew-expand-file-name (mew-summary-message-number) ;; - Mew 1.76
				 (mew-summary-folder-name)))))
    (save-excursion
      (setq x-pgp-sig-msg-to-verify-buffer
	    (set-buffer (get-buffer-create " *x-pgp-sig-tmp-article*")))
      (setq buffer-read-only nil)
      (erase-buffer)
      (x-pgp-sig-insert-file-contents-no-conversion file)
      (x-pgp-sig-delcr))))

(defun x-pgp-sig-mew-encode-message-header-with-semi ()
  "Encode the message header to MIME format with SEMI."
  (require 'eword-encode)
  (eword-encode-header))

(defun x-pgp-sig-mew-encode-message-header-with-tm ()
  "Encode the message header to MIME format with tm."
  (require 'tm-ew-e)
  (mime/encode-message-header))

(defun x-pgp-sig-mew-encode-message-header-with-toMime ()
  "Encode the message header to MIME format with toMime."
  (save-excursion
    (save-restriction
      (x-pgp-sig-narrow-to-header)
      (let (
	    ; Mule-2.x
	    (input-coding-system '*noconv*)
	    (output-coding-system mew-cs-mime-trans)
	    ; Emacs 20, XEmacs
	    (coding-system-for-read 'binary)
	    (coding-system-for-write mew-cs-mime-trans)
	    )
	(apply 'call-process-region (point-min) (point-max)
	       x-pgp-sig-toMime-program t t nil
	       x-pgp-sig-toMime-options)))))

(defun x-pgp-sig-mew-encode-message ()
  "Encode the message to the network code."
  (save-excursion
    (save-restriction
      (widen)
      (let ((inhibit-read-only t))
	(put-text-property (point-min) (point-max) 'read-only nil))
      (make-local-variable 'mail-header-separator)
      (setq mail-header-separator mew-header-separator
	    buffer-read-only nil)
      (and x-pgp-sig-mew-encode-message-header-function
	   (funcall x-pgp-sig-mew-encode-message-header-function))
      (goto-char (point-min))
      (and (re-search-forward "\\cj\\|\\ch\\|\\ck" nil t)
	   (mew-cs-encode-region (point-min) (point-max) mew-cs-mime-trans)))))

(defun x-pgp-sig-mew-encode-coding-and-sign (&optional force)
  "Encode the message to the network code and sign."
  (let ((inhibit-read-only t))
    (put-text-property (point-min) (point-max) 'read-only nil))
  (mew-cs-encode-region (point-min) (point-max) mew-cs-mime-trans)
  (x-pgp-sig-sign force))

;;; cmail
(defun x-pgp-sig-cmail-get-article ()
  "Get current message in article buffer."
  (let ((page (cmail-get-page-number-from-summary))
	(folder (get-buffer (cmail-folder-buffer cmail-current-folder)))
	beg end)
    (save-excursion
      (set-buffer folder)
      (cmail-n-page page)
      (setq beg (point))
      (cmail-n-page (1+ page))
      (forward-line -1)
      (setq end (point))
      (setq x-pgp-sig-msg-to-verify-buffer
	    (set-buffer (get-buffer-create " *x-pgp-sig-tmp-article*")))
      (setq buffer-read-only nil)
      (erase-buffer)
      (insert-buffer-substring folder beg end))))

;;; VM
(eval-when-compile
  ;; Pickup the macro `vm-select-folder-buffer'.
  (unless (load "vm-macro" t)
    (load "vm-misc" t) ;; For old VM.
    )
  ;; Pickup the macros 'vm-headers-of' and `vm-text-end-of'.
  (load "vm-message" t)
  )

(defun x-pgp-sig-vm-get-article ()
  (and (eq 'vm-summary-mode major-mode)
       (vm-follow-summary-cursor))
  (vm-select-folder-buffer)
  (let ((mp (car vm-message-pointer)))
    (save-restriction
      (widen)
      (copy-to-buffer
       (setq x-pgp-sig-msg-to-verify-buffer
	     (get-buffer-create " *x-pgp-sig-tmp-article*"))
       (vm-headers-of mp) (vm-text-end-of mp)))))

;;; Wanderlust
(defun x-pgp-sig-wl-get-article ()
  "Get current message in article buffer."
  (save-excursion
    (let ((message-buf (wl-message-get-original-buffer)))
      (wl-summary-set-message-buffer-or-redisplay)
      (setq x-pgp-sig-msg-to-verify-buffer
	    (set-buffer (get-buffer-create " *x-pgp-sig-tmp-article*")))
      (setq buffer-read-only nil)
      (erase-buffer)
      (insert-buffer-substring message-buf))))


(and (featurep 'x-pgp-sig) (run-hooks 'x-pgp-sig-reload-hook))
(provide 'x-pgp-sig)
(run-hooks 'x-pgp-sig-load-hook)
;;; x-pgp-sig.el ends here.
