;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;Advanced Micro Devices, Inc.
;;;June, 2001
;;;***************************************************************

(in-package "ACL2")

(local (include-book "../support/top"))

(include-book "bits")


;;;**********************************************************************
;;;                       SGN, SIG, and EXPO
;;;**********************************************************************

(defun expo (x)
  (cond ((or (not (rationalp x)) (= x 0)) 0)
	((< x 0) (expo (- x)))
	((< x 1) (1- (expo (* 2 x))))
	((< x 2) 0)
	(t (1+ (expo (/ x 2))))))

(defun sgn (x) 
  (if (or (not (rationalp x)) (= x 0))
      0
    (if (< x 0) -1 +1)))

(defun sig (x)
  (if (rationalp x)
      (if (< x 0)
          (- (* x (expt 2 (- (expo x)))))
        (* x (expt 2 (- (expo x)))))
    0))

(in-theory (disable sgn sig expo))

(defthm fp-rep
    (implies (rationalp x)
	     (equal x (* (sgn x) (sig x) (expt 2 (expo x)))))
  :rule-classes ())

(defthm fp-abs
    (implies (rationalp x)
	     (equal (abs x) (* (sig x) (expt 2 (expo x)))))
  :rule-classes ())

(defthm fp-rep-unique
    (implies (and (rationalp x)
		  (not (= x 0))
		  (rationalp m)
		  (<= 1 m)
		  (< m 2)
		  (integerp e)
		  (= (abs x) (* m (expt 2 e))))
	     (and (= m (sig x))
		  (= e (expo x))))
  :rule-classes ())

(defthm sgn*
    (implies (and (rationalp x) (rationalp y))
	     (= (sgn (* x y)) (* (sgn x) (sgn y)))))

(in-theory (disable sgn*))

(defthm expo-minus
    (= (expo (* -1 x))
       (expo x)))

(in-theory (disable expo-minus))

(defthm expo-lower-bound
    (implies (and (rationalp x)
		  (not (= x 0)))
	     (<= (expt 2 (expo x)) (abs x)))
  :rule-classes :linear)

(in-theory (disable expo-lower-bound))

(defthm expo-upper-bound
    (implies (and (rationalp x))
	     (< (abs x) (expt 2 (1+ (expo x)))))
  :rule-classes :linear)

(in-theory (disable expo-upper-bound))

(defthm bvecp-expo
    (implies (natp x)
	     (bvecp x (1+ (expo x)))))

(in-theory (disable bvecp-expo))

(defthm expo>=
    (implies (and (rationalp x)
		  (integerp n)
		  (>= x (expt 2 n)))
	     (>= (expo x) n))
  :rule-classes :linear)

(in-theory (disable expo>=))

(defthm expo<=
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (< x (* 2 (expt 2 n))))
	     (<= (expo x) n))
  :rule-classes :linear)

(in-theory (disable expo<=))

(defthm expo-squeeze
    (implies (and (rationalp x)
		  (integerp n)
		  (<= (expt 2 n) (abs x))
		  (< (abs x) (expt 2 (1+ n))))
	     (= (expo x) n))
  :rule-classes ())

(defthm expo-monotone
    (implies (and (rationalp x)
		  (not (= x 0))
		  (rationalp y)
		  (<= (abs x) (abs y)))
	     (<= (expo x) (expo y)))
  :rule-classes :linear)

(in-theory (disable expo-monotone))

(defthm expo-2**n
    (implies (integerp n)
	     (equal (expo (expt 2 n))
		    n)))

(defthm expo-shift
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n))
	     (= (expo (* (expt 2 n) x)) 
		(+ n (expo x)))))

(in-theory (disable expo-shift))

(defthm expo-x+2**k
    (implies (and (integerp k)
		  (rationalp x)
		  (> x 0)
		  (< (expo x) k))
	     (equal (expo (+ x (expt 2 k)))
		    k)))

(in-theory (disable expo-x+2**k))

(defthm expo-prod-lower
    (implies (and (rationalp x)
		  (not (= x 0))
		  (rationalp y)
		  (not (= y 0)))
	     (<= (+ (expo x) (expo y)) (expo (* x y))))
  :rule-classes :linear)

(in-theory (disable expo-prod-lower))

(defthm expo-prod-upper
    (implies (and (rationalp x)
		  (not (= x 0))
		  (rationalp y)
		  (not (= y 0)))
	     (>= (+ (expo x) (expo y) 1) (expo (* x y))))
  :rule-classes :linear)

(in-theory (disable expo-prod-upper))

(defthm mod-expo
    (implies (and (integerp x)
		  (> x 0))
	     (= (mod x (expt 2 (expo x)))
		(- x (expt 2 (expo x)))))
  :rule-classes ())

(defthm sig-minus
  (= (sig (* -1 x))
     (sig x)))

(in-theory (disable sig-minus))

(defthm sig-lower-bound
    (implies (and (rationalp x)
		  (not (= x 0)))
	     (<= 1 (sig x)))
  :rule-classes :linear)

(in-theory (disable sig-lower-bound))

(defthm sig-upper-bound
  (< (sig x) 2)
  :rule-classes :linear)

(in-theory (disable sig-upper-bound))

(defthm sig-shift
  (= (sig (* (expt 2 n) x)) 
     (sig x)))

(in-theory (disable sig-shift))

(defthm already-sig
  (implies (and (rationalp x)
                (<= 1 x)
                (< x 2))
           (= (sig x) x)))

(defthm sig-sig
    (equal (sig (sig x)) 
	   (sig x)))


;;;**********************************************************************
;;;                            EXACTP
;;;**********************************************************************

(defun exactp (x n)
  (integerp (* (sig x) (expt 2 (1- n)))))

(defthm exactp2
    (implies (and (rationalp x)
		  (integerp n))
	     (equal (exactp x n)
		    (integerp (* x (expt 2 (- (1- n) (expo x))))))))

(in-theory (disable exactp exactp2))

(defthm exact-neg
    (implies (and (rationalp x)
		  (integerp n))
	     (iff (exactp x n) (exactp (abs x) n)))
  :rule-classes ())

(defthm exactp-
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0)
		  (exactp x n))
	     (exactp (* -1 x) n)))

(in-theory (disable exactp-))

(defthm exactp-shift-iff
    (implies (and (rationalp x)
		  (integerp m)
		  (integerp n))
	     (iff (exactp x m)
		  (exactp (* (expt 2 n) x) m)))
  :rule-classes ())

(defthm exactp-<=
    (implies (and (rationalp x)
		  (integerp n)
		  (integerp m)
		  (<= m n)
		  (exactp x m))
	     (exactp x n)))

(in-theory (disable exactp-<=))

(defthm bvecp-exactp
  (implies (bvecp x n)
           (exactp x n)))

(defthm exactp-2**n
    (implies (and (integerp n)
		  (integerp m)
		  (> m 0))
	     (exactp (expt 2 n) m)))

(in-theory (disable exactp-2**n))

(defthm exactp-sig-x
  (equal (exactp (sig x) n)
         (exactp x n)))

(defthm exact-bits-1
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (= (expo x) (1- n))
		  (< k n))
	     (iff (integerp (/ x (expt 2 k)))
		  (exactp x (- n k))))
  :rule-classes ())

(defthm exact-bits-2
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (= (expo x) (1- n))
		  (< k n))
	     (iff (integerp (/ x (expt 2 k)))
		  (= (bits x (1- n) k)
		     (/ x (expt 2 k)))))
  :rule-classes ())

(defthm exact-bits-3
    (implies (and (natp x)
		  (natp n)
		  (natp k)
		  (= (expo x) (1- n))
		  (< k n))
	     (iff (integerp (/ x (expt 2 k)))
		  (= (bits x (1- k) 0)
		     0)))
  :rule-classes ())

(defthm exactp-prod
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp m)
		  (integerp n)
		  (exactp x m)
		  (exactp y n))
	     (exactp (* x y) (+ m n)))
  :rule-classes ())

(defthm exactp-x2
    (implies (and (rationalp x)
		  (integerp k)
		  (exactp x k)
		  (integerp n)
		  (exactp (* x x) (* 2 n)))
	     (exactp x n))
  :rule-classes ())

(defthm exact-k+1
    (implies (and (natp n)
		  (natp x)
		  (= (expo x) (1- n))
		  (natp k)
		  (< k (1- n))
		  (exactp x (- n k)))
	     (iff (exactp x (1- (- n k)))
		  (= (bitn x k) 0)))
  :rule-classes ())

(defun fp+ (x n)
  (+ x (expt 2 (- (1+ (expo x)) n))))

(defthm fp+1
    (implies (and (rationalp x)
		  (> x 0)
		  (rationalp y)
		  (> y x)
		  (integerp n)
		  (> n 0)
		  (exactp x n)
		  (exactp y n))
	     (>= y (fp+ x n)))
  :rule-classes ())

(defthm fp+2
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (exactp x n))
	     (exactp (fp+ x n) n))
  :rule-classes ())

(defthm expo-diff
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp k)
		  (integerp n)
		  (> n 0)
		  (> n k)
		  (exactp x n)
		  (exactp y n)
		  (<= (+ k (expo (- x y))) (expo x))
		  (<= (+ k (expo (- x y))) (expo y)))
	     (exactp (- x y) (- n k)))
  :rule-classes ())

(defthm expo-diff-0
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp n)
		  (> n 0)
		  (exactp x n)
		  (exactp y n)
		  (<= (expo (- x y)) (expo x))
		  (<= (expo (- x y)) (expo y)))
	     (exactp (- x y) n))
  :rule-classes ())

(defthm expo-diff-cor
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp n)
		  (> n 0)
		  (exactp x n)
		  (exactp y n)
		  (<= (abs (- x y)) (abs x))
		  (<= (abs (- x y)) (abs y)))
	     (exactp (- x y) n))
  :rule-classes ())

(defthm expo-diff-min
    (implies (and (rationalp x)
		  (rationalp y)
		  (> x 0)
		  (> y 0)
		  (integerp n)
		  (> n 0)
		  (exactp x n)
		  (exactp y n)
		  (not (= y x)))
	     (>= (expo (- y x)) (- (1+ (min (expo x) (expo y))) n)))
  :rule-classes ())

(defthm expo-diff-abs-any
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp n)
		  (> n 1)
		  (exactp x n)
		  (exactp y n))
	     (<= (abs (expo (- x y)))
		 (+ (max (abs (expo x)) (abs (expo y))) (1- n))))
  :rule-classes ())

