/**********************************************************************
 ** inp_funct - input handler functions
 **
 **    
 ** Reviewed through:
 **
 ** Copyright (C) 2000 George Noel (Slate), Ed Boraas
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef INP_FUNCT_C
#define INP_FUNCT_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "inp_funct.h"
#include "builder.h"
#include "input.h"
#include "objtype.h"
#include "user_dbase.h"
#include "gameflags.h"
#include "global.h"
#include "utils.h"
#include "login.h"
#include "comflags.h"
#include "adminflags.h"
#include "indflags.h"
#include "door.h"
#include "key.h"
#include "location.h"
#include "marker.h"
#include "mobile.h"
#include "merger.h"
#include "money.h"
#include "moveable.h"
#include "wearable.h"
#include "weapon.h"
#include "book.h"
#include "timespan.h"
#include "food.h"
#include "rope.h"
#include "newfuncts.h"
#include "memchk.h"
#include "race.h"
#include "inclination.h"
#include "talent.h"

/***********************************************************************
 ** main_bldr_handler - the main handler for the builder port
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int main_bldr_handler(MudObject *the_user, char *the_input) {

   Builder *the_builder;
   Input *new_input = NULL;
   Strings tmp_input;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return -1;

   the_builder = (Builder *) the_user;
   
   tmp_input = the_input;

   if (the_input != NULL && tmp_input.str_len() > 0)
   { 
      new_input = new_Input();
      new_input->handle_input(&tmp_input, the_builder);
   }
  
   if (new_input != NULL)
      delete_Input(new_input);

   return 1;
}


/***********************************************************************
 ** main_plr_handler - the main handler for the game port
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int main_plr_handler(MudObject *the_user, char *the_input) 
{
   Player    *the_player;
   Input     *new_input = NULL;
   Strings   tmp_input;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_input = the_input;
   if (the_input != NULL && tmp_input.str_len() > 0)
   { 
      new_input = new_Input();
      new_input->handle_input(&tmp_input, the_player);
   }

   if (new_input != NULL)
      delete_Input(new_input);

   return 1;
}


/***********************************************************************
 ** get_bname - gets the name of the builder
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int get_bname(MudObject *the_user, char *the_input) 
{
   Builder *the_builder;
   Strings builder_input;
   User_Dbase *tmp_udbase;
   Inp_Handler *tmp_handler;
   Flags       *oTmpFlags;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return -1;

   the_builder = (Builder *) the_user;
   
   builder_input = the_input;
   builder_input.remove_newline();
   /* get the builder's name */
   if (builder_input.str_len() > MAXNAMELEN)
   {
      the_builder->send_bldr("That name is too long!\n");
      return -1;
   }
   else if (builder_input.str_len() <= 0)
   {
      the_builder->send_bldr("\nCancelling log-on, bye!\n");
      the_builder->set_off();
      return -1;
   }
   else if (!valid_name(builder_input.str_show()))
   {
      the_builder->send_bldr("'%s' is not a valid name!\n\n", 
                                            builder_input.str_show());
      return -1;
   }
   else
   {
      builder_input.lowercase();
      builder_input.upper(0);


      the_builder->set_title(builder_input.str_show());

      tmp_udbase = mainstruct->get_user_database();

      if(    tmp_udbase->read_user( builder_input.str_show( ), the_builder ) <= 0
          || ( ( oTmpFlags = the_builder->get_gameflags( ) )
               && oTmpFlags->get_flag( GAMEFLAG_DELETED ) ) )
      {
         the_builder->
                   send_bldr("That user does not exist in our database.\n");
         the_builder->
                   send_bldr("Go to the game port at port %hd if you want to "
                             "add a new user.\n", the_config.gameport);
         the_builder->set_off();
         return -1;
      }
      
   }
   tmp_handler = the_builder->get_input_handler();
   tmp_handler->replace_input_handler(get_bpasswd, 
                                      "Please enter your password: ", 
                                      HANDLER_DATA_INT);
   tmp_handler->set_input_invis();
   return 1;
}

/***********************************************************************
 ** get_bpasswd - gets the request for a password from the user
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int get_bpasswd(MudObject *the_user, char *the_input) 
{
   Builder *the_builder;
   Strings builder_input;
   Strings the_password;
   Strings holder;
   Inp_Handler *tmp_handler;
   int        *counter;
   Flags      *gameflags;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return -1;

   the_builder = (Builder *) the_user;

   tmp_handler = the_builder->get_input_handler();

   if (tmp_handler->get_data() == NULL)
   {
      counter = new_int();
      *counter = 0;
      tmp_handler->set_data((void *) counter);
   }
   else
      counter = (int *) tmp_handler->get_data();

   builder_input = the_input;

   builder_input.remove_newline();
   if (builder_input.str_len() <= 0)
   {
      the_builder->send_bldr("\n\nCancelling log-on, bye!\n\n");
      the_builder->set_off();
      return -1;
   }
   
   encrypt_passwd(&builder_input, &the_password);
   if (!the_password.str_cmp(the_builder->get_passwd()))
   {
      the_builder->send_bldr("\nPassword incorrect.\n\n");
      *counter = *counter + 1;
      if (*counter == the_config.maxpasswdtries)
      {
         the_builder->send_bldr("Too many failures, bye!\n\n");
         the_builder->set_off();
         return -1;
      }
      else
      {
         tmp_handler->set_data((void *) counter);
      }
      return -1;
   }
   else
   {
      the_builder->send_bldr("\nCorrect password.\n\n");
   }

   if (mainstruct->get_builder(the_builder->get_title()) != NULL)
   {
      the_builder->send_bldr("That builder is already logged on.\n");
      tmp_handler->replace_input_handler(kick_off_builder, 
                             "Kick them off? (y/n) ", 0);
      return 0;
   }

   the_builder->set_pname(the_builder->get_title());

   // turn on their echo if it isnt yet
   the_builder->get_connection()->echo_on();

   gameflags = the_builder->get_gameflags();
   if (gameflags->get_flag(GAMEFLAG_COLOR))
      (the_builder->get_connection())->set_color(1);

   holder.sprintf("Log &+Gon&* by builder: %s", the_builder->get_name());
   mainstruct->log_event(holder.str_show());

   holder.sprintf("&+Y[&+WBuilder &+M%s&+W has logged "
           "onto the builder port.&+Y]&*\n", the_builder->get_name());
   mainstruct->send_all_players(holder.str_show(), NULL, ADMINFLAG_SEELOGIN);

   the_builder->send_bldr("\n&+CWelcome Builder &+M%s&+C!&*\n\n", 
                                                  the_builder->get_name());
      
   tmp_handler->pop_input_handler();
   return 1;
}


/***********************************************************************
 ** get_pname - gets the name of the player
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int get_pname(MudObject *the_user, char *the_input) 
{
   Player *the_player;
   Strings player_input;
   Strings playername;
   User_Dbase *tmp_udbase;
   Inp_Handler *tmp_handler;
   Flags       *oTmpFlags;
   int        result;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   /* get the player's name */
   player_input = the_input; 
   player_input.remove_newline();
   if (player_input.str_len() > MAXNAMELEN)
   {
      the_player->send_plr("That name is too long!\n\n");
      return -1;
   }
   else if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nCancelling log-on, bye!\n");
      the_player->set_off();
      return -1;
   }
   else if (!valid_name(player_input.str_show()))
   {
      the_player->send_plr("'%s' is not a valid name!\n\n", 
                                            player_input.str_show());
      return -1;
   }
   else
   {
      playername = player_input.str_show();
      playername.lowercase();
      playername.upper(0);

      tmp_udbase = mainstruct->get_user_database();

      the_player->set_title(playername.str_show());

      the_player->create_mail();

      if(((result = tmp_udbase->read_user( playername.str_show( ), the_player )) 
	 <= 0) || (( oTmpFlags = the_player->get_gameflags())
		   && oTmpFlags->get_flag( GAMEFLAG_DELETED)))

      {
	if (result == -2)
	{
	  the_player->send_plr("That player is dead and can't be used.\n"
			       "Please choose another.\n\n");
	  return 0;
	}
	else
	{
	  the_player->
	    send_plr("That player does not exist in our database.\n");
	  tmp_handler->replace_input_handler(ask_new_player, 
					     "Create a new player? (y/n) ", 0);
	  return 1;
	}   
      }

   }

   tmp_handler->replace_input_handler(get_ppasswd,
                                   "Please enter your password: ", 
                                   HANDLER_DATA_INT);
   tmp_handler->set_input_invis();

   return 1;
}


/***********************************************************************
 ** ask_new_player - finds out if this is a new player or not
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int ask_new_player(MudObject *the_user, char *the_input) 
{
   Player *the_player;
   Strings player_input;
   Inp_Handler *tmp_handler;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
      return -1;
   }

   if (!STRNCASECMP("yes", player_input.str_show(), player_input.str_len()))
   {
      the_player->send_plr("\n");
      tmp_handler->replace_input_handler(get_new_passwd1, 
                                          "Please enter a password: ", 0);
      tmp_handler->set_input_invis();
      return 1; 
   }
   else if (!STRNCASECMP("no", player_input.str_show(), 
                                                    player_input.str_len()))
   {
      the_player->set_initname();
      tmp_handler->replace_input_handler(get_pname, 
                                          "What shall I call you? ", 0);
      return 1; 
   }
   else
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
   }
   return -1;
}


/***********************************************************************
 ** get_ppasswd - asks for the password for the player
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int get_ppasswd(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Strings     the_password;
   Strings     holder;
   int         *count;
   Flags       *gameflags;
   Flags       *indflags;
   Location    *loc_holder;
   Mailer      *the_mailer;
   Race        *plr_race;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\n\nCancelling log-on, bye!\n\n");
      the_player->set_off();
      return -1;
   }

   if (tmp_handler->get_data() == NULL)
   {
      count = new_int();
      *count = 0;
      tmp_handler->set_data((void *) count);
   }
   else
      count = (int *) tmp_handler->get_data();

   encrypt_passwd(&player_input, &the_password);

   if (!the_password.str_cmp(the_player->get_passwd()))
   {
      the_player->send_plr("\nPassword incorrect.\n\n");
      *count = *count + 1;
      if (*count == the_config.maxpasswdtries)
      {
         the_player->send_plr("\nToo many failures, bye!\n\n");
         the_player->set_off();
         return -1;
      }
      else
      {
         tmp_handler->set_data((void *) count);
         return 0;
      }
   }


   the_player->send_plr("\nCorrect password.\n\n");


   if (mainstruct->get_player(the_player->get_title()) != NULL)
   {
      the_player->send_plr("That player is already logged on.\n");
      tmp_handler->replace_input_handler(kick_off_player, 
                             "Kick them off? (y/n) ", 0);
      return 0;
   }

   the_player->set_pname(the_player->get_title());

   /* if we can't get a start location, assign them to the default
      hard-coded location.  If that won't work, die */
   if (!the_player->get_location())
   {
      plr_race = mainstruct->get_race(the_player->get_race());

      if (plr_race == NULL)
      {
         the_player->set_loc_str(mainstruct->get_def_start_loc());   
      }
      else
         the_player->set_loc_str(plr_race->get_init_location());
   }
   if (the_player->reset_curr_loc() == -1)
   {
      mainstruct->log_error("Default start location not a valid location.",
                            "enter_new_player");
      holder.sprintf("%s@%s", the_config.entry_locname.str_show(),
                                       the_config.hardcoded_areaname.str_show());
      mainstruct->set_def_start_loc(holder.str_show());

      the_player->set_loc_str(mainstruct->get_def_start_loc());
      if (the_player->reset_curr_loc() == -1)
      {
         printf("Fatal Error!!! "
                "Could not get harcoded start location from database!!!\n");
         exit(0);
      }
   }

   // turn on their echo if it isnt yet
   the_player->get_connection()->echo_on();

   gameflags = the_player->get_gameflags();
   indflags = the_player->get_indflags();

   indflags->clr_flag(INDFLAG_CORPSE);
   indflags->clr_flag(INDFLAG_GHOST);
   the_player->set_health(the_player->get_maxhealth());

   if (gameflags->get_flag(GAMEFLAG_COLOR))
                     (the_player->get_connection())->set_color(1);

   holder.sprintf("Log &+Gon&* by player: %s", the_player->get_title());
   mainstruct->log_event(holder.str_show());

   /* read and create the player's last inventory */
   // the_player->read_inventory();

   loc_holder = the_player->get_loc(); 
   holder.sprintf("&+Y[&+WPlayer &+M%s&+W has entered "
                  "the game at &+c%s&+Y]&*\n", the_player->get_title(), 
                                    loc_holder->get_title());
   mainstruct->send_all_players(holder.str_show(), the_player, 
                                                 ADMINFLAG_SEELOGIN);

   holder.sprintf("%s enters the game.\n", the_player->get_title());

   gameflags->set_flag(GAMEFLAG_ONGAME);
   loc_holder->send_location(holder.str_show(), the_player);

   the_player->send_plr("\n&+CWelcome &+M%s&+C!&*\n\n", 
                                               the_player->get_name());

   loc_holder->show_location(the_player, gameflags->get_flag(GAMEFLAG_BRIEF));
   mainstruct->show_tod(the_player);
   the_mailer = the_player->get_mailer();
   if (the_mailer->has_newmail())
   {
      the_player->send_plr("&+WYou have NEW mail!&*\n");
      the_mailer->load_mail(Full);
      the_mailer->clr_newmail();
      the_mailer->write_mail();
   }

   mainstruct->sort_playerlist("exp");

   tmp_handler->pop_input_handler();
   return 1;
}


/***********************************************************************
 ** get_new_passwd1 - asks for a password for this player to use
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int get_new_passwd1(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Strings     the_password;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nCancelling new player creation\n\n");
      the_player->set_initname();
      tmp_handler->replace_input_handler(get_pname, 
                                              "What shall I call you? ", 0);
      return 0;
   }

   if (!validate_passwd(the_player, player_input.str_show()))
   {
      return -1;
   }

   encrypt_passwd(&player_input, &the_password);
   the_player->set_passwd(the_password.str_show());
  
   tmp_handler->replace_input_handler(get_new_passwd2, 
                      "\nPlease type the same password again: ", 0);
   tmp_handler->set_input_invis();

   return 1;
}


/***********************************************************************
 ** get_new_passwd2 - confirms the password
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int get_new_passwd2(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Strings     the_password;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nYou must type the same password both times.\n");
      tmp_handler->replace_input_handler(get_new_passwd1,
                                       "Please enter a password: ", 0);
      tmp_handler->set_input_invis();
      return 0;
   }

   encrypt_passwd(&player_input, &the_password);

   if (!the_password.str_cmp(the_player->get_passwd()))
   {
      the_player->send_plr("\nPlease type the same password both times.\n");
      tmp_handler->replace_input_handler(get_new_passwd1, 
                                         "Please enter a password: ", 0);
      tmp_handler->set_input_invis();
      return 0;
   }

   the_player->send_plr("\nPassword created.\n\n");

   // The next two lines should probably be moved somewhere more appropriate.
   // For now, this works :)
     
   tmp_handler->replace_input_handler(ask_use_color,                                 "Would you like to use color? (y/n) ", 0);

   return 1;
}



/***********************************************************************
 ** ask_use_color - figure out if the player wants to use color
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int ask_use_color(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Strings     the_file;
   Flags       *game_flags;
   Area_Dbase  *race_area;
   Object_List *the_dbase;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
      return -1;
   }

   game_flags = the_player->get_gameflags();
   if (!STRNCASECMP("yes", player_input.str_show(), player_input.str_len()))
   {
      game_flags->set_flag(GAMEFLAG_COLOR);
      (the_player->get_connection())->set_color(1);
   }
   else if (!STRNCASECMP("no", player_input.str_show(), 
                                                    player_input.str_len()))
   {
      game_flags->clr_flag(GAMEFLAG_COLOR);
      (the_player->get_connection())->set_color(0);
   }
   else
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
      return -1;
   }

   // First, we replace color with get_sex, then possibly stack on the
   // ask race handler
   tmp_handler->replace_input_handler(get_sex, 
                                 "Are you male, female, or neuter? (m/f/n) ", 0);

   // Now we see if they have more than one race loaded and if so, allow the
   // new player to pick one.  If only one, we just assign them that one
   the_dbase = mainstruct->get_dbase();
   if ((race_area = the_dbase->get_area("races")) == NULL)
     fault("Possibly fatal error!  Missing race area!\n");

   if (race_area->get_num_obj_loaded() < 1)
   {
     mainstruct->log_error("No races loaded.  "
		     "Needs at least one race to login new chars.", "get_sex");
     fault("Error, no races loaded.  Needs race to login new chars.\n");
     return -1;
   }
   if (race_area->get_num_obj_loaded() >= 2)
   {
     mainstruct->display_races(the_player);
     tmp_handler->push_input_handler(get_race, 
                                "What race will you be born into? ", 0);
   }
   else  // Assign them the one race
   {
     race_area->reset_list();
     the_player->set_race((race_area->get_next_obj())->get_name());
   }

   return 1;
}


/***********************************************************************
 ** get_race - gets the player's race
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int get_race(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Object_List *obj_list;
   Race        *the_race;
   Connection  *the_connection;
   int         show_desc = 0;
   Strings     the_word;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   the_word.assign_word(player_input.str_show(), 1);
   if ((the_word.str_show() != NULL) && 
       (!STRCASECMP("?", the_word.str_show())))
   {
      the_word.assign_word(player_input.str_show(), 2);
      if (the_word.str_show() == NULL)
      { 
         the_player->send_plr("format: ? <race>\n");
         return 0;
      }
      show_desc = 1;
   }

   obj_list = mainstruct->get_dbase();

   if ((the_race = obj_list->get_race_obj(the_word.str_show())) == NULL)
   {
      the_player->send_plr("I am not aware of the '%s' race.\n"
                  "Please select again.\n", the_word.str_show());
      return 0;
   }

   the_connection = the_player->get_connection();

   if (show_desc)
   {
      the_connection->start_paging();
      the_player->send_plr("%s\n", the_race->get_description());
      the_connection->end_paging(the_player);   
      return 1;
   }

   the_player->send_plr("\nYou have selected the '&+y%s&*' race.\n\n", 
                                               the_race->get_name());

   the_player->set_race(the_race->get_name());

   tmp_handler->replace_input_handler(confirm_race, 
                       "Would you like to use this race? (y/n) ", 0);

   the_connection = the_player->get_connection();
   the_connection->start_paging();

   the_player->send_plr("%s\n", the_race->get_description());
 
   the_connection->end_paging(the_player);
   
   return 1;
}


/***********************************************************************
 ** confirm_race - confirms that the user really wants this race after
 **                reading the race description
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int confirm_race(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Strings     the_file;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
      return -1;
   }

   if (!STRNCASECMP("yes", player_input.str_show(), player_input.str_len()))
   {
      tmp_handler->pop_input_handler();
      return 1;      
   }
   else if (!STRNCASECMP("no", player_input.str_show(), 
                                                    player_input.str_len()))
   {
      mainstruct->display_races(the_player);
      tmp_handler->replace_input_handler(get_race, 
                                "What race will you be born into? ", 0);

      return 1; 
   }
   else
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
      return -1;
   }


   return -1;
}


/***********************************************************************
 ** get_sex - finds out if they are male or female
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int get_sex(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Flags       *tmp_gameflags;
   Flags       *tmp_indflags;
   Strings     holder;
   Race        *the_race;
   Object_List *the_dbase;
   Strings     first_tutorial;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nPlease enter either male, female or neuter.\n");
      return 0;
   }

   tmp_gameflags = the_player->get_gameflags();
   tmp_indflags = the_player->get_indflags();

   if (player_input.str_n_cmp("male", player_input.str_len()))
   { 
      the_player->set_sex(SEX_MALE);
   }
   else if (player_input.str_n_cmp("female", player_input.str_len()))
   {
      the_player->set_sex(SEX_FEMALE);
   }
   else if (player_input.str_n_cmp("neuter", player_input.str_len()))
   {
       the_player->set_sex(SEX_NEUTER);
   }
   else
   {
      the_player->send_plr("That is not a valid choice.\n");
      return 0;
   }

   the_dbase = mainstruct->get_dbase();
   the_race = the_dbase->get_race_obj(the_player->get_race());

   the_player->set_birthday(time(0));
   the_player->set_desc(the_race->get_init_desc());
   the_player->add_rank_list(the_race->get_init_ability());
   the_player->set_strength(the_race->get_init_str());
   the_player->set_dexterity(the_race->get_init_dex());
   the_player->set_intel(the_race->get_init_intel());
   the_player->set_wisdom(the_race->get_init_wis());
   the_player->set_charisma(the_race->get_init_cha());
   the_player->set_health(the_player->get_maxhealth());
   the_player->set_talent_funds(the_config.talent_funds);

   the_player->get_gameflags()->
     set_flag(GAMEFLAG_FULLEXITS);     // enabled by default

   the_player->get_gameflags()->
     set_flag(GAMEFLAG_AUTODIAGNOSE);  // enabled by default

   holder.format_for_actions(the_race->get_init_brief(), the_player, NULL);
   the_player->push_brief("main", holder.str_show());

   // Now we fill up the stack with anything they need to select from

   // First, we put on the ask for intro one, this one will happen
   tmp_handler->replace_input_handler(ask_see_intro, 
    				"Do you want to display the intro? (y/n) ", 0);
   
   // Now we add on the talent one, if applicable
   if ((the_dbase->has_talents()) &&
       (the_race->get_allow_talent() != NULL))
   {
     the_player->load_talent_list();
     tmp_handler->push_input_handler(ask_talents, 
		     "Which talents would you prefer? (q when done) ", 0);
   }

   // Now add the inclinations, if applicable
   if ((the_dbase->has_inclinations()) && 
	 (the_race->get_allow_incl() != NULL))
   {
     tmp_handler->push_input_handler(ask_inclinations, 
					"Which inclination would you prefer? ", 0);
   }

   // now find the first tutorial and send them into the tutorial handler
   first_tutorial = the_race->get_next_tutorial(NULL);

   if (first_tutorial.str_show() != NULL)
   {
     // we mark which tutorial they are in  
     the_player->add_status("currenttutorial", first_tutorial.str_show());

     // If the tutorial is mandatory, just run it
     if (the_race->get_tutorial_mandatory(first_tutorial.str_show()))
     {
       return the_race->start_tutorial(first_tutorial.str_show(), the_player);
     }

     // Otherwise, replace the input handler with the question to run it or not
     tmp_handler->push_input_handler(handle_tutorial, 
                  the_race->get_tutorial_prompt(first_tutorial.str_show()), 0);
   }

   // display the list for the next thing displayed
   if (tmp_handler->get_command() == ask_inclinations)
     mainstruct->display_inclinations(the_player);
   else if (tmp_handler->get_command() == ask_talents)
     the_player->display_talents();
   

   the_player->send_plr("\n");
   return 1;
}



/***********************************************************************
 ** handle_tutorial - handles asking if the player wants to enter the
 **                   tutorial and sends them there if they choose to do so
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int handle_tutorial(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Strings     holder;
   Strings     cur_tutorial, next_tutorial;
   Race        *the_race;
   Object_List *the_dbase;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nPlease enter either yes or no.\n");
      return 0;
   }

   cur_tutorial = the_player->get_status_string("currenttutorial");
   if (cur_tutorial.str_show() == NULL)
      return 0;

   if ((the_race = mainstruct->get_race(the_player->get_race())) == NULL)
   {
      holder.sprintf("Player %s's race %s doesn't seem to exist.", 
                  the_player->get_name(), the_player->get_race());
      mainstruct->log_error(holder.str_show(), "handle_tutorial");
      return -1;
   }
   
   if (player_input.str_n_cmp("yes", player_input.str_len()))
   { 
      tmp_handler->pop_input_handler();
      return the_race->start_tutorial(cur_tutorial.str_show(), the_player);
   }
   else if (player_input.str_n_cmp("no", player_input.str_len()))
   {
      next_tutorial = the_race->get_next_tutorial(cur_tutorial.str_show());

      the_dbase = mainstruct->get_dbase();
      if (next_tutorial.str_show() == NULL)
      {
	tmp_handler->pop_input_handler();
	
	// display the list for the next thing displayed
        if (tmp_handler->get_command() == ask_inclinations)
	  mainstruct->display_inclinations(the_player);
	else if (tmp_handler->get_command() == ask_talents)
	  the_player->display_talents();

	the_player->send_plr("\n");
	return 0;
      }
     
      /* we mark which tutorial they are in */
      the_player->remove_status("currenttutorial");
      the_player->add_status("currenttutorial", next_tutorial.str_show());

      if (!the_race->get_tutorial_mandatory(next_tutorial.str_show()))
      {
         tmp_handler->replace_input_handler(handle_tutorial, 
                     the_race->get_tutorial_prompt(next_tutorial.str_show()), 0);
      }
      else
      {
	 tmp_handler->pop_input_handler();
	 the_race->start_tutorial(next_tutorial.str_show(), the_player);
	 
      }
      the_player->send_plr("\n");
   }
   else
   {
      the_player->send_plr("That is not a valid choice.\n");
      return 0;
   }

   return 1;
}


/***********************************************************************
 ** ask_inclinations - assigns an inclination to the player
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int ask_inclinations(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Strings     the_password;
   Object_List *obj_list;
   Inclination *the_inc;
   int         show_desc = 0;
   Strings     the_word;
   Connection  *the_connection;
   Race        *plr_race;
   Strings     allow_incl;
   Strings     holder;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   if ((plr_race = mainstruct->get_race(the_player->get_race())) == NULL)
   {
     holder.sprintf("Could not get player race '%s'", the_player->get_race());
     mainstruct->log_error(holder.str_show(), "ask_inclinations");
     return -1;
   }

   allow_incl = plr_race->get_allow_incl();
   
   player_input = the_input;
   player_input.remove_newline();

   the_word.assign_word(player_input.str_show(), 1);
   if ((the_word.str_show() != NULL) && 
       (!STRCASECMP("?", the_word.str_show())))
   {
      the_word.assign_word(player_input.str_show(), 2);
      if (the_word.str_show() == NULL)
      { 
         the_player->send_plr("format: ? <inclination>\n");
         return 0;
      }
      show_desc = 1;
   }

   obj_list = mainstruct->get_dbase();

   if (((the_inc = obj_list->get_inclination_obj(the_word.str_show())) == NULL) ||
       ((STRCASECMP(allow_incl.str_show(), "all")) && 
	(!allow_incl.find_in_str(the_inc->get_name()))))
   {
      the_player->send_plr("I am not aware of the '%s' inclination.\n"
                  "Please select again.\n", the_word.str_show());
      return 0;
   }

   the_connection = the_player->get_connection();

   if (show_desc)
   {
      the_connection->start_paging();
      the_player->send_plr("%s\n", the_inc->get_description());
      the_connection->end_paging(the_player);   
      return 1;
   }

   the_player->send_plr("\nYou have selected the '&+y%s&*' inclination.\n", 
                                               the_inc->get_name());

   the_inc->set_values(the_player);

   tmp_handler->pop_input_handler();

   if (tmp_handler->get_command() == ask_talents)
	  the_player->display_talents();

   the_player->send_plr("\n");
   return 1;
   
}


/***********************************************************************
 ** ask_talents - assigns talents to the player
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int ask_talents(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Area_Dbase  *talent_list;
   Talent      *the_talent;
   int         show_desc = 0;
   Strings     the_word;
   Connection  *the_connection;
   Race        *plr_race;
   Strings     allow_talent;
   Strings     holder;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   if ((plr_race = mainstruct->get_race(the_player->get_race())) == NULL)
   {
     holder.sprintf("Could not get player race '%s'", the_player->get_race());
     mainstruct->log_error(holder.str_show(), "ask_talents");
     return -1;
   }

   allow_talent = plr_race->get_allow_talent();

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   the_word.assign_word(player_input.str_show(), 1);
   if(    the_word.str_show() == NULL
       || ( the_word.str_show() != NULL
          && !STRCASECMP( "?", the_word.str_show( ) ) ) )
   {
      the_word.assign_word(player_input.str_show(), 2);
      if (the_word.str_show() == NULL)
      { 
         the_player->send_plr("format: ? <talent>\n");
         return 0;
      }
      show_desc = 1;
   }

   if (the_word.str_show() == NULL)
   {
     the_player->send_plr(
               "You must specify either 'q', '? <talent>' or a talent.\n");
     return 0;
   }

   if (!STRCASECMP(player_input.str_show(), "q"))
   {
      the_player->delete_talent_list();
      tmp_handler->pop_input_handler();
      the_player->send_plr("\n");
      return 1;      
   }

   talent_list = the_player->get_talent_list();

   if (((the_talent = (Talent *) talent_list->
                              get_areaobject(the_word.str_show())) == NULL) ||
       ((STRCASECMP(allow_talent.str_show(), "all")) && 
	(!allow_talent.find_in_str(the_talent->get_name()))))
   {
      the_player->send_plr("I am not aware of the '%s' talent.\n"
                             "Please select again.\n", the_word.str_show());
      return 0;
   }

   the_connection = the_player->get_connection();
   if (show_desc)
   {
      the_connection->start_paging();
      the_player->send_plr("%s\n", the_talent->get_description());
      the_connection->end_paging(the_player);   
      return 1;
   }

   if (the_talent->get_cost() > the_player->get_talent_funds())
   {
      the_player->send_plr(
                  "You don't have enough funds left to get that talent.\n");
      return 0;
   }

   /* run the special for the talent */
   the_talent->assign_talent(the_player);

   the_player->send_plr("\nYou have been granted the '&+y%s&*' trait.\n", 
                                                      the_talent->get_name());

   /* now make sure they can't do the talent again if it has run out */
   if (the_talent->get_allowed() > 1)
      the_talent->set_allowed(the_talent->get_allowed() - 1);
   else
      talent_list->delete_areaobject(the_talent);

   the_player->display_talents();
   
   return 1;
}


/***********************************************************************
 ** port_menu - allows selection of a port from the port menu
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int port_menu(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   int         show_desc = 0;
   Strings     the_word;
   Strings     the_file;
   Pager	   *the_pager;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   the_word.assign_word(player_input.str_show(), 1);

   if (the_word.str_show() == NULL)
   {
     the_player->send_plr(
               "You must specify either '1' for the game port or '2'\n"
			   "for the builder port.\n");
     return 0;
   }

   if (!STRCASECMP(player_input.str_show(), "1"))
   {
      /* first we push this on, so that when we pop the login banner, this
         input handler is waiting to be executed */
      tmp_handler->replace_input_handler(get_pname, "What shall I call you? ", 0);
      the_file.sprintf("%s/%s%s%s", the_config.basedir.str_show(), 
                                    BANNERDIR, "gamebanner", BANNEREXTENTION);

      /* now we push on the login banner */
      the_pager = new_Pager(the_file.str_show(), the_player->get_pager_lines());
      if (the_pager && the_pager->file_available())
	  { 
          if (the_pager->paged_read(the_player->get_connection()))
             delete_Pager(the_pager);
          else
		  {
             tmp_handler->push_input_handler(pager_funct, 
                "&+c<&+G******************&+Bpress return to continue, "
                "q to quit&+G**********************&+c>&*", HANDLER_DATA_PAGER);
             tmp_handler->set_data((void *) the_pager);
		  }
	  }    
      else
	  {
         the_player->send_plr("Error displaying login banner.\n");
         mainstruct->log_error("Error displaying login banner.\n", 
                                                         "login_player");
	  }
      
      return 1;      
   }
   else if (!STRCASECMP(player_input.str_show(), "2"))
   {
	  tmp_handler->pop_input_handler(); 
      mainstruct->switch_for_builder(the_player);
	  return 1;
   }
   else
   {
		the_player->send_plr("That is not a valid choice.\n");
		return 0;
   }


   tmp_handler->replace_input_handler(ask_talents, 
                     "Which talents would you prefer? (q when done) ", 0);
   
   return 1;
}


/***********************************************************************
 ** ask_see_intro - finds out if the user wants to display the intro
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int ask_see_intro(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Strings     the_file;
   Pager       *the_pager;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
      return -1;
   }


   if (!STRNCASECMP("yes", player_input.str_show(), player_input.str_len()))
   {
      the_file.sprintf("%s/%s%s%s", the_config.basedir.str_show(), 
                                        BANNERDIR, "intro", BANNEREXTENTION);

      /* we push on the login banner */
      the_pager = 
             new_Pager(the_file.str_show(), the_player->get_pager_lines());
      if (the_pager && the_pager->file_available())
      {  
         if (the_pager->paged_read(the_player->get_connection()))
	 {
            delete_Pager(the_pager);
            tmp_handler->pop_input_handler();
            enter_new_player(the_player->get_input_handler(), 
                             the_player->get_name(), NULL, NULL, NULL);
	 }
         else
         {
            tmp_handler->replace_input_handler(pager_funct, 
              "&+c<&+G******************&+Bpress return to continue, "
              "q to quit&+G**********************&+c>&*", HANDLER_DATA_PAGER);
            tmp_handler->set_data((void *) the_pager);
            tmp_handler->add_pop_function(enter_new_player, 
                                  the_player->get_name(), NULL, NULL, NULL);
         }
  
      }    
      else
      {
         the_player->send_plr("Error displaying login banner.\n");
         mainstruct->log_error("Error displaying login banner.\n", 
                                                         "ask_see_intro");
         tmp_handler->pop_input_handler();
         enter_new_player(the_player->get_input_handler(), 
                          the_player->get_name(), NULL, NULL, NULL);
      }
      return 1; 
   }
   else if (!STRNCASECMP("no", player_input.str_show(), 
                                                    player_input.str_len()))
   {
      tmp_handler->pop_input_handler();
      enter_new_player(the_player->get_input_handler(), 
                                 the_player->get_name(), NULL, NULL, NULL);
      return 1; 
   }
   else
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
      return -1;
   }


   return -1;
}

/***********************************************************************
 ** pager_funct - allows for the user to "hit enter to continue" when in
 **               the pager
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int pager_funct(MudObject *the_user, char *the_input)
{
   Player      *the_player = NULL;
   Builder     *the_builder = NULL;
   Strings     user_input;
   Inp_Handler *tmp_handler;
   Connection  *user_connection;
   Pager       *this_pager;

   if (the_user->get_type() == OBJ_TYPE_PLAYER)
   {
      the_player = (Player *) the_user;
      tmp_handler = the_player->get_input_handler();
      user_connection = the_player->get_connection();
   }
   else if (the_user->get_type() == OBJ_TYPE_BUILDER)
   {
      the_builder = (Builder *) the_user;
      tmp_handler = the_builder->get_input_handler();
      user_connection = the_builder->get_connection();
   }
   else
      return -1;

   user_input = the_input;
   user_input.remove_newline();

   this_pager = (Pager *) tmp_handler->get_data();
   if (user_input.str_len() < 0)
      return -1;

   if ((user_input.str_len() == 0) ||
       (!user_input.str_n_cmp("quit", user_input.str_len())))
   {
      if (this_pager->paged_read(user_connection))
      {
         tmp_handler->pop_input_handler();
         return 1;         
      }
      return 0;
   }
 
   tmp_handler->pop_input_handler();
   return 1;
}


/***********************************************************************
 ** delete_yes_no - for builder port, question on whether we should delete
 **                 or not
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int delete_yes_no(MudObject *the_user, char *the_input)
{
   Strings     tmp_word;
   Inp_Handler *this_handler;
   Builder     *the_builder;
   FILE        *delfile = NULL;
   Entity      *the_object;
   Area_Dbase  *the_dbase;
   Timespan    the_time;
   Strings     del_filename;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return -1;

   the_builder = (Builder *) the_user;
   this_handler = the_builder->get_input_handler();

   if ((the_object = the_builder->get_modify()) == NULL)
   {
      the_builder->send_bldr("Blargh!! Code Error!!\n");
      this_handler->pop_input_handler();
      return -1;
   }

   if ((the_dbase = the_builder->get_area()) == NULL)
   {
      the_builder->send_bldr("Unexpected Error encountered. Failed.\n");
      this_handler->pop_input_handler();
      return -1;
   }

   tmp_word.assign_word(the_input, 1);
     
   if (tmp_word.str_n_cmp("no", tmp_word.str_len()))
   {
      the_builder->send_bldr("Not deleted.\n");
      this_handler->pop_input_handler();
      return 0;
   }
   else if (!tmp_word.str_n_cmp("yes", tmp_word.str_len()))
   {
      the_builder->send_bldr("Please answer yes or no.\n");
      return 0;
   }

   del_filename.sprintf("%s/%s", the_config.basedir.str_show(), DELFILE);
   if ((delfile = xfopen(del_filename.str_show(), "a", "delete_file")) == NULL)
   {
      the_builder->send_bldr("Error opening deleted objects file. "
                             "Not deleted\n");
      return -1;
   }
      
   fprintf(delfile, "\n\nArea = %s\n", the_builder->get_areaname());

   the_object->write_object(delfile, 0);

   xfclose(delfile, "delete_file");

   the_builder->clr_modify();

   if (the_dbase->delete_areaobject(the_object) <= 0)
   {
      the_builder->send_bldr("Error, could not delete object.\n");
      return -1;
   }
   the_builder->send_bldr("Deleted.\n");
   this_handler->pop_input_handler();
   return 1;                              
}


/***********************************************************************
 ** kick_off_player - kicks off the player and logs in as them
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int kick_off_player(MudObject *the_user, char *the_input) 
{
   Player      *the_player;
   Strings     player_input;
   Inp_Handler *tmp_handler;
   Player      *remove_plr;
   Location    *loc_holder;
   Strings     holder;
   Flags       *gameflags;

   if (the_user->get_type() != OBJ_TYPE_PLAYER)
      return -1;

   the_player = (Player *) the_user;

   tmp_handler = the_player->get_input_handler();

   player_input = the_input;
   player_input.remove_newline();

   if (player_input.str_len() <= 0)
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
      return -1;
   }

   if (!STRNCASECMP("yes", player_input.str_show(), player_input.str_len()))
   {

      if ((remove_plr = mainstruct->get_player(the_player->get_title())) 
                                                                  == NULL)
      {
         the_player->send_plr("Fatal error, can't remove.\n");
         the_player->set_off();
         return -1; 
      }
      
      the_player->send_plr("Replacing character with you.\n");
      the_player->flush_player();
      remove_plr->send_plr("You have been kicked off by another login!\n");
      remove_plr->flush_player();

      remove_plr->swap_connection(the_player);

      loc_holder = remove_plr->get_loc();
      holder.sprintf("%s &+Greplaced&* by login at %s.", 
                remove_plr->get_title(), loc_holder->get_title());
      mainstruct->log_event(holder.str_show());

      holder.sprintf("&+Y[&+M%s&+W replaced by login at &+c%s&+Y]&*\n", 
                           remove_plr->get_title(), loc_holder->get_title());
      mainstruct->send_all_players(holder.str_show(), remove_plr, 
                                                 ADMINFLAG_SEELOGIN);
      gameflags = remove_plr->get_gameflags();
      loc_holder->show_location(remove_plr, 
                                        gameflags->get_flag(GAMEFLAG_BRIEF));
      remove_plr->display_prompt();

      the_player->set_off();
      return 1; 
   }
   else if (!STRNCASECMP("no", player_input.str_show(), 
                                                    player_input.str_len()))
   {
      the_player->set_initname();
      tmp_handler->replace_input_handler(get_pname, 
                                          "What shall I call you? ", 0);
      return 1; 
   }
   else
   {
      the_player->send_plr("\nPlease select yes or no.\n\n");
   }
   return -1;
}


/***********************************************************************
 ** kick_off_builder - kicks off the builder and logs in as them
 **
 ** returns:  1 for success, -1 for failure
 **
 ***********************************************************************/
int kick_off_builder(MudObject *the_user, char *the_input) 
{
   Builder *the_builder;
   Strings builder_input;
   Inp_Handler *tmp_handler;
   Builder *remove_bldr;
   Strings  holder;

   if (the_user->get_type() != OBJ_TYPE_BUILDER)
      return -1;

   the_builder = (Builder *) the_user;

   tmp_handler = the_builder->get_input_handler();

   builder_input = the_input;
   builder_input.remove_newline();

   if (builder_input.str_len() <= 0)
   {
      the_builder->send_bldr("\nPlease select yes or no.\n\n");
      return -1;
   }

   if (!STRNCASECMP("yes", builder_input.str_show(), 
                                                  builder_input.str_len()))
   {

      if ((remove_bldr = mainstruct->get_builder(the_builder->get_title())) 
                                                                  == NULL)
      {
         the_builder->send_bldr("Fatal error, can't remove.\n");
         the_builder->set_off();
         return -1; 
      }
      
      the_builder->send_bldr("Replacing builder with you.\n");
      the_builder->flush_builder();
      remove_bldr->send_bldr("You have been kicked off by another login!\n");
      remove_bldr->flush_builder();

      remove_bldr->swap_connection(the_builder);

      holder.sprintf("%s &+Greplaced&* by login at builder port.", 
                                                remove_bldr->get_title());
      mainstruct->log_event(holder.str_show());

      holder.sprintf("&+Y[&+M%s&+W replaced by login at builder port&+Y]&*\n",
                                              remove_bldr->get_title());
      mainstruct->send_all_players(holder.str_show(), NULL, 
                                                 ADMINFLAG_SEELOGIN);

      remove_bldr->display_prompt();

      the_builder->set_off();
      return 1; 
   }
   else if (!STRNCASECMP("no", builder_input.str_show(), 
                                                    builder_input.str_len()))
   {
      tmp_handler->replace_input_handler(get_bname, 
                                          "What shall I call you? ", 0);
      return 1; 
   }
   else
   {
      the_builder->send_bldr("\nPlease select yes or no.\n\n");
   }
   return -1;
}



#endif







