// ---------------------------------------------------------------------------
// - InputString.cpp                                                         -
// - standard object library - string stream class implementation            -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Character.hpp"
#include "InputString.hpp"

namespace aleph {

  // the input file supported quarks
  static const long QUARK_GET = String::intern ("get");
  static const long QUARK_SET = String::intern ("set");

  // create an empty string stream

  InputString::InputString (void)  {
    d_buffer.reset ();
  }

  // create a string stream with a buffer

  InputString::InputString (const String& data)  {
    d_buffer.reset    ();
    d_buffer.pushback (data);
  }

  // return the class name

  String InputString::repr (void) const {
    return "InputString";
  }

  // get one byte but do not remove it

  char InputString::get (void) const {
    char c = d_buffer.get ();
    return (c == nilc) ?  eofc : c;
    
  }
  
  // read one byte from the input stream.

  char InputString::read (void) {
    char c = d_buffer.read ();
    return (c == nilc) ? eofc : c;
  }

  // return true if we are at the eof

  bool InputString::iseof (void) const {
    return (d_buffer.length () == 0);
  }

  // return true if we can read a character

  bool InputString::valid (const long) const {
    return (d_buffer.length () != 0);
  }

  // set the stream with a string, reset the input buffer

  void InputString::set (const String& line) {
    d_buffer.reset    ();
    d_buffer.pushback (line);
  }

  // create a new input string in a generic way

  Object* InputString::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new InputString;

    // check for 1 argument
    if (argc == 1) {
      String sval = argv->getstring (0);
      return new InputString (sval);
    }
    throw Exception ("argument-error", 
		     "invalid arguments with with input string");
  }

  // apply this input string with a set of arguments and a quark

  Object* InputString::apply (Runnable* robj, Nameset* nset, const long quark,
			      Vector* argv) {

    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (quark == QUARK_GET) return new Character (get ());

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SET) {
	String val = argv->getstring (0);
	set (val);
	return nilp;
      }
    }    
    // call the input method
    return Input::apply (robj, nset, quark, argv);
  }
}
