/* $Id: ArkGraph.h,v 1.7 2002/12/26 20:32:19 mrq Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_GRAPH_H
#define ARK_GRAPH_H

#include <Ark/ArkMath.h>
#include <Ark/ArkLexer.h>

#include <map>
#include <vector>
#include <functional>

namespace Ark
{
   /// A gross graph class for pathfinding. It uses the A* algorithm without 
   /// the propagate down step, which is good enough for our needs but won't
   /// always find the shortest path.
   class Graph
   {
	 friend class Pathfinder;
	 friend void DrawGraph(Graph *g);

      public:
	 typedef unsigned short int NodeId;
	 typedef int Cost;

	 typedef std::vector<NodeId> Path;

	 struct Edge
	 {
	       NodeId m_Succ;
	       Cost m_Cost;
	 };

      protected:
	 typedef std::vector<Edge> EdgeList;

	 std::vector<EdgeList> m_Edges;
	 std::vector<Vector3> m_Positions;
	 int m_MinCostPerMeter;

      public:
	 Graph();
	 ~Graph();

	 void Read (const String &file);
	 void Read (Lexer &lexer);

	 void GetSuccessors(NodeId a, EdgeList const **  succ) const;
	 Cost EstimateCost(NodeId a, NodeId b) const;

	 Vector3 Position(NodeId a) {return m_Positions[a];}
	 int NumNodes() {return m_Positions.size();}
   };

   class Pathfinder
   {
      public:
	 Pathfinder(const Graph *graph);
	 ~Pathfinder();

	 void FindPath(Graph::NodeId a, Graph::NodeId b, Graph::Path &path);

     public:
	 enum { NULL_NODE = 0xFFFF };

	 struct PFNode
	 {
	       Graph::NodeId m_Node;

	       // g is the cost from the start to here,
	       // h the estimation of the cost from the start to the end
	       // passing by this node.
	       Graph::Cost m_g, m_h;
	 };
	 typedef std::greater<PFNode> Comp;

	 struct Marking
	 {
	       Graph::NodeId m_Previous;
	       Graph::Cost m_f, m_g;
	       Marking(): m_Previous (NULL_NODE),m_f(-1),m_g(-1) {}
	 };

	 typedef std::vector<PFNode> PFNodes;
	 typedef std::vector<Marking> MarkArray;

     private:
	 PFNodes m_Open, m_Visited;
	 MarkArray m_Mark;

	 PFNodes::iterator find_in_open(Graph::NodeId n);
	 inline bool in_open (Graph::NodeId n)
	 {
	    return m_Mark[(size_t)n].m_f != -1;
	 }

	 Graph::NodeId m_EndNode;
	 const Graph *m_Graph;
	 
   };
   
   inline bool operator == (const Pathfinder::PFNode &a,
		     const Pathfinder::PFNode &b)
      {return a.m_h == b.m_h && a.m_g == b.m_g;}

   inline bool operator < (const Pathfinder::PFNode &a,
		    const Pathfinder::PFNode &b)
      {return (a.m_h + a.m_g) < (b.m_h + b.m_g);}

   inline bool operator > (const Pathfinder::PFNode &a,
		    const Pathfinder::PFNode &b)
      {return (a.m_h + a.m_g) > (b.m_h + b.m_g);}

};

#endif
