/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	Class implementations for audio
 *	writing classes.
 *
 *	by Tony Sideris	(02:56AM Aug 11, 2001)
 *================================================*/
#include "arson.h"

#include <fstream>
#include <utility>

#include <kregexp.h>
#include <klocale.h>

#include "audiowriter.h"
#include "audiofile.h"
#include "tempfile.h"
#include "audiodoc.h"
#include "konfig.h"
#include "wavinfo.h"
#include "process.h"

/*========================================================*/
/*	Track class impl
 *========================================================*/

ArsonAudioWriter::Track::Track (const QString &path)
	: m_filename(path), m_length(0), m_pFifo(NULL)
{
}

ArsonAudioWriter::Track::Track (const Track &track)
	: m_pFifo(NULL)
{
	(*this) = track;
}

ArsonAudioWriter::Track &ArsonAudioWriter::Track::operator= (const Track &track)
{
	m_filename = track.m_filename;
	m_decoded = track.m_decoded;
	m_length = track.m_length;

	if (track.m_pFifo)
		m_pFifo = new ArsonFifo(*(track.m_pFifo));

	return *this;
}

ArsonAudioWriter::Track::~Track (void)
{
	//	Nothing...
}

/*========================================================*/

const QString &ArsonAudioWriter::Track::decodedPath (void) const
{
	if (m_decoded.isEmpty())
	{
		ArsonAudioFile af (filename());

		((ArsonAudioWriter::Track *) this)->m_decoded = af.decodedFile();
	}

	return m_decoded;
}

uint ArsonAudioWriter::Track::lengthInSeconds (void) const
{
	if (!m_length)
	{
		ArsonAudioFile af (filename());

		((ArsonAudioWriter::Track *) this)->m_length = af.length();
	}

	return m_length;
}

/*========================================================*/

bool ArsonAudioWriter::Track::createFifo (void)
{
	ArsonFifo *ptr = new ArsonFifo(decodedPath());

	if (ptr->valid())
	{
		ptr->setAutoDelete(false);
		m_pFifo = ptr;
	}
	else
		delete ptr;

	return (m_pFifo != NULL);
}

void ArsonAudioWriter::Track::deleteFifo (void)
{
	if (m_pFifo)
	{
		m_pFifo->unlink();

		delete m_pFifo;
		m_pFifo = NULL;
	}
}

/*========================================================*/

uint ArsonAudioWriter::Track::estimatedFileSize (void) const
{
	const uint size = QFileInfo(decodedPath()).size();

	if (size > 0)
		return size;
	
	/*	Total byte length of decoded track:
	 *	SAMPPERSEC * CHANS * (BITSPERSAMP / BITSPERBYTE) * LENGTHINSEC
	 */
	return (44100 * (2 * (16 / 8))) * lengthInSeconds();
}

bool ArsonAudioWriter::Track::burnable (void) const
{
	QFileInfo fi (decodedPath());

	if (fi.extension(false).lower() == "wav")
	{
		std::ifstream fin (decodedPath());
		bool result = false;

		if (fin)
		{
			wav::stdhdr *hdr = new wav::stdhdr(fin);

			if (hdr->valid())
			{
				const wav::fmt &fhdr = hdr->format();

				result = (fhdr.channels == 2 &&
					fhdr.samplesPerSec == 44100 &&
					fhdr.bitsPerSample == 16);

#ifdef ARSONDBG
				if (!result)
					Trace("bad wav file (%d %d %d)\n",
						fhdr.channels, fhdr.samplesPerSec,
						fhdr.bitsPerSample);
#endif
			}

			delete hdr;
		}

		return result;
	}

	return true;
}

/*========================================================*/
/*	Base class audio writers, provides common operations
 *	for burning audio tracks.
 *========================================================*/

ArsonAudioWriter::ArsonAudioWriter (ArsonProcessUI *pUI)
	: ArsonCdWriter(pUI)
{
	reset();
}

/*========================================================*/

void ArsonAudioWriter::reset (void)
{
	m_currentTrack = -1;
	m_flags = 0;
}

/*========================================================*/
/*	Err... This should actually be totalSeconds()
 *	but whatever...
 *========================================================*/

uint ArsonAudioWriter::totalMinutes (void) const
{
	uint index, total;

	for (total = index = 0; index < m_tracks.count(); ++index)
		total += track(index).lengthInSeconds();

	return total;
}

void ArsonAudioWriter::addTrack (ArsonFileListFileItem *pi)
{
	try {
		Track track (pi->local());
		m_tracks.append(track);
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/

bool ArsonAudioWriter::successful (ArsonProcess *ptr)
{
	int code = 0;
	const bool success = ptr->successful(&code);

	if (!success)
	{
		Trace("Process failed with code %d (normalized=%d, written=%d)\n",
			code, normalized(), written());

		/*	Normalize can fail with error code 2 if the
		 *	given track is already normalized, this is
		 *	not a fatal error, and we should not abort
		 *	because of it.
		 */
		return (code == 2 && normalized() && !written());
	}

	return true;
}

/*========================================================*/
/*	If the user aborts the burn during the processing
 *	(deocoding or fixing) of a track then remove the
 *	cached decoded track because it is not yet "valid."
 *========================================================*/

void ArsonAudioWriter::abort (void)
{
	Trace("abort: currentTrack=%d\n", m_currentTrack);
	
	if (m_currentTrack < trackCount())
		ArsonTempFile::deleteTemporaryFor(
			track(m_currentTrack).filename());

	ArsonCdWriter::abort();
}

/*========================================================*/
/*	Determines shich task to start
 *	next (decode/normalize/burn)
 *========================================================*/

void ArsonAudioWriter::nextTask (void)
{
	/*	The process:
	 *	1. Decode all encoded audio tracks (backup the WAVs to tmpdir)
	 *	2. Check each track to verify 44100Khz/16bps/stereo, run sox on broken tracks
	 *	3. Normalize all tracks if option is set
	 *	4. Burn the CD
	 */
	if (!decodeAudio() && !checkAudio())
	{
		m_pUI->setMaxProgress(0);

		if (!normalized() && opts().getLong(optNormal) != NORMALIZE_NONE)
			normalize();

		else if (!written())
			writeCd();
	}
}

/*========================================================*/
/*	Checks decoded tracks to see if they're burnable
 *========================================================*/

bool ArsonAudioWriter::checkAudio (void)
{
	//	Check each track for the proper format (44100/16/2)
	for (++m_currentTrack; m_currentTrack < trackCount(); ++m_currentTrack)
	{
		const Track &tr = track(m_currentTrack);

		if (!tr.burnable())
		{
			//	Run SOX on file to fix or whatever
			try
			{
				//	Clue the user in if sox is not installed
				if (!ACONFIG.program(ArsonConfig::PROGRAM_SOX)->valid())
					throw ArsonError(
						i18n("The track %1 is not valid (must be 44100Khz, 16bits/sample, stereo), and 'sox' is not available. Install 'sox' to enable arson to autofix invalid tracks such as these.")
						.arg(tr.filename()));

				//	Run sox on the broken track
				setProcess(
					new ArsonSoxProcess(this, tr.decodedPath()));
			}
			catch (ArsonError &err) {
				err.report();
				setProcess(NULL);
			}

			return true;
		}
	}

	return false;
}

/*========================================================*/
/*	Decodes the next decodable audio track
 *========================================================*/

bool ArsonAudioWriter::decodeAudio (void)
{
	ArsonAudioDecoderProcess *ptr;

	if (!decoded() && (ptr = nextDecodableTrack()))
	{
		try
		{
			const Track &tr = track(m_currentTrack);

			setProcess(ptr);

			if (ArsonProcessUI *pUI = ui())
			{
				if (pUI->maxProgress() != -1)
					pUI->setProgress(0, tr.lengthInSeconds());

				pUI->setText(
					i18n("Decoding file %1 ...")
					.arg(tr.filename()));
			}
		}
		catch (ArsonError &err) {
			err.report();
		}

		return true;
	}

	setDecoded();
	return false;
}

/*========================================================*/
/*	Finds the next decodable audio track, and creates
 *	the decoder.
 *========================================================*/

ArsonAudioDecoderProcess *ArsonAudioWriter::nextDecodableTrack (void)
{
	for (++m_currentTrack; m_currentTrack < trackCount(); m_currentTrack++)
	{
		ArsonAudioFile af (track(m_currentTrack).filename());
		const QString wav = track(m_currentTrack).decodedPath();

		try {
			if (ArsonAudioDecoderProcess *ptr = createDecoder(
					m_currentTrack, af, wav))
				return ptr;
		}
		catch (ArsonError &err) {
			err.report();
		}
	}

	//	Nothing more to decode, reset or the check
	m_currentTrack = -1;
	return NULL;
}

/*========================================================*/

ArsonAudioDecoderProcess *ArsonAudioWriter::createDecoder(
	int trackNo, ArsonAudioFile &af, const QString &wav)
{
	return af.decoder(this, wav);
}

/*========================================================*/
/*	(Optionally) Even out the volumes of all tracks
 *========================================================*/

void ArsonAudioWriter::normalize (void)
{
	try
	{
		ArsonNormalizeProcess *ptr = new ArsonNormalizeProcess(this);

		for (int index = 0; index < trackCount(); index++)
			ptr->addTrack(track(index).decodedPath());

		m_pUI->setText(i18n("Normalizing audio tracks..."));
		m_pUI->setProgress(0, 100);

		setNormalized();
		setProcess(ptr);
	}
	catch (ArsonError &err) {
		err.report();
	}
}

/*========================================================*/
/*	Bruns audio tracks ing DAO mode, using 'cdrdao'
 *========================================================*/

ArsonDaoWriter::ArsonDaoWriter (ArsonProcessUI *pUI)
	: ArsonAudioWriter(pUI), m_pCdInfo(NULL),
	m_pTOC(NULL)
{
	//	Nothing...
}

ArsonDaoWriter::~ArsonDaoWriter (void)
{
	delete m_pTOC;
}

/*========================================================*/
/*	Burns all tracks.
 *========================================================*/

ArsonWriterProcess *ArsonDaoWriter::createWriterProcess (void)
{
	if (!m_pTOC)
	{
		m_pTOC = new ArsonTocFile;
		m_pTOC->setCdInfo(m_pCdInfo);

		//	Fill the TOC file object with all tracks
		for (int index = 0; index < trackCount(); ++index)
			m_pTOC->addFile(track(index).decodedPath());

		//	Write the TOC file
		if (!m_pTOC->writeFile())
		{
			throw ArsonError(
				i18n("Failed to write TOC file: %1")
				.arg(m_pTOC->errorString()));
		}
	}

	const uint maxp = m_pUI->maxProgress();
	ArsonWriterProcess *ptr = new ArsonCdrdaoProcess(
		this, m_pTOC->filename());

	m_pUI->setMaxProgress(maxp);

	setWritten();
	return ptr;
}

/*========================================================*/

void ArsonDaoWriter::begin (const ArsonProcessOpts &opts)
{
	ArsonAudioWriter::begin(opts);

	nextTask();
}

/*========================================================*/

void ArsonDaoWriter::taskComplete (ArsonProcess *ptr)
{
	const bool success = successful(ptr);

	ArsonCdWriter::taskComplete(ptr);

	if (success)
		nextTask();
}

/*========================================================*/
/*	Burns audio tracks in TAO mode using 'cdrecord'.
 *	If 'On the Fly' is selected then all decoder output
 *	goes to a named pipe. Otherwise the normal decoder
 *	(async) loop is done.
 *========================================================*/

ArsonTaoWriter::ArsonTaoWriter (ArsonProcessUI *pUI)
	: ArsonAudioWriter(pUI)
{
	//	Nothing...
}

/*========================================================*/

bool ArsonTaoWriter::setupFifos (void)
{
	ArsonAudioDecoderProcess *ptr;

	while ((ptr = nextDecodableTrack()))
	{
		if (!ptr->execute())
		{
			Trace("FAILED TO EXECUTE DECODER\n");
			return false;
		}
	}

	return true;
}

/*========================================================*/

#define ON_THE_FLY		(opts().getBool(optOnTheFly))

void ArsonTaoWriter::begin (const ArsonProcessOpts &o)
{
	ArsonAudioWriter::begin(o);

	if (ON_THE_FLY)
		writeCd();
	else
		nextTask();
}

void ArsonTaoWriter::abort (void)
{
	deleteFifos();
	ArsonAudioWriter::abort();
}

/*========================================================*/

ArsonAudioDecoderProcess *ArsonTaoWriter::createDecoder(
	int trackNo, ArsonAudioFile &af, const QString &wav)
{
	ArsonAudioDecoderProcess *ptr = NULL;

	if ((ptr = ArsonAudioWriter::createDecoder(trackNo, af, wav)) && ON_THE_FLY)
	{
		if (!track(trackNo).createFifo())
			throw ArsonError(
				i18n("Failed to create FIFO %1").arg(wav));
	}

	return ptr;
}

/*========================================================*/

ArsonWriterProcess *ArsonTaoWriter::createWriterProcess (void)
{
	uint total = 0;
	ArsonCdrecordAudioProcess *ptr = new ArsonCdrecordAudioProcess(this);

	if (ON_THE_FLY && !setupFifos())
		return NULL;

	for (int index = 0; index < trackCount(); ++index)
	{
		ptr->addTrack(track(index).decodedPath());
		total += track(index).estimatedFileSize();
	}

	m_pUI->setProgress(0,
		total / (1024 * 1024));

	setWritten();
	return ptr;
}

/*========================================================*/

void ArsonTaoWriter::deleteFifos (void)
{
	for (int index = 0; index < trackCount(); ++index)
		track(index).deleteFifo();
}

/*========================================================*/

void ArsonTaoWriter::taskComplete (ArsonProcess *ptr)
{
	const bool success = successful(ptr);

	/*	Reset the decoder loop state variables
	 *	so that the Simulate/Burn feature works,
	 *	also delete a FIFOs.
	 */
	if (ON_THE_FLY && ptr == process())
	{
		deleteFifos();
		reset();
	}

	ArsonAudioWriter::taskComplete(ptr);

	//	Continue the async decode loop
	if (success && !ON_THE_FLY)
		nextTask();
}

/*========================================================*/
