/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
#ifndef __MP3INFO_H__
#define __MP3INFO_H__

#include <stdio.h>
#include <string>
#include <map>

/*========================================================*/

namespace mp3
{
	typedef unsigned short ushort;
	typedef unsigned long ulong;
	typedef unsigned char byte;
	typedef unsigned short tbl_bitrate[16];
	typedef ulong tbl_samplerate[4];

	struct BITINFO {
		BITINFO (int _bit, int _size)
			: bit(_bit), size(_size) {}

		inline ulong getValue (ulong hdr) const
		{	return (hdr >> bit) & ((1 << size) - 1); }
		inline const char *boolString (ulong hdr) const
		{	return getValue(hdr) ? "yes" : "no"; }

		ulong bit : 16;
		ulong size : 16;
	};

	const BITINFO biFrameSync		= BITINFO(21,11);
	const BITINFO biVersion			= BITINFO(19,2);
	const BITINFO biLayer			= BITINFO(17,2);
	const BITINFO biNoCRC			= BITINFO(16,1);
	const BITINFO biBitrateIndex	= BITINFO(12,4);
	const BITINFO biSampleIndex		= BITINFO(10,2);
	const BITINFO biPadding			= BITINFO(9,1);
	const BITINFO biPrivate			= BITINFO(8,1);
	const BITINFO biChannel			= BITINFO(6,2);
	const BITINFO biModeExt			= BITINFO(4,2);
	const BITINFO biCopyright		= BITINFO(3,1);
	const BITINFO biOriginal		= BITINFO(2,1);
	const BITINFO biEmphasis		= BITINFO(0,2);

	/*========================================================*/

	class Header
	{
	public:
		Header (void) : m_data(0) {}

		static tbl_samplerate sampleRateTbl[4];
		static tbl_bitrate bitRateTbl[5];

		enum { STEREO, JSTEREO, DUALCHAN, SINGCHAN };	//	Channel modes
		enum { NONE, MS5015, RESV, CCITJ715 };			//	Emphsis
		enum { V25, V0, V2, V1 };						//	Mpeg version IDs
		enum { L0, L3, L2, L1 };						//	Layer IDs

		bool find (byte *buf, ulong size, ulong *ptr);
		ulong bitrate (void) const;
		void load (byte *buf);
		bool valid (void) const;

		inline ulong binary (void) const
		{	return m_data; }
		inline ulong pos (void) const
		{	return m_hdrpos; }
		inline void reset (void)
		{	m_data = 0; }

		inline ulong samplerate (void) const
		{	return sampleRateTbl[biVersion.getValue(m_data)][biSampleIndex.getValue(m_data)];	}

		inline ulong framelength (void) const
		{
			return (biLayer.getValue(m_data) == L1)
				? (12 * (bitrate() * 1000) / samplerate() + biPadding.getValue(m_data)) * 4
				: 144 * (bitrate() * 1000) / samplerate() + biPadding.getValue(m_data);
		}

		inline const char *versionString (void) const {
			const char *sz[] =
			{ "2.5", "(unknown)", "2.0", "1.0" };

			return sz[version()];
		}

		inline ulong version (void) const
		{	return biVersion.getValue(m_data); }

		inline const char *layerString (void) const {
			const char *sz[] =
			{ "(unknown)", "III", "II", "I" };

			return sz[layer()];
		}

		inline ulong layer (void) const
		{	return biLayer.getValue(m_data); }

		inline const char *channelmodeString (void) const {
			const char *sz[] =
			{ "Stereo", "Joint Stereo", "Dual channel", "Single channel" };

			return sz[channelmode()];
		}

		inline ulong channelmode (void) const
		{	return biChannel.getValue(m_data); }

		inline const char *emphasisString (void) const {
			const char *sz[] =
			{ "none", "50/15 ms", "reserved", "CCIT J.17" };

			return sz[emphasis()];
		}

		inline ulong emphasis (void) const
		{	return biEmphasis.getValue(m_data); }

		inline ulong frameSync (void) const
		{	return biFrameSync.getValue(m_data); }

	private:
		ulong m_data,
			m_hdrpos;
	};

	/*========================================================*/

	class Id3v2;
	class Vbr;
	class Tag;

	class Info
	{
	public:
		Info (void);
		~Info (void);

		bool load (const char *filename);
		ulong bitrate (void) const;
		ulong frames (void) const;

		inline ulong playtime (void) const
		{	return ((8 * m_filelen) / 1000) / bitrate();	}
		inline const Header &header (void) const
		{	return m_header; }
		inline ulong fileLength (void) const
		{	return m_filelen; }
		inline const Id3v2 *id3v2 (void) const
		{	return m_pId3; }
		inline const Vbr *vbr (void) const
		{	return m_pVBR; }
		inline const Tag *tag (void) const
		{	return m_pTag; }

	private:
		ulong m_filelen;
		Header m_header;
		Id3v2 *m_pId3;
		Vbr *m_pVBR;
		Tag *m_pTag;
	};

	/*========================================================*/

	class Vbr
	{
	public:
		Vbr (void);
		bool find (byte *buf, ulong pos, ulong *ptr);

		inline ulong frames (void) const
		{	return m_frames; }

		enum {
			FRAMES = 0x01,
			BYTES = 0x02,
			TOC = 0x04,
			VBRSCALE = 0x08,
		};

	private:
		ulong fromBuf (byte *buf, int index);

		ulong m_frames;
	};

	/*========================================================*/

	class Tag
	{
	public:
		Tag (void);

		bool find (byte *buf);

		inline const char *comment (void) const { return m_comment.c_str(); }
		inline const char *title (void) const { return m_title.c_str(); }
		inline const char *artist (void) const { return m_artist.c_str(); }
		inline const char *album (void) const { return m_album.c_str(); }
		inline ulong year (void) const { return m_year; }
		inline byte genre (void) const { return m_genre; }

	private:
		std::string getString (byte *ptr, int pos);

		std::string m_comment;
		std::string m_title;
		std::string m_artist;
		std::string m_album;
		ulong m_year;
		byte m_genre;
	};

	class Id3v2
	{
		class frame
		{
		public:
			frame (void);
			bool read (FILE *file);

			std::string frameType (void) const
			{ return std::string(m_type, 4); }
			ulong size (void) const
			{ return m_size; }
			std::string data (void) const
			{ return m_data; }

		private:
			char m_type[4];
			ulong m_size;
			ushort m_flags;
			std::string m_data;
		};

	public:
		Id3v2 (void);

		bool read (FILE *file);

		const char *value (const char *key) const;
		const char *value (int key) const;

		enum {
			ARTIST,
			TITLE,
			ALBUM,
			YEAR,
			TRACKNO,
			COUNT
		};
		
	private:
		typedef std::map<std::string,std::string> FRAMES;
		
		enum {
			UNSYNC = 0x01,
			EXTHDR = 0x02,
			EXPERM = 0x04,
			FOOTER = 0x08,
		};

		byte m_verHi, m_verLo;
		FRAMES m_frames;
		byte m_flags;
		ulong m_size;
	};
};

/*========================================================*/
#endif	/*	__MP3INFO_H__	*/
