<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    menumaint-defs.php                                      */
/* Author:      Paul Waite                                              */
/* Description: Definitions for maintenance of menus. Menu structure is */
/*              stored in the Axyl database in the ax_menu and the      */
/*              ax_menuoption tables. This module contains classes      */
/*              which allow this to be maintained via web forms.        */
/*                                                                      */
/* ******************************************************************** */
/** @package menu */

/** Menu classes */
include_once("menu-defs.php");

// ......................................................................
/** Create new menuoption */
define("NEW_MENUOPTION", -1);

// ......................................................................
/**
* Menu Maintainer
* @package menu
*/
class menumaintainer extends HTMLObject {
  // Public
  /** The name of the menu */
  var $menu_name = "";
  /** The language of the menu (0 = default) */
  var $language = 0;
  /** The name of the language of the menu */
  var $lang_desc = "default";
  /** The description of the menu */
  var $menu_desc = "";

  // Private
  /** The ID of the menu
      @access private */
  var $menu_id = false;
  /** Array of groups allowed to access menu
      @access private */
  var $user_groups = array();
  /** Menu active flag
      @access private */
  var $active = true;
  /** The parent ID of the menu-options being edited
      @access private */
  var $edit_parent_id = 0;
  /** The ID of the current menuoption being edited, if any
      @access private */
  var $edit_menuop_id;
  /** The level of the menu-options
      @access private */
  var $menu_level = 0;
  /** Whether this menu exists in the database
      @access private */
  var $exists = false;
  /** The array of menu options
      @access private */
  var $menuoptions = array();
  /** The name of our form
      @access private */
  var $formname = "";
  // ....................................................................
  /**
  * Constructor
  * Create a new menumaintainer.
  * @param string $id The unique name/identity of the menu.
  * @param integer $pid The optional id of a parent menuoption.
  * @param boolean $multilang If true, offer multi-language options
  */
  function menumaintainer($menuname="", $pid=0, $multilang=MULTILANG_DISABLED) {
    global $edit_parent_id, $sel_menu_id, $edit_menu_id, $menuopid;
    global $menumode, $menu_name, $language;
    global $RESPONSE;

    // Mode
    if (isset($menumode)) $this->mode = $menumode;
    else $this->mode = "viewing";

    // Initialise..
    $this->multilang = $multilang;
    $this->formname  = "menufm";

    // When you have one of these, you need this script..
    $RESPONSE->add_script(
        "function menugo(mode) {\n"
      . " var rc=true;\n"
      . " if(mode=='delete') {\n"
      . "  var msg = \"WARNING:\\n\\n\";\n"
      . "  msg+=\"Do you really want to delete this menu?\\n\\n\";\n"
      . "  rc = confirm(msg);\n"
      . " }\n"
      . " if(rc) {\n"
      . "  document.forms." . $this->formname . ".menumode.value=mode;\n"
      . "  document.forms." . $this->formname . ".submit();\n"
      . " }\n"
      . "}\n"
      );

    // Set up menu name..
    if ($menuname != "") {
      $this->menu_name = $menuname;
    }
    elseif (isset($menu_name)) {
      $this->menu_name = $menu_name;
    }

    // Set up menu language..
    if ($this->multilang && isset($language)) {
      $this->language = $language;
    }

    if ($this->mode != "createnew") {
      // Obtain a menu ID to work with..
      // A menu we are selecting..
      if (isset($sel_menu_id)) {
        $this->menu_id = $sel_menu_id;
      }
      // A menu we are editing currently..
      elseif (isset($edit_menu_id)) {
        $this->menu_id = $edit_menu_id;
      }

      // A language we are selecting..
      // Save parent ID..
      if (isset($edit_parent_id)) {
        $this->edit_parent_id = $edit_parent_id;
      }
      else {
        $this->edit_parent_id = $pid;
      }

      // Edit request overrides parent ID..
      if ($this->mode == "edit" && isset($menuopid)) {
        $m = new menuoption($menuopid);
        if ($m->exists) {
          $this->edit_parent_id = $m->parent_id;
          $this->edit_menuop_id = $menuopid;
          $this->mode = "editing";
        }
      }
    }
    // Process anything POSTed via form..
    $this->POSTprocess();

    // Edit new, or get the existing menu..
    if ($this->mode != "createnew") {
      $this->get();
    }

  } // menumaintainer
  // ....................................................................
  /**
  * Get the menu
  * Retrieves the specified menu from database. If it doesn't exist
  * then we create a new one.
  * @return boolean True if the record was acquired successfully
  */
  function get() {
    debug_trace($this);
    $this->exists = false;

    // Two method of finding the menu - by ID and by Name/language..
    $qByName  = "SELECT * FROM ax_menu m, ax_language l";
    $qByName .= " WHERE m.lang_id=$this->language";
    $qByName .= "   AND l.lang_id=m.lang_id";
    if ($this->menu_name != "") {
      $qByName .= " AND menu_name='" . addslashes($this->menu_name) . "'";
    }
    $qByID  = "SELECT * FROM ax_menu m, ax_language l";
    $qByID .= " WHERE m.menu_id=$this->menu_id";
    $qByID .= "   AND l.lang_id=m.lang_id";

    // Try and find it..
    $found = false;
    if ($this->menu_id != false) {
      $mnuQ = dbrecordset($qByID);
      $found = $mnuQ->hasdata;
    }
    // Fallback procedure..
    if (!$found) {
      $mnuQ = dbrecordset($qByName);
    }
    if ($mnuQ->hasdata) {
      $this->menu_id     = $mnuQ->field("menu_id");
      $this->menu_name   = $mnuQ->field("menu_name");
      $this->language    = $mnuQ->field("lang_id");
      $this->lang_desc   = $mnuQ->field("lang_desc");
      $this->menu_desc   = $mnuQ->field("menu_desc");
      $this->user_groups = explode(",", $mnuQ->field("menu_user_groups"));
      $this->active      = $mnuQ->istrue("active");
      $this->exists = true;
      $this->menu_level = 0;

      // Get menuoptions as required..
      $q  = "SELECT *";
      $q .= "  FROM ax_menuoption";
      $q .= " WHERE menu_id=$this->menu_id";
      if ($this->edit_parent_id > 0) {
        $q .= " AND parent_id=$this->edit_parent_id";
      }
      else {
        $q .= " AND menu_level=0";
      }
      $q .= " ORDER BY display_order";
      $mnuQ = dbrecordset($q);
      if ($mnuQ->hasdata) {
        do {
          $menuoptionid = $mnuQ->field("menuoption_id");
          $menuoption = new menuoption();
          $menuoption->menu_id        = $this->menu_id;
          $menuoption->menuoptionid   = $menuoptionid;
          $menuoption->parent_id      = $mnuQ->field("parent_id");
          $menuoption->user_groups    = explode(",", $mnuQ->field("user_groups"));
          $menuoption->user_type      = $mnuQ->field("user_type");
          $menuoption->menu_level     = $mnuQ->field("menu_level");
          $menuoption->label          = $mnuQ->field("label");
          $menuoption->description    = $mnuQ->field("description");
          $menuoption->display_order  = $mnuQ->field("display_order");
          $menuoption->action         = $mnuQ->field("action");
          $menuoption->target         = $mnuQ->field("target");
          $menuoption->sitepage       = $mnuQ->field("sitepage");
          $menuoption->sitepage_parms = $mnuQ->field("sitepage_parms");
          $menuoption->auth_code      = $mnuQ->istrue("auth_code");
          $menuoption->active         = $mnuQ->istrue("active");
          $menuoption->last_modified  = $mnuQ->field("last_modified");
          $menuoption->width          = $mnuQ->field("width");
          $menuoption->height         = $mnuQ->field("height");
          $menuoption->is_parent      = $mnuQ->istrue("is_parent");
          $menuoption->exists = true;
          $menuoption->syncparms();
          // Stash menuoption..
          $this->menuoptions[$menuoptionid] = $menuoption;
        } while ($mnuQ->get_next());
        $this->menu_level = $menuoption->menu_level;
      }
    }
    debug_trace();
    // Return true if at least the menu exists..
    return $this->exists;
  } // get
  // ....................................................................
  /**
  * Save this menu to the database. Create a new one if it doesn't
  * already exist.
  */
  function put() {
    debug_trace($this);
    // Deal with brand new menu..
    if ($this->exists) {
      $mnuQ = new dbupdate("ax_menu");
      $mnuQ->where("menu_id=$this->menu_id");
    }
    else {
      $mnuQ = new dbinsert("ax_menu");
      $this->menu_id = get_next_sequencevalue("seq_menu_id", "ax_menu", "menu_id");
      $mnuQ->set("menu_id", $this->menu_id);
    }
    $mnuQ->set("menu_name",        $this->menu_name);
    $mnuQ->set("menu_desc",        $this->menu_desc);
    $mnuQ->set("menu_user_groups", implode(",", $this->user_groups));
    $mnuQ->set("active",           $this->active);
    $mnuQ->set("last_modified",    'now()');
    $this->exists = $mnuQ->execute();

    // Save any menuoptions..
    if (isset($this->menuoptions) && count($this->menuoptions) > 0) {
      foreach ($this->menuoptions as $menuoption) {
        $menuoption->put();
      } // foreach
    }
    debug_trace();
  } // put
  // ....................................................................
  /**
  * This adds a new menuoption to the list, but does not add any records
  * to the database.
  */
  function add_menuoption() {
    debug_trace($this);
    $menuoption = new menuoption();
    $menuoption->menu_id = $this->menu_id;
    $menuoption->menu_level = $this->menu_level;
    $menuoption->display_order = 999;
    $this->menuoptions[$menuoption->menuoptionid] = $menuoption;
    debug_trace();
  } // add_menuoption
  // ....................................................................
  /**
  * Remove menuoption from the menu. Just removes the given menuoption
  * from the list in this object - no database changes are made.
  * @param integer $id The id of the menuoption to remove
  */
  function remove_menuoption($id) {
    debug_trace($this);
    if (isset($this->menuoptions[$id])) {
      $menuoption->delete();
      unset($this->menuoptions[$id]);
    }
    debug_trace();
  } // remove_menuoption
  // ....................................................................
  /**
  * Delete this menu from the database.
  */
  function delete() {
    debug_trace($this);
    if ($this->exists) {
      // RI will delete the menuoptions of this menu..
      $del = new dbdelete("ax_menu");
      $del->where("menu_id=$this->menu_id");
      $del->execute();
      $this->menuoptions = array();
    }
    debug_trace();
  } // delete
  // ....................................................................
  /**
  * Render the menu editing suite.
  * @return string The HTML for the editing suite form etc.
  * @access private
  */
  function editform() {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;

    // Width of large form elements..
    $ewidth = "300px";
    $pwidth = "110px";

    // Generic combo for user groups..
    $cboGroups = new form_combofield();
    $cboGroups->multiselect = true;
    $cboGroups->setclass("axlistbox");
    $cboGroups->setstyle("width:$ewidth;");
    $cboGroups->set_size(4);
    $cboGroups->additem("", "Any");
    $gQ = dbrecordset("SELECT * FROM ax_group ORDER BY group_desc");
    $cboGroups->add_querydata($gQ, "group_desc", "group_desc");

    // Generic combo for site pages..
    $cboWebPages = new form_combofield();
    $cboWebPages->setclass("axcombo");
    $cboWebPages->additem(MENU_ITEM_SUBMENU);
    $cboWebPages->additem(MENU_ITEM_SPACER);
    $cboWebPages->additem(MENU_ITEM_SEPARATOR);
    $cboWebPages->additem(MENU_ITEM_URL);
    $gQ = dbrecordset("SELECT * FROM ax_sitepage WHERE enabled=TRUE ORDER BY page_title");
    if ($gQ->hasdata) {
      do {
        $pgtitle = $gQ->field("page_title");
        $pgpath  = $gQ->field("page_path");
        $pgfile  = basename($pgpath);
        $cboWebPages->additem($pgpath, "$pgtitle ($pgfile)");
      } while ($gQ->get_next());
    }

    // Initialise content..
    $s = "";

    // ..................................................................
    // KEYFIELD and RECORD MAINTAINER
    // Menuoption listbox
    // Declared here so we can create the maintainer and register buttons
    // before they are used in the form.
    //
    // This is the keyfield listbox which controls the maintainance
    // process. It lists all records being maintained..
    $menuoption_listbox = new form_combofield("menuoption_id");
    $menuoption_listbox->setclass("axlistbox");
    // Make a new record maintainer, and attach the buttons..
    $maintainer = new recmaintainer($this->formname, $menuoption_listbox);

    // Create buttons..
    $bup    = new form_imagebutton("_up",    "", "", "$LIBDIR/img/_up.gif",     "Move up",             57, 15);
    $bdown  = new form_imagebutton("_down",  "", "", "$LIBDIR/img/_down.gif",   "Move down",           57, 15);
    $bdel   = new form_imagebutton("_del",   "", "", "$LIBDIR/img/_delete.gif", "Delete menu option",  57, 15);
    $badd   = new form_imagebutton("_add",   "", "", "$LIBDIR/img/_add.gif",    "Add new menu option", 57, 15);
    $bsave  = new form_imagebutton("_save",  "", "", "$LIBDIR/img/_save.gif",   "Save menu settings",  57, 15);
    $bdone  = new form_imagebutton("_done",  "", "", "$LIBDIR/img/_done.gif",   "Done with editing",   57, 15);
    $breset = new form_imagebutton("_reset", "", "", "$LIBDIR/img/_reset.gif",  "Reset form",          57, 15);
    $btop   = new form_imagebutton("_top",   "", "", "$LIBDIR/img/_top.gif",    "Top of menu tree",    57, 15);
    $bprev  = new form_imagebutton("_prev",  "", "", "$LIBDIR/img/_prev.gif",   "Previous menu level", 57, 15);
    $bnext  = new form_imagebutton("_next",  "", "", "$LIBDIR/img/_next.gif",   "Next sub-menu level", 57, 15);
    $bnewpg = new form_imagebutton("_newpg", "", "", "$LIBDIR/img/_create.gif", "Create new webpage",  57, 15);
    $breset->set_onclick("document.forms.$this->formname.reset()");

    // Special handling for Next button
    $RESPONSE->add_script(
        "var rc=false;\n"
      . "function checkforsubmenu() {\n"
      . "  fm = document.forms.$this->formname;\n"
      . "  if (fm.menuoption_id != null && fm.menuoption_id.options.length != 0) {\n"
      . "    if (fm.sitepage.value != '') {\n"
      . "      alert('The current item is not a sub-menu.');\n"
      . "      rc= false;\n"
      . "    }\n"
      . "    else rc = true;\n"
      . "  }\n"
      . "}\n"
      );
    $bnext->set_onclick("checkforsubmenu();return rc;");

    // Register all relevant buttons to the maintainer..
    $maintainer->register_button("up" ,   $bup);
    $maintainer->register_button("down",  $bdown);
    $maintainer->register_button("add",   $badd);
    $maintainer->register_button("save",  $bsave);
    $maintainer->register_button("del",   $bdel);
    $maintainer->register_button("reset", $breset);

    // Control table..
    $Ted = new table("menu_$this->menu_id");
    $Ted->setpadding(2);

    // ..................................................................
    // Toolbar..
    $toolbar[] = $bdone;
    $toolbar[] = $bsave;
    $Tbar = new table("toolbar");
    $Tbar->tr("axtitle");
    $Tbar->th("<b>EDITING</b> [$this->menu_id]", "axtitle");
    $tools = "";
    foreach ($toolbar as $button) {
      $tools .= $button->render();
    }
    $Tbar->th($tools, "text-align:right");
    $Ted->thead();
    $Ted->tr();
    $Ted->td( $Tbar->render() );
    $Ted->td_colspan(3);

    // ..................................................................
    // Global Menu Properties..
    $Ted->tr("axhdg");
    $Ted->td("<b>MENU PROPERTIES</b>", "axhdg");
    $Ted->td_colspan(3);

    // ..................................................................
    // Menu name & Active flag
    $menu_name = new form_textfield("menu_name", "", $this->menu_name);
    $menu_name->setstyle("width:$ewidth;");
    $menu_name->setclass("axtxtbox");
    $menu_active = new form_checkbox("menu_active");
    $menu_active->setclass("axchkbox");
    $menu_active->checked = $this->active;
    $Ted->tr("axbglite");
    $Ted->td("Menu name:", "axfg");
    $Ted->td( $menu_name->render() . "&nbsp;&nbsp;" . "Active:&nbsp;" . $menu_active->render(), "axfg" );
    $Ted->td_colspan(2);

    // Multi-language selector..
    if ($this->multilang) {
      $cboLangs = new form_combofield("language");
      $cboLangs->setclass("axcombo");
      $cboLangs->setstyle("width:$ewidth;");
      $LQ = dbrecordset("SELECT * FROM ax_language ORDER BY display_order");
      $cboLangs->add_querydata($LQ, "lang_id", "lang_desc");
      $cboLangs->setvalue($this->language);
      // Insert this combo..
      $Ted->tr("axbglite");
      $Ted->td("Menu language:", "axfg");
      $Ted->td( $cboLangs->render() );
      $Ted->td_colspan(2);
    }
    else {
      $hidLang = new form_hiddenfield("language", $this->language);
      $Ted->tr("axbglite");
      $Ted->td( $hidLang->render() );
      $Ted->td_colspan(3);
    }

    // Menu description
    $menu_desc = new form_textfield("menu_desc", "", $this->menu_desc);
    $menu_desc->setstyle("width:$ewidth;");
    $menu_desc->setclass("axtxtbox");
    $Ted->tr("axbgdark");
    $Ted->td("Description:", "axfg");
    $Ted->td( $menu_desc->render() );
    $Ted->td_colspan(2);

    // Menu group access
    $menu_user_groups = $cboGroups;
    $menu_user_groups->name = "menu_user_groups";
    $menu_user_groups->setvalue($this->user_groups);
    $Ted->tr("axbglite");
    $Ted->td("Group access:", "axfg");
    $Ted->td_alignment("", "top");
    $Ted->td( $menu_user_groups->render() );
    $Ted->td( "&nbsp;" );

    // ..................................................................
    // Menu Options Editing Fields..
    $Ted->tr("axhdg");
    $Ted->td("<b>MENU ITEMS</b>", "axhdg");
    $Ted->td_colspan(3);

    // Continue defining listbox..
    $menuoption_listbox->setstyle("width:$ewidth;");
    $menuoption_listbox->size = 10;
    $default_parent_id = 0;
    foreach ($this->menuoptions as $m) {
      // Populate listbox..
      $menuoption_listbox->additem($m->menuoptionid, $m->label);

      // Set the displayed menu action..
      $actionbits = explode("?", $m->action);
      $menuaction = $actionbits[0];
      if ($menuaction == "" || $menuaction == $m->sitepage) {
        $menuaction = MENU_ITEM_URL;
      }
      else {
        $menuaction = $m->sitepage;
      }

      // Populate maintainer data. The maintainer add_record method
      // requires an associative array keyed on listbox key id..
      $rec = array(
              "parent_id"      => $m->parent_id,
              "label"          => $m->label,
              "description"    => $m->description,
              "action"         => $menuaction,
              "target"         => $m->target,
              "sitepage"       => $m->sitepage,
              "sitepage_parms" => $m->sitepage_parms,
              "is_parent"      => (($m->is_parent) ? "t" : "f"),
              "user_groups"    => implode(",", $m->user_groups),
              "user_type"      => $m->user_type,
              "auth_code"      => (($m->auth_code) ? "t" : "f"),
              "width"          => $m->width,
              "height"         => $m->height,
              "active"         => (($m->active) ? "t" : "f")
              );
      $maintainer->add_record($m->menuoptionid, $rec);

      // If this is the menuoption being edited, make sure
      // that it gets focussed..
      if (isset($this->edit_menuop_id) && $m->menuoptionid == $this->edit_menuop_id) {
        $maintainer->initial_record($m->menuoptionid);
      }
      // Save default as the last value..
      $default_parent_id = $m->parent_id;
    }
    // Now set the defaults for each of the fields. These are
    // necessary for when a new record is created..
    $defaults = array(
              "parent_id"      => $default_parent_id,
              "label"          => MENU_ITEM_SUBMENU,
              "description"    => "",
              "action"         => MENU_ITEM_URL,
              "target"         => "",
              "sitepage"       => MENU_ITEM_SUBMENU,
              "sitepage_parms" => "",
              "is_parent"      => "f",
              "user_groups"    => "",
              "user_type"      => "",
              "auth_code"      => "f",
              "width"          => 100,
              "height"         => 18,
              "active"         => "t"
              );
    $maintainer->add_defaults($defaults);

    // The listbox field..
    $menuoption_listbox->setvalue($m->menuoptionid);
    $Ted->tr("axbgdark");
    $Ted->td(
          $btop->render()   . "<br>"
        . $bnext->render() . "<br>"
        . $bprev->render()
        );
    $Ted->td_alignment("", "top");
    $Ted->td( $menuoption_listbox->render() );
    $Ted->td(
          $bup->render()   . "<br>"
        . $bdown->render() . "<br>"
        . $bdel->render()  . "<br>"
        . $badd->render()
        );
    $Ted->td_alignment("right", "top");

    // ..................................................................
    $Ted->tr("axhdg");
    $Ted->td("<b>MENU ITEM SETTINGS</b>", "axhdg");
    $Ted->td_colspan(3);

    // ..................................................................
    // Parent field..
    $menuop_parents = new form_combofield("parent_id", "", $m->parent_id);
    $maintainer->register_field($menuop_parents);
    $menuop_parents->setstyle("width:$ewidth;");
    $menuop_parents->setclass("axcombo");
    $menuop_parents->additem("0", "TOP LEVEL");
    if ($this->exists) {
      $q  = "SELECT * FROM ax_menuoption";
      $q .= " WHERE menu_id=$this->menu_id";
      $q .= "   AND is_parent";
      $q .= " ORDER BY label";
      $moQ = dbrecordset($q);
      if ($moQ->hasdata) {
        do {
          $mid = $moQ->field("menuoption_id");
          $mlabel = $moQ->field("label");
          $menuop_parents->additem($mid, $mlabel);
        } while ($moQ->get_next());
      }
    }
    $Ted->tr("axbglite");
    $Ted->td( "Child of:", "axfg" );
    $Ted->td( $menuop_parents->render() );
    $Ted->td( "&nbsp;" );

    // ..................................................................
    // Menuoption label field..
    $menuop_label = new form_textfield("label", "", $m->label);
    $maintainer->register_field($menuop_label);
    $menuop_label->setstyle("width:$ewidth;");
    $menuop_label->setclass("axtxtbox");
    $Ted->tr("axbglite");
    $Ted->td( "Label:", "axfg" );
    $Ted->td( $menuop_label->render() );
    $Ted->td( "&nbsp;" );

    // ..................................................................
    // Menuoption description field..
    $menuop_desc = new form_textfield("description", "", $m->description);
    $maintainer->register_field($menuop_desc);
    $menuop_desc->setstyle("width:$ewidth;");
    $menuop_desc->setclass("axtxtbox");
    $Ted->tr("axbgdark");
    $Ted->td( "Description:", "axfg" );
    $Ted->td( $menuop_desc->render() );
    $Ted->td( "&nbsp;" );

    // ..................................................................
    // Menuoption is parent field..
    $menuop_is_parent = new form_checkbox("is_parent");
    $maintainer->register_field($menuop_is_parent);
    $menuop_is_parent->checked = $m->is_parent;
    $menuop_is_parent->setclass("axchkbox");
    $Ted->tr("axbgdark");
    $Ted->td( "Is parent of sub-menu:", "axfg" );
    $Ted->td( $menuop_is_parent->render() );
    $Ted->td( "&nbsp;" );

    // ..................................................................
    // Menuoption sitepage and sitepage parms, action & target fields..
    $menuop_spage = $cboWebPages;
    $menuop_spage->name = "sitepage";
    $maintainer->register_field($menuop_spage);
    $menuop_spage->setvalue($m->sitepage);
    $menuop_spage->setstyle("width:$ewidth;");

    $menuop_action = new form_textfield("action");
    $maintainer->register_field($menuop_action);
    $menuop_action->setstyle("width:$ewidth;");
    $menuop_action->setclass("axtxtbox");
    $menuop_action->set_onblur("setAdhocURL(this,'" . $this->formname . "','_')");

    $menuop_spage_parms = new form_textfield("sitepage_parms", "", $m->sitepage_parms);
    $maintainer->register_field($menuop_spage_parms);
    $menuop_spage_parms->setstyle("width:$ewidth;");
    $menuop_spage_parms->setclass("axtxtbox");

    $menuop_target = new form_textfield("target");
    $maintainer->register_field($menuop_target);
    $menuop_target->setstyle("width:$pwidth;");
    $menuop_target->setclass("axtxtbox");

    $Ted->tr("axbglite");
    $Ted->td( "Target webpage:", "axfg" );
    $Ted->td( $menuop_spage->render() );
    $Ted->td( "&nbsp;" );
    $Ted->tr("axbglite");
    $Ted->td( "or:", "axfg" );
    $Ted->td_alignment("right");
    $Ted->td( $menuop_action->render() );
    $Ted->td( "&nbsp;" );
    $Ted->tr("axbglite");
    $Ted->td( "URL Parameters:", "axfg" );
    $Ted->td( $menuop_spage_parms->render() );
    $Ted->td( "&nbsp;" );
    $Ted->tr("axbglite");
    $Ted->td( "Target frame:", "axfg" );
    $Ted->td( $menuop_target->render() );
    $Ted->td( "&nbsp;" );

    $RESPONSE->add_script(
        "function setAdhocURL(elem, fm, pfx) {\n"
      . " if(elem.value!='" . MENU_ITEM_URL . "') {\n"
      . "  pgcombo=eval('document.forms.'+fm+'.sitepage');\n"
      . "  if(pgcombo!=null) {\n"
      . "   comboSet(pgcombo, '" . MENU_ITEM_URL . "');\n"
      . "   changedValue(fm, pgcombo, pfx);\n"
      . "  }\n"
      . " }\n"
      . "}\n"
        );

    // ..................................................................
    // Menuoption user groups field..
    $menuop_ugroups = $cboGroups;
    $menuop_ugroups->name = "user_groups";
    $maintainer->register_field($menuop_ugroups);
    $menuop_ugroups->setvalue($m->user_groups);
    $Ted->tr("axbgdark");
    $Ted->td( "Permitted groups:", "axfg" );
    $Ted->td_alignment("", "top");
    $Ted->td( $menuop_ugroups->render() );
    $Ted->td_alignment("", "top");
    $Ted->td( "&nbsp;" );

    // ..................................................................
    // User type and authorisation code
    $menuop_utype = new form_combofield("user_type", "", $m->user_type);
    $menuop_utype->setclass("axcombo");
    $maintainer->register_field($menuop_utype);
    $menuop_utype->setstyle("width:$ewidth;");
    $menuop_utype->additem("", "Anyone");
    $menuop_utype->additem("sys",  "System engineer only");
    $menuop_auth = new form_checkbox("auth_code");
    $menuop_auth->setclass("axchkbox");
    $menuop_auth->checked = $m->auth_code;
    $maintainer->register_field($menuop_auth);
    $Ted->tr("axbglite");
    $Ted->td( "Permitted user type:", "axfg" );
    $Ted->td( $menuop_utype->render() );
    $Ted->td( "Auth code reqd:&nbsp;" . $menuop_auth->render(), "axfg" );
    $Ted->td_alignment("right");

    // ..................................................................
    // Width, height and active
    $menuop_width = new form_textfield("width", "", $m->width);
    $maintainer->register_field($menuop_width);
    $menuop_width->setstyle("width:60px;");
    $menuop_width->setclass("axtxtbox");
    $menuop_height = new form_textfield("height", "", $m->height);
    $maintainer->register_field($menuop_height);
    $menuop_height->setstyle("width:60px;");
    $menuop_height->setclass("axtxtbox");
    $menuop_active = new form_checkbox("active");
    $menuop_active->checked = $m->active;
    $maintainer->register_field($menuop_active);
    $Ted->tr("axbgdark");
    $Ted->td( "HVmenu sizing:", "axfg" );
    $Tin = new table("size");
    $Tin->tr();
    $Tin->td( $menuop_width->render() . " wide", "axfg" );
    $Tin->td( " x ", "axfg" );
    $Tin->td_alignment("center");
    $Tin->td( $menuop_height->render() . " high", "axfg" );
    $Ted->td( $Tin->render() );
    $Ted->td("Active:&nbsp;" . $menuop_active->render(), "axfg" );
    $Ted->td_alignment("right");

    // ..................................................................
    // Render the whole form..
    $Ted->tr("axtitle");
    $Ted->td("&nbsp;", "axtitle");
    $Ted->td_colspan(3);

    $Ted->set_width_profile("25%,55%,20%");
    $Ted->inherit_attributes($this);
    $s .= $Ted->render();

    // Render the maintainer. This adds the Javascript data structures
    // and renders the hidden fields for submitting changed field data..
    $s .= $maintainer->render();

  // ....................................................................
    debug_trace();
    // Return the html..
    return $s;
  } // editform
  // ....................................................................
  /** Renders a menu entry for the hierarchical list of options being
  * generated recursively.
  * @access private
  */
  function menu_entry($rendermode, $prefix, $mopid, $mno_children, $mno_details) {
    debug_trace($this);
    global $RESPONSE;
    $childcount = 0;
    $submnu = new table();
    $submnu->setstyle("margin-left:40px;");

    // Store current menu option..
    $details = explode("|", $mno_details[$mopid]);
    $menu_level = $details[0];
    $label = $details[1];
    $action = $details[3];
    if ($menu_level == 0) $label = "<b>$label</b>";
    switch ($rendermode) {
      case "editmap":
        $maintainer_href  = "$RESPONSE->requested?edit_menu_id=" . rawurlencode($this->menu_id);
        $maintainer_href .= "&menuopid=$mopid";
        $maintainer_href .= "&menumode=edit";
        break;
      case "sitemap":
        $maintainer_href = $action;
        break;
    }
    $submnu->tr();
    $submnu->td("<a href=\"$maintainer_href\">$label</a>");

    // And children..
    if (isset($mno_children[$mopid]) && $mno_children[$mopid] != "") {
      $childoptions = explode("|", $mno_children[$mopid]);
      $childcount = count($childoptions);
      $pfxcount = 1;
      foreach ($childoptions as $childmopid) {
        $childprefix = $prefix . "|" . $pfxcount;
        $submnu->tr();
        $submnu->td( $this->menu_entry($rendermode, $childprefix, $childmopid, $mno_children, $mno_details) );
        $pfxcount += 1;
      }
    }
    debug_trace();
    return $submnu->render();
  } // menu_entry
  // ....................................................................
  /**
  * Render the whole menu layout hierarchy as a sitemap, which simply
  * gives links to each menu page.
  * @return string The HTML
  */
  function sitemap() {
    return $this->menulayout("sitemap");
  } // sitemap
  // ....................................................................
  /**
  * Render the whole menu layout content. We have two modes of operation,
  * one to return the hierarchy as a sitemap, which simply gives links
  * to each menu page, and one which acts as a navigator to go and edit
  * the menu options at any point in the menu.
  * @param string $rendermode Mode of display: 'editmap' or 'sitemap'
  * @return string The HTML
  * @access private
  */
  function menulayout($rendermode="editmap") {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;
    global $edit_menu_id;

    // Initialise content..
    $s = "";

    // Render dropdown menu if more than one menu..
    $menucombo = false;
    if ($rendermode == "editmap") {
      $menus = dbrecordset("SELECT * FROM ax_menu m, ax_language l WHERE l.lang_id=m.lang_id ORDER BY menu_name");
      if ($menus->rowcount > 0) {
        $menucombo = new form_combofield("sel_menu_id");
        do {
          $menu_id = $menus->field("menu_id");
          $menuname = $menus->field("menu_name");
          $menudesc = $menus->field("menu_desc");
          $menulang = $menus->field("lang_desc");
          $menucombo->additem($menu_id, "$menuname ($menulang)");
        } while ($menus->get_next());
        $menucombo->setvalue($this->menu_id);
        $menucombo->set_onchange("menugo('viewing')");
      }
    }

    // Create buttons..
    $bed = new form_imagebutton("_edit", "", "", "$LIBDIR/img/_edit.gif", "Edit this menu", 42, 15);
    $bnew = new form_imagebutton("_new", "", "", "$LIBDIR/img/_new.gif", "Create new menu", 42, 15);
    $bdel = new form_imagebutton("_del", "", "", "$LIBDIR/img/_delete.gif", "Delete this menu", 57, 15);
    $bed->set_onclick("menugo('editing')");
    $bnew->set_onclick("menugo('createnew')");
    $bdel->set_onclick("menugo('delete')");

    // Menu layout table..
    $Tvw = new table();
    if ($rendermode == "editmap") {
      $Tvw->tr("axtitle");
      $Tvw->td("Menu Layout", "axtitle");
      // Select, new, delete toolbar..
      $Toolbar = new table("toolbar");
      $Toolbar->setpadding(2);
      $Toolbar->setwidth("");
      if ($menucombo) {
        $Toolbar->td( $menucombo->render() );
      }
      $Toolbar->td( $bed->render() );
      $Toolbar->td( $bnew->render() );
      $Toolbar->td( $bdel->render() );
      $Tvw->td( $Toolbar->render() );
      $Tvw->td_alignment("right");
    }

    // MENU ITEM DETAIL..
    if ($this->exists) {
      $q  = "SELECT *";
      $q .= "  FROM ax_menu m, ax_menuoption mo";
      $q .= " WHERE m.menu_id=$this->menu_id";
      $q .= "   AND mo.menu_id=m.menu_id";
      $q .= "   AND mo.active=TRUE";
      $q .= " ORDER BY mo.menu_level,mo.parent_id,mo.display_order";
      $item = dbrecordset($q);
      if ($item->hasdata) {
        $topcount = 0;
        do {
          $mopid        = $item->field("menuoption_id");
          $mnu_ugroups  = $item->field("user_groups");
          $mnu_usertype = $item->field("user_type");
          if ($mnu_ugroups == "" || $RESPONSE->ismemberof_group_in($mnu_ugroups)) {
            if ($mnu_usertype == "" || ($RESPONSE->user_type == $mnu_usertype)) {
              $parent_id   = $item->field("parent_id");
              $menu_level  = $item->field("menu_level");
              $label       = $item->field("label");
              $description = $item->field("description");
              $action      = $item->field("action");
              $authcode    = $item->field("auth_code");
              if ($menu_level == 0) {
                $topcount += 1;
                $mno_top[$topcount] = $mopid;
              }
              $mno_details[$mopid] = "$menu_level|$label|$description|$action|$authcode";
              $mno_level[$mopid]   = $menu_level;
              $mno_parent[$mopid]  = $parent_id;
              if ($parent_id != "") {
                if (isset($mno_children[$parent_id])) {
                  $mno_children[$parent_id] .= "|";
                }
                $mno_children[$parent_id] .= $mopid;
              }
            } // user type check
          } // memberof
        } while ($item->get_next());

        // Store each row..
        $toplevels = new table();
        $toplevels->setpadding(4);
        while (list($item_no, $mopid) = each($mno_top)) {
          $prefix = "$item_no";
          $toplevels->tr();
          $toplevels->td( $this->menu_entry($rendermode, $prefix, $mopid, $mno_children, $mno_details) );
        }
        $Tvw->tr();
        $Tvw->td( $toplevels->render() );
        $Tvw->td_colspan(2);
      }
      else {
        $Tvw->tr();
        $Tvw->td( "No menu items defined.", "axfg" );
        $Tvw->td_colspan(2);
      }
    }
    else {
      $Tvw->tr();
      $Tvw->td( "Empty menu.", "axfg" );
      $Tvw->td_colspan(2);
    }

    // Render the menu viewer table..
    $s .= $Tvw->render();
    return $s;
  } // menulayout
  // ....................................................................
  /**
  * Render the block content according to the mode of operation
  * we are in. Possible modes: 'viewing', 'editing', 'saving'.
  * @return string The HTML
  */
  function html() {
    debug_trace($this);
    global $RESPONSE;

    $s = "";

    // Start form for editing..
    if ($RESPONSE->ismemberof_group_in("Editor,Admin")) {
      $s .= "<form name=\"$this->formname\" method=\"post\">\n";
    }
    switch($this->mode) {
      case "editing":
        $this->mode = "saving";
        $s .= $this->editform();
        break;
      case "createnew":
        $this->mode = "savingnew";
        $s .= $this->editform();
        break;
      default:
        $this->mode = "viewing";
        $s .= $this->menulayout();
    } // switch

    // Finish the form..
    if ($RESPONSE->ismemberof_group_in("Editor,Admin")) {
      // Include action hidden field, and block ID
      $mode   = new form_hiddenfield("menumode",       $this->mode);
      $mid    = new form_hiddenfield("edit_menu_id",   $this->menu_id);
      $mlevel = new form_hiddenfield("menulevel",      $this->menu_level);
      $pid    = new form_hiddenfield("edit_parent_id", $this->edit_parent_id);
      $s .= $mode->render() . $mid->render() . $pid->render() . $mlevel->render();
      $s .= "</form>\n";
    }
    debug_trace();
    return $s;
  } // html
  // ....................................................................
  /**
  * Process a block edit form POST.
  * Assume that the fields have been submitted in a form as named
  * in the config, and grab the POSTed values. This method is executed
  * from the constructor usually, before anything is read in from
  * the database. We get first shot to change data here.
  * @access private
  */
  function POSTprocess() {
    debug_trace($this);
    global $HTTP_POST_VARS, $RESPONSE;
    if (isset($HTTP_POST_VARS["menumode"])) {
      global $_done_x, $_edit_x;
      global $edit_menu_id, $edit_parent_id, $menuoption_id, $menulevel;
      $this->mode = $HTTP_POST_VARS["menumode"];
      debugbr("POSTprocess: mode: $this->mode");
      switch ($this->mode) {
        case "viewing":
          if (isset($_edit_x)) {
            $this->mode = "editing";
          }
          break;

        case "delete":
          if ($this->menu_id != false) {
            $mdel = new dbdelete("ax_menu");
            $mdel->where("menu_id=$this->menu_id");
            $mdel->execute();
          }
          break;

        case "createnew":
          $this->menu_id = false;
          $this->menu_name = "(new menu name)";
          $this->menu_desc = "(new menu description)";
          break;

        case "savingnew":
          $this->put();
          $this->get();
          $edit_menu_id = $this->menu_id;
        case "saving":
          if ($edit_menu_id == $this->menu_id) {
            global $_save_x, $_cancel_x, $_done_x;
            global $_prev_x, $_next_x, $_top_x, $_newpg_x;
            global $_recmaintpost_form;
            global $_recmaintpost_data;
            global $_recmaintpost_flds;
            global $_recmaintpost_dels;
            global $_recmaintpost_order;

            // Let me out of here..
            if (isset($_done_x) || isset($_cancel_x)) {
              // Drop through to viewing..
              $this->mode = "viewing";
            }
            // Go to top level of menu..
            elseif ( isset($_top_x)) {
              $this->edit_parent_id = 0;
              $this->mode = "editing";
            }
            // Go to previous menu level..
            elseif ( isset($_prev_x)) {
              if ($edit_parent_id > 0) {
                $mo = new menuoption($edit_parent_id);
                if ($mo->exists) {
                  $edit_parent_id = $mo->parent_id;
                  $this->edit_parent_id = $mo->parent_id;
                }
              }
              $this->mode = "editing";
            }
            // Go to to next menu level..
            elseif ( isset($_next_x)) {
              if (isset($menuoption_id) && $menuoption_id != 0) {
                $this->edit_parent_id = $menuoption_id;
                $edit_parent_id = $menuoption_id;
              }
              $this->mode = "editing";
            }
            // Posted record maintenance data..
            elseif ( isset($_recmaintpost_form)
                  && $_recmaintpost_form == $this->formname) {
              // Deal with deletes..
              if (isset($_recmaintpost_dels) && $_recmaintpost_dels != "") {
                $menuoption_delids = explode(FIELD_DELIM, $_recmaintpost_dels);
                foreach ($menuoption_delids as $delmenuoptionid) {
                  // First, clear out microsite references to this menu option..
                  $mopnul = new dbupdate("ax_microsite_page");
                  $mopnul->set("menuoption_id", NULLVALUE);
                  $mopnul->where("menuoption_id=$delmenuoptionid");
                  $mopnul->execute();
                  // Remove menuoption..
                  $mopdel = new dbdelete("ax_menuoption");
                  $mopdel->where("menuoption_id=$delmenuoptionid");
                  $mopdel->execute();
                }
              }
              // Menu option adds and saves..
              if (isset($_recmaintpost_data) && $_recmaintpost_data != "") {
                $menuoptionrecs = explode(RECORD_DELIM, $_recmaintpost_data);
                $menuoption_fields = explode(",", $_recmaintpost_flds);
                foreach ($menuoptionrecs as $menuoptionrec) {
                  $menuoption_values = explode(FIELD_DELIM, $menuoptionrec);
                  $menuoptionid = array_shift($menuoption_values);
                  // Cater for new creations..
                  if (strstr($menuoptionid, "NEW_")) {
                    $savedid = $menuoptionid;
                    $menuoptionid = get_next_sequencevalue(
                                        "seq_menuoption_id",
                                        "ax_menuoption",
                                        "menuoption_id"
                                        );
                    $im = new dbinsert("ax_menuoption");
                    $im->set("menuoption_id", $menuoptionid);
                    $im->set("menu_id",       $this->menu_id);
                    $im->set("parent_id",     $this->edit_parent_id);
                    $im->set("menu_level",    $menulevel);
                    $im->set("display_order", 999);
                    $im->execute();
                    // Fix up potential re-ordering id..
                    if (isset($_recmaintpost_order)) {
                      $_recmaintpost_order = str_replace($savedid, $menuoptionid, $_recmaintpost_order);
                    }
                  }
                  // Update the menuoption data..
                  $um = new dbupdate("ax_menuoption");
                  $um->where("menuoption_id=$menuoptionid");
                  $pos = 0; $sitepage = ""; $sitepage_parms = "";
                  foreach ($menuoption_fields as $menuoption_field) {
                    // Set the menu level..
                    if ($menuoption_field == "parent_id") {
                      $new_parid = $menuoption_values[$pos];
                      if ($new_parid != $this->edit_parent_id) {
                        if ($new_parid == 0) {
                          $menulevel = 0;
                        }
                        else {
                          $q  = "SELECT menu_level FROM ax_menuoption";
                          $q .= " WHERE menuoption_id=$new_parid";
                          $lq = dbrecordset($q);
                          if ($lq->hasdata) {
                            $menulevel = $lq->field("menu_level") + 1;
                          }
                        }
                        $um->set("menu_level", $menulevel);
                      }
                    }
                    // Record sitepage settings for later..
                    if ($menuoption_field == "sitepage") {
                      $sitepage = $menuoption_values[$pos];
                    }
                    elseif ($menuoption_field == "sitepage_parms") {
                      $sitepage_parms = $menuoption_values[$pos];
                    }

                    if ($menuoption_field == "action") {
                      // Defer action setting 'til later..
                      $menuaction = $menuoption_values[$pos++];
                    }
                    else {
                      // All other 'normal' field settings..
                      $um->set($menuoption_field, $menuoption_values[$pos++]);
                    }
                  } // foreach

                  // Post-processing for label and menu action etc..
                  switch ($sitepage) {
                    // Special-function menu items first..
                    case MENU_ITEM_SPACER:
                      $um->set("label", MENU_ITEM_SPACER);
                      $menuaction = "";
                      break;
                    case MENU_ITEM_SEPARATOR:
                      $um->set("label", MENU_ITEM_SEPARATOR);
                      $menuaction = "";
                      break;
                    case MENU_ITEM_SUBMENU:
                      $menuaction = "";
                      break;
                    case MENU_ITEM_URL:
                      // Leave menuaction as-is..
                      break;
                    // Normal menu items last..
                    default:
                      $menuaction = $sitepage;
                  } // switch

                  // Set the resulting menu action..
                  if ($menuaction != "") {
                    // Fix leading slash for action and sitepage if reqd..
                    if (substr($menuaction, 0, 1) != "/" && !protocol_prefixed($menuaction)) {
                      $menuaction = "/$menuaction";
                      if ($sitepage != MENU_ITEM_URL) {
                        $um->set("sitepage", $menuaction);
                      }
                    }
                    if ($sitepage_parms != "") {
                      $menuaction .= "?" . $sitepage_parms;
                    }
                  }
                  $um->set("action", $menuaction);

                  // Set lastmodified, and fire it off..
                  $um->set("last_modified", 'now()');
                  $um->execute();
                } // foreach menuoptionrecs
              }
              // Save menu properties..
              global $menu_name, $language, $menu_desc, $menu_user_groups, $menu_active;
              $mu = new dbupdate("ax_menu");
              $mu->where("menu_id='" . addslashes($this->menu_id) . "'");
              $mu->set("menu_name", $menu_name);
              $mu->set("lang_id", $language);
              $mu->set("menu_name", $menu_name);
              $mu->set("active", isset($menu_active));
              $mu->set("last_modified", 'now()');
              $mu->execute();

              // Check/save menuoption ordering..
              if (isset($_recmaintpost_order) && $_recmaintpost_order != "") {
                $ord = 1;
                $idlist = explode(FIELD_DELIM, $_recmaintpost_order);
                foreach ($idlist as $menuoptionid) {
                  $upd = new dbupdate("ax_menuoption");
                  $upd->where("menuoption_id=$menuoptionid");
                  $upd->set("display_order", $ord);
                  $upd->set("last_modified", 'now()');
                  $upd->execute();
                  $ord += 1;
                }
              }
              // Drop through to viewing..
              $this->mode = "viewing";
            } // if our menu being saved
          }
          break;
      } // switch
    }
    debug_trace();
  } // POSTprocess

} // menumaintainer class

// ----------------------------------------------------------------------
?>