<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    microsite-defs.php                                      */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing a microsite. The microsite     */
/*              definitions are defined and stored in the Axyl database */
/*              and 'published' onto disk by this interface. This is    */
/*              therefore Add/Modify/Delete/Publish for a microsite.    */
/*                                                                      */
/* ******************************************************************** */
/** @package microsite */

/** Media catalog classes */
include_once("catalog-defs.php");

// Minimum plugins slot we provide to edit page plugins
define("MIN_PLUGIN_SLOTS", 7);

// ......................................................................
/**
* The Microsite Class
* Encapsulates a microsite definition. Holds the meta-data associated
* with a microsite, and allows this data to be modified via a dedicated
* form/POSTprocess. Also allows the microsite to be requested for 'publishing'
* to the appropriate destination for microsites, depending on webserver
* requirements/configuration.
* @package microsite
*/
class microsite extends HTMLObject {
  // Public
  /** The name of the microsite */
  var $microsite_name = "";
  /** The description of the microsite */
  var $microsite_desc = "";
  /** The domain of the microsite */
  var $microsite_domain = "";
  /** The ID of the microsite menu */
  var $menu_id;
  /** The main stylesheet of the microsite */
  var $css = "";
  /** The IE stylesheet of the microsite */
  var $css_ie = "";
  /** The Netscape stylesheet of the microsite */
  var $css_ns = "";
  /** Whether microsite publish is being requested */
  var $req_microsite_publish = false;
  /** Whether microsite removal is being requested */
  var $req_microsite_remove = false;
  /** Whether microsite is currently installed (built) */
  var $currently_installed = false;
  /** When microsite was last installed (timestamp) */
  var $last_installed_ts;
  /** When microsite was last modified (timestamp) */
  var $last_modified_ts;

  // Private
  /** Whether this microsite exists in database
      @access private */
  var $exists = false;
  /** Templates in this microsite (array)
      @access private */
  var $templates = array();
  /** Pages in this microsite (array)
      @access private */
  var $pages = array();
  /** Media items in this microsite (array)
      @access private */
  var $catalogitems = array();
  /** mode of operation
      @access private */
  var $mode = "";
  /** The name of our form
      @access private */
  var $formname = "";
  /** An array of error messages to report
      @access private */
  var $errmsgs = array();
  /** Number of plugin slots we provide to edit page plugins
      @access private */
  var $plugin_slots = MIN_PLUGIN_SLOTS;
  // ....................................................................
  /**
  * Constructor
  * Create a new microsite instance.
  * @param string $name The unique name/identity of the microsite.
  */
  function microsite($name="") {
    global $mode, $microsite_name;
    global $RESPONSE;

    // Mode
    if (isset($mode)) $this->mode = $mode;
    else $this->mode = "editing";

    // Set up microsite name..
    if ($name != "") {
      $this->microsite_name = $name;
    }
    elseif (isset($microsite_name)) {
      $this->microsite_name = $microsite_name;
    }
    else {
      // Try for the first one we can get..
      $ms = dbrecordset("SELECT microsite_name FROM ax_microsite ORDER BY microsite_desc,microsite_name");
      if ($ms->hasdata) {
        $this->microsite_name = $ms->field("microsite_name");
      }
    }

    // Set the form name we will use..
    $this->formname  = "fm_microsite";

    // Process anything POSTed via form..
    $this->POSTprocess();

    // Edit new, or get the existing microsite..
    if ($this->mode != "adding") {
      $this->get();
    }

  } // microsite
  // ....................................................................
  /**
  * Get the microsite
  * Retrieves the specified microsite from database. If it doesn't exist
  * then we create a new one.
  * @return boolean True if the record was acquired successfully
  */
  function get($name="") {
    debug_trace($this);
    $this->exists = false;
    if ($name != "") {
      $this->microsite_name = $name;
    }
    if ($this->microsite_name != "") {
      $q  = "SELECT * FROM ax_microsite";
      $q .= " WHERE microsite_name='" . addslashes($this->microsite_name) . "'";
      $mQ = dbrecordset($q);
      if ($mQ->hasdata) {
        $this->exists = true;
        $this->microsite_name        = $mQ->field("microsite_name");
        $this->microsite_desc        = $mQ->field("microsite_desc");
        $this->microsite_domain      = $mQ->field("microsite_domain");
        $this->css                   = $mQ->field("css");
        $this->css_ie                = $mQ->field("css_ie");
        $this->css_ns                = $mQ->field("css_ns");
        $this->req_microsite_publish = $mQ->istrue("req_microsite_publish");
        $this->req_microsite_remove  = $mQ->istrue("req_microsite_remove");
        $this->currently_installed   = $mQ->istrue("currently_installed");
        $this->last_installed_ts     = datetime_to_timestamp($mQ->field("last_installed"));
        $this->last_modified_ts      = datetime_to_timestamp($mQ->field("last_modified"));
        if ($mQ->field("menu_id") != "") {
          $this->menu_id = $mQ->field("menu_id");
        }
        // Get microsite pages
        $q  = "SELECT *";
        $q .= "  FROM ax_microsite_page";
        $q .= " WHERE microsite_name='" . addslashes($this->microsite_name) . "'";
        $q .= " ORDER BY display_order";
        $mpQ = dbrecordset($q);
        if ($mpQ->hasdata) {
          do {
            $pgid = $mpQ->field("microsite_page_id");
            $page = new microsite_page(
                      $pgid,
                      $this->microsite_name,
                      $mpQ->field("page_title"),
                      $mpQ->field("cache_seconds"),
                      $mpQ->istrue("corepage"),
                      $mpQ->istrue("microsite_homepage"),
                      $mpQ->istrue("enabled"),
                      $mpQ->field("menuoption_label"),
                      $mpQ->field("menuoption_id"),
                      $mpQ->field("display_order"),
                      $mpQ->field("microsite_template_id"),
                      $mpQ->field("page_id")
                      );
            // Retrieve associated info..
            $page->get_info();
            $this->pages[$pgid] = $page;
          } while ($mpQ->get_next());
        }

        // Get templates
        $q  = "SELECT *";
        $q .= "  FROM ax_microsite_template";
        $q .= " WHERE microsite_name='" . addslashes($this->microsite_name) . "'";
        $mtQ = dbrecordset($q);
        if ($mtQ->hasdata) {
          do {
            $mtid = $mtQ->field("microsite_template_id");
            $template = new microsite_template(
                      $mtid,
                      $mtQ->field("microsite_name"),
                      $mtQ->field("template_name"),
                      $mtQ->field("template_type"),
                      $mtQ->field("template_content")
                      );
            $this->templates[$mtid] = $template;
          } while ($mtQ->get_next());
        }

        // Get microsite media
        $q  = "SELECT m.* FROM ax_microsite_media m, ax_catalog c";
        $q .= " WHERE m.microsite_name='" . addslashes($this->microsite_name) . "'";
        $q .= "   AND c.cat_id=m.cat_id";
        $q .= " ORDER BY c.mime_category,c.upload_timestamp DESC";
        $mmQ = dbrecordset($q);
        if ($mmQ->hasdata) {
          do {
            $cat_id = $mmQ->field("cat_id");
            $catitem = new catalogitem($cat_id);
            if ($catitem->valid) {
              $this->catalogitems[$cat_id] = $catitem;
            }
          } while ($mmQ->get_next());
        }
      }
    }
    debug_trace();
    // Return true if at least the microsite exists..
    return $this->exists;
  } // get
  // ....................................................................
  /**
  * Save this microsite to the database. Create a new one if it doesn't
  * already exist.
  */
  function save() {
    debug_trace($this);
    // Deal with brand new microsite..
    if ($this->exists) {
      $mQ = new dbupdate("ax_microsite");
      $mQ->where("microsite_name='" . addslashes($this->microsite_name) . "'");
    }
    else {
      $mQ = new dbinsert("ax_microsite");
      $mQ->set("microsite_name", $this->microsite_name);
    }
    $mQ->set("microsite_desc", $this->microsite_desc);
    $mQ->set("microsite_domain", $this->microsite_domain);
    $mQ->set("css", $this->css);
    $mQ->set("css_ie", $this->css_ie);
    $mQ->set("css_ns", $this->css_ns);
    $mQ->set("req_microsite_publish", $this->req_microsite_publish);
    $mQ->set("req_microsite_remove", $this->req_microsite_remove);
    $mQ->set("currently_installed", $this->currently_installed);
    if (isset($this->menu_id) && $this->menu_id != "") {
      $mQ->set("menu_id", $this->menu_id);
    }
    else {
      $mQ->set("menu_id", NULLVALUE);
    }
    $mQ->set("last_modified", 'now()');
    $this->exists = $mQ->execute();

    // Save microsite pages & any attached templates..
    foreach ($this->pages as $pgid => $page) {
      $page->save();
    }

    // Save media..
    $delm = new dbdelete("ax_microsite_media");
    $delm->where("microsite_name='" . addslashes($this->microsite_name) . "'");
    $delm->execute();
    foreach ($this->catalogitems as $cat_id => $catitem) {
      $inm = new dbinsert("ax_microsite_media");
      $inm->set("microsite_name", $this->microsite_name);
      $inm->set("cat_id", $cat_id);
      $inm->execute();
    }

    debug_trace();
  } // save
  // ....................................................................
  /**
  * Initialise this object to default values. Eg. this is done after a
  * delete, so we don't see the deleted object data.
  */
  function initialise() {
    $this->exists = false;
    $this->microsite_name = "";
    $this->microsite_desc = "";
    $this->microsite_domain = "";
    if (isset($this->menu_id)) unset($this->menu_id);
    $this->css = "";
    $this->css_ie = "";
    $this->css_ns = "";
    $this->req_microsite_publish = false;
    $this->req_microsite_remove = false;
    $this->currently_installed = false;
    if (isset($this->last_installed_ts)) unset($this->last_installed_ts);
    if (isset($this->last_modified_ts)) unset($this->last_modified_ts);
    $this->templates = array();
    $this->pages = array();
    $this->catalogitems = array();
    $this->errmsgs = array();
    $this->plugin_slots = MIN_PLUGIN_SLOTS;
  } // initialise
  // ....................................................................
  /**
  * Delete this microsite from the database. We do not rely on RI to
  * delete all of the associated records, since the database in use
  * might not support it.
  */
  function delete() {
    debug_trace($this);
    if ($this->exists) {
      start_transaction();

      // Delete all stories associated with this microsite..
      include_once("story-defs.php");
      $q  = "SELECT * FROM ax_microsite_story";
      $q .= " WHERE microsite_name='" . addslashes($this->microsite_name) . "'";
      $storyQ = dbrecordset($q);
      if ($storyQ->hasdata) {
        do {
          $sid = $storyQ->field("story_id");
          $story = new story($sid);
          if ($story->valid) {
            // Flag as deleted & unindex Lucene
            $story->delete_story();
            // Delete story record completely now..
            $stdel = new dbdelete("ax_story");
            $stdel->where("story_id", $sid);
            $stdel->execute();
          }
        } while($storyQ->get_next());
      }

      // Delete all pages. This deletes page-menuoption, page-template,
      // and the microsite page record itself..
      if (count($this->pages) > 0) {
        foreach ($this->pages as $pgid => $page) {
          $page->delete();
        }
      }

      // Delete associated menuoptions and menu..
      if (isset($this->menu_id)) {
        $del = new dbdelete("ax_menuoption");
        $del->where("menu_id=$this->menu_id");
        $del->execute();

        $del = new dbdelete("ax_menu");
        $del->where("menu_id=$this->menu_id");
        $del->execute();
      }

      // Delete all associated templates..
      if (count($this->templates) > 0) {
        $del = new dbdelete("ax_microsite_template");
        $del->where("microsite_name='" . addslashes($this->microsite_name) . "'");
        $del->execute();
      }

      // Delete all associated microsite media..
      if (count($this->catalogitems) > 0) {
        // Remove the ones uploaded for this site from catalog..
        foreach ($this->catalogitems as $catitem) {
          if ($catitem->category == $this->microsite_name) {
            $catitem->delete();
          }
        }
        // Remove all microsite link records..
        $del = new dbdelete("ax_microsite_media");
        $del->where("microsite_name='" . addslashes($this->microsite_name) . "'");
        $del->execute();
      }

      // Finally, delete the site record iteself..
      $del = new dbdelete("ax_microsite");
      $del->where("microsite_name='" . addslashes($this->microsite_name) . "'");
      $del->execute();

      // Final act..
      commit();

      // Remove physical file structures..
      $this->unpublish();

      // Initialise local variables..
      $this->initialise();
    }
    debug_trace();
  } // delete
  // ....................................................................
  /**
  * Remove publishing for this microsite. We just set a database flag
  * so that the cron script will do the work. Note that this doesn't
  * remove the actual definition of the microsite - just the bits and
  * pieces (files) which are the physical instance of it, and which
  * make it viewable to the internet.
  */
  function remove_request() {
    $requp = new dbupdate("ax_microsite");
    $requp->set("req_microsite_publish", false);
    $requp->set("req_microsite_remove", true);
    $requp->where("microsite_name='" . addslashes($this->microsite_name) . "'");
    $requp->execute();
  } // remove_request
  // ....................................................................
  /**
  * Request publishing for this microsite. We just set a database flag
  * so that the cron script will do the work.
  */
  function publish_request() {
    $requp = new dbupdate("ax_microsite");
    $requp->set("req_microsite_publish", true);
    $requp->set("req_microsite_remove", false);
    $requp->where("microsite_name='" . addslashes($this->microsite_name) . "'");
    $requp->execute();
  } // publish_request
  // ....................................................................
  /**
  * Publish this microsite. We build the dir structures which actually
  * create this as an Axyl theme-website under the ./var directory, which
  * is writeable to the webserver. If the microsite has been published
  * already, then a symlink to our microsite will exist in the ./templates
  * directory. If not, then we request it to be done by a cron-driven
  * command-line Php script which has the required permissions.
  */
  function publish() {
    global $RESPONSE, $TEMPLATESDIR, $CMDIR;

    // Document root for whole site..
    $DOCROOT = $RESPONSE->site_docroot;

    // These are the website-relative paths..
    $pub_dir = "$CMDIR/microsites";
    if (!is_dir($DOCROOT . $pub_dir)) {
      mkpath($DOCROOT . $pub_dir);
      chmod($DOCROOT . $pub_dir, 0775);
    }
    $pub_themedir = "$pub_dir/themes/$this->microsite_name";
    $pub_pagesdir = "$pub_dir/pages/$this->microsite_name";

    // Assert the presence of physical directories..
    if (!is_dir($DOCROOT . $pub_themedir)) {
      mkpath($DOCROOT . $pub_themedir);
      chmod($DOCROOT . $pub_themedir, 0775);
    }
    if (!is_dir($DOCROOT . $pub_pagesdir)) {
      mkpath($DOCROOT . $pub_pagesdir);
      chmod($DOCROOT . $pub_pagesdir, 0775);
    }

    // After the microsite is installed (symlinked), the website-relative
    // path for content will be as follows. This is required for menu
    // options.
    $real_pagesdir = "$CMDIR/$this->microsite_name";

    // MENU
    if (!isset($this->menu_id) || $this->menu_id == "") {
      $mnuQ = new dbinsert("ax_menu");
      $this->menu_id = get_next_sequencevalue("seq_menu_id", "ax_menu", "menu_id");
      $mnuQ->set("menu_id", $this->menu_id);
      $mnuQ->set("menu_name", $this->microsite_name . "_menu");
      $mnuQ->set("menu_desc", "Menu generated for microsite " . $this->microsite_name);
      $mnuQ->set("last_modified", 'now()');
      $mnuQ->execute();
      $this->save();
    }

    // STYLESHEETS
    $css = new outputfile($DOCROOT . "$pub_themedir/sitestyle.css");
    if ($css->opened) {
      $css->write($this->css);
      $css->closefile();
    }
    $css_ie = new outputfile($DOCROOT . "$pub_themedir/sitestyle_ie.css");
    if ($css_ie->opened) {
      $css_ie->write($this->css_ie);
      $css_ie->closefile();
    }

    // TEMPLATES
    foreach ($this->templates as $tpid => $template) {
      $template->publishto($pub_themedir);
    }

    // IMAGES & OTHER MEDIA
    if (!is_dir($DOCROOT . "$pub_themedir/img")) {
      mkpath($DOCROOT . "$pub_themedir/img");
      chmod($DOCROOT . "$pub_themedir/img", 0775);
    }
    foreach ($this->catalogitems as $cat_id => $catitem) {
      if (copy($DOCROOT . $catitem->filepath, $DOCROOT . "$pub_themedir/img/" . basename($catitem->filepath))) {
        debugbr("publish: copied media: $catitem->filepath --> $pub_themedir/img/" . basename($catitem->filepath), DBG_DEBUG);
      }
    }

    // MICROSITE PAGES
    foreach ($this->pages as $pgid => $page) {
      debugbr("publish: microsite page $page->microsite_page_id start (menu=$this->menu_id)", DBG_DEBUG);
      $page->publishto($pub_pagesdir, $real_pagesdir, $this->menu_id);
    }

    // REQUEST SYMLINKAGE
    if (!$this->currently_installed || !is_link($DOCROOT . "$TEMPLATESDIR/$this->microsite_name")) {
      $this->publish_request();
      debugbr("publish: making publish request for site linkage", DBG_DEBUG);
    }
    else {
      debugbr("publish: site linkage exists (microsite is installed/live)", DBG_DEBUG);
    }

  } // publish
  // ....................................................................
  /**
  * Un-publish this microsite. This just removes the physical contents
  * of the microsite which were created in the ./var area.
  */
  function unpublish() {
    global $RESPONSE, $TEMPLATESDIR, $CMDIR;

    // Document root for whole site..
    $DOCROOT = $RESPONSE->site_docroot;

    // These are the website-relative paths..
    $pub_dir = "$CMDIR/microsites";
    $real_pub_themedir = $DOCROOT . "$pub_dir/themes/$this->microsite_name";
    $real_pub_pagesdir = $DOCROOT . "$pub_dir/pages/$this->microsite_name";

    if (is_dir($real_pub_themedir)) {
      shell_exec("rm -rf $real_pub_themedir");
    }
    if (is_dir($real_pub_pagesdir)) {
      shell_exec("rm -rf $real_pub_pagesdir");
    }

  } // unpublish
  // ....................................................................
  /**
  * Return a copy of the template object named as specified, and with the
  * given type. Or false if it wasn't found.
  * @return mixed The template object by name & type, or else false
  * @access private
  */
  function get_template_by_name_and_type($tpname, $tptype) {
    foreach ($this->templates as $tpid => $template) {
      if ($template->template_name == $tpname && $template->template_type == $tptype) {
        return $template;
      }
    }
    return false;
  } // get_template_by_name_and_type
  // ....................................................................
  /**
  * Return a copy of the page object which has a menuoption label
  * as specified.
  * @return mixed The page object by menu label, or else false
  * @access private
  */
  function get_page_by_menulabel($menulabel) {
    foreach ($this->pages as $pgid => $page) {
      if ($page->menuoption_label == $menulabel) {
        return $page;
      }
    }
    return false;
  } // get_page_by_menulabel
  // ....................................................................
  /**
  * Return a copy of the page object which has a sitepage ID
  * as specified.
  * @return mixed The page object by sitepage ID, or else false
  * @access private
  */
  function get_page_by_sitepage_id($page_id) {
    foreach ($this->pages as $pgid => $page) {
      if ($page->page_id == $page_id) {
        return $page;
      }
    }
    return false;
  } // get_page_by_sitepage_id
  // ....................................................................
  /**
  * Render the microsite editing suite.
  * @return string The HTML for the editing suite form etc.
  * @access private
  */
  function editform() {
    debug_trace($this);
    global $LIBDIR;
    global $RESPONSE;
    global $MIN_PLUGIN_SLOTS;

    // Widths..
    $form_width  = "640";
    $num_widthpx = "70px";
    $sml_widthpx = ceil(0.30 * $form_width) . "px";
    $mid_widthpx = ceil(0.45 * $form_width) . "px";
    $big_widthpx = ceil(0.60 * $form_width) . "px";
    $combo_widthpx = $big_widthpx;

    // Heights
    $tall_heightpx  = "250px";
    $short_heightpx = "100px";

    // Initialise content..
    $s = "";

    // Create toolbar buttons..
    $bpub  = new form_imagebutton("_publish", "", "", "$LIBDIR/img/_publish.gif", "Publish to web",       57, 15);
    $brem  = new form_imagebutton("_remove",  "", "", "$LIBDIR/img/_remove.gif",  "Remove from web",      57, 15);
    $bdel  = new form_imagebutton("_delete",  "", "", "$LIBDIR/img/_delete.gif",  "Delete microsite",     57, 15);
    $badd  = new form_imagebutton("_add",     "", "", "$LIBDIR/img/_add.gif",     "Add new",              57, 15);
    $bcan  = new form_imagebutton("_cancel",  "", "", "$LIBDIR/img/_cancel.gif",  "Cancel add",           57, 15);
    $bsave = new form_imagebutton("_save",    "", "", "$LIBDIR/img/_save.gif",    "Save changes",         57, 15);
    $bdef  = new form_imagebutton("_defcss",  "", "", "$LIBDIR/img/_default.gif", "Reset to default CSS", 57, 15);
    $bdef->setstyle("padding-left:5px");

    $bpub->set_onclick("microgo('publish')");
    $brem->set_onclick("microgo('remove')");
    $bdel->set_onclick("microgo('delete')");
    $badd->set_onclick("microgo('add')");
    $bcan->set_onclick("microgo('cancel')");
    $bdef->set_onclick("microgo('default_css')");

    // Buttons for templates maintainer..
    $tm_badd = new form_imagebutton("_addtm",  "", "", "$LIBDIR/img/_add.gif",     "Add new template",           57, 15);
    $tm_bdel = new form_imagebutton("_deltm",  "", "", "$LIBDIR/img/_delete.gif",  "Delete template",            57, 15);
    $tm_bdef = new form_imagebutton("_deftm",  "", "", "$LIBDIR/img/_default.gif", "Reset to default templates", 57, 15);
    $tm_bdef->set_onclick("microgo('default_templates')");

    // Buttons for pages maintainer..
    $pg_badd  = new form_imagebutton("_addpg",  "", "", "$LIBDIR/img/_add.gif",    "Add new page", 57, 15);
    $pg_bdel  = new form_imagebutton("_delpg",  "", "", "$LIBDIR/img/_delete.gif", "Delete page",  57, 15);
    $pg_bup   = new form_imagebutton("_uppg",   "", "", "$LIBDIR/img/_up.gif",     "Move up",      57, 15);
    $pg_bdown = new form_imagebutton("_downpg", "", "", "$LIBDIR/img/_down.gif",   "Move down",    57, 15);
    $pg_bdef  = new form_imagebutton("_defpg",  "", "", "$LIBDIR/img/_default.gif", "Install default pages", 57, 15);
    $pg_bdef->set_onclick("microgo('default_pages')");

    // Buttons for media maintainer..
    $me_badd = new form_imagebutton("_addme",  "", "", "$LIBDIR/img/_add.gif",     "Add new media",          57, 15);
    $me_bdel = new form_imagebutton("_delme",  "", "", "$LIBDIR/img/_delete.gif",  "Delete media",           57, 15);
    $me_bdef = new form_imagebutton("_defme",  "", "", "$LIBDIR/img/_default.gif", "Reset to default media", 57, 15);
    $me_bdef->set_onclick("microgo('default_images')");

    // ..................................................................
    // MICROSITE TEMPLATES MAINTAINER
    // Declared here so we can create the maintainer and register buttons
    // before they are used in the form, later on.

    $templates_listbox = new form_combofield("microsite_template_id");
    $templates_listbox->setclass("axlistbox");
    $templates_listbox->setstyle("width:$combo_widthpx;");
    $templates_listbox->size = 5;

    // Make a new templates record maintainer, and attach the buttons..
    $templates_maintainer = new recmaintainer($this->formname, $templates_listbox, "templates_");

    // Register templates maintainer buttons..
    $templates_maintainer->register_button("store", $bsave); // Save button is common to all
    $templates_maintainer->register_button("store", $bpub);
    $templates_maintainer->register_button("store", $bdef);
    $templates_maintainer->register_button("store", $pg_bdef);
    $templates_maintainer->register_button("store", $tm_bdef);
    $templates_maintainer->register_button("store", $me_bdef);
    $templates_maintainer->register_button("add",   $tm_badd);
    $templates_maintainer->register_button("del",   $tm_bdel);

    // ..................................................................
    // MICROSITE PAGES MAINTAINER
    // Declared here so we can create the maintainer and register buttons
    // before they are used in the form, later on.

    $pages_listbox = new form_combofield("microsite_page_id");
    $pages_listbox->setclass("axlistbox");
    $pages_listbox->setstyle("width:$combo_widthpx;");
    $pages_listbox->size = 10;

    // Make a new pages record maintainer, and attach the buttons..
    $pages_maintainer = new recmaintainer($this->formname, $pages_listbox, "pages_");

    // Register pages maintainer buttons..
    $pages_maintainer->register_button("store", $bsave); // Save button is common to all
    $pages_maintainer->register_button("store", $bpub);
    $pages_maintainer->register_button("store", $bdef);
    $pages_maintainer->register_button("store", $pg_bdef);
    $pages_maintainer->register_button("store", $tm_bdef);
    $pages_maintainer->register_button("store", $me_bdef);
    $pages_maintainer->register_button("add",   $pg_badd);
    $pages_maintainer->register_button("del",   $pg_bdel);
    $pages_maintainer->register_button("up" ,   $pg_bup);
    $pages_maintainer->register_button("down",  $pg_bdown);

    // ..................................................................
    // MICROSITE MEDIA MAINTAINER
    // Declared here so we can create the maintainer and register buttons
    // before they are used in the form, later on.

    $media_listbox = new form_combofield("microsite_cat_id");
    $media_listbox->setclass("axlistbox");
    $media_listbox->setstyle("width:$combo_widthpx;");
    $media_listbox->size = 15;

    // Make a new media record maintainer, and attach the buttons..
    $media_maintainer = new recmaintainer($this->formname, $media_listbox, "media_");

    // Register media maintainer buttons..
    $media_maintainer->register_button("store", $bsave); // Save button is common to all
    $media_maintainer->register_button("store", $bpub);
    $media_maintainer->register_button("store", $bdef);
    $media_maintainer->register_button("store", $pg_bdef);
    $media_maintainer->register_button("store", $tm_bdef);
    $media_maintainer->register_button("store", $me_bdef);
    $media_maintainer->register_button("add",   $me_badd);
    $media_maintainer->register_button("del",   $me_bdel);

    // Control table..
    $Ted = new table("microsite_$this->microsite_name");
    $Ted->setpadding(2);
    $Ted->setwidth("640");
    $Ted->setalign("center");

    // ..................................................................
    // HEADING & NAVBAR
    // Navigation to edit existing microsites
    $MicroSel = new form_combofield("microsite_name", "", $this->microsite_name);
    $MicroSel->additem("");
    $MicroSel->setclass("axcombo");
    $micros = dbrecordset("SELECT * FROM ax_microsite ORDER BY microsite_desc");
    if ($micros->hasdata) {
      $microsite_name = $micros->field("microsite_name");
      $desc = $micros->field("microsite_desc");
      $dom  = $micros->field("microsite_domain");
      $MicroSel->additem($microsite_name, (dom != "") ? $dom : $desc);
    }
    $Ted->tr("axtitle");
    $Ted->th("<b>MICROSITE</b> <small>$this->microsite_name</small>", "axtitle");
    if ($this->mode != "adding") {
      $Ted->td( $MicroSel->render() );
      $Ted->td_alignment("right");
    }
    else {
      $Ted->td_colspan(2);
    }

    // ..................................................................
    // TOOLBAR
    // Apply the hook for form submit just prior to rendering..
    if ($this->mode == "adding") {
      $bsave->set_onclick("microgo('adding')", SCRIPT_APPEND);
      $toolbar[] = $bsave;
      $toolbar[] = $bcan;
    }
    else {
      $bsave->set_onclick("microgo('save')", SCRIPT_APPEND);
      $toolbar[] = $badd;
      if ($this->microsite_name != "") {
        $toolbar[] = $bsave;
        // Mere Authors can't do this stuff, only Editors..
        if ($RESPONSE->ismemberof_group_in("Editor")) {
          // Request miscrosite is published to web
          $toolbar[] = $bpub;
          if ($this->currently_installed) {
            if (!$this->req_microsite_remove) {
              // Request microsite is removed from web
              $toolbar[] = $brem;
            }
          }
          else {
            if (!$this->req_microsite_publish) {
              // Delete the whole microsite definition
              $toolbar[] = $bdel;
            }
          }
        }
      }
    }
    $Tbar = new table("toolbar");
    $Tbar->setpadding(1);
    $Tbar->setwidth("");
    $Tbar->tr();
    foreach ($toolbar as $tool) {
      $Tbar->td($tool->render());
    }
    $Ted->tr("axhdg");
    $Ted->td( $Tbar->render() );
    $Ted->td_colspan(2);
    $Ted->td_alignment("right");

    // ..................................................................
    // Error messages - if any
    if (count($this->errmsgs) > 0) {
      $Ted->tr("axhdg");
      $Ted->td("<b>ERRORS</b>", "axhdg");
      $Ted->td_colspan(2);

      foreach ($this->errmsgs as $errmsg) {
        $Ted->tr("axbglite");
        $Ted->td( $errmsg, "axerror" );
        $Ted->td_colspan(2);
        $Ted->td_alignment("center");
      }
    }

    // ..................................................................
    // Global Microsite Properties..
    $Ted->tr("axhdg");
    $Ted->td("<b>MICROSITE PROPERTIES</b>", "axhdg");
    $Ted->td_colspan(2);

    // ..................................................................
    // Microsite name - adding only..
    if ($this->mode == "adding") {
      $Fld = new form_textfield("new_microsite_name", "", $this->microsite_name);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->setclass("axtxtbox");
      $Ted->tr("axbgdark");
      $Ted->td("New name (no spaces):", "axfg");
      $Ted->td( $Fld->render() );
    }

    // ..................................................................
    // Microsite description
    $Fld = new form_textfield("microsite_desc", "", $this->microsite_desc);
    $Fld->setstyle("width:$big_widthpx;");
    $Fld->setclass("axtxtbox");
    $Ted->tr("axbgdark");
    $Ted->td("Description:", "axfg");
    $Ted->td( $Fld->render() );

    // ..................................................................
    // Microsite domain
    $Fld = new form_textfield("microsite_domain", "", $this->microsite_domain);
    $Fld->setstyle("width:$mid_widthpx;");
    $Fld->setclass("axtxtbox");
    $Ted->tr("axbglite");
    $Ted->td("Domain (URL):", "axfg");
    $Ted->td( $Fld->render() );

    // ..................................................................
    // Microsite CSS
    if ($this->mode != "adding") {
      $Fld = new form_memofield("css", "", $this->css);
      $Fld->set_wrapmode("off");
      $Fld->setstyle("width:$big_widthpx;height:$tall_heightpx;");
      $Fld->setclass("axmemo");

      $T = new table("css");
      $T->tr();
      $T->td( $Fld->render() );
      $T->td( $bdef->render() );
      $T->td_alignment("", "top");
      $T->set_width_profile("60%,40%");

      $Ted->tr("axbgdark");
      $Ted->td("CSS:", "axfg");
      $Ted->td_alignment("", "top");
      $Ted->td( $T->render() );
    }

    // ..................................................................
    // Microsite CSS overrides for Internet Explorer
    // This one has a "default" button so we do it in a table..
    if ($this->mode != "adding") {
      $Fld = new form_memofield("css_ie", "", $this->css_ie);
      $Fld->set_wrapmode("off");
      $Fld->setstyle("width:$big_widthpx;height:$short_heightpx;");
      $Fld->setclass("axmemo");
      $Ted->tr("axbglite");
      $Ted->td("CSS (IE overrides):", "axfg");
      $Ted->td_alignment("", "top");
      $Ted->td( $Fld->render() );
    }

    // ..................................................................
    // Microsite installed flag - display-only
    if ($this->mode != "adding") {
      $Fld = new form_labelfield("", $this->currently_installed ? "Yes" : "No");
      $Ted->tr("axbgdark");
      $Ted->td("Published to web:", "axfg");
      $Ted->td( $Fld->render() );
    }

    // ..................................................................
    // Microsite last-installed timestamp - display-only
    if ($this->mode != "adding" && $this->currently_installed && $this->last_installed_ts != "") {
      $Fld = new form_labelfield("", timestamp_to_displaydate(NICE_DATETIME, $this->last_installed_ts));
      $Ted->tr("axbglite");
      $Ted->td("Last published:", "axfg");
      $Ted->td( $Fld->render() );
    }
    // ..................................................................
    // MICROSITE TEMPLATES, PAGES and MEDIA etc.
    if ($this->mode != "adding") {
      // ..................................................................
      // MICROSITE TEMPLATES
      $Ted->tr("axhdg");
      $Ted->td("<b>MICROSITE TEMPLATES</b>", "axhdg");
      $Ted->td_colspan(2);

      foreach ($this->templates as $tpid => $template) {
        // Populate listbox..
        $templates_listbox->additem(
              $template->microsite_template_id,
              $template->template_name . " (" . $template->template_type . ")"
              );

        // Populate maintainer data. The maintainer add_record method
        // requires an associative array keyed on listbox key id..
        $rec = array(
                "template_name"    => $template->template_name,
                "template_type"    => $template->template_type,
                "template_content" => $template->template_content
                );
        $templates_maintainer->add_record($template->microsite_template_id, $rec);
      }
      // Now set the defaults for each of the fields. These are
      // necessary for when a new record is created..
      $defaults = array(
                "template_name"    => "new_template",
                "template_type"    => "html",
                "template_content" => ""
                );
      $templates_maintainer->add_defaults($defaults);

      // The listbox field..
      $templates_listbox->setvalue($template->microsite_template_id);

      $Tin = new table("templates");
      $Tin->tr();
      $Tin->td( $templates_listbox->render() );
      $Tin->td(
          $tm_badd->render() . "<br>"
        . $tm_bdel->render() . "<br>"
        . $tm_bdef->render()
        );
      $Tin->td_alignment("", "top");
      $Tin->set_width_profile("50%,50%");

      $Ted->tr("axbgdark");
      $Ted->td( "&nbsp;" );
      $Ted->td( $Tin->render() );

      // ..................................................................
      $Ted->tr("axbglite");
      $Ted->td("<b>TEMPLATE SETTINGS</b>", "axfg");
      $Ted->td_colspan(2);

      // ..................................................................
      // Microsite template name
      $Fld = new form_textfield("template_name", "", $template->template_name);
      $templates_maintainer->register_field($Fld);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->setclass("axtxtbox");
      $Ted->tr("axbgdark");
      $Ted->td( "Template name:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite menuoption lable
      $Fld = new form_combofield("template_type", "", $template->template_type);
      $Fld->additem("html", "HTML");
      $Fld->additem("wml", "WML");
      $templates_maintainer->register_field($Fld);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->setclass("axcombo");
      $Ted->tr("axbglite");
      $Ted->td( "Type:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Template content
      $Fld = new form_memofield("template_content", "", $template->template_content);
      $Fld->set_wrapmode("off");
      $templates_maintainer->register_field($Fld);
      $Fld->setstyle("width:$big_widthpx;height:$tall_heightpx;");
      $Fld->setclass("axmemo");
      $Ted->tr("axbgdark");
      $Ted->td("Template content", "axfg");
      $Ted->td_alignment("", "top");
      $Ted->td( $Fld->render() );

      $Ted->tr("axbgdark");
      $Ted->td( "", "axfg" );
      $Ted->td( "The template above is simply an HTML (or WML) page, with 'plugin areas' "
              . "provided by HTML comments, eg. &LT;!--MAIN_CONTENT--&GT;. You can edit "
              . "the template in-situ, or use some third-party HTML editor and paste it "
              . "into the form here and then save it.",
                "axnote"
                );

      // ..................................................................
      // MICROSITE PAGES
      $Ted->tr("axhdg");
      $Ted->td("<b>MICROSITE PAGES</b>", "axhdg");
      $Ted->td_colspan(2);

      // Load up our page plugins data..
      $q  = "SELECT *";
      $q .= "  FROM ax_microsite_page p, ax_microsite_page_plugin pp,";
      $q .= "       ax_plugin_area pa, ax_plugin_content pc";
      $q .= " WHERE p.microsite_name='" . addslashes($this->microsite_name) . "'";
      $q .= "   AND pp.microsite_page_id=p.microsite_page_id";
      $q .= "   AND pa.plugin_pattern=pp.plugin_pattern";
      $q .= "   AND pc.plugin_content=pp.plugin_content";
      $pQ = dbrecordset($q);
      $plugins = array();
      $plugin_counts = array();
      if ($pQ->hasdata) {
        do {
          $pageid = $pQ->field("microsite_page_id");
          $plugin_pattern     = $pQ->field("plugin_pattern");
          $plugin_content     = $pQ->field("plugin_content");
          $plugins[$pageid][] = "$plugin_pattern|$plugin_content";
          if (!isset($plugin_counts[$pageid])) {
            $plugin_counts[$pageid] = 0;
          }
          $plugin_counts[$pageid] += 1;
        } while ($pQ->get_next());
      }
      if (count($plugin_counts) > 0) {
        $this->plugin_slots = max(max($plugin_counts) + 2, MIN_PLUGIN_SLOTS);
      }
      else {
        $this->plugin_slots = MIN_PLUGIN_SLOTS;
      }

      foreach ($this->pages as $pgid => $page) {
        // Populate listbox..
        $page_desc = $page->page_title;
        if ($page->microsite_homepage) {
          $page_desc .= " (home)";
        }
        $pages_listbox->additem($page->microsite_page_id, $page_desc);

        // Populate maintainer data. The maintainer add_record method
        // requires an associative array keyed on listbox key id..
        $rec = array(
                "page_title"                 => $page->page_title,
                "menuoption_label"           => $page->menuoption_label,
                "page_microsite_template_id" => $page->microsite_template_id,
                "cache_minutes"              => (int)($page->cache_seconds / 60),
                "microsite_homepage"         => ($page->microsite_homepage) ? "t" : "f",
                "enabled"                    => ($page->enabled) ? "t" : "f"
                );
        // Add the plugin stuff..
        $tot = 0;
        if (isset($plugins[$page->microsite_page_id])) {
          $plugin_infos = $plugins[$page->microsite_page_id];
          sort($plugin_infos);
          foreach ($plugin_infos as $plugin_info) {
            $bits = explode("|", $plugin_info);
            $pattern = $bits[0];
            $content = $bits[1];
            $rec["plugin_pattern_$tot"] = $pattern;
            $rec["plugin_content_$tot"] = $content;
            $tot += 1;
            if ($tot >= $this->plugin_slots) break;
          }
        }
        while ($tot < $this->plugin_slots) {
            $rec["plugin_pattern_$tot"] = "";
            $rec["plugin_content_$tot"] = "";
            $tot += 1;
        }
        $pages_maintainer->add_record($page->microsite_page_id, $rec);
      }
      // Now set the defaults for each of the fields. These are
      // necessary for when a new record is created..
      $defaults = array(
                "page_title"                 => "New Page",
                "menuoption_label"           => "New Page",
                "page_microsite_template_id" => "",
                "cache_minutes"              => 0,
                "microsite_homepage"         => 'f',
                "enabled"                    => 't',
                "plugin_pattern_0"           => 'FOOTER',
                "plugin_content_0"           => 'cm',
                "plugin_pattern_1"           => 'HEADER',
                "plugin_content_1"           => 'cm',
                "plugin_pattern_2"           => 'MAIN_MENU',
                "plugin_content_2"           => 'main_menu',
                "plugin_pattern_3"           => 'MAIN_CONTENT',
                "plugin_content_3"           => 'cm',
                "plugin_pattern_4"           => 'RIGHT_SIDEBAR',
                "plugin_content_4"           => 'cm'
                );
      for ($i = 5; $i < $this->plugin_slots; $i++) {
        $defaults[] = "plugin_pattern_$i";
        $defaults[] = "plugin_content_$i";
      }
      $pages_maintainer->add_defaults($defaults);

      // The listbox field..
      $pages_listbox->setvalue($page->microsite_page_id);

      $pgbuttons =
        $pg_badd->render() . "<br>"
      . $pg_bdel->render() . "<br>"
      . $pg_bup->render()  . "<br>"
      . $pg_bdown->render();

      // Only allow them to install default pages if they
      // actually have some templates defined..
      if ($this->get_template_by_name_and_type("main", "html") !== false) {
        $pgbuttons .= "<br>" . $pg_bdef->render();
      }

      $Tin = new table("pages");
      $Tin->tr();
      $Tin->td( $pages_listbox->render() );
      $Tin->td($pgbuttons);
      $Tin->td_alignment("", "top");
      $Tin->set_width_profile("50%,50%");

      $Ted->tr("axbgdark");
      $Ted->td( "&nbsp;" );
      $Ted->td( $Tin->render() );

      // ..................................................................
      $Ted->tr("axbglite");
      $Ted->td("<b>PAGE SETTINGS</b>", "axfg");
      $Ted->td_colspan(2);

      // ..................................................................
      // Microsite page title
      $Fld = new form_textfield("page_title", "", $page->page_title);
      $pages_maintainer->register_field($Fld);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->setclass("axtxtbox");
      $Ted->tr("axbgdark");
      $Ted->td( "Page title:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite menuoption lable
      $Fld = new form_textfield("menuoption_label", "", $page->menuoption_label);
      $pages_maintainer->register_field($Fld);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->setclass("axtxtbox");
      $Ted->tr("axbglite");
      $Ted->td( "Menu label:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite template
      $Fld = new form_combofield("page_microsite_template_id", "", $page->microsite_template_id);
      $Fld->additem("");
      foreach ($this->templates as $tp) {
        $Fld->additem($tp->microsite_template_id, $tp->template_name);
      }
      $pages_maintainer->register_field($Fld);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->setclass("axcombo");
      $Ted->tr("axbglite");
      $Ted->td( "Template:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite cache minutes (stored as seconds)
      $Fld = new form_textfield("cache_minutes", "", (int)($page->cache_seconds / 60));
      $pages_maintainer->register_field($Fld);
      $Fld->setstyle("width:$num_widthpx;");
      $Fld->setclass("axnumbox");
      $Fld->set_onblur("limitInt(this, 0, 99999, 0)");
      $Ted->tr("axbglite");
      $Ted->td( "Cache minutes:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite homepage flag
      $Fld = new form_checkbox("microsite_homepage");
      $pages_maintainer->register_field($Fld);
      $Fld->checked = $page->microsite_homepage;
      $Fld->setclass("axchkbox");
      $Ted->tr("axbgdark");
      $Ted->td( "Is microsite homepage:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite page enabled flag
      $Fld = new form_checkbox("enabled");
      $pages_maintainer->register_field($Fld);
      $Fld->checked = $page->enabled;
      $Fld->setclass("axchkbox");
      $Ted->tr("axbgdark");
      $Ted->td( "Enable for publish:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite page content definitions
      $pattcombo = new form_combofield("");
      $pattcombo->setclass("axcombo");
      $pattcombo->setstyle("width:200px");
      $patts = dbrecordset("SELECT * FROM ax_plugin_area");
      $pattcombo->additem("");
      $pattcombo->add_querydata($patts, "plugin_pattern", "plugin_pattern");
      $contcombo = new form_combofield("");
      $contcombo->setclass("axcombo");
      $contcombo->setstyle("width:200px");
      $conts = dbrecordset("SELECT * FROM ax_plugin_content");
      $contcombo->additem("");
      $contcombo->add_querydata($conts, "plugin_content", "plugin_content_desc");

      $Tin = new table("plugins");
      $Tin->setpadding(1);
      $Tin->tr();
      $Tin->td( "Plugin Area" );
      $Tin->td( "Contains" );
      for ($i = 0; $i < $this->plugin_slots; $i++) {
        $pattcombo->setname("plugin_pattern_$i");
        $pages_maintainer->register_field($pattcombo);
        $contcombo->setname("plugin_content_$i");
        $pages_maintainer->register_field($contcombo);
        $Tin->tr();
        $Tin->td( $pattcombo->render() );
        $Tin->td( $contcombo->render() );
      }

      $Ted->tr("axbglite");
      $Ted->td( "Plugins:", "axfg" );
      $Ted->td_alignment("", "top");
      $Ted->td( $Tin->render() );
      $Ted->tr("axbglite");
      $Ted->td( "", "axfg" );
      $Ted->td( "The plugin areas above correspond to the ones in your templates "
              . "named in HTML comments, eg. &LT;!--MAIN_CONTENT--&GT;. When you "
              . "assign content to one of these using the right-hand menus, then "
              . "that content will turn up in the plugin area.",
                "axnote"
                );

      // ..................................................................
      // MICROSITE MEDIA
      $Ted->tr("axhdg");
      $Ted->td("<b>MICROSITE LOCAL MEDIA</b>", "axhdg");
      $Ted->td_colspan(2);

      foreach ($this->catalogitems as $cat_id => $catitem) {
        // Populate listbox..
        $catitem_desc = $catitem->cat_name;
        if ($catitem->filepath != "") {
          $catitem_desc .= " (" . basename($catitem->filepath) . ")";
        }
        $media_listbox->additem(
              $cat_id,
              $catitem_desc
              );

        // Populate maintainer data. The maintainer add_record method
        // requires an associative array keyed on listbox key id..
        $rec = array(
                "cat_id"        => $cat_id,
                "cat_name"      => $catitem->cat_name,
                "mime_category" => $catitem->mime_category,
                "mime_type"     => $catitem->mime_type,
                "cat_desc"      => $catitem->cat_desc
                );
        $media_maintainer->add_record($cat_id, $rec);
      }
      // Now set the defaults for each of the fields. These are
      // necessary for when a new record is created..
      $defaults = array(
                "cat_id"        => 0,
                "cat_name"      => "",
                "mime_category" => "",
                "mime_type"     => "",
                "cat_desc"      => ""
                );
      $media_maintainer->add_defaults($defaults);

      // The listbox field..
      $media_listbox->setvalue($catitem->cat_id);
      $media_listbox->set_onchange("cat_id_preview_func(this.value)", SCRIPT_APPEND);

      // Ensure the first item gets previewed..
      if (count($this->catalogitems) > 0) {
        reset($this->catalogitems);
        list($cat_id, $catitem) = each($this->catalogitems);
        $RESPONSE->set_onload(
              "cat_id_preview_func($cat_id);"
              );
      }

      $Tin = new table("media");
      $Tin->tr();
      $Tin->td( $media_listbox->render() );
      $Tin->td(
          $me_badd->render() . "<br>"
        . $me_bdel->render() . "<br>"
        . $me_bdef->render()
        );
      $Tin->td_alignment("", "top");
      $Tin->set_width_profile("50%,50%");

      $Ted->tr("axbgdark");
      $Ted->td( "&nbsp;" );
      $Ted->td( $Tin->render() );

      $Ted->tr("axbgdark");
      $Ted->td( "&nbsp;" );
      $Ted->td( "These media files are stored in the images directory which is local "
              . "to this microsite. They can be used in templates by referring to them "
              . "with a path like: 'img/foo.gif' and the proper path will be prefixed. "
              . "Note that these media files are also placed into the global Media "
              . "Catalog, and are therefore also available to be picked from there.",
              "axnote"
              );

      // ..................................................................
      $Ted->tr("axbglite");
      $Ted->td("<b>MEDIA DETAILS</b>", "axfg");
      $Ted->td_colspan(2);

      // ..................................................................
      // Media catalog item
      $q  = "SELECT * FROM ax_catalog c";
      $q .= " ORDER BY c.mime_category";
      $cat = dbrecordset($q);
      $Fld = new form_imagecombo("cat_id");
      $Fld->set_formname($this->formname);
      if ($cat->hasdata) {
        do {
          $Fld->additem(
              $cat->field("cat_id"),
              $cat->field("cat_name") . " (" . $cat->field("mime_category") . ")",
              $cat->field("filepath")
              );
        } while ($cat->get_next());
      }
      $media_maintainer->register_field($Fld);
      $Fld->setstyle("width:$big_widthpx;");
      $Fld->setclass("axcombo");
      $Ted->tr("axbgdark");
      $Ted->td( "Media item:", "axfg" );
      $Ted->td_alignment("", "top");
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite media name
      $Fld = new form_textfield("cat_name");
      $media_maintainer->register_field($Fld);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->set_displayonly();
      $Fld->setclass("axtxtbox");
      $Ted->tr("axbglite");
      $Ted->td( "Name:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite media mime category
      $Fld = new form_textfield("mime_category");
      $media_maintainer->register_field($Fld);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->set_displayonly();
      $Fld->setclass("axtxtbox");
      $Ted->tr("axbglite");
      $Ted->td( "Mime category:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite media mime type
      $Fld = new form_textfield("mime_type");
      $media_maintainer->register_field($Fld);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->set_displayonly();
      $Fld->setclass("axtxtbox");
      $Ted->tr("axbglite");
      $Ted->td( "Mime type:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite media description
      $Fld = new form_textfield("cat_desc");
      $media_maintainer->register_field($Fld);
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->set_displayonly();
      $Fld->setclass("axtxtbox");
      $Ted->tr("axbglite");
      $Ted->td( "Description:", "axfg" );
      $Ted->td( $Fld->render() );

      // ..................................................................
      // Microsite media new file upload
      // File upload field - allows them to add media on the
      // fly to go with a story..
      $Fld = new form_fileuploadfield("new_media");
      $Fld->setsize(38);
      $Ted->tr("axbgdark");
      $Ted->td( "New media:", "axfg" );
      $Ted->td( $Fld->render() );

      // And the 'name' to go with it..
      $Fld = new form_textfield("new_media_name");
      $Fld->setstyle("width:$mid_widthpx;");
      $Fld->setclass("axtxtbox");
      $Ted->tr("axbglite");
      $Ted->td( "New media name:", "axfg" );
      $Ted->td( $Fld->render() );
    } // not adding mode

    // ..................................................................
    // Render the whole form..
    $Ted->tr("axtitle");
    $Ted->td("&nbsp;", "axtitle");
    $Ted->td_colspan(2);

    $Ted->inherit_attributes($this);
    $Ted->set_width_profile("25%,75%");
    $s .= $Ted->render();

    // Render the maintainers. This adds the Javascript data structures
    // and renders the hidden fields for submitting changed field data..
    $s .= $pages_maintainer->render();
    $s .= $templates_maintainer->render();
    $s .= $media_maintainer->render();

    // When you have one of these, you need this script..
    $RESPONSE->add_script(
        "function microgo(mode) {\n"
      . " var rc=true;\n"
      . " if(mode=='delete') {\n"
      . "  var msg = \"CONFIRM:\\n\\n\";\n"
      . "  msg+='Do you really want to delete this microsite definition?\\n';\n"
      . "  msg+='This step is permanent!\\n\\n';\n"
      . "  rc = confirm(msg);\n"
      . " }\n"
      . " if(mode=='remove') {\n"
      . "  var msg = \"CONFIRM:\\n\\n\";\n"
      . "  msg+='Do you really want to remove the microsite from view?\\n';\n"
      . "  msg+='(the PUBLISH function can be used to make it available again)\\n\\n';\n"
      . "  rc = confirm(msg);\n"
      . " }\n"
      . " if(rc) {\n"
      . "  document.forms." . $this->formname . ".mode.value=mode;\n"
      . "  document.forms." . $this->formname . ".submit();\n"
      . " }\n"
      . "}\n"
      );
  // ....................................................................
    debug_trace();
    // Return the html..
    return $s;
  } // editform
  // ....................................................................
  /**
  * Render the block content according to the mode of operation
  * we are in. Possible modes: 'viewing', 'editing', 'saving'.
  * @return string The HTML
  */
  function html() {
    debug_trace($this);
    global $RESPONSE;

    // Something to propagate the mode..
    $hidmode = new form_hiddenfield("mode", $this->mode);

    $s  = "";
    $s .= "<form name=\"$this->formname\" method=\"post\" enctype=\"multipart/form-data\">\n";
    $s .= $this->editform();
    $s .= $hidmode->render();
    $s .= "</form>\n";

    debug_trace();
    return $s;
  } // html
  // ....................................................................
  /**
  * Process a block edit form POST for "saving" the data. This is called
  * from the main POSTprocess method.
  * @access private
  */
  function POSTprocess_save() {
    global $RESPONSE;
    global $pages_recmaintpost_form;

    // Sanity check for posted record maintenance data..
    if ( isset($pages_recmaintpost_form)
          && $pages_recmaintpost_form == $this->formname) {

      // Access to all recmaint posted data..
      global $pages_recmaintpost_form,  $templates_recmaintpost_form,  $media_recmaintpost_form;
      global $pages_recmaintpost_data,  $templates_recmaintpost_data,  $media_recmaintpost_data;
      global $pages_recmaintpost_flds,  $templates_recmaintpost_flds,  $media_recmaintpost_flds;
      global $pages_recmaintpost_dels,  $templates_recmaintpost_dels,  $media_recmaintpost_dels;
      global $pages_recmaintpost_order, $templates_recmaintpost_order, $media_recmaintpost_order;

      // MICROSITE PAGES
      // Page deletes..
      if (isset($pages_recmaintpost_dels) && $pages_recmaintpost_dels != "") {
        $pages_delids = explode(FIELD_DELIM, $pages_recmaintpost_dels);
        foreach ($pages_delids as $delpageid) {
          $page = new microsite_page($delpageid);
          $page->get();
          // Delete page & all associated entities..
          $page->delete();
        }
      }
      // Page adds and saves..
      if (isset($pages_recmaintpost_data) && $pages_recmaintpost_data != "") {
        $page_recs = explode(RECORD_DELIM, $pages_recmaintpost_data);
        $page_fields = explode(",", $pages_recmaintpost_flds);
        foreach ($page_recs as $page_rec) {
          $page_values = explode(FIELD_DELIM, $page_rec);
          $microsite_page_id = array_shift($page_values);
          // Cater for new creations..
          if (strstr($microsite_page_id, "NEW_")) {
            $savedid = $microsite_page_id;
            $page = new microsite_page(NEW_MICROSITE_PAGE, $this->microsite_name);
            $page->save();
            $microsite_page_id = $page->microsite_page_id;
            // Fix up potential re-ordering id..
            if (isset($pages_recmaintpost_order)) {
              $pages_recmaintpost_order = str_replace($savedid, $microsite_page_id, $pages_recmaintpost_order);
            }
          }
          // Remove existing plugins, so we can add them back..
          $delpi = new dbdelete("ax_microsite_page_plugin");
          $delpi->where("microsite_page_id=$microsite_page_id");
          $delpi->execute();
          // Update the microsite page data..
          $um = new dbupdate("ax_microsite_page");
          $um->where("microsite_page_id=$microsite_page_id");
          $pos = 0;
          $plugins = array();
          $plugin_count = 0;
          foreach ($page_fields as $page_field) {
            $field_value = $page_values[$pos++];
            // Make sure there's only ever one homepage..
            if ($page_field == "microsite_homepage" && $field_value == "t") {
              $hpunset = new dbupdate("ax_microsite_page");
              $hpunset->set("microsite_homepage", false);
              $hpunset->where("microsite_name='" . addslashes($this->microsite_name) . "'");
              $hpunset->where("AND microsite_homepage=TRUE");
              $hpunset->execute();
            }
            if ($page_field == "page_microsite_template_id") {
              $page_field = "microsite_template_id";
              if ($field_value == "") {
                $field_value = NULLVALUE;
              }
            }
            if ($page_field == "cache_minutes") {
              $page_field  = "cache_seconds";
              $field_value = (int)($field_value * 60);
            }
            // Process either plugins, or normal data..
            if (substr($page_field, 0, 6) == "plugin") {
              if ($field_value != "") {
                $plugin_field = substr($page_field, 7, 7);
                if ($plugin_field == "pattern") {
                  $plugin_count += 1;
                }
                $plugin_index = (int)(substr($page_field, 15));
                $plugins[$plugin_index][$plugin_field] = $field_value;
              }
            }
            else {
              if ($page_value == "") {
                $page_value = NULLVALUE;
              }
              $um->set($page_field, $field_value);
            }
          } // foreach
          $um->execute();
          // Process any plugins we got..
          if ($plugin_count > 0) {
            for ($ix = 0; $ix < $plugin_count; $ix++) {
              if ($plugins[$ix]["pattern"] != "" && $plugins[$ix]["content"] != "") {
                $plin = new dbinsert("ax_microsite_page_plugin");
                $pluginid = get_next_sequencevalue("seq_plugin_id", "ax_microsite_page_plugin", "plugin_id");
                $plin->set("plugin_id", $pluginid);
                $plin->set("microsite_page_id", $microsite_page_id);
                $plin->set("plugin_pattern", $plugins[$ix]["pattern"]);
                $plin->set("plugin_content", $plugins[$ix]["content"]);
                $plin->execute();
              }
            }
          }
        } // foreach
      }
      // Check/save page ordering..
      if (isset($pages_recmaintpost_order) && $pages_recmaintpost_order != "") {
        $ord = 1;
        $idlist = explode(FIELD_DELIM, $pages_recmaintpost_order);
        foreach ($idlist as $microsite_page_id) {
          $upd = new dbupdate("ax_microsite_page");
          $upd->where("microsite_page_id=$microsite_page_id");
          $upd->set("display_order", $ord);
          $upd->execute();
          $ord += 1;
        }
      }
      // MICROSITE TEMPLATES
      // Template deletes..
      if (isset($templates_recmaintpost_dels) && $templates_recmaintpost_dels != "") {
        $templates_delids = explode(FIELD_DELIM, $templates_recmaintpost_dels);
        foreach ($templates_delids as $deltemplateid) {
          $template = new microsite_template($deltemplateid);
          $template->get();
          $template->delete();
        }
      }
      // Template adds and saves..
      if (isset($templates_recmaintpost_data) && $templates_recmaintpost_data != "") {
        $template_recs = explode(RECORD_DELIM, $templates_recmaintpost_data);
        $template_fields = explode(",", $templates_recmaintpost_flds);
        foreach ($template_recs as $template_rec) {
          $template_values = explode(FIELD_DELIM, $template_rec);
          $microsite_template_id = array_shift($template_values);
          // Cater for new creations..
          if (strstr($microsite_template_id, "NEW_")) {
            $savedid = $microsite_template_id;
            $template = new microsite_template(NEW_MICROSITE_TEMPLATE, $this->microsite_name);
            $template->save();
            $microsite_template_id = $template->microsite_template_id;
          }
          // Update the microsite template data..
          $um = new dbupdate("ax_microsite_template");
          $um->where("microsite_template_id=$microsite_template_id");
          $pos = 0;
          foreach ($template_fields as $template_field) {
            // Set the current value assignment..
            $um->set($template_field, $template_values[$pos++]);
          } // foreach
          $um->execute();
        } // foreach
      }
      // MICROSITE MEDIA
      // Media deletes..
      if (isset($media_recmaintpost_dels) && $media_recmaintpost_dels != "") {
        $media_delids = explode(FIELD_DELIM, $media_recmaintpost_dels);
        foreach ($media_delids as $delcat_id) {
          $medel = new dbdelete("ax_microsite_media");
          $medel->where("microsite_name='" . addslashes($this->microsite_name) . "'");
          $medel->where("AND cat_id=$delcat_id");
          $medel->execute();
        }
      }
      // Media adds and saves..
      if (isset($media_recmaintpost_data) && $media_recmaintpost_data != "") {
        $media_recs = explode(RECORD_DELIM, $media_recmaintpost_data);
        $media_fields = explode(",", $media_recmaintpost_flds);
        foreach ($media_recs as $media_rec) {
          $media_values = explode(FIELD_DELIM, $media_rec);
          $dummy = array_shift($media_values);
          if (strstr($dummy, "NEW_")) {
            $im = new dbinsert("ax_microsite_media");
            $pos = 0;
            foreach ($media_fields as $media_field) {
              $media_value = $media_values[$pos++];
              if ($media_field == "cat_id") {
                $cat_id = $media_value;
                $im->set($media_field, $cat_id);
              }
            } // foreach
            $im->set("microsite_name", $this->microsite_name);
            if (isset($cat_id) && $cat_id != 0) {
              $im->execute();
            }
          }
        } // foreach
      }
      // Uploaded media - automatically added to microsite..
      global $new_media_name;
      $newci = new catalogitem();
      $errmsgs = $newci->upload($new_media_name, $this->microsite_name, "Created for microsite $this->microsite_name");
      if ($newci->valid) {
        $mmin = new dbinsert("ax_microsite_media");
        $mmin->set("microsite_name", $this->microsite_name);
        $mmin->set("cat_id", $newci->cat_id);
        $mmin->execute();
      }

      // Save the global microsite information..
      global $microsite_desc, $microsite_domain;
      global $css, $css_ie;
      $mup = new dbupdate("ax_microsite");
      $mup->set("microsite_desc", $microsite_desc);
      $mup->set("microsite_domain", $microsite_domain);
      $mup->set("css", $css);
      $mup->set("css_ie", $css_ie);
      $mup->set("last_modified", 'now()');
      $mup->execute();

      $this->mode = "editing";
    } // if our microsite being saved

  } // POSTprocess_save
  // ....................................................................
  /**
  * Process a block edit form POST.
  * Assume that the fields have been submitted in a form as named
  * in the config, and grab the POSTed values. This method is executed
  * from the constructor usually, before anything is read in from
  * the database. We get first shot to change data here.
  * @access private
  */
  function POSTprocess() {
    debug_trace($this);
    global $HTTP_POST_VARS, $RESPONSE;
    global $INCDIR, $CATALOGDIR, $IMAGESDIR;
    if (isset($HTTP_POST_VARS["mode"])) {
      $this->mode = $HTTP_POST_VARS["mode"];
      debugbr("POSTprocess: initial mode: $this->mode", DBG_DEBUG);
      switch ($this->mode) {
        case "publish":
          // Publish microsite instance to the internet..
          $this->POSTprocess_save();
          $this->get();
          $this->publish();
          $this->mode = "editing";
          break;

        case "remove":
          // Remove microsite instance from the internet..
          $this->remove_request();
          $this->mode = "editing";
          break;

        case "delete":
          // Delete microsite definition completely..
          $this->get();
          $this->delete();
          // Try for the first microsite we can get..
          $ms = dbrecordset(
                        "SELECT microsite_name"
                      . "  FROM ax_microsite"
                      . " ORDER BY microsite_desc,microsite_name"
                      );
          if ($ms->hasdata) {
            $this->microsite_name = $ms->field("microsite_name");
          }
          $this->mode = "editing";
          break;

        case "add":
          // Add new microsite definition..
          $this->microsite_name = "new_microsite";
          $this->microsite_desc = "My New Microsite";
          $this->mode = "adding";
          break;

        case "cancel":
          // Cancelling add mode..
          $this->mode = "editing";
          break;

        case "adding":
          global $new_microsite_name;
          $this->microsite_name = $new_microsite_name;
          $exists = dbrecordset("SELECT * FROM ax_microsite WHERE microsite_name='" . addslashes($new_microsite_name) . "'");
          if ($exists->hasdata) {
            $this->errmsgs[] = "Microsite '$new_microsite_name' already exists. Pick another name.";
            $this->mode = "editing";
            return;
          }
          // Insert the microsite record..
          $min = new dbinsert("ax_microsite");
          $min->set("microsite_name", $new_microsite_name);
          if (!$min->execute()) {
            $this->errmsgs[] = "Unable to create microsite '$new_microsite_name'. Please inform your sysadmin.";
            $this->mode = "editing";
            return;
          }
          // If all went well, save it..
          $this->POSTprocess_save();
          break;

        case "save":
          $this->POSTprocess_save();
          break;

        case "default_css":
          $this->POSTprocess_save();
          $sheets = $RESPONSE->get_stylesheets("", "");
          $csspath = $RESPONSE->site_docroot . $sheets["ss"];
          $cssin = new inputfile($csspath);
          if ($cssin->opened) {
            $css = $cssin->readall();
            $cssin->closefile();
            if ($css !== false) {
              $this->get();
              $this->css = $css;
              $this->save();
            }
          }
          break;

        case "default_templates":
          $this->POSTprocess_save();
          $this->get();
          // Get array of available default templates..
          $template_paths = get_dirlist(
                          $RESPONSE->site_docroot . $INCDIR,
                          false, // no recursion
                          $regex="/.*default_.*/"
                          );
          // Create new templates, or redefine existing..
          foreach ($template_paths as $template_path) {
            $tpfile = new inputfile($template_path);
            if ($tpfile->opened) {
              $tpcontent = $tpfile->readall();
              if ($tpcontent !== false) {
                $tptype = get_file_extn($template_path);
                $tpname = basename($template_path);
                $tpname = str_replace("default_template_", "", $tpname);
                $tpname = get_file_stem($tpname);
                $template = $this->get_template_by_name_and_type($tpname, $tptype);
                if ($template === false) {
                  $template = new microsite_template(NEW_MICROSITE_TEMPLATE, $this->microsite_name);
                }
                // Assert values and store it..
                $template->template_name = $tpname;
                $template->template_type = $tptype;
                $template->template_content = $tpcontent;
                $template->save();
                $this->templates[$template->microsite_template_id] = $template;
              }
            } // opened
          }
          // Now fix up our page templates..
          $newpages = array();
          foreach ($this->pages as $pgid => $page) {
            if (isset($page->template)) {
              $tpname = $page->template->template_name;
              $tptype = $page->template->template_type;
              $template = $this->get_template_by_name_and_type($tpname, $tptype);
              if ($template !== false) {
                $page->template == $template;
              }
            }
            $newpages[$pgid] = $page;
          }
          $this->pages = $newpages;
          // Save microsite, including all pages data..
          $this->save();
          break;

        case "default_pages":
          $this->POSTprocess_save();
          $this->get();
          // Add the default pages to this new microsite object.
          // Select template to use. We want to find the "main"
          // template by its ID here..
          $main_template  = $this->get_template_by_name_and_type("main", "html");

          // Install default Cm pages..
          $default_pages = array(
            "cm-home.php"     => "HOME|Home page|index|MAIN_MENU~tree_menu%HEADER~cm%FOOTER~cm%MAIN_CONTENT~feat%RIGHT_SIDEBAR~latest",
            "cm-about.php"    => "ABOUT|About|main|MAIN_MENU~tree_menu%HEADER~cm%FOOTER~cm%MAIN_CONTENT~cm%SITE_SEARCH~search_form%MAIN_CONTENT~search_results",
            "cm-products.php" => "PRODUCTS|Products|main|MAIN_MENU~tree_menu%HEADER~cm%FOOTER~cm%MAIN_CONTENT~cm%SITE_SEARCH~search_form%MAIN_CONTENT~search_results",
            "cm-services.php" => "SERVICES|Services|main|MAIN_MENU~tree_menu%HEADER~cm%FOOTER~cm%MAIN_CONTENT~cm%SITE_SEARCH~search_form%MAIN_CONTENT~search_results",
            "cm-contact.php"  => "CONTACT|Contact us|main|MAIN_MENU~tree_menu%HEADER~cm%FOOTER~cm%MAIN_CONTENT~cm%SITE_SEARCH~search_form%MAIN_CONTENT~search_results"
            );
          foreach ($default_pages as $page_path => $page_info) {
            $info = explode("|", $page_info);
            $menu_label = $info[0];
            if ($this->get_page_by_menulabel($menu_label) === false) {
              $page_desc  = $info[1];
              $template_name = $info[2];
              $template = $this->get_template_by_name_and_type($template_name, "html");
              if ($template === false) {
                $template = $main_template;
              }
              $page = new microsite_page(
                  NEW_MICROSITE_PAGE,
                  $this->microsite_name,
                  $page_desc,
                  0,           // cache secs
                  false,       // corepage
                  ($page_path == "cm-home.php") ? true : false, // homepage
                  true,        // enabled
                  $menu_label,
                  "",          // menuopt ID
                  "",          // order
                  $template->template_id
                  );
              $page->save();
              // Plugins defined for the page..
              if (trim($info[3]) != "") {
                $plugins = explode("%", $info[3]);
                if (count($plugins) > 0) {
                  foreach ($plugins as $plugin_info) {
                    $plugin_bits = explode("~", $plugin_info);
                    $plugin_pattern = $plugin_bits[0];
                    $plugin_content = $plugin_bits[1];
                    if ($plugin_pattern != "" && $plugin_content != "") {
                      $plin = new dbinsert("ax_microsite_page_plugin");
                      $pluginid = get_next_sequencevalue("seq_plugin_id", "ax_microsite_page_plugin", "plugin_id");
                      $plin->set("plugin_id", $pluginid);
                      $plin->set("microsite_page_id", $page->microsite_page_id);
                      $plin->set("plugin_pattern", $plugin_pattern);
                      $plin->set("plugin_content", $plugin_content);
                      $plin->execute();
                    }
                  }
                }
              }
            } // not found by menu label
          } // foreach

          // Now install core Axyl pages..
          $core_pages = array(
            "/axyl-news.php"        => "LATEST NEWS|Latest news",
            "/axyl-story-admin.php" => "NEWS EDITOR|News editor",
            "/axyl-catalog.php"     => "MEDIA CATALOG|Media catalog",
            "/axyl-forums.php"      => "FORUMS|Forums",
            "/axyl-login.php"       => "LOGIN|Member login"
            );
          foreach ($core_pages as $page_path => $page_info) {
            $info = explode("|", $page_info);
            $menu_label = $info[0];
            $page_desc  = $info[1];
            $pg = dbrecordset("SELECT * FROM ax_sitepage WHERE page_path='$page_path'");
            if ($pg->hasdata) {
              $spgid = $pg->field("page_id");
              if ($this->get_page_by_sitepage_id($spgid) === false) {
                $page = new microsite_page(
                    NEW_MICROSITE_PAGE,
                    $this->microsite_name,
                    $page_desc,
                    0,           // cache secs
                    true,        // corepage
                    false,       // homepage
                    true,        // enabled
                    $menu_label,
                    "",          // menuopt ID
                    "",          // order
                    $main_template->template_id,
                    $spgid
                    );
                $page->save();
              }
            }
          } // foreach
          break;

        case "default_images":
          include_once("catalog-defs.php");
          $this->POSTprocess_save();
          $this->get();
          // These are the images we consider worthy of copying across
          // from the 'top' Axyl website, to this microsite. If these
          // images aren't found, we do nothing. If we already have them
          // on this microsite, we do nothing.
          $default_images = array(
                "axyl_logo.png"    => "Axyl logo",
                "loginBot.gif"     => "Axyl login bottom piece",
                "masterTile.gif"   => "Axyl main tile",
                "footerTile.gif"   => "Axyl footer tile",
                "topTile.gif"      => "Axyl header tile",
                "axyl_news.gif"    => "Axyl news icon",
                "axylMini.gif"     => "Axyl popup logo",
                "axyl01.gif"       => "Axyl logo piece",
                "axyl01b.gif"      => "Axyl line piece",
                "axyl02.gif"       => "Axyl black block",
                "axyl03.gif"       => "Axyl black/olive block",
                "axyl04.gif"       => "Axyl search left angle",
                "axyl05.gif"       => "Axyl search main piece",
                "axyl06.gif"       => "Axyl search right angle",
                "axyl08.gif"       => "Axyl white block",
                "axyl09.gif"       => "Axyl copyright footer",
                "buildingBack.jpg" => "Right sidebar black",
                "corner.gif"       => "Axyl corner piece",
                "corner1.gif"      => "Axyl corner piece"
                );
          // Get existing media catalog, complete..
          $catalog = new catalog();
          $catalog->search();

          // Get array of available default templates..
          $media_paths = get_dirlist(
                            $RESPONSE->site_docroot . $IMAGESDIR,
                            false // no recursion
                            );
          foreach ($media_paths as $media_path) {
            $media_fname = basename($media_path);
            if (isset($default_images[$media_fname])) {
              $media_filepath = str_replace($RESPONSE->site_docroot, "", $media_path);
              $catitem = $catalog->get_catalogitem_by_filepath($media_filepath);
              if ($catitem === false) {
                $name = ucfirst(get_file_stem($media_fname));
                $desc = $default_images[$media_fname];
                $keywords = "axyl default";

                // Make new catalog item & save..
                $catitem = new catalogitem();
                // Move file into place, and create DB record..
                $cataloged = $catitem->create(
                                $media_path, $media_path, $media_filepath,
                                "",
                                0, 0,
                                $name, $this->microsite_name, $desc, $keywords
                                );
              }
              else {
                $cataloged = true;
              }
              if ($cataloged && !isset($this->catalogitems[$catitem->cat_id])) {
                $mmin = new dbinsert("ax_microsite_media");
                $mmin->set("cat_id", $catitem->cat_id);
                $mmin->set("microsite_name", $this->microsite_name);
                $mmin->execute();
                $this->catalogitems[$catitem->cat_id] = $catitem;
              }
            }
          } // foreach
          break;

      } // switch
    }
    debugbr("POSTprocess: final mode: $this->mode", DBG_DEBUG);
    debug_trace();
  } // POSTprocess

} // micrositemaintainer class

// ----------------------------------------------------------------------
define("NEW_MICROSITE_PAGE", -1);
/**
* Microsite Page - a container class.
* @package microsite
*/
class microsite_page {
  // Public
  /** Unique microsite page ID */
  var $microsite_page_id;
  /** Name of microsite this page is for */
  var $microsite_name;
  /** Title of this page */
  var $page_title = "";
  /** How many seconds to cache this page */
  var $cache_seconds = 0;
  /** Whether this page is an Axyl core page */
  var $corepage = false;
  /** Whether this page is the microsite home page */
  var $microsite_homepage = false;
  /** Whether this page is to be published */
  var $enabled = true;
  /** Label for menu option of this page */
  var $menuoption_label = "";
  /** Order (for displaying menu lables) */
  var $display_order = 999;
  /** Axyl sitepage ID (Fk) */
  var $page_id;
  /** Microsite template ID (Fk) */
  var $microsite_template_id;
  /** Axyl sitepage page filepath */
  var $page_path;
  // Private
  /** The microsite template object applied to this page
      @access private */
  var $template;
  /** Menuoption ID - FK to ax_menuoption record */
  var $menuoption_id;
  // ....................................................................
  /**
  * Constructor
  * Create a new microsite_page
  * @param integer $id The unique microsite page ID
  * @param string $microsite_name The name of the microsite this is for
  * @param string $title The title of this page
  * @param boolean $homepage True if this page is the site homepage (indexpage)
  * @param boolean $enabled True if this page is ok to publish
  * @param string $label The label to use for the menu option for this page
  * @param integer $order The display order (for the menu option)
  * @param integer $template_id The ID of the template record of this page (Fk)
  * @param integer $page_id The ID of the Axyl sitepage record of this page (Fk)
  */
  function microsite_page(
        $id=NEW_MICROSITE_PAGE,
        $microsite_name="",
        $title="",
        $cache_seconds=0,
        $corepage=false,
        $homepage=false,
        $enabled=true,
        $label="",
        $mopid="",
        $order="",
        $template_id="",
        $page_id=""
  ) {
    $this->microsite_page_id = $id;
    $this->microsite_name = $microsite_name;
    $this->page_title = $title;
    $this->cache_seconds = (is_numeric($cache_seconds) ? $cache_seconds : 0);
    $this->corepage = $corepage;
    $this->microsite_homepage = $homepage;
    $this->enabled = $enabled;
    $this->menuoption_label = $label;
    if ($mopid != "") {
      $this->menuoption_id = $mopid;
    }
    if ($order != "") {
      $this->display_order = $order;
    }
    if ($template_id != "") {
      $this->microsite_template_id = $template_id;
    }
    if ($page_id != "") {
      $this->page_id = $page_id;
    }
  } // constructor
  // ....................................................................
  /**
  * Get the microsite page details from the databse.
  * @return boolean True if we successfully got the record
  */
  function get($pageid="") {
    $res = false;
    if ($pageid != "") {
      $this->microsite_page_id = $pageid;
    }
    if (isset($this->microsite_page_id) && $this->microsite_page_id != NEW_MICROSITE_PAGE) {
      $q  = "SELECT *";
      $q .= "  FROM ax_microsite_page";
      $q .= " WHERE microsite_page_id=$this->microsite_page_id";
      $mpQ = dbrecordset($q);
      if ($mpQ->hasdata) {
        $this->microsite_name     = $mpQ->field("microsite_name");
        $this->page_title         = $mpQ->field("page_title");
        $this->cache_seconds      = $mpQ->field("cache_seconds");
        $this->corepage           = $mpQ->istrue("corepage");
        $this->microsite_homepage = $mpQ->istrue("microsite_homepage");
        $this->enabled            = $mpQ->istrue("enabled");
        $this->menuoption_label   = $mpQ->field("menuoption_label");
        $this->display_order      = $mpQ->field("display_order");
        if ($mpQ->field("menuoption_id") != "") {
          $this->menuoption_id    = $mpQ->field("menuoption_id");
        }
        if ($mpQ->field("microsite_template_id") != "") {
          $this->microsite_template_id = $mpQ->field("microsite_template_id");
        }
        if ($mpQ->field("page_id") != "") {
          $this->page_id = $mpQ->field("page_id");
        }
        $res = true;
      }
    }
    return $res;
  } // get
  // ....................................................................
  /**
  * Get the associated details for this page, from the database. This
  * includes the template(s) associated with the page, and also the
  * menuoption_id linking to the associated menu option record.
  */
  function get_info() {
    if (isset($this->template)) {
      unset($this->template);
    }
    if (isset($this->microsite_page_id)) {
      // Template
      if (isset($this->microsite_template_id) && $this->microsite_template_id != "") {
        $q  = "SELECT *";
        $q .= "  FROM ax_microsite_template";
        $q .= " WHERE microsite_template_id=$this->microsite_template_id";
        $tptQ = dbrecordset($q);
        if ($tptQ->hasdata) {
          $this->template = new microsite_template(
                    $tptQ->field("microsite_template_id"),
                    $tptQ->field("microsite_name"),
                    $tptQ->field("template_name"),
                    $tptQ->field("template_type"),
                    $tptQ->field("template_content")
                    );
        }
      }
      // Sitepage
      if (isset($this->page_id) && $this->page_id != "") {
        $q  = "SELECT *";
        $q .= "  FROM ax_sitepage";
        $q .= " WHERE page_id=$this->page_id";
        $pgQ = dbrecordset($q);
        if ($pgQ->hasdata) {
          $this->page_path = $pgQ->field("page_path");
        }
      }
    }
  } // get_info
  // ....................................................................
  /**
  * Delete this page from the database. We also delete all other records
  * associated with this page, and do not rely on RI.
  */
  function delete() {
    global $RESPONSE;
    $res = false;
    if (isset($this->microsite_page_id) && $this->microsite_page_id != NEW_MICROSITE_PAGE) {
      start_transaction();
      // Page plugins..
      $del = new dbdelete("ax_microsite_page_plugin");
      $del->where("microsite_page_id=$this->microsite_page_id");
      $del->execute();

      // Delete associated menuoption..
      if (isset($this->menuoption_id) && $this->menuoption_id != "") {
        $del = new dbdelete("ax_menuoption");
        $del->where("menuoption_id=$this->menuoption_id");
        if ($del->execute()) {
          unset($this->menuoption_id);
        }
      }

      // Delete associated sitepage. Note that we don't delete it
      // if it is a corepage, since those are Axyl system pages..
      if (isset($this->page_id) && $this->page_id != "") {
        if (!$this->corepage) {
          $del = new dbdelete("ax_sitepage");
          $del->where("page_id=$this->page_id");
        }
        unset($this->page_id);
        unset($this->page_path);
      }

      // Delete the page itself..
      $del = new dbdelete("ax_microsite_page");
      $del->where("microsite_page_id=$this->microsite_page_id");
      $del->execute();

      if ( commit() ) {
        $res = true;
        if (!$this->corepage) {
          if (isset($this->page_path) && $this->page_path != "") {
            $DOCROOT = $RESPONSE->site_docroot;
            if (file_exists("$DOCROOT/$this->page_path")) {
              unlink("$DOCROOT/$this->page_path");
              unset($this->page_path);
            }
          }
        }
      }
    }
    return $res;
  } // delete
  // ....................................................................
  /**
  * Update or create the microsite page. Also saves the template data
  * associated with it.
  * @return boolean True if we successfully saved the record
  */
  function save() {
    $res = false;
    if (isset($this->microsite_name) && $this->microsite_name != "") {
      if ($this->microsite_page_id == NEW_MICROSITE_PAGE) {
        $create_new = true;
        $this->microsite_page_id =
                get_next_sequencevalue(
                    "seq_microsite_page_id",
                    "ax_microsite_page",
                    "microsite_page_id"
                    );
        $pgQ = new dbinsert("ax_microsite_page");
        $pgQ->set("microsite_page_id", $this->microsite_page_id);
      }
      else {
        $pgQ = new dbupdate("ax_microsite_page");
        $pgQ->where("microsite_page_id=$this->microsite_page_id");
      }
      // Set data..
      $pgQ->set("microsite_name",     $this->microsite_name);
      $pgQ->set("page_title",         $this->page_title);
      $pgQ->set("cache_seconds",      $this->cache_seconds);
      $pgQ->set("corepage",           $this->corepage);
      $pgQ->set("microsite_homepage", $this->microsite_homepage);
      $pgQ->set("enabled",            $this->enabled);
      $pgQ->set("menuoption_label",   $this->menuoption_label);
      $pgQ->set("display_order",      $this->display_order);
      if (isset($this->menuoption_id) && $this->menuoption_id != "") {
        $pgQ->set("menuoption_id", $this->menuoption_id);
      }
      else {
        $pgQ->set("menuoption_id", NULLVALUE);
      }
      if (isset($this->microsite_template_id) && $this->microsite_template_id != "") {
        $pgQ->set("microsite_template_id", $this->microsite_template_id);
      }
      else {
        $pgQ->set("microsite_template_id", NULLVALUE);
      }
      if (isset($this->page_id) && $this->page_id != "") {
        $pgQ->set("page_id", $this->page_id);
      }
      else {
        $pgQ->set("page_id", NULLVALUE);
      }
      $res = $pgQ->execute();
    }
    return $res;
  } // save
  // ....................................................................
  /**
  * Publish this microsite page. We create a sitepage record, and save
  * a FK to it, create a menuoption for the page and FK to that, and
  * we create a physical page from the microsite template.
  * @param string $pubdir Web-relative path to dir to save the page file
  * @param string $realdir Path for menuoption - for page access when published
  * @param integer $menu_id ID of the menu to create menuoptions for
  */
  function publishto($pubdir, $realdir, $menu_id) {
    global $RESPONSE, $TEMPLATESDIR, $INCDIR, $CMDIR;

    // Document root for whole site..
    $DOCROOT = $RESPONSE->site_docroot;

    // Have we got a sitepage at all yet..
    if (!$this->corepage) {
      if (!isset($this->page_id) || $this->page_id == "") {
        $this->page_id = get_next_sequencevalue("seq_page_id", "ax_sitepage", "page_id");
        $pgup = new dbinsert("ax_sitepage");
        $pgup->set("page_id", $this->page_id);
        $pgup->set("managed", true);
        $pgup->set("enabled", true);
        debugbr("publish page: inserting new page $this->page_id", DBG_DEBUG);
      }
      else {
        $pgup = new dbupdate("ax_sitepage");
        $pgup->where("page_id=$this->page_id");
        debugbr("publish page: updating existing page $this->page_id", DBG_DEBUG);
      }
      if ($this->page_title != "") {
        $pagename = $this->page_title;
      }
      elseif ($this->menuoption_label != "") {
        $pagename = $this->menuoption_label;
      }
      else {
        $this->pagename = $this->page_id;
      }
      $this->page_path = "$realdir/cm-" . str_replace(" ", "_", strtolower($pagename)) . ".php";
      $phys_path = $DOCROOT . "$pubdir/cm-" . str_replace(" ", "_", strtolower($pagename)) . ".php";
      $pgup->set("page_title", $this->page_title);
      $pgup->set("page_path", $this->page_path);
      $pgup->execute();

      // Always refresh the file itself, and reassert the template
      // and title settings..
      if (file_exists($phys_path)) {
        unlink($phys_path);
        debugbr("publish page: unlinking existing page: $phys_path", DBG_DEBUG);
      }

      // Make a new physical page from the template..
      $template_page = $DOCROOT . "$INCDIR/microsite-template.inc";
      if (isset($this->template)) {
        $page_template = $this->template->template_name;
      }
      else {
        $page_template = "main";
      }
      if (file_exists($template_page)) {
        // Generate a unique layout ID for CM..
        $layout_id = $this->microsite_name . "_"
                  . $this->microsite_page_id . "_"
                  . str_replace(" ", "_", $this->page_title);
        // Make our new page on disk..
        $newpage = new outputfile($phys_path);
        $newpage->template($template_page);
        $newpage->replace("microsite-template.inc", $this->page_path);
        $newpage->replace("My Layout Title", $this->page_title);
        $newpage->replace("My Template Name", $page_template);

        // Are we caching the page?..
        if ($this->cache_seconds > 0) {
          $newpage->replace(
              "// THIS PAGE IS NOT CACHED (do not remove this line!)",
              "\$RESPONSE->cache($this->cache_seconds);"
              );
        }

        // Plugin content we will insert..
        $PLUGIN_CONTENT = "";

        // Build up our plugins replacement string..
        $q  = "SELECT * FROM ax_microsite_page_plugin pp, ax_plugin_content pc";
        $q .= " WHERE pp.microsite_page_id=$this->microsite_page_id";
        $q .= "   AND pc.plugin_content=pp.plugin_content";
        $q .= " ORDER BY replace_content";
        $pluginQ = dbrecordset($q);
        if ($pluginQ->hasdata) {
          do {
            $pluginid        = $pluginQ->field("plugin_id");
            $plugin_pattern  = $pluginQ->field("plugin_pattern");
            $generator       = $pluginQ->field("generator");
            $generator_type  = $pluginQ->field("generator_type");
            $content_type    = $pluginQ->field("plugin_content");
            $replace_content = $pluginQ->istrue("replace_content");
            // Generate the plugin content according to type..
            switch ($generator_type) {
              case "cm":
                $content = new cm_plugin_content(
                              $layout_id . "_"
                            . strtolower($plugin_pattern)
                            );
                if ($content->valid) {
                  $plugin_content = $content->render();
                }
                break;
              case "func":
                $content = new func_plugin_content($generator);
                if ($content->valid) {
                  $plugin_content = $generator;
                }
                break;
              case "defer":
                $plugin_content = "\"" . addslashes($generator) . "\"";
                break;
              case "lit":
                $plugin_content = "\"" . addslashes($generator) . "\"";
                break;
              case "file":
                $content = new file_plugin_content($generator);
                if (!$content->valid) {
                  $content = new file_plugin_content($RESPONSE->site_docroot . $generator);
                }
                if ($content->valid) {
                  $plugin_content = "\"" . addslashes($content->render()) . "\"";
                }
                break;
            } // switch

            // Render the complete plugin..
            $s  = "\n// Plugin ID #$pluginid\n";
            if ($replace_content && $generator_type != "cm" && $generator_type != "defer") {
              // Replace existing content, if content is immediately available
              // and is not a nullstring..
              $arg = "\"" . strtoupper($plugin_pattern) . "\", \$content";
              $s .= "\$content = $plugin_content;\n";
              $s .= "if (trim(\$content) != \"\") {\n";
              $s .= "  \$RESPONSE->plugin_replace($arg);\n";
              $s .= "}";
            }
            else {
              // Append content - ie. the usual plugin methodology..
              $arg = "\"" . strtoupper($plugin_pattern) . "\", " . $plugin_content;
              $s .= "\$RESPONSE->plugin($arg);";
            }
            // Accumulate..
            $PLUGIN_CONTENT .= "$s\n";

          } while ($pluginQ->get_next());

          if ($PLUGIN_CONTENT != "") {
            $newpage->replace(
                "// PLUGIN CONTENT PLACEMENT (do not remove this line!)",
                $PLUGIN_CONTENT
                );
          }
        }
        $newpage->closefile();
        debugbr("publish page: writing the page out to disk: $phys_path", DBG_DEBUG);
      }
    } // if corepage

    // Update or create microsite menuoption..
    if (isset($this->menuoption_id) && $this->menuoption_id != "") {
      $mnuQ = new dbupdate("ax_menuoption");
      $mnuQ->where("menuoption_id=$this->menuoption_id");
    }
    else {
      $this->menuoption_id = get_next_sequencevalue("seq_menuoption_id", "ax_menuoption", "menuoption_id");
      $mnuQ = new dbinsert("ax_menuoption");
      $mnuQ->set("menuoption_id", $this->menuoption_id);
      $mnuQ->set("width",         189); // a default width (px)
      debugbr("publish page: creating new menuoption $this->menuoption_id", DBG_DEBUG);
    }
    $mnuQ->set("menu_id",       $menu_id);
    $mnuQ->set("parent_id",     0);
    $mnuQ->set("menu_level",    0);
    $mnuQ->set("label",         $this->menuoption_label);
    $mnuQ->set("description",   $this->page_title);
    $mnuQ->set("display_order", $this->display_order);
    $mnuQ->set("action",        $this->page_path);
    $mnuQ->set("sitepage",      $this->page_path);
    $mnuQ->set("active",        $this->enabled);
    $mnuQ->set("last_modified", 'now()');
    $mnuQ->execute();

    // Save any new ID's etc..
    $this->save();
  } // publishto

} // class microsite_page

// ----------------------------------------------------------------------
define("NEW_MICROSITE_TEMPLATE", -1);
/**
* Microsite Template - a container class.
* @package microsite
*/
class microsite_template {
  // Public
  /** Unique microsite template ID */
  var $microsite_template_id;
  /** Template name */
  var $template_name = "";
  /** Template type 'html' or 'wml' */
  var $template_type = "html";
  /** Template content */
  var $template_content = "";

  // Private
  /** The name of the microsite this template is for
      @access private */
  var $microsite_name;
  // ....................................................................
  /**
  * Constructor
  * Create a new microsite_template
  * @param integer $id The unique microsite template ID
  * @param string $microsite_name The name of the microsite this is for
  * @param string $name The name of this temaplate
  * @param string $type The type of this temaplate 'html' or 'wml'
  * @param string $content The actual template content
  */
  function microsite_template($id=NEW_MICROSITE_TEMPLATE, $microsite_name="", $name="", $type="html", $content="") {
    $this->microsite_template_id = $id;
    $this->microsite_name = $microsite_name;
    $this->template_name = $name;
    $this->template_type = $type;
    $this->template_content = $content;
  }// microsite_template
  // ....................................................................
  /**
  * Get the current microsite template, as identified by the ID, from
  * the database.
  * @return boolean True if we successfully got the record
  */
  function get() {
    $res = false;
    if (isset($this->microsite_template_id)
    && $this->microsite_template_id != NEW_MICROSITE_TEMPLATE) {
      $q  = "SELECT * FROM ax_microsite_template";
      $q .= " WHERE microsite_template_id=$this->microsite_template_id";
      $tpQ = dbrecordset($q);
      if ($tpQ->hasdata) {
        $this->microsite_name   = $tpQ->field("microsite_name");
        $this->template_name    = $tpQ->field("template_name");
        $this->template_type    = $tpQ->field("template_type");
        $this->template_content = $tpQ->field("template_content");
        $res = true;
      }
    }
    return $res;
  } // get
  // ....................................................................
  /**
  * Update or create the microsite template to the database.
  * @return boolean True if we successfully saved the record
  */
  function save() {
    if (isset($this->microsite_name) && $this->microsite_name != "") {
      if ($this->microsite_template_id == NEW_MICROSITE_TEMPLATE) {
        $this->microsite_template_id =
                get_next_sequencevalue(
                    "seq_microsite_template_id",
                    "ax_microsite_template",
                    "microsite_template_id"
                    );
        $mstQ = new dbinsert("ax_microsite_template");
        $mstQ->set("microsite_template_id", $this->microsite_template_id);
      }
      else {
        $mstQ = new dbupdate("ax_microsite_template");
        $mstQ->where("microsite_template_id=$this->microsite_template_id");
      }
      // Set data..
      $mstQ->set("microsite_name",   $this->microsite_name);
      $mstQ->set("template_name",    $this->template_name);
      $mstQ->set("template_type",    $this->template_type);
      $mstQ->set("template_content", $this->template_content);
      return $mstQ->execute();
    }
  } // save
  // ....................................................................
  /**
  * Delete the current microsite template, from the database.
  * @return boolean True if we successfully deleted the record
  */
  function delete() {
    if (isset($this->microsite_template_id)
    && $this->microsite_template_id != NEW_MICROSITE_TEMPLATE) {
      $tpdel = new dbdelete("ax_microsite_template");
      $tpdel->where("microsite_template_id=$this->microsite_template_id");
      return $tpdel->execute();
    }
  } // delete
    // ....................................................................
  /**
  * Publish the template(s) into the given directory. The filename will
  * be as per the Axyl standard, depending on template name and type.
  * @param string $dirpath Path to a directory to publish this template.
  */
  function publishto($pubdir) {
    global $RESPONSE;
    $template_file = "template_" . $this->template_name . "." . strtolower($this->template_type);
    $template_path = $RESPONSE->site_docroot . "$pubdir/$template_file";
    $template = new outputfile($template_path);
    $template->write($this->template_content);
    $template->closefile();
  } // publish

} // class microsite_template

// --------------------------------------------------------------------
?>