
/*
 *  Diverse SLab audio routines.
 *  Copyright (c) by Nick Copeland <nick.copeland@ntlworld.com> 1996,2002
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */


/*
 * These are largely device specific operations for the audio devices. Some of
 * the calls are general, but may eventually be used by several of the SLab
 * applications.
 *
 * This file contains the ALSA specific calls.
 */

#ifdef DEBUG
#include <stdio.h>
#endif
#include <unistd.h>
#include <fcntl.h>
#ifdef SUBFRAGMENT
#include <malloc.h>
#endif
#include <stdlib.h>

/*
 * Audio device structure format definitions.
 */
#include "slabaudiodev.h"
#include "slabalsadev.h"

#ifdef ALSA_SUPPORT
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR <= 5)

/*
 * Setup audio parameters with device support.
 */
setAudioALSAparam(audioDev, devID, param, left, right)
duplexDev *audioDev;
char *param;
int devID;
short left, right;
{
	int cont;

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("setAudioALSAparam(%s, %i, %i): %i, %i\n", param, left, right,
			devID, audioDev->devID);

	if ((cont = getAlsaCapability(audioDev, param)) == -1)
		return;

	setAlsaValue(audioDev, cont, 0, left);
	setAlsaValue(audioDev, cont, 1, right);

	return;
}

/*
 * Binned since alsa support was implemented.
 *
 * This is a device specific routine. We take the fd, parameter, and left
 * right values, and configure the device. These routines should be used by the
 * engine to do the actual audio horsework.
 *
 * This is actually an internal routine, it should not be called from outside
 * of this file.
 */
SL_setAudioALSAparameter(fd, audioDev, param, valueL, valueR)
duplexDev *audioDev;
{
	return(0);
}

checkAudioALSAcaps(audioDev, devID, fd)
duplexDev *audioDev;
int devID, fd;
{
}

closeALSAmixer(audioDev)
duplexDev *audioDev;
{
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("closeALSAmixer(): %08x\n", alsaDev[audioDev->devID].mh);

	if (alsaDev[audioDev->devID].mh != (snd_mixer_t *) NULL)
	{
		int err;

		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("real closeALSAmixer(): %08x\n",
				alsaDev[audioDev->devID].mh);
		snd_mixer_close((void *) alsaDev[audioDev->devID].mh);
		if ((err = snd_ctl_close(alsaDev[audioDev->devID].ch)) < 0) {
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("SND CTL Close error: %s\n", snd_strerror(err));
		}
	}

	alsaDev[audioDev->devID].mh = NULL;
	alsaDev[audioDev->devID].ch = NULL;
}

openALSAmixer(audioDev)
duplexDev *audioDev;
{
	int card = 0, dev = 0, err, eindex = 0;

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("openALSAmixer(): %08x, %s %s\n", alsaDev[audioDev->devID].mh,
			audioDev->devName, audioDev->mixerName);

	if (alsaDev[audioDev->devID].mh != (snd_mixer_t *) NULL)
		closeALSAmixer(audioDev);

	card = atoi(&audioDev->devName[0]);
	dev = atoi(&audioDev->mixerName[0]);

	if ((err = snd_ctl_open(&alsaDev[audioDev->devID].ch, card)) < 0) {
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("SND CTL Open error: %s\n", snd_strerror(err));
		return -1;
	}
	if ((err = snd_ctl_hw_info(alsaDev[audioDev->devID].ch,
		&alsaDev[audioDev->devID].hwInfo)) < 0) {
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Mixer info error: %s\n", snd_strerror(err));
		return -1;
	} else {
		if (audioDev->cflags & SLAB_AUDIODBG)
		{
			printf("Found HW Device:\n");
			printf("  type - %i\n", alsaDev[audioDev->devID].hwInfo.type);
			printf("  PCM - %i\n", alsaDev[audioDev->devID].hwInfo.pcmdevs);
			printf("  mixers - %i\n",
				alsaDev[audioDev->devID].hwInfo.mixerdevs);
			printf("  ID - %s\n", alsaDev[audioDev->devID].hwInfo.id);
			printf("  abbr - %s\n",
				alsaDev[audioDev->devID].hwInfo.abbreviation);
			printf("  name - %s\n", alsaDev[audioDev->devID].hwInfo.name);
			printf("  longname - %s\n",
				alsaDev[audioDev->devID].hwInfo.longname);
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 4)
			printf("  switches - %i\n",
				alsaDev[audioDev->devID].hwInfo.switches);
#endif
		}
	}

	if (snd_mixer_open((void *) &alsaDev[audioDev->devID].mh, card, dev) < 0)
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("  FAILED\n");
		return -1;
	} else {
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("mixer %i:%i open succeeded: %08x\n",
				card, dev, alsaDev[audioDev->devID].mh);
	}

	if ((err = snd_mixer_info(alsaDev[audioDev->devID].mh,
		&alsaDev[audioDev->devID].mixerInfo)) < 0) {
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Mixer info error: %s\n", snd_strerror(err));
	} else {
		if (audioDev->cflags & SLAB_AUDIODBG)
		{
			printf("Found Mixing Device:\n");
			printf("  type - %i\n", alsaDev[audioDev->devID].mixerInfo.type);
			printf("  ATTR - %i\n", alsaDev[audioDev->devID].mixerInfo.attrib);
			printf("  ID - %s\n", alsaDev[audioDev->devID].mixerInfo.id);
			printf("  name - %s\n", alsaDev[audioDev->devID].mixerInfo.name);
			printf("  elements - %i\n",
				alsaDev[audioDev->devID].mixerInfo.elements);
			printf("  groups - %i\n",
				alsaDev[audioDev->devID].mixerInfo.groups);
#if (SND_LIB_MAJOR == 0 && SND_LIB_MINOR == 4)
			printf("  switches - %i\n",
				alsaDev[audioDev->devID].mixerInfo.switches);
#endif
		}
	}

	/* GROUPS list */

	alsaDev[audioDev->devID].gsInfo.groups_size =
		alsaDev[audioDev->devID].mixerInfo.groups;
	alsaDev[audioDev->devID].gsInfo.pgroups = alsaDev[audioDev->devID].pgroups;

	bzero(alsaDev[audioDev->devID].gInfo, sizeof(snd_mixer_group_t) * 64);

	if ((err = snd_mixer_groups(alsaDev[audioDev->devID].mh,
		&alsaDev[audioDev->devID].gsInfo)) < 0){
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Mixer group error: %s\n", snd_strerror(err));
	} else {
		int i;

		if (audioDev->cflags & SLAB_AUDIODBG)
		{
			printf("\nFound groups\n");
			printf("  size - %i\n",
				alsaDev[audioDev->devID].gsInfo.groups_size);
			printf("  groups - %i\n", alsaDev[audioDev->devID].gsInfo.groups);
			printf("  over - %i\n\n",
				alsaDev[audioDev->devID].gsInfo.groups_over);
		}

		for (i = 0; i < alsaDev[audioDev->devID].gsInfo.groups; i++)
		{
			if (alsaDev[audioDev->devID].gsInfo.pgroups == NULL)
			{
				if (audioDev->cflags & SLAB_AUDIODBG)
					printf("    no groups listed\n");
				break;
			}
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("    %s, %i, index %i\n",
				&alsaDev[audioDev->devID].gsInfo.pgroups[i].name,
				alsaDev[audioDev->devID].gsInfo.pgroups[i].index, i);

			strcpy(alsaDev[audioDev->devID].gInfo[i].gid.name,
				alsaDev[audioDev->devID].gsInfo.pgroups[i].name);
			alsaDev[audioDev->devID].gInfo[i].gid.index =
				alsaDev[audioDev->devID].gsInfo.pgroups[i].index;

			/* GROUP check */
			if ((err = snd_mixer_group_read(alsaDev[audioDev->devID].mh,
				&alsaDev[audioDev->devID].gInfo[i]))
				< 0) {
				if (audioDev->cflags & SLAB_AUDIODBG)
					printf("Mixer group error: %s\n", snd_strerror(err));
			} else {
				int j, k;

				/*
				 * What the fuck. We do not seem to be able to know the element
				 * count for any given group in advance. This would imply that
				 * we need to do this call twice. Once to get the _over count,
				 * then allocate that count, and call again?
				 * I'm missing something.
				 */
				alsaDev[audioDev->devID].gInfo[i].elements_size =
					alsaDev[audioDev->devID].gInfo[i].elements_over;
				alsaDev[audioDev->devID].gInfo[i].pelements =
					&alsaDev[audioDev->devID].pelements[eindex];

				eindex += alsaDev[audioDev->devID].gInfo[i].elements_size;

				snd_mixer_group_read(alsaDev[audioDev->devID].mh,
					&alsaDev[audioDev->devID].gInfo[i]);

				if (audioDev->cflags & SLAB_AUDIODBG)
				{
					printf("      size - %i\n",
					alsaDev[audioDev->devID].gInfo[i].elements_size);
					printf("      groups - %i\n",
						alsaDev[audioDev->devID].gInfo[i].elements);
					printf("      over - %i\n",
						alsaDev[audioDev->devID].gInfo[i].elements_over);
				}
				for (j = 0; j < alsaDev[audioDev->devID].gInfo[i].elements_size;
					j++)
				{
					if (alsaDev[audioDev->devID].gInfo[i].pelements[j].name[0]
						!= '\0')
						if (audioDev->cflags & SLAB_AUDIODBG)
							printf("        %s, %i, %i\n",
								&alsaDev[audioDev->devID]
									.gInfo[i].pelements[j].name[0],
								alsaDev[audioDev->devID]
									.gInfo[i].pelements[j].index,
								alsaDev[audioDev->devID]
									.gInfo[i].pelements[j].type);

					strcpy(alsaDev[audioDev->devID].eData[i].eid.name,
						alsaDev[audioDev->devID].gInfo[i].pelements[j].name);
					alsaDev[audioDev->devID].eData[i].eid.index =
						alsaDev[audioDev->devID].gInfo[i].pelements[j].index;
					alsaDev[audioDev->devID].eData[i].eid.type =
						alsaDev[audioDev->devID].gInfo[i].pelements[j].type;

					/* ELEMENT check - bin this for later developments
					if ((err =
						snd_mixer_element_info(alsaDev[audioDev->devID].mh,
						&alsaDev[audioDev->devID].eData[i])) >= 0)
					{
						if (audioDev->cflags & SLAB_AUDIODBG)
							printf("GOT STUFF\n");
					}
					 */
				}

				if (audioDev->cflags & SLAB_AUDIODBG)
				{
					printf("      caps - %i\n",
						alsaDev[audioDev->devID].gInfo[i].caps);
					printf("      channels - %i\n",
						alsaDev[audioDev->devID].gInfo[i].channels);
					printf("      mute - %i\n",
						alsaDev[audioDev->devID].gInfo[i].mute);
					printf("      capt - %i\n",
						alsaDev[audioDev->devID].gInfo[i].capture);
					printf("      captgrp - %i\n",
						alsaDev[audioDev->devID].gInfo[i].capture_group);
					printf("      min - %i\n",
						alsaDev[audioDev->devID].gInfo[i].min);
					printf("      max - %i\n",
						alsaDev[audioDev->devID].gInfo[i].max);
				}

				for (k = 0; k < 32; k++)
				{
					if ((1 << k) & alsaDev[audioDev->devID].gInfo[i].channels)
					{
						if (audioDev->cflags & SLAB_AUDIODBG)
							printf("      vol%i - %i\n", k,
							alsaDev[audioDev->devID].gInfo[i].volume.values[k]);
					}
				}

				/*
				 * For testing at least, unmute everything, and give it some
				 * reasonable signal level.
				alsaDev[audioDev->devID].gInfo[i].mute = 3;
				for (k = 0; k < 32; k++)
				{
					if ((1<<k) & alsaDev[audioDev->devID].gInfo[i].channels)
					{
						alsaDev[audioDev->devID].gInfo[i].volume.values[k] =
							alsaDev[audioDev->devID].gInfo[i].min;
					}
				}

				snd_mixer_group_write(alsaDev[audioDev->devID].mh,
					&alsaDev[audioDev->devID].gInfo[i]);
				if ((err =
					snd_mixer_group_write(alsaDev[audioDev->devID].mh,
						&alsaDev[audioDev->devID].gInfo[i]))
						< 0)
					if (audioDev->cflags & SLAB_AUDIODBG)
						printf("snd_mixer_group_write: %s\n",
							snd_strerror(err));
				 */
			}
		}
	}

	/*
	 * ELEMENTS list - Not going to use this part. Maybe later.
	 *
	audioDev->esInfo.elements_size =
		alsaDev[audioDev->devID].mixerInfo.elements;
	audioDev->esInfo.pelements = &audioDev->pelements[eindex];
	eindex += audioDev->esInfo.elements_size;

	if ((err = snd_mixer_elements(alsaDev[audioDev->devID].mh,
		&audioDev->esInfo))
		< 0) {
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Mixer element error: %s\n", snd_strerror(err));
	} else {
		int i;
		snd_mixer_element_t *ePtr;

		if (audioDev->cflags & SLAB_AUDIODBG)
		{
			printf("\nFound elements\n");
			printf("  size - %i\n", audioDev->esInfo.elements_size);
			printf("  elements - %i\n", audioDev->esInfo.elements);
			printf("  over - %i\n", audioDev->esInfo.elements_over);
		}

		for (i = 0; i < audioDev->esInfo.elements; i++)
		{
			if (audioDev->esInfo.pelements[i].name[0] != '\0')
				if (audioDev->cflags & SLAB_AUDIODBG)
					printf("    %s, %i, %i\n",
						&audioDev->esInfo.pelements[i].name,
						audioDev->esInfo.pelements[i].index,
						audioDev->esInfo.pelements[i].type);

			strcpy(alsaDev[audioDev->devID].eData[i].eid.name,
				audioDev->esInfo.pelements[i].name);
			alsaDev[audioDev->devID].eData[i].eid.index =
				audioDev->esInfo.pelements[i].index;
			alsaDev[audioDev->devID].eData[i].eid.type =
				audioDev->esInfo.pelements[i].type;

			if ((err = snd_mixer_element_info(alsaDev[audioDev->devID].mh,
				&alsaDev[audioDev->devID].eData[i])) >= 0)
			{
				if (audioDev->cflags & SLAB_AUDIODBG)
					printf("GOT STUFF 2\n");
			}
		}
	}
	*/
	return(0);
}

validAlsaDev(audioDev, index)
duplexDev *audioDev;
{
	if (index >= alsaDev[audioDev->devID].mixerInfo.groups)
		return 0;

	return 1;
}

static char NO_NAME[8] = "none";

char *
getAlsaDeviceName(audioDev)
duplexDev *audioDev;
{
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("GetAlsaDeviceName %i\n", audioDev->devID,
			&alsaDev[audioDev->devID].hwInfo.name[0]);

	if ((alsaDev[audioDev->devID].hwInfo.name == NULL)
		|| (&alsaDev[audioDev->devID].hwInfo.name[0] == NULL)
		|| (alsaDev[audioDev->devID].hwInfo.name[0] == '\0'))
		return(&NO_NAME[0]);

	return &alsaDev[audioDev->devID].hwInfo.name[0];
}

char *
getAlsaName(audioDev, cont)
duplexDev *audioDev;
{
	if (cont < 0)
		return(&NO_NAME[0]);
	if (cont < alsaDev[audioDev->devID].mixerInfo.groups)
		return((char *) &alsaDev[audioDev->devID].gsInfo.pgroups[cont].name[0]);

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("getAlsaName %i: %s\n", cont,
			&alsaDev[audioDev->devID].gsInfo.pgroups[cont].name);

	return(&NO_NAME[0]);
}

getAlsaMutability(audioDev, cont)
duplexDev *audioDev;
{
	if (cont < 0)
		return(-2);
	/*
	 * No idea why I cannot find this flag in the header files - 0.4.1e?
	 */
	if (alsaDev[audioDev->devID].gInfo[cont].caps & 0x4)
	{
		if (alsaDev[audioDev->devID].gInfo[cont].mute == 0)
			return 0;
		return 1;
	}
	/*
	 * -2 indicates no support
	 */
	return -2;
}

getAlsaRecordability(audioDev, cont)
duplexDev *audioDev;
{
	if (cont < 0)
		return(-2);

	/*
	 * No idea why I cannot find this flag in the header files - 0.4.1e?
	 * Also, this is unlikely to be correct flag. Net results is that every
	 * mixer group functions as "recordable" in SLab gui. Could be worse.
	 */
	if (alsaDev[audioDev->devID].gInfo[cont].caps & 0x1)
	{
		if (alsaDev[audioDev->devID].gInfo[cont].capture == 0)
			return 0;
		return 1;
	}
	/*
	 * -2 indicates no support
	 */
	return -2;
}

setAlsaRecordSource(audioDev, cont, position)
duplexDev *audioDev;
{
	int err;
	snd_mixer_group_t param;

	if (cont < 0)
		return(-2);
	/*
	 * Get the current value
	 */
	bzero(&param, sizeof(snd_mixer_group_t));
	strcpy(param.gid.name, alsaDev[audioDev->devID].gsInfo.pgroups[cont].name);
	param.gid.index = alsaDev[audioDev->devID].gsInfo.pgroups[cont].index;

	if ((err = snd_mixer_group_read(alsaDev[audioDev->devID].mh, &param)) < 0)
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("snd_mixer_group_read: %s\n", snd_strerror(err));

	if (position == 0)
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Clearing ALSA record source %s\n", param.gid.name);
		param.capture = 0;
	} else {
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("Setting ALSA record source %s\n", param.gid.name);
		param.capture = 3;
	}

	if ((err =
		snd_mixer_group_write(alsaDev[audioDev->devID].mh, &param)) < 0)
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("snd_mixer_group_write: %s\n", snd_strerror(err));

	return(1);
}

getAlsaStereoStatus(audioDev, cont)
duplexDev *audioDev;
{
	int k, accum = 0;

	if (cont < 0)
		return(-2);

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("getAlsaStereoStatus()\n");

	for (k = 0; k < 32; k++)
	{
		if ((1<<k) & alsaDev[audioDev->devID].gInfo[cont].channels)
			accum++;
	}
	return(accum);
}

setAlsaMute(audioDev, cont, onoff)
duplexDev *audioDev;
{
	int err;
	snd_mixer_group_t param;

	if (cont < 0)
		return(-2);
	/*
	 * Get the current value
	 */
	bzero(&param, sizeof(snd_mixer_group_t));
	strcpy(param.gid.name, alsaDev[audioDev->devID].gsInfo.pgroups[cont].name);
	param.gid.index = alsaDev[audioDev->devID].gsInfo.pgroups[cont].index;
	if ((err = snd_mixer_group_read(alsaDev[audioDev->devID].mh, &param)) < 0)
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("snd_mixer_group_read: %s\n", snd_strerror(err));

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("setAlsaMute()\n");

	if (onoff)
		param.mute = 3;
	else
		param.mute = 0;

	if ((err = snd_mixer_group_write(alsaDev[audioDev->devID].mh, &param)) < 0)
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("snd_mixer_group_write: %s\n", snd_strerror(err));

	if ((err = snd_mixer_group_read(alsaDev[audioDev->devID].mh, &param)) < 0)
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("snd_mixer_group_read: %s\n", snd_strerror(err));

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("configured %s (%i, %i)\n", param.gid.name, cont, param.mute);

	return(1);
}

getAlsaValue(audioDev, cont, side)
duplexDev *audioDev;
{
	if (side == 1)
		side = SND_MIXER_CHN_FRONT_LEFT;
	if (side == 2)
		side = SND_MIXER_CHN_FRONT_RIGHT;

	if (cont < 0)
		return(-2);

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("getAlsaValue(%i, %i)\n", cont, side);

	return (alsaDev[audioDev->devID].gInfo[cont].volume.values[side]);
}

setAlsaValue(audioDev, cont, side, value)
duplexDev *audioDev;
{
	int tVal, err;
	snd_mixer_group_t param;

	if (cont < 0)
		return(-2);

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("setAlsaValue(%i, %i, %i)\n", cont, side, value);

	bzero(&param, sizeof(snd_mixer_group_t));

	strcpy(param.gid.name, alsaDev[audioDev->devID].gsInfo.pgroups[cont].name);
	param.gid.index = alsaDev[audioDev->devID].gsInfo.pgroups[cont].index;

	if ((err = snd_mixer_group_read(alsaDev[audioDev->devID].mh, &param)) < 0)
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("snd_mixer_group_read: %s\n", snd_strerror(err));

	if (side == 1)
		side = SND_MIXER_CHN_FRONT_LEFT;
	else if (side == 2)
		side = SND_MIXER_CHN_FRONT_RIGHT;
	else
		side -= 1;

	/*
	 * If we do not support this channel, return.
	 */
	if (((1<<side) & param.channels) == 0)
		return;

	tVal = (param.max - param.min) * value / 100 + param.min;
	/*
	 * Some settings cause problems with busy, going to try missing max values.
	 */
	if (tVal >= param.max)
		tVal = param.max - 1;

	param.volume.values[side] = tVal;

	if ((err = snd_mixer_group_write(alsaDev[audioDev->devID].mh, &param)) < 0)
	{
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("snd_mixer_group_write: %s\n", snd_strerror(err));
		/*
		 * This is the sledgehammer approach. If we get busy returns, which we
		 * do for some of the more esoteric controllers, then close the mixer
		 * and open it again.
		 */
		closeALSAmixer(audioDev);
		openALSAmixer(audioDev);
	}

	if ((err = snd_mixer_group_read(alsaDev[audioDev->devID].mh, &param)) < 0)
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("snd_mixer_group_read: %s\n", snd_strerror(err));

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("configured %s (%i, %i, %i)\n", param.gid.name, cont, side,
			param.volume.values[side]);

	return(1);
}

getAlsaCapability(audioDev, controller)
duplexDev *audioDev;
int controller;
{
	int i, result = -1;

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("getAlsaCapability(%i)\n", controller);

	if (controller < alsaDev[audioDev->devID].gsInfo.groups_size)
		return(controller);
/*
	for  (i = 0; i < alsaDev[audioDev->devID].mixerInfo.groups; i++)
	{
		if (strcmp(controller, alsaDev[audioDev->devID].gInfo[i].gid.name) == 0)
			return i;
	}
*/
	return(result);
}

getAlsaCapByName(audioDev, name)
duplexDev *audioDev;
char *name;
{
	int i;

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("AlsaCapByName %x %s\n", audioDev, name);

	for (i = 0; i < 32; i++)
	{
		if (&alsaDev[audioDev->devID].gsInfo.pgroups[i].name == NULL)
			return -1;

		if (strcmp(alsaDev[audioDev->devID].gsInfo.pgroups[i].name, name) == 0)
			return i;
	}
	return -1;
}
#else /* ALSA Minor vers0ion <= 5 */
/*
 * This is for ALSA 0.9, there were such massive alterations to the audio
 * interface definitions that a general rewrite was required.
 * This is always painful stuff, hopefully after this release things will
 * be a bit more stable.
 */

validAlsaDev(duplexDev *audioDev, int cont)
{
	if (cont >= alsaDev[audioDev->devID].elem_count)
		return(0);
	return(1);
}

const char *
getAlsaName(duplexDev *audioDev, int cont)
{
	snd_mixer_selem_id_t *sid;

	sid = (snd_mixer_selem_id_t *)(((char *) alsaDev[audioDev->devID].mixer_sid)
		+ snd_mixer_selem_id_sizeof() * cont);

if (audioDev->cflags & SLAB_AUDIODBG)
printf("getAlsaName(%i): \"%s\"\n", cont, snd_mixer_selem_id_get_name(sid));
	return(snd_mixer_selem_id_get_name(sid));
}

getAlsaRecordability(duplexDev *audioDev, int cont)
{
	snd_mixer_elem_t *elem;
	snd_mixer_selem_id_t *sid;

if (audioDev->cflags & SLAB_AUDIODBG)
printf("getRecordability\n");
	sid = (snd_mixer_selem_id_t *)(((char *) alsaDev[audioDev->devID].mixer_sid)
		+ snd_mixer_selem_id_sizeof() * cont);

    elem = snd_mixer_find_selem(alsaDev[audioDev->devID].mh, sid);

	if (snd_mixer_selem_has_capture_switch(elem))
		return(0);

	return(-2);
}

getAlsaStereoStatus(duplexDev *audioDev, int cont)
{
if (audioDev->cflags & SLAB_AUDIODBG)
printf("getStereoStatus(%i): %i\n", cont);
	/*
	 * Cannot find a stereo status switch, so for now assume that only 
	 * Master Mono and Mic are mono.
	 */
	if (strcmp(getAlsaName(audioDev, cont), "Master Mono") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "Mic") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "Center") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "LFE") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "Wave Center") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "Wave LFE") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "Phone") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "PC Speaker") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "Headphone LFE") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "Headphone Center") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "3D Control - Switch") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "Mic Boost (+20dB)") == 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "External Amplifier Power Down")
		== 0)
		return(1);
	if (strcmp(getAlsaName(audioDev, cont), "3D Control Sigmatel - Depth") == 0)
		return(1);

	return(2);
}

getAlsaMutability(duplexDev *audioDev, int cont)
{
	snd_mixer_elem_t *elem;
	snd_mixer_selem_id_t *sid;

if (audioDev->cflags & SLAB_AUDIODBG)
printf("getMutability\n");
	sid = (snd_mixer_selem_id_t *)(((char *) alsaDev[audioDev->devID].mixer_sid)
		+ snd_mixer_selem_id_sizeof() * cont);

    elem = snd_mixer_find_selem(alsaDev[audioDev->devID].mh, sid);

	/*
	 * If we have playback capability then we also have mutability.
	 */
	if (snd_mixer_selem_has_playback_switch(elem))
		return(1);

	return(0);
}

getAlsaCapability(duplexDev *audioDev, int cont)
{
if (audioDev->cflags & SLAB_AUDIODBG)
printf("getAlsaCapability(%i)\n", cont);

	if (cont >= alsaDev[audioDev->devID].elem_count)
		return(-1);
	return(cont);
}

getAlsaCapByName(duplexDev *audioDev, char *name)
{
	snd_mixer_selem_id_t *sid;
	int cont;

	if (name[strlen(name) - 1] == ' ')
		name[strlen(name) - 1] = '\0';

if (audioDev->cflags & SLAB_AUDIODBG)
printf("getAlsaCapByName(%s)\n", name);
	for (cont = 0; cont < alsaDev[audioDev->devID].elem_count; cont++)
	{
		sid = (snd_mixer_selem_id_t *)(((char *)
			alsaDev[audioDev->devID].mixer_sid)
			+ snd_mixer_selem_id_sizeof() * cont);

		if (strcmp(snd_mixer_selem_id_get_name(sid), name) == 0)
			return(cont);
	}
	return(-1);
}

getAlsaValue(duplexDev *audioDev, int cont, int side)
{
	snd_mixer_elem_t *elem;
	snd_mixer_selem_id_t *sid;
	long vmin, vmax, vol;

	sid = (snd_mixer_selem_id_t *)(((char *) alsaDev[audioDev->devID].mixer_sid)
		+ snd_mixer_selem_id_sizeof() * cont);

    elem = snd_mixer_find_selem(alsaDev[audioDev->devID].mh, sid);

if (audioDev->cflags & SLAB_AUDIODBG)
printf("getAlsaValue\n");

	if (snd_mixer_selem_has_playback_volume(elem))
	{
    	snd_mixer_selem_get_playback_volume_range(elem, &vmin, &vmax);
	    snd_mixer_selem_get_playback_volume(elem, side, &vol);
	} else {
    	snd_mixer_selem_get_capture_volume_range(elem, &vmin, &vmax);
	    snd_mixer_selem_get_capture_volume(elem, side, &vol);
	}

	return(vol * 100 / (vmax - vmin));
}

setAudioALSAparam(duplexDev *audioDev, int devID, char *param,
	short left, short right)
{
	int cont;

if (audioDev->cflags & SLAB_AUDIODBG)
printf("setAudioALSAparam(%i)\n", devID);

	if ((cont = getAlsaCapability(audioDev, devID)) == -1)
	{
if (audioDev->cflags & SLAB_AUDIODBG)
printf("could not find capability \"%s\"\n", param);
		return;
	}

	setAlsaValue(audioDev, cont, 1, left);
	if (getAlsaStereoStatus(audioDev, cont) > 1)
		setAlsaValue(audioDev, cont, 2, right);

	return(0);
}

setAlsaRecordSource(duplexDev *audioDev, int cont, int position)
{
	snd_mixer_elem_t *elem;
	snd_mixer_selem_id_t *sid;

if (audioDev->cflags & SLAB_AUDIODBG)
printf("setAlsaRecordSource\n");
	sid = (snd_mixer_selem_id_t *)(((char *) alsaDev[audioDev->devID].mixer_sid)
		+ snd_mixer_selem_id_sizeof() * cont);

    elem = snd_mixer_find_selem(alsaDev[audioDev->devID].mh, sid);

	if (snd_mixer_selem_has_capture_switch(elem))
	{
		snd_mixer_selem_set_capture_switch(elem, 0, position);
		snd_mixer_selem_set_capture_switch(elem, 1, position);
	}

	return(0);
}

setAlsaValue(duplexDev *audioDev, int cont, int side, int value)
{
	snd_mixer_elem_t *elem;
	snd_mixer_selem_id_t *sid;
	long vmin, vmax, vol;

	if ((--side == 1) && (getAlsaStereoStatus(audioDev, cont) < 2))
		return;

	sid = (snd_mixer_selem_id_t *)(((char *) alsaDev[audioDev->devID].mixer_sid)
		+ snd_mixer_selem_id_sizeof() * cont);

    elem = snd_mixer_find_selem(alsaDev[audioDev->devID].mh, sid);

if (audioDev->cflags & SLAB_AUDIODBG)
printf("setAlsaValue(%i, %i, %i)\n", cont, side, value);

	if (snd_mixer_selem_has_playback_volume(elem))
	{
if (audioDev->cflags & SLAB_AUDIODBG)
printf("HAS PLAYBACK FOUND\n");
    	snd_mixer_selem_get_playback_volume_range(elem, &vmin, &vmax);
	} else {
if (audioDev->cflags & SLAB_AUDIODBG)
printf("HAS CAPTURE FOUND\n");
    	snd_mixer_selem_get_capture_volume_range(elem, &vmin, &vmax);
	}

	vol = value * (vmax - vmin) / 100;

	/*
	 * If you get violations in this code, with assert failures in the sound
	 * library then it is likely this controller is either mono, or does not
	 * support continuous control (is a switch). I should work on the correct
	 * way to determine the capabilities of a device
	 */
	if (snd_mixer_selem_has_playback_volume(elem))
	{
if (audioDev->cflags & SLAB_AUDIODBG)
printf("PLAYBACK VOLUME\n");

    	if (snd_mixer_selem_set_playback_volume(elem, side, vol) < -1)
			printf("failed to set value\n");
	} else if (snd_mixer_selem_has_capture_volume(elem)) {
if (audioDev->cflags & SLAB_AUDIODBG)
printf("CAPTURE VOLUME\n");
	   	if (snd_mixer_selem_set_capture_volume(elem, side, vol) < -1)
			printf("failed to set value\n");
	}

	return(0);
}

const char *
getAlsaDeviceName(duplexDev *audioDev)
{
if (audioDev->cflags & SLAB_AUDIODBG)
printf("setDeviceName(%s)\n", alsaDev[audioDev->devID].name);
	return(alsaDev[audioDev->devID].name);
}

closeALSAmixer(audioDev)
duplexDev *audioDev;
{
	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("closeALSAmixer(): %08x\n", alsaDev[audioDev->devID].mh);

	if (alsaDev[audioDev->devID].mh != (snd_mixer_t *) NULL)
	{
		int err;

		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("real closeALSAmixer(): %08x\n",
				alsaDev[audioDev->devID].mh);

		if ((err = snd_mixer_close((void *) alsaDev[audioDev->devID].mh)) < 0)
		{
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("SND Mixer Close error: %s\n", snd_strerror(err));
		}
		if ((err = snd_ctl_close(alsaDev[audioDev->devID].ch)) < 0) {
			if (audioDev->cflags & SLAB_AUDIODBG)
				printf("SND CTL Close error: %s\n", snd_strerror(err));
		}
	}

	alsaDev[audioDev->devID].mh = NULL;
	alsaDev[audioDev->devID].ch = NULL;

	return(0);
}

checkAudioALSAcaps()
{
	/*
	 * Deprecated.
	 */
	return(-1);
}

setAlsaMute(duplexDev *audioDev, int cont, int onoff)
{
	snd_mixer_elem_t *elem;
	snd_mixer_selem_id_t *sid;
	int join;

if (audioDev->cflags & SLAB_AUDIODBG)
printf("setAlsaMute(%i, %i)\n", cont, onoff);
	sid = (snd_mixer_selem_id_t *)(((char *) alsaDev[audioDev->devID].mixer_sid)
		+ snd_mixer_selem_id_sizeof() * cont);

    elem = snd_mixer_find_selem(alsaDev[audioDev->devID].mh, sid);

join = snd_mixer_selem_has_playback_volume_joined(elem);
if (audioDev->cflags & SLAB_AUDIODBG)
printf("joined on device %i is %i\n", cont, join);

	if (snd_mixer_selem_has_playback_switch(elem))
	{
		snd_mixer_selem_set_playback_switch(elem, 0, 1 - onoff);
		if (getAlsaStereoStatus(audioDev, cont) > 1)
			snd_mixer_selem_set_playback_switch(elem, 1, 1 - onoff);
	}

	return(1);
}

openALSAmixer(duplexDev *audioDev)
{
	int err, selem_count, elem_index = 0, mixer_n_selems = 0;
	snd_mixer_selem_id_t *sid;
  	snd_mixer_elem_t *elem;

	snd_ctl_card_info_alloca(&alsaDev[audioDev->devID].hwInfo);

	if (alsaDev[audioDev->devID].ch != NULL)
		return;

	if ((err = snd_ctl_open(&alsaDev[audioDev->devID].ch,
		&audioDev->mixerName[0], 0)) < 0)
	{
		printf("Could not open control interface\n");
		return(-1);
	}
	if ((err = snd_ctl_card_info(alsaDev[audioDev->devID].ch,
		alsaDev[audioDev->devID].hwInfo)) < 0)
	{
		printf("Could not get hardware info\n");
		return(-1);
	}

	alsaDev[audioDev->devID].name =
		strdup(snd_ctl_card_info_get_name(alsaDev[audioDev->devID].hwInfo));

	if (audioDev->cflags & SLAB_AUDIODBG)
	{
		printf("Found: %s\n", alsaDev[audioDev->devID].name);
		printf("Hardware: %s\n",
			snd_ctl_card_info_get_mixername(alsaDev[audioDev->devID].hwInfo));
	}

	if ((err = snd_mixer_open(&alsaDev[audioDev->devID].mh, 0)) < 0)
	{
		printf("Could not get mixer\n");
		return(-1);
	}

	if ((err = snd_mixer_attach(alsaDev[audioDev->devID].mh,
		&audioDev->mixerName[0])) < 0)
	{
		printf("Could not attach to mixer %s\n", audioDev->mixerName);
		return(-1);
	}
	if ((err = snd_mixer_selem_register(alsaDev[audioDev->devID].mh,
		NULL, NULL)) < 0)
	{
		printf("Could not get mixer\n");
		return(-1);
	}
/*
	snd_mixer_set_callback(alsaDev[audioDev->devID].mh, mixer_event);
*/
	if ((err = snd_mixer_load (alsaDev[audioDev->devID].mh)) < 0)
	{
		printf("Could not get mixer\n");
		return(-1);
	}
	selem_count = snd_mixer_get_count(alsaDev[audioDev->devID].mh);

	alsaDev[audioDev->devID].mixer_sid
		= malloc(snd_mixer_selem_id_sizeof() * selem_count);

	for (alsaDev[audioDev->devID].eInfo[mixer_n_selems]
			= snd_mixer_first_elem(alsaDev[audioDev->devID].mh);
		alsaDev[audioDev->devID].eInfo[mixer_n_selems];
		alsaDev[audioDev->devID].eInfo[mixer_n_selems]
			= snd_mixer_elem_next(
				alsaDev[audioDev->devID].eInfo[mixer_n_selems - 1]))
	{
		sid = (snd_mixer_selem_id_t *)(((char *)
			alsaDev[audioDev->devID].mixer_sid)
			+ snd_mixer_selem_id_sizeof() * mixer_n_selems);
		snd_mixer_selem_get_id(alsaDev[audioDev->devID].eInfo[mixer_n_selems],
			sid);
		if (!snd_mixer_selem_is_active(alsaDev[audioDev->devID].eInfo[
			mixer_n_selems]))
			break;
		mixer_n_selems++;
	}

	if (audioDev->cflags & SLAB_AUDIODBG)
		printf("found %i elements\n", mixer_n_selems);

	alsaDev[audioDev->devID].elem_count = mixer_n_selems;

    for (elem_index = 0; elem_index < mixer_n_selems; elem_index++)
	{
		sid = (snd_mixer_selem_id_t *)(((char *)
			alsaDev[audioDev->devID].mixer_sid)
			+ snd_mixer_selem_id_sizeof() * elem_index);
		if (audioDev->cflags & SLAB_AUDIODBG)
			printf("	%s\n", snd_mixer_selem_id_get_name(sid));
	}
}

#endif /* ALSA VERSION */
#endif /* ALSA_SUPPORT over most of file */
