/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * Copyright  1994-2001 Roxen Internet Software
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_language.pike,v 1.37 2004/01/08 20:53:19 vida Exp $
 */

#include <module.h>
inherit "module";

#include <camas/globals.h>

inherit "caudiumlib";

//
//! module: CAMAS: Language
//!  The Language Module for CAMAS.<br />
//!  This module handle all internationalization parts of CAMAS. It will
//!  translate internals messages into localized messages that CAMAS can handle
//!  <br />This is the module where you specify the default language used in
//!  CAMAS.<br /><b>This module is automatically selected if you select "CAMAS:
//!  Main module".</b>
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_language.pike,v 1.37 2004/01/08 20:53:19 vida Exp $
//

constant cvs_version = "$Id: camas_language.pike,v 1.37 2004/01/08 20:53:19 vida Exp $";
constant module_type = MODULE_PROVIDER;
constant module_name = "CAMAS: Language Module";
constant module_doc  = "The Language Module for CAMAS.<br />"
                       "This module handle all internationalization parts of "
                       "CAMAS. It will tranlate internals messages into "
                       "localized messages that CAMAS can handle.<br />"
                       "This is the module where you specify the default "
                       "language used in CAMAS.<br /><b>This module is "
                       "automatically selected if you select \"CAMAS: Main "
                       "module\".</b>";
constant module_unique = 1;
constant thread_safe = 1;		// I think this module should be :)

#if constant(thread_create)
object global_lock = Thread.Mutex ();
#endif

mapping lang_progs = ([ "english" : 0 ]);
mapping lang_progs_short = ([ "en" : "english" ]);
mapping lang_charsets = ([ "english" : "iso-8859-1" ]);
mapping lang_replacement_strings = ([ "english" : "" ]);
int langs_loaded = 0;

void create () {
#ifdef CAMAS_DEBUG
  defvar ("debug", 0, "Debug", TYPE_FLAG,
          "When on, debug messages will be logged in Caudium's debug logfile. "
          "This information is very useful to the developers when fixing bugs.");
#endif

  defvar ("langfiledir", "modules/camas/languages/", "Language files directory", TYPE_STRING,
          "The directory where you store the language files (camas_*.pike)."
          "You can leave this empty if you're only going to use english.");

  load_langs ();

  defvar("defaultlang", "english", "Default user interface language",
         TYPE_MULTIPLE_STRING,
         "The default language for users. The language at the login prompt."
         "This is the name of the language as it's presented at the user "
         "preferences screen.",
	 indices (lang_progs));

  defvar("browserlang", 1, "Default language from browser", TYPE_FLAG,
         "Decides wheter the default language should be selected from the browser's "
         "settings if possible. If all your users use the same langauge you may want "
         "to disable this, since many users don't bother changing their language "
         "preferences in their browsers.");
#if constant(String.Elite)
  defvar("leetp", 0, "Leet level", TYPE_INT,
         "The leet level of Camas, use 0 if you are not an hacker<br/>"
         "100 is the maximum leet level, used only by gurus.");
#endif
}

string check_variable (string var, mixed value) {
  switch (var) {
  case "langfiledir":
    if (sizeof (value)) {
      array stat = file_stat (value);
      if (!stat || stat[1] != -2)
        return "Must be a directory or empty string.";
    }
    langs_loaded = 0;
    break;
  }
}

void start () {
  load_langs ();
}

string status ()
{
  string out = "CAMAS Language Module.";
  out += "<br />Provides these languages :<ul>";
  foreach (sort (indices (lang_progs)), string lang)
    out += "<li>" + lang + "</li>";
  out += "</ul>";
  return out;
}

string query_provides ()
{
  return ("camas_lang");
}

/*
 * What we provide here
 */

//
//! method: int version(void)
//!  Give the CAMAS_AUTH api version
//!  supported by the module
//! returns:
//!  the version of the API
//! note:
//!  The base API is 1. But if we provide v2, we
//!  *MUST* be backward compatible with v1.
//
int version ()
{
  return 1;
}

void load_langs () {
#if constant(thread_create)
  object lock = global_lock->lock();
#endif
  mixed global_err = catch {
    CDEBUG("Loading langs...");
    if (!langs_loaded) {
      mixed err;

      if (sizeof (QUERY (langfiledir))) {
        array files = get_dir (QUERY (langfiledir));
        if (files) {
          CDEBUG(sprintf ("\nCAMAS: Looking up languages files in %s, there %s %d file%s...",
         QUERY (langfiledir),
                           (sizeof (files) == 1) ? "is" : "are",
                           sizeof (files),
                           (sizeof (files) == 1) ? "" : "s"));
          foreach (files, string file) {
            if (file[0..5] == "camas_" && (lower_case (file[sizeof (file) - 5..]) == ".pike" ||
                                           lower_case (file[sizeof (file) - 8..]) == ".pike.gz")) {
              CDEBUG("CAMAS: Considering lang file " + file);
              object lang_prog;
              string lang_file = combine_path (QUERY (langfiledir), file);
              object ee = ErrorContainer();
              master()->set_inhibit_compile_errors (ee);
              err = catch (lang_prog = compile_string (Stdio.read_bytes (lang_file))());
              if(err && (file[sizeof(file)-3..] == ".gz"))
              {
#if constant(Gz.File)
                /* Gz.File is for Gzip files, Gz.deflate|inflate only implement the algorithm used in
                   gzip but not gzip itself (gzip files contains a CRC and a few more changes)
                   Gz.File is only available in Pike 7.4+ 
                  -- vida */
                object file = Gz.File();
                if(file->open(lang_file, "rb"))
                {
                  err = catch (lang_prog = compile_string (file->read())());
                  file->close();
                }
#else
                report_warning("CAMAS: Gzip support not available in your pike, can't load language file " + 
                  file + "\n");
#endif
              }
              master()->set_inhibit_compile_errors (0);
              if (!err && lang_prog->imho_lang) {
                CDEBUG ("CAMAS: Loaded lang " + lang_prog->imho_lang() + " from file " + file);
                lang_progs[lang_prog->imho_lang()] = lang_prog;
                if (lang_prog->imho_lang_short)
                  lang_progs_short[lang_prog->imho_lang_short()] = (lang_prog->imho_lang ());
                if (lang_prog->imho_lang_charset)
                  lang_charsets[lang_prog->imho_lang()] = (lang_prog->imho_lang_charset ());
                if (lang_prog->imho_lang_replacement_string)
                  lang_replacement_strings[lang_prog->imho_lang()] = (lang_prog->imho_lang_replacement_string ());
                // FIXME: what is this?!
                // Roxen 2.x dumps precompiled files as .o-files which pike
                // loads a lot faster. Support for this in Caudium doesn't
                // yet exist. -- david
#if 0
                if (roxen->dump)
                  roxen->dump (lang_file);
#endif
              }
              else
                if (err)
                {
                  CDEBUG("CAMAS: could not load language file: " + err[0]);
                  if(strlen(ee->get()))
                  {
                    CDEBUG(sprintf("%s", ee->get()));
                  }
                  else if(err)
                    CDEBUG(describe_backtrace(err));
                }
            }
          }
        }
        else
          CDEBUG("No files were found in '" +  QUERY (langfiledir) + "'");
      }
      else
        CDEBUG("Can't open directory '" + QUERY (langfiledir) + "'");
    }
    else
      CDEBUG("Langs already loaded, won't load them this time");
  };
#if constant(thread_create)
  destruct(lock);
#endif
  if(global_err)
    report_error("error in camas_language.pike: %s\n", describe_backtrace(global_err));
}

#include <camas/defaultlanguage.h>

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: debug
//! When on, debug messages will be logged in Caudium's debug logfile. This information is very useful to the developers when fixing bugs.
//!  type: TYPE_FLAG
//!  name: Debug
//
//! defvar: langfiledir
//! The directory where you store the language files (camas_*.pike).You can leave this empty if you're only going to use english.
//!  type: TYPE_STRING
//!  name: Language files directory
//
//! defvar: defaultlang
//! The default language for users. The language at the login prompt.This is the name of the language as it's presented at the user preferences screen.
//!  type: TYPE_MULTIPLE_STRING
//!  name: Default user interface language
//
//! defvar: browserlang
//! Decides wheter the default language should be selected from the browser's settings if possible. If all your users use the same langauge you may want to disable this, since many users don't bother changing their language preferences in their browsers.
//!  type: TYPE_FLAG
//!  name: Default language from browser
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

