/*
 * Caudium - An extensible World Wide Web server
 * Copyright  2000-2004 The Caudium Group
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*
 * $Id: camas_layout_manager.pike,v 1.55.2.4 2004/03/09 19:01:03 vida Exp $
 */

#include <module.h>
#include <camas/globals.h>

inherit "module";
inherit "caudiumlib";

inherit "modules/filesystems/filesystem" : filesystem;

//
//! module: CAMAS: Layout Manager
//!  Layout Manager Module for CAMAS.<br />
//!  This module provides CAMAS the names of layouts (e.g. skins) that can
//!  be used. You can use this module to force a layout different than the
//!  default one when there is user definied "Layout Filesystem" in the
//!  current configuration.<br />
//!  <b>This module is automatically selected if you select "CAMAS: Main
//!  module".</b>
//! inherits: module
//! inherits: caudiumlib
//! type: MODULE_PROVIDER
//! cvs_version: $Id: camas_layout_manager.pike,v 1.55.2.4 2004/03/09 19:01:03 vida Exp $
//

constant cvs_version = "$Id: camas_layout_manager.pike,v 1.55.2.4 2004/03/09 19:01:03 vida Exp $";
constant module_type = MODULE_PROVIDER|MODULE_LOCATION;
constant module_name = "CAMAS: Layout Manager";
constant module_doc  = "Layout Manager Module for CAMAS.<br />"
                       "This module provides to CAMAS the names of layouts (e.g. "
                       "skins) that can be used. You can use this module to "
                       "force a layout different than the default one when you "
                       "added a \"Layout Filesystem\" module in the current "
                       "configuration.<br />To have a login layout different "
                       "the default one, simply pass the ?layout=nameofyoulayout"
                       "argument to CAMAS or set Default properties -&gt; "
                       "User interface to the name of your layout in the CAMAS "
                       "main module.<br /><b>This module is automatically "
                       "selected if you select \"CAMAS: Main module\".</b>";
constant module_unique = 1; // another CAMAS layout manager is pointless
constant thread_safe = 1;		// should be

#if constant(thread_create)
object global_lock = Thread.Mutex();
#endif

mapping (string:object) layouts = ([ ]);  // mapping for storing layout (name:object)
private string loginlayout = "Caudium WWW";


/* Caudium modules methods */

//! method: create
//!  Constructor
void create (object conf)
{
  filesystem::create();
  module_dependencies(conf, ({ "camas_layout_default" }));
  
  defvar ("nointernal",
          0,
          "Hide the internal layout",
          TYPE_FLAG,
          "If custom layouts are available, hide the Camas internal layout from the users.",
          0,
          lambda () { return (sizeof (my_configuration ()->get_providers ("camas_layout")) == 1); });

  defvar ("nodefault",
          0,
          "Don't add a 'virtual' layout named Default",
          TYPE_FLAG,
          "By default Camas will add a layout named 'Default' which maps to the layout available at "
          "the login screen. If you set this to Yes, Camas will not add this fake layout in the users's "
          "select box.");

#ifdef CAMAS_DEBUG 
  defvar ("debug",
          0,
          "Debug",
          TYPE_FLAG,
          "When on, debug messages will be logged in Caudium's debug logfile. "
          "This information is very useful to the developers when fixing bugs.");
#endif
  
  // override some values and don't show these in the CIF, it will hurt the user
  killvar("searchpath");
  defvar("searchpath", "", "Search path", TYPE_STRING|VAR_EXPERT,
      "Don't touch this, it is automatically set by Camas");

  killvar("mountpoint");
  defvar("mountpoint", "/", "Mountpoint", TYPE_STRING|VAR_EXPERT);

  killvar(".files");
  defvar(".files", 0, "Directory Settings: Show hidden files", TYPE_FLAG|VAR_EXPERT);

  killvar("dir");
  defvar("dir", 1, "Directory Settings: Enable directory listings per default", TYPE_FLAG|VAR_EXPERT);

  killvar("tilde");
  defvar("tilde", 0, "Directory Settings: Show backup files", TYPE_FLAG|VAR_EXPERT);

  killvar("put");
  defvar("put", 0, "Allowed Access Methods: PUT", TYPE_FLAG|VAR_EXPERT);
  
  killvar("appe");
  defvar("appe", 0, "Allowed Access Methods: APPE", TYPE_FLAG|VAR_EXPERT);
  
  killvar("delete");
  defvar("delete", 0, "Allowed Access Methods: DELETE", TYPE_FLAG|VAR_EXPERT);

  killvar("method_mkdir");
  defvar("method_mkdir", 0, "Allowed Access Methods: MKDIR", TYPE_FLAG|VAR_EXPERT);
  
  killvar("method_mv");
  defvar("method_mv", 0, "Allowed Access Methods: MV", TYPE_FLAG|VAR_EXPERT);

  killvar("method_chmod");
  defvar("method_chmod", 0, "Allowed Access Methods: CHMOD", TYPE_FLAG|VAR_EXPERT);

}

int|string check_variable (string var, mixed value)
{
  if(var == "loginlayout")
  {
    if(search(list_layouts(), value) == -1)
      return "This layout doesn't exist. Did you removed the module managing it ? Please reload this module "
        "or restart Caudium.";
    loginlayout = value;
  }
  return 0;
}

string get_defaultloginlayout(object conf)
{
  array (object) providers = conf->get_providers ("camas_layout");
  if (sizeof (providers))
    foreach (providers, object p)
      if(p && p->is_default)
        return p->name();
  return loginlayout;

}

void register_layouts()
{
  array _layouts = list_layouts() - ({ "Default" });
  string old_default_loginlayout;
  catch(old_default_loginlayout = QUERY(loginlayout));
  string defaultloginlayout;
  if(search(_layouts, old_default_loginlayout || get_defaultloginlayout(my_configuration())) != -1)
    defaultloginlayout = old_default_loginlayout || get_defaultloginlayout(my_configuration());
  else
    defaultloginlayout = _layouts[0];
  defvar("loginlayout", defaultloginlayout, "Default layout for login prompt",
       TYPE_STRING_LIST, 	 
         "<p>The default layout to use for the login prompt.</p>",
        _layouts);
}

//! method: start (int num, object conf)
//!  Function called when the module is started
void start (int num, object conf)
{
  init_layouts (conf);
  filesystem::start();
}

//! method: stop (int num, object conf)
//!  Function called when the module is stopped
void stop (int num, object conf)
{
#ifdef CAMAS_DEBUG
  CDEBUG("camas_layout_manager stopped");
#endif
}

//! method: query_location ()
//!  Returns the mount point of the module
//!  The mount point is located under CAMAS one
string query_location ()
{
  object camas_main = my_configuration ()->get_provider ("camas_main");
  if(!objectp(camas_main))
  {
    report_error("CAMAS main module not present");
    return "CAMAS main module not present";
  }
  
  return camas_main->query_location() + "_camastemplates/";
}

//! method: status ()
//!  Displays status information in the caudium configuration interface
//! returns:
//!  The list of available layouts in HTML form to display in the CIF
string status ()
{
  string out = "CAMAS Layouts Manager Module :<br />";
  
  if (sizeof (layouts))
  {
    array tmp = list_layouts();
    
    if (sizeof (tmp))
    {
      out += sprintf ("Provides th%ss%s layout%s :<ul>",
                      (sizeof (tmp) == 1) ? "i" : "e",
                      (sizeof (tmp) == 1) ? "" : "e",
                      (sizeof (tmp) == 1) ? "" : "s");
      
      foreach (tmp, string l)
        out += "<li>" + l + "</li>";

      out += "</ul>";
    }
    else
      out += "Default Layout";
  }
  return out;
}

//! method: mixed find_file (string path, object id)
//!  Method called when a request is made for an URL within this module mount point.
//!  This module only allows GET method and file return.
//! arg: string path
//!  URL minus the module mountpoint
//! arg: object id
//!  The Caudium id object
mixed find_file (string path, object id)
{
  object data;                                               // data to send back to the client
  array(string) path_elements  = path / "/";   // store each path element into an array
  path_elements -= ({ "" });
  if(!sizeof(path_elements))
    return 0;
  string layout = HTTP_DECODE_STRING(path_elements[0]);      // name of the layout
  string layout_root = "";

  if (layouts[layout] && layouts[layout]->layout_root)
    layout_root = layouts[layout]->layout_root ();

  if (!sizeof(layout_root))
  {
    CDEBUG("find_file(): root directory for the layout "+layout+" can't be found");
    return 0;
  }

  /* ".." not allowed in path */
  /* shouldn't appear         */
  if(has_value(path_elements,".."))
  {
    CDEBUG(".. not allowed in path");
    return 0;
  }
  
  string path2 = "";
  for(int i=1; i<sizeof(path_elements); i++)
  {
    path2 += "/" + path_elements[i];
  }

  string real_path = layout_root + path2;     // path of the file requested in the filesystem

  CDEBUG("real_path for request '"+query_location()+"/"+path+"' is '"+real_path+"'");

  return filesystem::find_file(real_path, id);
}

//! method: stat_file (string path, object id)
//!  Returns informations about the files
//! arg: string path
//!  The filesystem path of the file wanted to be checked
//! arg: object id
//!  The Caudium id object
array(int) stat_file (string path, object id)
{
  return filesystem::stat_file(path, id);
}


/* Layout management */

//! method: void init_layouts (object conf)
//!  Initialisation of available layouts 
//!  Available layouts are listed in the global layouts mapping
//! arg: object conf
void init_layouts (object conf)
{
  array (object) providers = conf->get_providers ("camas_layout");
  if (sizeof (providers))
  {
    foreach (providers, object p)
    {
      if (p && p->name () != "NONE")
      {
        if(layouts[p->name()] && layouts[p->name()] != p)
          CDEBUG(sprintf("init_layouts: Overwriting layout [%s] old "
             "module is %O, new is %O", p->name(), layouts[p->name()], p));
#if constant (thread_create)
        object lock = global_lock->lock();
#endif
        mixed err = catch {
          layouts[p->name ()] = p;
        };
#if constant (thread_create)
        destruct(lock);
#endif      
        if(err)
          report_error("error in camas_layout_manager.pike: %s\n", describe_backtrace(err));
        else
          CDEBUG(sprintf("init_layouts: Added layouts[%s]=%O", p->name(), p));
      }
      else
        report_warning("init_layouts: Provider %O is not a valid "
              "CAMAS layout provider\n", p);
    }
    register_layouts();
    
    foreach (providers, object p)
      if (p && p->name () != "NONE")
      {
      
#if constant (thread_create)
        object lock = global_lock->lock();
#endif
        mixed err = catch {
          if(QUERY(loginlayout) == p->name())
          {
            loginlayout = p->name ();
            // this is the login layout, add a fake layout named 'Default' linking to it
            if(!QUERY(nodefault))
            {
              layouts["Default"] = layouts[get_login_layout()];
              CDEBUG(sprintf("init_layouts: Layout %s set as default layout",
                    get_login_layout()));
            }
          }
        };    
#if constant (thread_create)
        destruct(lock);
#endif      
        if(err)
          report_error("error in camas_layout_manager.pike: %s\n", describe_backtrace(err));
      }
  }
}

//! method: add_layout (object p)
//!  Add layout p to the current layout list
//! arg: object p
//!  The layout object to add
//! note: this doesn't reload the CIF
void add_layout (object p)
{
  if(p)
  {
    CDEBUG("add_layout: adding layout "+p->name());
    if(layouts[p->name()] && layouts[p->name()] != p)
        CDEBUG(sprintf("add_layout: Overwriting layout [%s] old "
          "module is %O, new is %O", p->name(), layouts[p->name()], p));

#if constant (thread_create)
    object lock = global_lock->lock();
#endif
    
    mixed err = catch {
      layouts[p->name ()] = p;
      register_layouts();
      if(QUERY(loginlayout) == p->name())
      {
        loginlayout = p->name ();
        // this is the login layout, add a fake layout named 'Default' linking to it
        if(!QUERY(nodefault))
        {
          layouts["Default"] = layouts[get_login_layout()];
          CDEBUG(sprintf("add_layout: Layout %s set as default layout",
              get_login_layout()));
        }
      }
    };
    
#if constant (thread_create)
    destruct(lock);
#endif
    if(err)
      report_error("error in camas_layout_manager.pike: %s\n", describe_backtrace(err));
    else
      CDEBUG(sprintf("add_layout: Added layouts[%s]=%O", p->name(), p));
  }
  else
    report_warning("add_layout: Provider %O is not a valid Camas layout module",
          p);
}

//! method: remove_layout (object p)
//!  Delete layout p from the current layout list
//! arg: object p
//!  The layout object to remove
void remove_layout (object p)
{
  if (p && layouts[p->name ()])
  {
#if constant (thread_create)
    object lock = global_lock->lock();
#endif
    
    mixed err = catch {
      m_delete (layouts, p->name ());
      register_layouts();
      CDEBUG(sprintf("remove_layout: Removing layout %s", p->name()));
      if(QUERY(loginlayout) == p->name())
      {
        array layout_names = indices(layouts);
        if(layout_names && sizeof(layout_names))
          loginlayout = layout_names[0];
        // there are no more layouts, load the default one...
        else
          module_dependencies(my_configuration(), ({ "camas_layout_default" }));
        if(!QUERY(nodefault))
        {
          layouts["Default"] = layouts[get_login_layout()];
        }
        CDEBUG("remove_layout: Layout was a default layout, new default "
             "layout set to " + get_login_layout());
      }
    };
    
#if constant (thread_create)
    destruct(lock);
#endif
    
    if(err)
      report_error("error in camas_layout_manager.pike: %s\n", describe_backtrace(err));
  }
  else
    report_warning("remove_layout: Provider %O can't be removed, not a valid "
        "Camas layout", p);
}

//! method: query_provide ()
//!  Returns the name of the service this module provides
string query_provides ()
{
  return ("camas_layout_manager");
}

//! method: list_layouts ()
//!  List all the layouts registered
//! returns:
//!  array containing all the available layouts
array (string) list_layouts ()
{
  array lay = sort (indices (layouts));
  if(sizeof(layouts) > 1 && QUERY(nointernal))
  {
	array result = ({ });
  	foreach(lay, string lay_name)
	    	if(!layouts[lay_name]->is_default)
			result += ({ lay_name });
	return result;
  }
  return lay;
}

/* Layouts */

//! method: string get_login_layout ()
//!  Returns the default layout name for the login screen
string get_login_layout()
{
  return loginlayout;
}

//! method: string get_layout_data(string layout, int screen, string camasframe)
//!  Returns the contents of the layout given the layout name, screen id and
//!  CAMAS frame name
//! arg: string layout
//!  Layout name
//! arg: int screen
//!  Screen id
//! arg: string camasframe
//!  CAMAS frame name
string get_layout_data(string layout, int screen, string camasframe)
{
  string layout_data = "";
 
  // defaulting to _top if no frame is given
  if(!stringp(camasframe))
    camasframe = "_top";
  
  if (layouts[layout] && layouts[layout]->get_layout_data)
    layout_data = layouts[layout]->get_layout_data (screen, camasframe);

#ifdef CAMAS_DEBUG
  if (!layout_data)
    CDEBUG("Did not find frame '"+camasframe+"' for screen '"+screen+"' in layout '"+ layout +"'");
#endif
 
  if(!stringp(layout_data))
    layout_data="";
  
  // TODO: see what to do with frames / old IMHO layouts / new default screen
  object layout_fs = layouts[get_login_layout()];
  if(!layout_fs)
  {
    report_error("CAMAS error: Login layout can't be loaded, trying to load "
    "one layout anyway\n");
    foreach(indices(layouts), string layout)
    {
      if(layouts[layout])
      {
        layout_fs = layouts[layout];
        break;
      }
    }
  }
  if(!layout_fs)
  {
    string errmsg = "CAMAS error: No layouts were found, Camas cannot work.\n";
    report_error(errmsg);
    return "<html><body><strong>" + errmsg + "</strong></body></html>";
  }
  return (sizeof(layout_data)) ? layout_data : layout_fs->get_layout_data (screen, camasframe);
}



/* Buttons */

//! method: string get_button_mode (string layout)
//!  Returns the button mode for a given layout
//! arg: string layout
//!  The layout name we want the button mode
string get_button_mode (string layout)
{
  string _mode = "";

  if (layouts[layout] && layouts[layout]->button_mode)
  {
    CDEBUG("Asking layout");
    _mode = layouts[layout]->button_mode (layout);
  }

  CDEBUG("Button mode for layout "+layout+" is "+_mode);

  return _mode;
}

//! method: get_gbutton_syntax (string layout)
//!  Returns the gbutton syntax data given the layout name
//! arg: string layout
//!  The layout name
string get_gbutton_syntax (string layout)
{
  string _gbutton_syntax = "";

  if(layouts[layout] && layouts[layout]->get_gbutton_syntax)
    _gbutton_syntax = layouts[layout]->get_gbutton_syntax();

  return _gbutton_syntax;
}

//! method: get_basicbutton_list (string layout)
//!  Returns the basic button data given the layout name
//! arg: string layout
//!  The layout name
mapping get_basicbutton_list (string layout)
{
  mapping _basic_button_list = ([ ]);

  if(layouts[layout] && layouts[layout]->get_basicbutton_list)
    _basic_button_list = layouts[layout]->get_basicbutton_list();

  return _basic_button_list;
}

//! method: get_icon_extension (string layout)
//!  Returns the file extension of the icon images given the layout name
//! arg: string layout
//!  The layout name
string get_icon_extension (string layout)
{
	string _icon_extension = "";
	
	if(layouts[layout] && layouts[layout]->get_icon_extension)
		_icon_extension = layouts[layout]->get_icon_extension();
	
	return _icon_extension;
}

/* START AUTOGENERATED DEFVAR DOCS */

//! defvar: nodefault
//! If custom layouts are available, hide the 'Default' layout from the users.
//!  type: TYPE_FLAG
//!  name: Hide the Default layout
//
//! defvar: debug
//! When on, debug messages will be logged in Caudium's debug logfile. This information is very useful to the developers when fixing bugs.
//!  type: TYPE_FLAG
//!  name: Debug
//

/*
 * If you visit a file that doesn't contain these lines at its end, please
 * cut and paste everything from here to that file.
 */

/*
 * Local Variables:
 * c-basic-offset: 2
 * End:
 *
 * vim: softtabstop=2 tabstop=2 expandtab autoindent formatoptions=croqlt smartindent cindent shiftwidth=2
 */

