/* GTK - The GIMP Toolkit
 * Copyright (C) 1995-1997 Peter Mattis, Spencer Kimball and Josh MacDonald
 * GtkSelBox Copyright (C) 2000 Stefan Ondrejicka <ondrej@idata.sk>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gtkselbox.h"
#include <gtk/gtk.h>

#if GTK_CHECK_VERSION(1,3,0)
#define STYLE_GET_XTHICK(widget) GTK_WIDGET (widget)->style->xthickness
#define STYLE_GET_YTHICK(widget) GTK_WIDGET (widget)->style->ythickness
#else
#define STYLE_GET_XTHICK(widget) GTK_WIDGET (widget)->style->klass->xthickness
#define STYLE_GET_YTHICK(widget) GTK_WIDGET (widget)->style->klass->ythickness
#endif

enum {
        ARG_0,
	ARG_THICKNESS
};

static void	gtk_selbox_class_init	(GtkSelBoxClass		*klass);
static void	gtk_selbox_init		(GtkSelBox		*selbox);
static void	gtk_selbox_get_arg	(GtkObject		*object,
					 GtkArg			*arg,
					 guint			arg_id);
static void	gtk_selbox_set_arg	(GtkObject		*object,
					 GtkArg			*arg,
					 guint			arg_id);
static void	gtk_selbox_size_request (GtkWidget		*widget,
					 GtkRequisition		*requisition);
static void	gtk_selbox_size_allocate(GtkWidget		*widget,
					 GtkAllocation		*allocation);
static void	gtk_selbox_paint	(GtkWidget		*widget,
					 GdkRectangle		*area);
static void	gtk_selbox_draw		(GtkWidget		*widget,
					 GdkRectangle		*area);
static gint	gtk_selbox_expose	(GtkWidget		*widget,
					 GdkEventExpose		*event);
static void	gtk_selbox_realize	(GtkWidget		*widget);



guint gtk_selbox_get_type ()
{
        static guint selbox_type = 0;

        if (!selbox_type)
        {
                GtkTypeInfo selbox_info =
                {
                        "GtkSelBox",
                        sizeof (GtkSelBox),
                        sizeof (GtkSelBoxClass),
                        (GtkClassInitFunc) gtk_selbox_class_init,
                        (GtkObjectInitFunc) gtk_selbox_init,
			NULL ,
			NULL ,
			NULL,
                };

                selbox_type = gtk_type_unique (gtk_frame_get_type (), &selbox_info);
        }

        return selbox_type;
}

static void gtk_selbox_class_init (GtkSelBoxClass *klass)
{
        GtkWidgetClass *widget_class;
        GtkObjectClass *object_class;

        widget_class = (GtkWidgetClass*) klass;
        object_class = (GtkObjectClass*) klass;

        widget_class->size_request = gtk_selbox_size_request;
        widget_class->size_allocate = gtk_selbox_size_allocate;
	widget_class->draw = gtk_selbox_draw;
	widget_class->expose_event = gtk_selbox_expose;
	widget_class->realize = gtk_selbox_realize;

        gtk_object_add_arg_type ("GtkSelBox::thickness", GTK_TYPE_INT, GTK_ARG_READWRITE, ARG_THICKNESS);

	object_class->set_arg = gtk_selbox_set_arg;
	object_class->get_arg = gtk_selbox_get_arg;

}

static void gtk_selbox_init (GtkSelBox *selbox)
{
	selbox->thickness = 3;
	selbox->selected = FALSE;
	selbox->with_frame = FALSE;
	selbox->with_window = FALSE;

	selbox->was_selected = FALSE;
}

static void gtk_selbox_get_arg (
GtkObject 	*object,
GtkArg		*arg,
guint		arg_id)
{
	GtkSelBox *selbox = GTK_SELBOX(object);

        switch (arg_id)
        {
		case ARG_THICKNESS:
			GTK_VALUE_INT (*arg) = selbox->thickness;
			break;
                default:
                        arg->type = GTK_TYPE_INVALID;
                        break;
        }
}

static void gtk_selbox_set_arg (
GtkObject 	*object,
GtkArg		*arg,
guint		arg_id)
{
	GtkSelBox *selbox = GTK_SELBOX(object);

        switch (arg_id)
        {
		case ARG_THICKNESS:
			gtk_selbox_set_thickness(selbox, GTK_VALUE_INT (*arg));
			break;
                default:
                        arg->type = GTK_TYPE_INVALID;
                        break;
	}
}

GtkWidget* gtk_selbox_new (
gboolean	with_frame,
gboolean	with_window)
{
	GtkWidget *selbox;

	selbox = gtk_type_new (gtk_selbox_get_type ());

	GTK_SELBOX(selbox)->with_frame = with_frame;
	GTK_SELBOX(selbox)->with_window = with_window;

	if (with_window)
	{
		 GTK_WIDGET_UNSET_FLAGS(selbox, GTK_NO_WINDOW);
	}
	else
	{
		 GTK_WIDGET_SET_FLAGS(selbox, GTK_NO_WINDOW);
	}

	return selbox;
}

static void gtk_selbox_size_request (
GtkWidget	*widget,
GtkRequisition	*requisition)
{
	GtkBin *bin = GTK_BIN(widget);

	requisition->width = (GTK_SELBOX (widget)->with_frame ? 
			 STYLE_GET_XTHICK(widget) : 0) * 2 +
			GTK_SELBOX (widget)->thickness * 2;

	requisition->height = (GTK_SELBOX (widget)->with_frame ?
			 STYLE_GET_YTHICK(widget) : 0) * 2 +
			GTK_SELBOX (widget)->thickness * 2;

	if (bin->child && GTK_WIDGET_VISIBLE (bin->child))
	{
		GtkRequisition child_requisition;
		gtk_widget_size_request (bin->child, &child_requisition);

		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static void gtk_selbox_size_allocate (
GtkWidget	*widget,
GtkAllocation	*allocation)
{
	if (GTK_WIDGET_REALIZED (widget) && GTK_SELBOX(widget)->with_window)
	{
		gdk_window_move_resize (widget->window, 
			allocation->x,
			allocation->y,
			allocation->width,
			allocation->height);
	}

	if (GTK_WIDGET_MAPPED (widget) &&
	    ((widget->allocation.x != allocation->x) ||
	     (widget->allocation.y != allocation->y) ||
	     (widget->allocation.width != allocation->width) ||
	     (widget->allocation.height != allocation->height)) &&
	    (widget->allocation.width != 0) &&
	    (widget->allocation.height != 0))
        {
		gtk_widget_queue_clear (widget);
	}

	widget->allocation = *allocation;

	if (GTK_BIN(widget)->child && GTK_WIDGET_VISIBLE (GTK_BIN(widget)->child))
	{
		GtkAllocation child_allocation;

		child_allocation.x = (GTK_SELBOX (widget)->with_frame ? 
			 STYLE_GET_XTHICK(widget) : 0) +
			GTK_SELBOX (widget)->thickness;

		child_allocation.width = MAX(1, (gint)allocation->width - child_allocation.x * 2);

		child_allocation.y = (GTK_SELBOX (widget)->with_frame ?
			 STYLE_GET_YTHICK(widget) : 0) +
			GTK_SELBOX (widget)->thickness;

		child_allocation.height = MAX (1, ((gint)allocation->height - child_allocation.y * 2));

		if (!GTK_SELBOX(widget)->with_window)
		{
			child_allocation.x += allocation->x;
			child_allocation.y += allocation->y;
		}

		gtk_widget_size_allocate (GTK_BIN(widget)->child, &child_allocation);
	}
}

static void gtk_selbox_paint (GtkWidget    *widget,
			      GdkRectangle *area)
{
	if (GTK_WIDGET_DRAWABLE (widget))
	{
		gint px,py,x,y,w,h,i;
		GdkGC *gc;

		if (GTK_SELBOX(widget)->with_window)
		{
			x = 0; y = 0;
		}
		else
		{
			x = widget->allocation.x;
			y = widget->allocation.y;
		}

		px = (GTK_SELBOX (widget)->with_frame ? 
			 STYLE_GET_XTHICK(widget) : 0);

		py = (GTK_SELBOX (widget)->with_frame ?
			 STYLE_GET_YTHICK(widget) : 0);

		w = widget->allocation.width - 2 * px - 1;
		h = widget->allocation.height - 2 * py - 1;

		gc = GTK_SELBOX(widget)->selected ? widget->style->light_gc[GTK_STATE_SELECTED] : widget->style->bg_gc[GTK_STATE_NORMAL];

		if (GTK_SELBOX(widget)->selected ||
		    (!GTK_SELBOX(widget)->selected && GTK_SELBOX(widget)->was_selected))
		{
			if (area)
				    gdk_gc_set_clip_rectangle (gc, area);

			for (i = GTK_SELBOX(widget)->thickness ; i ; i --)
			{
				gdk_draw_rectangle (widget->window, gc,
					FALSE, px+x, px+y, w, h);

				x ++; w -= 2;
				y ++; h -= 2;
			}
			GTK_SELBOX(widget)->was_selected = FALSE;
		}

		if (GTK_SELBOX(widget)->with_frame)
		{
			if (GTK_SELBOX(widget)->with_window)
			{
				x = 0; y = 0;
			}
			else
			{
				x = widget->allocation.x;
				y = widget->allocation.y;
			}

			gtk_paint_shadow (widget->style, widget->window,
				GTK_STATE_NORMAL, GTK_FRAME(widget)->shadow_type,
				area, widget, "frame",
				x ,
				y  ,
				widget->allocation.width,
				widget->allocation.height);
		}
	}
}

static void gtk_selbox_draw (GtkWidget    *widget,
			      GdkRectangle *area)
{
	if (GTK_WIDGET_DRAWABLE (widget))
	{
		GdkRectangle child_area;
		GdkRectangle tmp_area;

		tmp_area = *area;

		gtk_selbox_paint (widget, &tmp_area);

		if (GTK_BIN (widget)->child && gtk_widget_intersect (GTK_BIN (widget)->child, &tmp_area, &child_area))
			gtk_widget_draw (GTK_BIN (widget)->child, &child_area);
	}
}

gint gtk_selbox_expose (
GtkWidget	*widget,
GdkEventExpose	*event)
{
	if (GTK_WIDGET_DRAWABLE (widget))
	{
		GtkBin *bin = GTK_BIN (widget);
		GdkEventExpose child_event;

		gtk_selbox_paint (widget, &event->area);

		child_event = *event;
		if (bin->child &&
		    GTK_WIDGET_NO_WINDOW (bin->child) &&
		    gtk_widget_intersect (bin->child, &event->area, &child_event.area))
#if 1
			gtk_widget_event (bin->child, (GdkEvent*) &child_event);
#else
			GTK_WIDGET_CLASS(((GtkObject*)(bin->child))->klass)->draw(bin->child, &child_event.area);
#endif
	}

	return FALSE;
}

static void
gtk_selbox_realize (GtkWidget *widget)
{
        GdkWindowAttr attributes;
        gint attributes_mask;
        gint border_width;

	GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);

	if (GTK_SELBOX(widget)->with_window)
	{
		border_width = GTK_SELBOX (widget)->with_frame ?
			 STYLE_GET_XTHICK(widget) : 0;

		attributes.x = widget->allocation.x;
		attributes.y = widget->allocation.y;
		attributes.width = widget->allocation.width;
		attributes.height = widget->allocation.height;

		attributes.window_type = GDK_WINDOW_CHILD;
		attributes.wclass = GDK_INPUT_OUTPUT;
		attributes.visual = gtk_widget_get_visual (widget);
		attributes.colormap = gtk_widget_get_colormap (widget);
		attributes.event_mask = gtk_widget_get_events (widget)
			| gtk_widget_get_extension_events (widget)
			| GDK_BUTTON_MOTION_MASK
			| GDK_POINTER_MOTION_MASK
			| GDK_BUTTON_PRESS_MASK
			| GDK_BUTTON_RELEASE_MASK
			| GDK_EXPOSURE_MASK
			| GDK_ENTER_NOTIFY_MASK
			| GDK_LEAVE_NOTIFY_MASK;

		attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;

		widget->window = gdk_window_new (gtk_widget_get_parent_window (widget), 
		&attributes, attributes_mask);
		gdk_window_set_user_data (widget->window, widget);
	}
	else
	{
		if (widget->parent)
		{
			widget->window = gtk_widget_get_parent_window (widget);
			gdk_window_ref (widget->window);
		}
	}
	widget->style = gtk_style_attach (widget->style, widget->window);
	gtk_style_set_background (widget->style, widget->window, GTK_STATE_NORMAL);
}

void gtk_selbox_set_thickness (
GtkSelBox	*selbox,
gint		thickness)
{
	if (selbox->thickness == thickness)
		return;

	selbox->thickness = thickness;

	if (GTK_WIDGET_VISIBLE (selbox))
		gtk_widget_queue_resize (GTK_WIDGET (selbox));
}

void gtk_selbox_set_selected (
GtkSelBox	*selbox,
gint		selected)
{
	if ((selbox->selected && selected) ||
	    (!selbox->selected && !selected))
		return;

	selbox->was_selected = selbox->selected;

	selbox->selected = selected;

	if (GTK_WIDGET_VISIBLE (selbox))
		gtk_widget_queue_draw (GTK_WIDGET (selbox));
}

