;; For debugging output:
;;  (cmsg-add :debug-parsing)

(in-package :odcl)

(defclass test-parsing-input ()
  ((items      :initarg  :items
	       :accessor items)
   (csv-format :initarg  :csv-format
	       :accessor csv-format)
   (text       :initarg  :text
	       :accessor text)))

(defun %copy-test-parsing-input (source)
  (make-instance 'test-parsing-input
		 :items      (copy-list (items source))
		 :csv-format (copy-list (csv-format source))
		 :text       (copy-seq (text source))))

(defparameter *test-parsing-simple*
  (let ((items '("A" "B" "no" "56")))
    (make-instance 'test-parsing-input
		   :items      items
		   :csv-format '((:field-1 string)
				 (:field-2 string)
				 (:field-3 string)
				 (:field-4 string))
		   :text       (format nil "~{\"~A\"~^,~}" items))))

(defparameter *test-parsing-mac*
  (let ((result (%copy-test-parsing-input *test-parsing-simple*)))
    (setf (text result) (format nil "~A~A~A~A"
				(text result) #\Return (text result) #\Return))
    result))

(defparameter *test-parsing-dos*
  (let ((result (%copy-test-parsing-input *test-parsing-simple*)))
    (setf (text result) (format nil "~A~A~A~A~A~A"
				(text result) #\Return #\Newline
				(text result) #\Return #\Newline))
    result))

(defparameter *test-parsing-unix*
  (let ((result (%copy-test-parsing-input *test-parsing-simple*)))
    (setf (text result) (format nil "~A~A~A~A"
				(text result) #\Newline (text result) #\Newline))
    result))

(defparameter *test-parsing-tab-separated*
  (let ((items '("a one" "a two" "a three" "and a four")))
    (make-instance 'test-parsing-input
		   :items      items
		   :csv-format '((:field-1 string)
				 (:field-2 string)
				 (:field-3 string)
				 (:field-4 string))
		   :text       (format nil "~{\"~A\"~^	~}" items))))

(defparameter *test-parsing-escaped-comma*
  (let ((items '("one" "two\\,TWO" "three" "four")))
    (make-instance 'test-parsing-input
		   :items      items
		   :csv-format '((:field-1 string)
				 (:field-2 string)
				 (:field-3 string)
				 (:field-4 string))
		   :text       (format nil "~{~A~^,~}" items))))

(defparameter *test-parsing-quoted-comma*
  (let ((items '("one" "two,TWO" "three" "four")))
    (make-instance 'test-parsing-input
		   :items      items
		   :csv-format '((:field-1 string)
				 (:field-2 string)
				 (:field-3 string)
				 (:field-4 string))
		   :text       (format nil "~{\"~A\"~^,~}" items))))

(defparameter *test-parsing-escaped-quote*
  (let ((items '("one" "two\\\"TWO" "\\\"three\\\"THREE\\\"" "four")))
    (make-instance 'test-parsing-input
		   :items      items
		   :csv-format '((:field-1 string)
				 (:field-2 string)
				 (:field-3 string)
				 (:field-4 string))
		   :text       (format nil "~{~A~^,~}" items))))

(defparameter *test-parsing-escaped-newline*
  (let ((items (list "one" "two" (format nil "~A\\~C~A" "before" #\Newline "after") "four")))
    (make-instance 'test-parsing-input
		   :items      items
		   :csv-format '((:field-1 string)
				 (:field-2 string)
				 (:field-3 string)
				 (:field-4 string))
		   :text       (format nil "~{~A~^,~}" items))))

(defmacro assert-string= (s1 s2)
  (let ((lhs (gensym "S1-")) (rhs (gensym "S2-")))
    `(let ((,lhs (remove #\\ ,s1)) (,rhs ,s2))
      (cmsg-c :debug-parsing "comparing \"~A\" and \"~A\"" ,lhs ,rhs)
      (unless (string= ,lhs ,rhs)
	(error "Strings not equal: \"~A\" \"~A\"" ,lhs ,rhs)))))

(defregression (:parsing 1)
    "Simple test with :list output."
  (let* ((input *test-parsing-simple*)
	 (stream (make-string-input-stream (text input)))
	 (result (read-csv-record stream (csv-format input))))
    (cmsg-c :debug-parsing ">>~A<<" (text input))
    (do ((i 0 (1+ i))
	 (end (length (items input))))
	((= i end) nil)
      (assert-string= (nth i (items input))
		      (nth i result)))))

(defregression (:parsing 2)
    "Simple test with :alist output."
  (let* ((input *test-parsing-simple*)
	 (stream (make-string-input-stream (text input)))
	 (result (read-csv-record stream (csv-format input) :output-format :alist))
	 (i 0))
    (cmsg-c :debug-parsing ">>~A<<" (text input))
    (cmsg-c :debug-parsing ">>~S<<" result)
    (dolist (field (csv-format input))
      (let* ((key (car field))
	     (expected (nth i (items input)))
	     (found (cdr (assoc key result))))
	(assert-string= expected found)
	(incf i)))))

(defun %test-parsing-line-endings (input)
  (let* ((stream (make-string-input-stream (text input)))
	 (discard (read-csv-record stream (csv-format input)))
	 (result (read-csv-record stream (csv-format input))))
    (cmsg-c :debug-parsing "First record: ~S" discard)
    (do ((i 0 (1+ i))
	 (end (length (items input))))
	((= i end) nil)
      (assert-string= (nth i (items input))
		      (nth i result)))))

(defregression (:parsing 20)
    "Mac line endings."
  (%test-parsing-line-endings *test-parsing-mac*))

(defregression (:parsing 21)
    "DOS line endings."
  (%test-parsing-line-endings *test-parsing-dos*))

(defregression (:parsing 22)
    "Unix line endings."
  (%test-parsing-line-endings *test-parsing-unix*))

(defregression (:parsing 23)
    "Tab-separated."
  (let* ((input *test-parsing-tab-separated*)
	 (stream (make-string-input-stream (text input)))
	 (result (read-csv-record stream (csv-format input)
				  :sep-char #\Tab)))
    (cmsg-c :debug-parsing ">>~A<<" (text input))
    (do ((i 0 (1+ i))
	 (end (length (items input))))
	((= i end) nil)
      (assert-string= (nth i (items input))
		      (nth i result)))))

(defregression (:parsing 24)
    "Quoted separator."
  (let* ((input *test-parsing-quoted-comma*)
	 (stream (make-string-input-stream (text input)))
	 (result (read-csv-record stream (csv-format input))))
    (cmsg-c :debug-parsing ">>~A<<" (text input))
    (do ((i 0 (1+ i))
	 (end (length (items input))))
	((= i end) nil)
      (assert-string= (nth i (items input))
		      (nth i result)))))

(defregression (:parsing 25)
    "Escaped separator."
  (let* ((input *test-parsing-escaped-comma*)
	 (stream (make-string-input-stream (text input)))
	 (result (read-csv-record stream (csv-format input))))
    (cmsg-c :debug-parsing ">>~A<<" (text input))
    (do ((i 0 (1+ i))
	 (end (length (items input))))
	((= i end) nil)
      (assert-string= (nth i (items input))
		      (nth i result)))))

(defregression (:parsing 26)
    "Escaped quote character."
  (let* ((input *test-parsing-escaped-quote*)
	 (stream (make-string-input-stream (text input)))
	 (result (read-csv-record stream (csv-format input))))
    (cmsg-c :debug-parsing ">>~A<<" (text input))
    (do ((i 0 (1+ i))
	 (end (length (items input))))
	((= i end) nil)
      (assert-string= (nth i (items input))
		      (nth i result)))))

(defregression (:parsing 27)
    "Escaped record separator (newline)."
  (let* ((input *test-parsing-escaped-newline*)
	 (stream (make-string-input-stream (text input)))
	 (result (read-csv-record stream (csv-format input))))
    (cmsg-c :debug-parsing ">>~A<<" (text input))
    (do ((i 0 (1+ i))
	 (end (length (items input))))
	((= i end) nil)
      (assert-string= (nth i (items input))
		      (nth i result)))))

(defregression (:parsing 28)
    "Format value function is NIL."
  (let* ((items '("hello world" "a bc" "A B C"))
	 (input "\"hello world\", a bc , \"A B C\"")
	 (stream (make-string-input-stream input))
	 (format '((:f1 string) (:f2 string) (:f3)))
	 (result (read-csv-record stream format)))
    (do ((i 0 (1+ i))
	 (end (length items)))
	((= i end) nil)
      (if (cdr (nth i format))
	  (assert-string= (nth i items) (nth i result))
	  (progn
	    (cmsg-c :debug-parsing "expecting \"~A\" to be NIL" (nth i items))
	    (assert (not (nth i result))))))))
