// CoMET - The Crimson Fields Map Editing Tool
// Copyright (C) 2002-2005 Jens Granseuer
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//

////////////////////////////////////////////////////////////////////////
// edwidget.cpp
////////////////////////////////////////////////////////////////////////

#include <stdlib.h>

#include "edwidget.h"
#include "misc.h"

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::NumberWidget
// DESCRIPTION: Create a new NumberWidget.
// PARAMETERS : id     - widget identifier
//              x      - left edge of widget
//              y      - top edge of widget
//              w      - widget width
//              h      - widget height
//              number - initial value to display
//              min    - minimum allowed value
//              max    - maximum allowed value
//              flags  - widget flags (see widget.h for details)
//              title  - widget title
//              window - widget parent window
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

NumberWidget::NumberWidget( short id, short x, short y, unsigned short w, unsigned short h,
            short number, short min, short max, unsigned short flags,
            const char *title, Window *window ) :
    StringWidget( id, x, y, w, h, itoa(number, numbuf), strlen(itoa(max,numbuf2)),
                  flags, title, window ) {
  num = number;
  minval = min;
  maxval = max;
  SetValidator( this );
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::SetNumber
// DESCRIPTION: Fill the widget with a value.
// PARAMETERS : number - new widget value
//              upd    - whether to update the display (default is true)
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void NumberWidget::SetNumber( short number, bool upd /* = true */ ) {
  num = MIN( MAX( minval, number ), maxval );

  SetString( itoa( num, numbuf ), upd );
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::ValidateKey
// DESCRIPTION: Only accept numbers for input.
// PARAMETERS : str - string currently entered in widget (not used)
//              key - char to be entered
//              pos - position at which to enter the char
// RETURNS    : TRUE if key is accepted, FALSE if refused
////////////////////////////////////////////////////////////////////////

bool NumberWidget::ValidateKey( const char *str, unsigned short key,
                                unsigned short pos ) const {
  return ((key >= '0') && (key <= '9')) || ((key == '-') && (pos == 0));
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::SetMin
// DESCRIPTION: Set a new minimum value.
// PARAMETERS : min - new minimum
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void NumberWidget::SetMin( short min ) {
  minval = min;

  if ( Number() < min ) SetNumber( min );
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::SetMax
// DESCRIPTION: Set a new maximum value.
// PARAMETERS : max - new maximum
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void NumberWidget::SetMax( short max ) {
  maxval = max;

  if ( Number() > max ) SetNumber( max );
}

////////////////////////////////////////////////////////////////////////
// NAME       : NumberWidget::Release
// DESCRIPTION: When the widget is released make sure the number which
//              was entered is inside the requested boundaries and
//              adjust it if necessary.
// PARAMETERS : -
// RETURNS    : -
////////////////////////////////////////////////////////////////////////

void NumberWidget::Release( void ) {
  const char *str = String();
  short n = (str ? atoi(str) : 0);
  SetNumber( n );
  StringWidget::Release();
}

