import java.io.InputStream;
import java.io.IOException;
import java.io.FileNotFoundException;

/**
 * Interface for picking content from files.
 *
 * A class implementing the FileParser interface should accept a data source,
 * parse it, and provide access to the discovered document attributes. 
 *
 * These attributes are:
 * <ul>
 *   <li>Title</li>
 *   <li>Author</li>
 *   <li>Description</li>
 *   <li>Keywords</li>
 * </ul>
 *
 * If the data source provides hyperlinks, it should also parse:
 * <ul>
 *   <li>URLs</li>
 * </ul>
 *
 * @author Ben Secrest &lt;blsecres@users.sourceforge.net&gt;
 * @version $Id: FileParser.java,v 1.11 2002/08/27 00:07:26 blsecres Exp $
 */
public interface FileParser {
    /**
     * Perform parsing on the given source.
     * @param file The IGFile object to open a file for and fill in attributes
     * @throws IOException if an error occurs while reading a file
     * @throws FileNotFoundException if the file reference in the IGFile object
     *		does not exist
     */
    public void parse(IGFile file) throws IOException, FileNotFoundException;


    /**
     * Perform parsing on an open stream.
     * @param file The IGFile to fill in attributes for
     * @param stream The data source for parsing
     * @throws IOException if an error occurs while reading data
     */
    public void parse(IGFile file, InputStream stream)
	throws IOException, StreamResetException;


    /**
     * Set the desired attributes to extract
     * @param wanted A set of bits describing preferences
     */
    public void setWantedItems(IGKeySet wanted);


    /**
     * Set the logger to use with this module
     * @param logObj The object to use for logging data
     */
    public void setLog(IGLog logObj);


    /**
     * Get the file extensions this parser can handle
     * @return String array of file extensions
     */
    public String[] getExtensions();


    /**
     * Get the mime types this parser can handle
     * @return String array of mime types
     */
    public String[] getMimeTypes();


    /**
     * Get the file magic for files this parser can handle
     * @return Array of byte arrays containing magic signature
     */
    public FileMagic getMagic();
}
