/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Copyright 2004 Todd Kulesza
 *
 * Authors:
 * 		Todd Kulesza <todd@dropline.net>
 *		Davyd Madeley <davyd@ucc.asn.au>
 */

#include "utils.h"
#include "drivel_request.h"

#include <stdarg.h>
#include <string.h>

static void
add_item (DrivelRequest *dr, const gchar *key, const gchar *value)
{
	DrivelRequestItem *dri;
	
	dri = g_new0 (DrivelRequestItem, 1);
	dri->key = g_strdup (key);
	dri->value = g_strdup (value);
	
	dr->items = g_slist_prepend (dr->items, dri);
	
	return;
}

/* Create an empty DrivelRequest */
DrivelRequest*
drivel_request_new (void)
{
	DrivelRequest *dr;
	
	dr = g_new0 (DrivelRequest, 1);
	dr->items = NULL;
	dr->current = NULL;
	dr->type = REQUEST_TYPE_NONE;
	dr->protocol = REQUEST_PROTOCOL_NONE;
	dr->api = BLOG_API_UNKNOWN;
	dr->request_values = g_hash_table_new_full (g_str_hash, g_str_equal,
			hash_table_item_free, hash_table_item_free);
	dr->data = NULL;
	
	return dr;
}

/* Create a DrivelRequest with a type and multiple items, terminated by NULL */
DrivelRequest*
drivel_request_new_with_items (DrivelRequestType type, 
		DrivelRequestProtocol protocol, DrivelBlogAPI api, ...)
{
	DrivelRequest *dr;
	va_list ap;
	const gchar *key, *value;
	
	dr = drivel_request_new ();
	drivel_request_set_type (dr, type);
	drivel_request_set_protocol (dr, protocol);
	drivel_request_set_api (dr, api);
	
	va_start (ap, api);
	
	do
	{
		key = va_arg (ap, gchar*);
		if (key)
		{
			value = va_arg (ap, gchar*);
			add_item (dr, key, value);
		}
	} while (key);
	
	va_end (ap);
	
	return dr;
}

/* Add multiple items to a DrivelRequest.  Terminate the list with NULL. */
void
drivel_request_add_items (DrivelRequest *dr, ...)
{
	va_list ap;
	const gchar *key, *value;
	
	va_start (ap, dr);
	
	do
	{
		key = va_arg (ap, gchar*);
		if (key)
		{
			value = va_arg (ap, gchar*);
			add_item (dr, key, value);
		}
	} while (key);
	
	va_end (ap);
	
	return;
}

/* Remove multiple items by their key names.  Terminate the list with NULL. */
void
drivel_request_remove_items (DrivelRequest *dr, ...)
{
	va_list ap;
	const gchar *key;
	GSList *current;
	gboolean found;
	
	do
	{
		key = va_arg (ap, gchar*);
		if (key)
		{
			found = FALSE;
			for (current = dr->items; current && !found; current = current->next)
			{
				DrivelRequestItem *item;
				
				item = current->data;
				if (!strcmp (item->key, key))
				{
					found = TRUE;
					g_free (item->key);
					g_free (item->value);
					g_free (item);
					dr->items = g_slist_delete_link (dr->items, current);
				}
			}
		}
	} while (key);
	
	va_end (ap);
	
	return;
}

/* Set the type of request */
void
drivel_request_set_type (DrivelRequest *dr, DrivelRequestType type)
{
	dr->type = type;
	
	return;
}

/* Get the type of request */
DrivelRequestType
drivel_request_get_type (DrivelRequest *dr)
{
	return (dr->type);
}

/* Set the network protocol */
void
drivel_request_set_protocol (DrivelRequest *dr, DrivelRequestProtocol protocol)
{
	dr->protocol = protocol;
	
	return;
}

/* Get the network protocol */
DrivelRequestProtocol
drivel_request_get_protocol (DrivelRequest *dr)
{
	return (dr->protocol);
}

/* Set the journal API */
void
drivel_request_set_api (DrivelRequest *dr, DrivelBlogAPI api)
{
	dr->api = api;
	
	return;
}

/* Get the journal API */
DrivelBlogAPI
drivel_request_get_api (DrivelRequest *dr)
{
	return dr->api;
}

/* Set the returned network data */
void
drivel_request_set_data (DrivelRequest *dr, DrivelRequestData *data)
{
	dr->data = data;
	
	return;
}

/* Get a pointer to the returned network data */
DrivelRequestData*
drivel_request_get_data (DrivelRequest *dr)
{
	return (dr->data);
}

/* Insert a key/value pair into the request_values hash table */
void
drivel_request_value_insert (DrivelRequest *dr, const gchar *key, const gchar *value)
{
	g_hash_table_insert (dr->request_values, g_strdup (key), g_strdup (value));
	
	return;
}

/* Lookup a value given a key in the request_values hash table */
const gchar*
drivel_request_value_lookup (DrivelRequest *dr, const gchar *key)
{
	gchar *value;
	
	value = g_hash_table_lookup (dr->request_values, key);
	
	return value;
}

/* Clear the request_values hash table */
void
drivel_request_clear_values (DrivelRequest *dr)
{
	hash_table_clear (dr->request_values);
	
	return;
}

/* Set the DrivelRequest to the first item, returns FALSE if no items are
   available. */
gboolean
drivel_request_start (DrivelRequest *dr)
{
	dr->current = dr->items;
	
	return (!(!(dr->current)));
}

/* Iterate to the next item in the list, returns FALSE if no items are
   available. */
gboolean
drivel_request_next (DrivelRequest *dr)
{
	dr->current = dr->current->next;
	
	return (!(!(dr->current)));
}

/* Returns the current item in the list, or NULL if no items are available. */
DrivelRequestItem*
drivel_request_get_current_item (DrivelRequest *dr)
{
	DrivelRequestItem *item;
	
	if (dr->current)
		item = dr->current->data;
	else
		item = NULL;
	
	return item;
}

/* Looks for 'key' in 'items', returns 'value' if found */
const gchar*
drivel_request_item_lookup (DrivelRequest *dr, const gchar *key)
{
	DrivelRequestItem *item;
	gboolean valid;
	const gchar *value = NULL;
	
	valid = drivel_request_start (dr);
	while (valid)
	{
		item = drivel_request_get_current_item (dr);
		if (!strcmp (key, item->key))
		{
			value = item->value;
			valid = FALSE;
		}
		else
			valid = drivel_request_next (dr);
	}
	
	return value;
}
/* Frees the memory used by the object and the list. */
void
drivel_request_free (DrivelRequest *dr)
{
	GSList *current;
	DrivelRequestItem *item;
	
	for (current = dr->items; current; current = current->next)
	{
		item = current->data;
		g_free (item->key);
		g_free (item->value);
		g_free (item);
	}
	g_slist_free (dr->items);
	hash_table_clear (dr->request_values);
	if (dr->data)
	{
		g_free (dr->data->data);
		g_free (dr->data);
	}
	g_free (dr);
	
	return;
}
