/*  $Header: /cvsroot/dvipdfmx/src/pdffont.c,v 1.5 2004/03/11 11:50:22 hirata Exp $

    This is dvipdfmx, an eXtended version of dvipdfm by Mark A. Wicks.

    Copyright (C) 2002 by Jin-Hwan Cho and Shunsaku Hirata,
    the dvipdfmx project team <dvipdfmx@project.ktug.or.kr>
    
    Copyright (C) 1998, 1999 by Mark A. Wicks <mwicks@kettering.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.
    
    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.
    
    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA.
*/

#if HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "system.h"
#include "mfileio.h"
#include "mem.h"
#include "error.h"

#include "pdfobj.h"

#include "encodings.h"
#include "cmap.h"
#include "unicode.h"

#include "dpxutil.h"

#include "type1.h"
#include "type1c.h"
#include "ttf.h"
#include "pkfont.h"

#include "pdffont.h"

static int __verbose = 0;

#define PDFFONT_DEBUG_STR "Font"
#define PDFFONT_DEBUG     3

void
pdf_font_set_verbose (void)
{
  __verbose++;
}

int
pdf_font_get_verbose (void)
{
  return __verbose;
}

void
pdf_font_set_dpi (int font_dpi)
{
  PKFont_set_dpi(font_dpi);
}

void
pdf_font_make_tag (char *tag)
{
  int i;
  char ch;
  static char first = 1;

  if (first) {
    srand(time(NULL));
    first = 0;
  }

  for (i = 0; i < 6; i++) {
    ch = rand() % 26;
    tag[i] = ch + 'A';
  }
  tag[6] = '\0';

}

static pdf_font *
pdf_font_new (void)
{
  pdf_font *font;

  font = NEW(1, pdf_font);

  font->ident = NULL;
  font->subtype = -1;

  memset(font->fontname, 0, PDF_NAME_LEN_MAX*sizeof(char));

  font->filename  = NULL;
  font->embedding =  1;
  font->encoding_id = -1;

  font->indirect   = NULL;
  font->fontdict   = NULL;
  font->descriptor = NULL;

  font->used_chars = NULL;
  font->flags = 0;

  return font;
}

static void
pdf_font_flush (pdf_font *font)
{
  if (font) {
    if (font->indirect)   pdf_release_obj(font->indirect);
    if (font->fontdict)   pdf_release_obj(font->fontdict);
    if (font->descriptor) pdf_release_obj(font->descriptor);

    font->indirect   = NULL;
    font->fontdict   = NULL;
    font->descriptor = NULL;
  }

  return;
}

static void
pdf_font_release (pdf_font *font)
{
  if (font) {
    if (font->ident)      RELEASE(font->ident);
    if (font->filename)   RELEASE(font->filename);
    if (font->used_chars) RELEASE(font->used_chars);

    if (font->indirect)   ERROR("%s: Object not flushed.", PDFFONT_DEBUG_STR);
    if (font->fontdict)   ERROR("%s: Object not flushed.", PDFFONT_DEBUG_STR);
    if (font->descriptor) ERROR("%s: Object not flushed.", PDFFONT_DEBUG_STR);
    RELEASE(font);
  }

  return;
}

#define CACHE_ALLOC_SIZE 16u

static struct pdf_font_cache {
  int num;
  int max;
  pdf_font **fonts;
} *font_cache = NULL;

void
pdf_font_init (void)
{
  if (font_cache)
    return;

  font_cache = NEW(1, struct pdf_font_cache);
  font_cache->num = 0;
  font_cache->max = CACHE_ALLOC_SIZE;
  font_cache->fonts = NEW(font_cache->max, pdf_font *);
}

#define CHECK_ID(n) do {\
                        if (!font_cache)\
                           ERROR("%s: Font cache not initialized.", PDFFONT_DEBUG_STR);\
                        if ((n) < 0 || (n) >= font_cache->num)\
                           ERROR("%s: Invalid ID %d", PDFFONT_DEBUG_STR, (n));\
                    } while (0)


pdf_obj *
pdf_font_get_resource (int font_id)
{
  pdf_font *font;

  CHECK_ID(font_id);
  font = font_cache->fonts[font_id];
  if (!font->indirect)
    font->indirect = pdf_ref_obj(font->fontdict);

  return pdf_link_obj(font->indirect);
}

char *
pdf_font_get_usedchar (int font_id)
{
  pdf_font *font;

  CHECK_ID(font_id);
  font = font_cache->fonts[font_id];
  if (!font->used_chars) {
    font->used_chars = NEW(256, char);
    memset(font->used_chars, 0, 256*sizeof(char));
  }

  return font->used_chars;
}

int
pdf_font_get_fonttype (int font_id)
{
  pdf_font *font;

  CHECK_ID(font_id);
  font = font_cache->fonts[font_id];

  return font->subtype;
}

char *
pdf_font_get_fontname (int font_id)
{
  pdf_font *font;

  CHECK_ID(font_id);
  font = font_cache->fonts[font_id];

  return font->fontname;
}

int
pdf_font_findfont (const char *map_name, int encoding_id,
		   double font_scale, double design_size, int embedding)
{
  pdf_font *font;
  int       font_id;

  if (!font_cache)
    pdf_font_init();
  ASSERT(font_cache);

  for (font_id = 0; font_id < font_cache->num; font_id++) {
    int found;

    font = font_cache->fonts[font_id];
    switch (font->subtype) {
    case PDFFONT_FONTTYPE_TYPE1:
      found = Type1Font_match(font, map_name, encoding_id, font_scale, design_size, 1);
      break;
    case PDFFONT_FONTTYPE_TYPE1C:
      found = Type1CFont_match(font, map_name, encoding_id, font_scale, design_size, 1);
      break;
    case PDFFONT_FONTTYPE_TRUETYPE:
      found = TTFont_match(font, map_name, encoding_id, font_scale, design_size, 1);
      break;
    case PDFFONT_FONTTYPE_TYPE3:
      found = PKFont_match(font, map_name, encoding_id, font_scale, design_size, 1);
      break;
    default:
      found = 0;
      ERROR("Unknown font type: %d", font->subtype);
    }
    if (found) {
      return font_id;
    }
  }

  font_id = font_cache->num;
  font    = pdf_font_new();

#ifdef OPENTYPE_PREFERED
  if (Type1CFont_open(font, map_name, encoding_id, font_scale, design_size, 1) >= 0) {
    font->subtype = PDFFONT_FONTTYPE_TYPE1C;
  } else if (Type1Font_open(font, map_name, encoding_id, font_scale, design_size, 1) >= 0) {
    font->subtype = PDFFONT_FONTTYPE_TYPE1;
#else
  if (Type1Font_open(font, map_name, encoding_id, font_scale, design_size, 1) >= 0) {
    font->subtype = PDFFONT_FONTTYPE_TYPE1;
  } else if (Type1CFont_open(font, map_name, encoding_id, font_scale, design_size, 1) >= 0) {
    font->subtype = PDFFONT_FONTTYPE_TYPE1C;
#endif
  } else if (TTFont_open(font, map_name, encoding_id, font_scale, design_size, 1) >= 0) {
    font->subtype = PDFFONT_FONTTYPE_TRUETYPE;
  } else if (PKFont_open(font, map_name, encoding_id, font_scale, design_size, 1) >= 0) {
    font->subtype = PDFFONT_FONTTYPE_TYPE3;
  } else {
    pdf_font_release(font);
    return -1;
  }

  if (font_cache->num >= font_cache->max) {
    font_cache->max  += CACHE_ALLOC_SIZE;
    font_cache->fonts = RENEW(font_cache->fonts, font_cache->max, pdf_font *);
  }
  font_cache->fonts[font_id] = font;
  font_cache->num += 1;

  return font_id;
}

void
pdf_font_close (void)
{
  int font_id;

  if (!font_cache)
    return;

  for (font_id = 0; font_id < font_cache->num; font_id++) {
    pdf_font *font;

    font = font_cache->fonts[font_id];
    if (__verbose) {
      MESG("(%s", font->ident);
      if (__verbose > 2) {
	MESG("[%s][%s+%s]", font->filename, font->uniqueID, font->fontname);
      } else if (__verbose > 1) {
	MESG("[%s][%s]", font->filename, font->fontname);
	if (font->encoding_id >= 0) {
	  Encoding *encoding = Encoding_cache_get(font->encoding_id);
	  MESG("[%s]", Encoding_get_name(encoding));
	} else {
	  MESG("[Built-in]");
	}
      }
    }

    switch (font->subtype) {
    case PDFFONT_FONTTYPE_TYPE1:
      if (__verbose) MESG("[Type1]");
      Type1Font_dofont (font);
      break;
    case PDFFONT_FONTTYPE_TYPE1C:
      if (__verbose) MESG("[Type1C]");
      Type1CFont_dofont (font);
      break;
    case PDFFONT_FONTTYPE_TRUETYPE:
      if (__verbose) MESG("[TrueType]");
      TTFont_dofont (font);
      break;
    case PDFFONT_FONTTYPE_TYPE3:
      if (__verbose) MESG("[Type3/PK]");
      PKFont_dofont (font);
      break;
    default:
      ERROR("Unknown font type: %d", font->subtype);
    }
    pdf_font_flush(font);
    pdf_font_release(font);

    if (__verbose)
      MESG(")");
  }
  RELEASE(font_cache->fonts);
  RELEASE(font_cache);
  font_cache = NULL;

  return;
}
