;;; emms-modeline.el --- Modeline and titlebar infos for emms

;; Copyright (C) 2004  Free Software Foundation, Inc.

;; Author: Mario Domgrgen <kanaldrache@gmx.de>
;; Keywords: multimedia

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This file is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with GNU Emacs; see the file COPYING.  If not, write to
;; the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
;; Boston, MA 02111-1307, USA.

;;; Commentary:
;;
;; To activate put simply the following line in your Emacs:
;;
;;   (require 'emms-modeline)
;;   (emms-modeline 1)

;;; Code:

(require 'emms)

(defgroup emms-modeline nil
  "Showing information on modeline and titlebar"
  :prefix "emms-modeline-"
  :group 'emms)

(defcustom emms-modeline-modeline-function 'emms-modeline-playlist-current
  "Function for showing infos in modeline or nil if don't want to."
  :type '(choice (const :tag "Don't show info on modeline" nil) function)
  :group 'emms-modeline)

(defcustom emms-modeline-titlebar-function nil
  "Function for showing infos in titlebar or nil if you don't want to."
  :type '(choice (const :tag "Don't show info on titlebar" nil) function)
  :group 'emms-modeline)

(defcustom emms-modeline-format " [ %s ] "
  "String used for displaying the current track in modeline and titlebar."
  :type 'string
  :group 'emms)

(defun emms-modeline-playlist-current ()
  "Format the currently playing song"
  (format emms-modeline-format (emms-playlist-current)))

(defvar emms-modeline-initial-titlebar frame-title-format)

(defun emms-modeline (arg)
  "Turn on `emms-modeline' if ARG is positive, off otherwise."
  (interactive "p")
  (or global-mode-string (setq global-mode-string '("")))
  (if (and arg (> arg 0))
      (progn
	(add-hook 'emms-playlist-current-changed-hook 'emms-modeline-alter)
	(add-hook 'emms-player-finished-hook 'emms-modeline-blank)
	(add-hook 'emms-player-stopped-hook 'emms-modeline-blank)
	(add-hook 'emms-player-started-hook 'emms-modeline-alter)	
	(when (and emms-modeline-modeline-function
		   (not (member 'emms-modeline-string global-mode-string)))
	  (setq global-mode-string
		(append global-mode-string
			'(emms-modeline-string))))
	(when emms-player-playing-p (emms-modeline-alter)))
    (remove-hook 'emms-playlist-current-changed-hook 'emms-modeline-alter)
    (remove-hook 'emms-player-finished-hook 'emms-modeline-blank)
    (remove-hook 'emms-player-stopped-hook 'emms-modeline-blank)
    (remove-hook 'emms-player-started-hook 'emms-modeline-alter)
    (emms-modeline-restore-titlebar)
    (emms-modeline-restore-modeline)))

(defun emms-modeline-alter ()
  "Alter modeline/titlebar."
  (emms-modeline-alter-modeline)
  (emms-modeline-alter-titlebar))

(defun emms-modeline-alter-modeline ()
  "Update the modeline with song info."
  (when emms-modeline-modeline-function
    (setq emms-modeline-string
	  (funcall emms-modeline-modeline-function))
    (force-mode-line-update)))

(defun emms-modeline-alter-titlebar ()
  "Update the titlebar with song info."
  (when emms-modeline-titlebar-function
    (setq frame-title-format
	  (list "" emms-modeline-initial-titlebar (funcall emms-modeline-titlebar-function)))))


(defun emms-modeline-blank ()
  "Blank modeline and titlebar but not quit `emms-modeline'."
  (setq emms-modeline-string nil)
  (force-mode-line-update)
  (emms-modeline-restore-titlebar))
	
(defun emms-modeline-restore-modeline ()
  "Restore the modeline."
  (when emms-modeline-modeline-function
    (setq global-mode-string
	  (remove 'emms-modeline-string global-mode-string))
    (force-mode-line-update)))

(defun emms-modeline-restore-titlebar ()
  "Restore the modeline."
  (when emms-modeline-titlebar-function
    (setq frame-title-format
	  (list emms-modeline-initial-titlebar))))

(provide 'emms-modeline)
;;; emms-modeline.el ends here
