
#include <glib.h>
#include "entity.h"

static gint unique_enode_watcher_id = 0;

void
enode_event_parent (ENode * parent, ENode * child)
{
    GSList *tmp;
    ENodeEventWatcher *watcher;

    ECHECK_RET (parent != NULL);
    ECHECK_RET (child != NULL);

    /* Check to be sure its not already parented */
    ECHECK_RET (!ENODE_FLAG_ISSET (child, ENODE_PARENTED));

    EDEBUG (("enode-event", "enode_event_parent for parent %s, child %s",
	     parent->element->str, child->element->str));

    tmp = parent->event_watchers;
    while (tmp) {
	watcher = tmp->data;

	EDEBUG (("enode-event",
		 "Checking parent %s for childattr settings, watcher flags %d",
		 parent->element->str, watcher->flags));
	/* If it's a child watcher, we propogate. */
	if (watcher->flags & ENODE_CHILDCHANGEFLAGS)
	    child->event_watchers =
		g_slist_append (child->event_watchers, watcher);

	tmp = tmp->next;
    }

    element_parent_notify (parent, child);

    /* We call the new_child notifyer here because all new children are also
     * immediately renderered. */
    tmp = child->event_watchers;
    while (tmp) {
	watcher = tmp->data;

	/* TODO: Do we have to check the dispatcher type ? */
	if (watcher->flags & ENODE_ONCHILDNEW)
	    if (watcher->dispatch) {
		EventNewChild dispatch = watcher->dispatch;
		dispatch (child, watcher->user_data);
		EDEBUG (("enode-event",
			 "'parent' dispatched on node %s for watcher flag %d",
			 child->element->str, watcher->flags));
	    }
	tmp = tmp->next;
    }
}


void
enode_event_delete (ENode * node)
{
    GSList *tmp;
    ENodeEventWatcher *watcher;

    ECHECK_RET (node != NULL);

    tmp = node->event_watchers;
    while (tmp) {
	watcher = tmp->data;

	/* TODO: Do we have to check the dispatcher type ? */
	if (watcher->flags & ENODE_ONCHILDDELETE)
	    if (watcher->dispatch) {
		EventDelete dispatch = watcher->dispatch;
		dispatch (node, watcher->user_data);
	    }

	/* FIXME: Look to see if this is the node that implemented the child
	 * watch, and if so, delete full tree */

	tmp = tmp->next;
    }

    g_slist_free (node->event_watchers);
    element_destroy_notify (node);
}

void
enode_event_set_attrib (ENode * node, EBuf * attr, EBuf * value, gint notify_renderer)
{
    ENodeEventWatcher *watcher;
    GSList *tmp;

    ECHECK_RET (node != NULL);
    ECHECK_RET (attr != NULL);
    ECHECK_RET (value != NULL);

    tmp = node->event_watchers;
    while (tmp) {
	watcher = tmp->data;

	/* TODO: Do we have to check the dispatcher type ? */
	if (watcher->flags & ENODE_ONCHILDATTRIBCHANGE ||
	    watcher->flags & ENODE_ONATTRIBCHANGE) if (watcher->dispatch) {
		EventAttribChange dispatch = watcher->dispatch;
		dispatch (node, attr, value, watcher->user_data);
		EDEBUG (("enode-event",
			 "'set_attrib' dispatched on node %s for watcher flag %d",
			 node->element->str, watcher->flags));
	    }
	tmp = tmp->next;
    }

    if (notify_renderer)
	element_set_attrib_notify (node, attr, value);

}

/* 
 * void         enode_event_get_attrib (ENode *node, EBuf *attr); */

void
enode_event_set_data (ENode * node, EBuf * data)
{
    ECHECK_RET (node != NULL);
    element_set_data_notify (node, data);
}

static gint
enode_event_install_child_watcher_callback (ENodeTreeWalk * walker)
{
    ENode *curnode = walker->curnode;
    ENodeEventWatcher *watcher = walker->user_data1;

    /* We only set the flag in the children if it is rendered, otherwise, it
     * * will be added again when the new node is parented (see *
     * enode_event_parent ()). */
    if (ENODE_FLAG_ISSET (curnode, ENODE_RENDERED))
	curnode->event_watchers =
	    g_slist_append (curnode->event_watchers, watcher);

    EDEBUG (("enode-event",
	     "Installed child watcher on %s", curnode->element->str));
    return (TRUE);
}

static void
enode_event_install_child_watcher (ENode * node, ENodeEventWatcher * watcher)
{
    ENodeTreeWalk *walker;

    node->event_watchers = g_slist_append (node->event_watchers, watcher);

    walker = enode_treewalk_new (node);
    walker->user_data1 = watcher;
    enode_treewalk (walker, enode_event_install_child_watcher_callback, NULL);
    enode_treewalk_free (walker);
}

gint
enode_event_watch_attrib (ENode * node,
			  EventAttribChange onchange, gpointer user_data)
{
    ENodeEventWatcher *watcher;

    ECHECK_RETVAL (node != NULL, -1);
    ECHECK_RETVAL (onchange != NULL, -1);

    watcher = g_new (ENodeEventWatcher, 1);

    watcher->flags = ENODE_ONATTRIBCHANGE;
    watcher->dispatch = onchange;
    watcher->node = node;
    watcher->user_data = user_data;
    watcher->id = unique_enode_watcher_id++;

    node->event_watchers = g_slist_append (node->event_watchers, watcher);

    EDEBUG (("enode-event",
	     "Installed ONATTRIBCHANGE watcher on %s", node->element->str));

    return (watcher->id);
}

gint
enode_event_watch_child_attrib (ENode * node,
				EventAttribChange onchange, gpointer user_data)
{
    ENodeEventWatcher *watcher;

    ECHECK_RETVAL (node != NULL, -1);
    ECHECK_RETVAL (onchange != NULL, -1);

    watcher = g_new (ENodeEventWatcher, 1);

    watcher->flags = ENODE_ONCHILDATTRIBCHANGE;
    watcher->dispatch = onchange;
    watcher->node = node;
    watcher->user_data = user_data;
    watcher->id = unique_enode_watcher_id++;

    EDEBUG (("enode-event", "node %s has installed child_attrib watcher..",
	     node->element->str));

    enode_event_install_child_watcher (node, watcher);

    return (watcher->id);
}

/* 
 * void         enode_event_watch_data (ENode *node, gpointer onchange); void
 *      enode_event_watch_child_data (ENode *node, gpointer onchange);
 * 
 * void         enode_event_watch_delete (ENode *node, gpointer onchange);
 * 
 */

gint
enode_event_watch_child_delete (ENode * node, EventDelete onchange,
				gpointer user_data)
{
    ENodeEventWatcher *watcher;

    ECHECK_RETVAL (node != NULL, -1);
    ECHECK_RETVAL (onchange != NULL, -1);

    watcher = g_new (ENodeEventWatcher, 1);

    watcher->flags = ENODE_ONCHILDDELETE;
    watcher->dispatch = onchange;
    watcher->node = node;
    watcher->user_data = user_data;
    watcher->id = unique_enode_watcher_id++;

    EDEBUG (("enode-event", "node %s has installed child_delete watcher..",
	     node->element->str));

    enode_event_install_child_watcher (node, watcher);

    return (watcher->id);
}

gint
enode_event_watch_child_new (ENode * node,
			     EventNewChild onchange, gpointer user_data)
{
    ENodeEventWatcher *watcher;

    ECHECK_RETVAL (node != NULL, -1);
    ECHECK_RETVAL (onchange != NULL, -1);

    watcher = g_new (ENodeEventWatcher, 1);

    watcher->flags = ENODE_ONCHILDNEW;
    watcher->dispatch = onchange;
    watcher->node = node;
    watcher->user_data = user_data;
    watcher->id = unique_enode_watcher_id++;

    EDEBUG (("enode-event", "node %s has installed child_new watcher..",
	     node->element->str));

    enode_event_install_child_watcher (node, watcher);

    return (watcher->id);
}

/* 
 * void         enode_event_release (ENode *node, gint watcher_id) */


