#include <glib.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "entity.h"
#include "modulegen.h"

#include "rargv.h"

#if 0
/* Handle all termination signals. */
static void
signal_handle_end (int sig)
{
    EDEBUG (("entity-exec-class", "got sig=%i\n", sig));

    fprintf (stderr, "Exiting with sig%i\n", sig);

    exit (1);			/* For now we must exit... */
}

#endif 

/* Another brick in the -Wall */
void perl_init (RendererFlags flags);
void tcl_init (RendererFlags flags);
void c_init (RendererFlags flags);
void python_init (RendererFlags flags);
void rendgtk_init (RendererFlags flags);
void mzscheme_init (RendererFlags flags);
void javascript_init (RendererFlags flags);

/* Builtins */
void rendgeneric_init (RendererFlags flags);
void entity_lang_init (RendererFlags flags);
void rendcsink_init (RendererFlags flags);
void rendresolver_init (RendererFlags flags);


void
entity_initialize_modules (void)
{
    /* These are built in, and are always initialized */
    rendgeneric_init (RENDERER_REGISTER | RENDERER_INIT);
    entity_lang_init (RENDERER_REGISTER | RENDERER_INIT);
#ifdef USE_CSINK
    rendcsink_init (RENDERER_REGISTER | RENDERER_INIT);
    rendresolver_init (RENDERER_REGISTER | RENDERER_INIT);
#endif
#ifdef STATIC_PERL
    perl_init (RENDERER_REGISTER | RENDERER_INIT);
#endif
#ifdef STATIC_TCL
    tcl_init (RENDERER_REGISTER | RENDERER_INIT);
#endif
#ifdef STATIC_C
    c_init (RENDERER_REGISTER | RENDERER_INIT);
#endif
#ifdef STATIC_PYTHON
    python_init (RENDERER_REGISTER | RENDERER_INIT);
#endif
#ifdef STATIC_GTK
    rendgtk_init (RENDERER_REGISTER | RENDERER_INIT);
#endif
#ifdef USE_MZSCHEME
    mzscheme_init (RENDERER_REGISTER | RENDERER_INIT);
#endif
#ifdef STATIC_JAVASCRIPT
    javascript_init (RENDERER_REGISTER | RENDERER_INIT);
#endif
}

static void
entity_create_dotdirs (void)
{
    /* broken on win32 FIXME */
#ifndef WIN32
    gchar *path;

    /* Insure entity home is created */
    path = g_strconcat (ENTITY_HOME, NULL);

    if (-1 == mkdir (path, 0700)) {
	if (errno != EEXIST) {	/* Hope this is a dir. */
	    g_error ("creating directory %s: %s", path, g_strerror (errno));
	}
    }
    g_free (path);
#endif
}


static void
print_usage (void)
{
    g_print ("Usage: %s  [-Vhm] [-g <domain>] [-l <config path>] [-a <files>] [file]\n"
	     "  -V           Print version and exit.\n"
	     "  -g           Turn on debug messages. "
	     "Accepts domain as argument, or 'all'\n"
	     "  -a           Load all files on command line into Entity.\n"
	     "  -m           Generate module config files.\n"
	     "  -l           Specify alternate location for libraries and configuration directories.\n"
	     "  -h           Print this message and exit.\n"
	     "%s %s, compiled: %s %s\n"
	     "Report bugs to <entity-devel@lists.sourceforge.net>\n",
	     PACKAGE, PACKAGE, VERSION, __DATE__, __TIME__);
}


int
main (int argc, gchar * argv[])
{
    GSList *files = NULL;
    GSList *tmp;
    gint load_all_files = FALSE;

    entity_init (&argc, &argv);

    RARGV_SWITCH {
	case 'g':		/* Debugging flag: -g <domains>. */
	    edebug_enable (RARGV_NEXT);
	    break;

	case 'a':		/* Load all files on command line into entity. */
	    load_all_files = TRUE;
	    break;

	case 'h':		/* Help: -h or -\?. */
	    /* Fall through. */
	case '?':
	    print_usage ();
	    exit (0);
	    /* NOTREACHED */
	    break;

	case 'V':		/* Print the version. */
	    g_print ("%s %s\n", PACKAGE, VERSION);
	    exit (0);
	    /* NOTREACHED */
	    break;
            
#ifndef WIN32
	case 'm':		/* Create a config file from loaded libs. */
	    modulegen (NULL);
	    break;
            
	case 't':		/* Create a config file from loaded libs. */
	    modulegen (NULL);
	    break;
#endif				/* ifndef WIN32 */
	
        case 'l':		/* Set alternative directory for config tree. 
				 */
            optarg = RARGV_NEXT;
            if (!optarg) {
                g_print ("Option explected for -l\n");
                print_usage ();
                exit (0);
            }
	    econfig_set_attr ("config-location", g_strdup (optarg));
	    break;
	
        case 'c':		/* The execution class to start file on:
				 * -c <class>. */
            optarg = RARGV_NEXT;
            if (!optarg) {
                g_print ("Option explected for -c\n");
                print_usage ();
                exit (0);
            }
	    econfig_set_attr ("exec-class", g_strstrip (optarg));
	    break;

        case '*':

            if (!files || load_all_files) { 
                files = g_slist_append (files, RARGV_CURRENT);
            }
            break;

	default:	
            g_print ("Unknown flag: -%c\n", RARGV_CURRENT_FLAG);
            print_usage ();
            exit (0);
            break;

    } RARGV_CLOSE;

    /* Make sure entity dotdir exists. */
    entity_create_dotdirs ();
    
    /* Initialize renderers. */
    renderers_init ();

    /* Initialize builtin modules. */
    entity_initialize_modules ();

    /* Load this file up in the execution class specified.  This also ends up
     * calling the mainloop. */
    if (files) {
        for (tmp = files; tmp; tmp = tmp->next) {
            xml_parse_file (NULL, tmp->data);
        }
    } else {
	print_usage ();
	exit (0);
    }

    g_slist_free (files);
   
    if (!entity_mainloop_is_registered ()) {
	g_warning ("No mainloop was registered for use in Entity.");
	g_warning ("This usually indicates that no tag was used which has a renderer that supplies a mainloop.");
	exit (1);
    }

    /* Loop waiting on the tree. */
    entity_mainloop_run ();

    return (0);
}


