#include <gtk/gtk.h>
#include "entity.h"
#include "gtk-common.h"
#include "gtk-widget-attr.h"

static EBuf *
find_parent_enode_with_attrib (ENode * node, gchar * search, gchar * attrib)
{
    ENode *parent;
    EBuf *value;

    parent = node;
    while (parent) {
	value = enode_attrib (parent, attrib, NULL);
	if (ebuf_not_empty (value))
	    return (value);
	parent = enode_parent (parent, search);
    }

    return (NULL);
}


static void
rendgtk_tree_item_onselect_callback (GtkWidget * widget, GtkWidget * item,
				     gpointer user_data)
{
    ENode *child_node;
    EBuf *function;

    child_node = gtk_object_get_data (GTK_OBJECT (item), "xml-node");

    if (!child_node)
	return;

    EDEBUG (("tree-renderer", "selected node %s.%s", child_node->element->str,
	     enode_attrib_str (child_node, "name", NULL)));
    enode_attrib_quiet (child_node, "selected", ebuf_new_with_true ());

    function = find_parent_enode_with_attrib (child_node, "tree", "onselect");
    if (function)
	enode_call_ignore_return (child_node, function->str, "");
}

static void
rendgtk_tree_item_ondeselect_callback (GtkWidget * widget, GtkWidget * item,
				       gpointer user_data)
{
    ENode *node = user_data;
    ENode *child;
    EBuf *function;

    child = gtk_object_get_data (GTK_OBJECT (item), "xml-node");
    if (!child)
	return;

    EDEBUG (("tree-renderer", "deselected node %s.%s", child->element->str,
	     enode_attrib_str (child, "name", NULL)));

    enode_attrib_quiet (child, "selected", ebuf_new_with_str (""));

    function = find_parent_enode_with_attrib (child, "tree", "onselect");
    if (function)
	enode_call_ignore_return (node, function->str, "");
}


static void
rendgtk_tree_item_expand_callback (GtkWidget * widget, gpointer user_data)
{
    ENode *node = user_data;
    EBuf *function;

    enode_attrib_quiet (node, "expanded", ebuf_new_with_true ());

    function = find_parent_enode_with_attrib (node, "tree", "onexpand");
    if (function)
	enode_call_ignore_return (node, function->str, "");
}

static void
rendgtk_tree_item_collapse_callback (GtkWidget * widget, gpointer user_data)
{
    ENode *node = user_data;
    EBuf *function;

    enode_attrib_quiet (node, "expanded", ebuf_new_with_false ());

    function = find_parent_enode_with_attrib (node, "tree", "oncollapse");
    if (function)
	enode_call_ignore_return (node, function->str, "");
}

static void
rendgtk_tree_item_onselectchange_callback (GtkWidget * widget,
					   gpointer user_data)
{
    ENode *node = user_data;
    EBuf *function;

    EDEBUG (
	    ("tree-renderer", "selection change event - node %s.%s",
	     node->element->str, enode_attrib_str (node, "name", NULL)));

    function =
	find_parent_enode_with_attrib (node, "tree", "onselectionchange");
    if (function)
	enode_call_ignore_return (node, function->str, "");
}

static gint
rendgtk_tree_selection_type_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *tree;

    tree = enode_get_kv (node, "tree-widget");

    EDEBUG (("tree-renderer", "selection-type set to %s", value->str));

    if (!tree) {
	EDEBUG (("tree-renderer", "umm, no tree ?!?!"));
	return (TRUE);
    }


    if (ebuf_equal_str (value, "multiple")) {
	gtk_tree_set_selection_mode (GTK_TREE (tree), GTK_SELECTION_MULTIPLE);
	return (TRUE);
    }

    if (ebuf_equal_str (value, "browse")) {
	gtk_tree_set_selection_mode (GTK_TREE (tree), GTK_SELECTION_BROWSE);
	return (TRUE);
    }

    if (ebuf_equal_str (value, "extended")) {
	gtk_tree_set_selection_mode (GTK_TREE (tree), GTK_SELECTION_EXTENDED);
	return (TRUE);
    }

    /* Single is default */
    gtk_tree_set_selection_mode (GTK_TREE (tree), GTK_SELECTION_SINGLE);

    return (TRUE);
}


static gint
rendgtk_tree_select_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *tree_item;
    GtkWidget *tree;

    tree_item = enode_get_kv (node, "tree-item-widget");
    tree = enode_get_kv (node, "tree-widget");

    if (!tree_item || !tree)
	return (TRUE);

    if (erend_value_is_true (value))
	gtk_tree_select_child (GTK_TREE (tree), tree_item);
    else
	gtk_tree_unselect_child (GTK_TREE (tree), tree_item);

    return (TRUE);
}


static gint
rendgtk_tree_expanded_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *tree_item;

    tree_item = enode_get_kv (node, "tree-item-widget");

    if (!tree_item)
	return (TRUE);

    if (erend_value_is_true (value))
	gtk_tree_item_expand (GTK_TREE_ITEM (tree_item));
    else
	gtk_tree_item_collapse (GTK_TREE_ITEM (tree_item));

    return (TRUE);
}

void
rendgtk_tree_destroy (ENode * node)
{
    GtkWidget *tree_item;

    ENode *parent_node;

    EDEBUG (("tree-renderer", "Destroying tree of some sorts.."));

    /* Check to see if this is a toplevel tree. */
    parent_node = enode_parent (node, NULL);
    if (!ebuf_equal_str (parent_node->element, "tree")) {
	GtkWidget *widget = enode_get_kv (node, "tree-widget");
	EDEBUG (("tree-renderer", "Destroying top level tree"));
	if (widget)
	    gtk_widget_destroy (widget);
    } else {
	/* we are a child of a tree, and we need to remove ourselves properly 
	 * from the tree */
	tree_item = enode_get_kv (node, "tree-item-widget");

	EDEBUG (("tree-renderer", "Destroying tree item"));
	/* gtk_tree_item_remove_subtree (GTK_TREE_ITEM (tree_item)); */
	/* I think the above would work if we also got rid of the tree */

	/* Cruelty to dumb trees */
	if (tree_item)
	    gtk_widget_hide (tree_item);
    }
}


static gint
rendgtk_tree_expandable_attr_set (ENode * node, EBuf * attr, EBuf * value)
{
    GtkWidget *tree;

    tree = enode_get_kv (node, "tree-widget");

    EDEBUG (
	    ("tree-renderer", "checking for tree widget - %p - node %s.%s",
	     node, node->element->str, enode_attrib_str (node, "name", NULL)));

    /* We have to do this because this function is not always called by the
     * renderer */
    value = enode_attrib (node, "expandable", NULL);

    if (ebuf_empty (value) || erend_value_is_true (value)) {
	GtkWidget *tree_item;
	tree_item = enode_get_kv (node, "tree-item-widget");

	EDEBUG (
		("tree-renderer",
		 "seeing if we need to create a tree widget for this node."));
	if (!tree && tree_item) {

	    EDEBUG (("tree-renderer", "Yep! Creating a tree widget for node"));

	    tree = gtk_tree_new ();
	    enode_set_kv (node, "tree-widget", tree);

	    gtk_signal_connect (GTK_OBJECT (tree), "select-child",
				rendgtk_tree_item_onselect_callback, node);
	    gtk_signal_connect (GTK_OBJECT (tree), "selection-changed",
				rendgtk_tree_item_onselectchange_callback,
				node);
	    gtk_signal_connect (GTK_OBJECT (tree), "unselect-child",
				rendgtk_tree_item_ondeselect_callback, node);

	    EDEBUG (("tree-renderer", "parenting %p to %p", tree, tree_item));
	    gtk_widget_show (tree);

	    if (tree_item->parent)
		gtk_tree_item_set_subtree (GTK_TREE_ITEM (tree_item), tree);
	}
    } else {
	EDEBUG (("tree-renderer", "Destroying tree widget for node!"));

	if (tree)
	    gtk_widget_destroy (tree);
	enode_set_kv (node, "tree-widget", NULL);
    }
    return (TRUE);
}

static void
rendgtk_tree_render (ENode * node)
{
    ENode *parent;
    GtkWidget *tree_item;
    GtkWidget *hbox;
    GtkWidget *tree;

    /* Check to see if this is a toplevel tree. */
    parent = enode_parent (node, NULL);
    if (!ebuf_equal_str (parent->element, "tree")) {
	tree = gtk_tree_new ();

	enode_set_kv (node, "top-widget", tree);
	enode_set_kv (node, "tree-widget", tree);
	rendgtk_show_cond (node, tree);

	/* Connect signals */
	gtk_signal_connect (GTK_OBJECT (tree), "select-child",
			    rendgtk_tree_item_onselect_callback, node);
	gtk_signal_connect (GTK_OBJECT (tree), "selection-changed",
			    rendgtk_tree_item_onselectchange_callback, node);
	gtk_signal_connect (GTK_OBJECT (tree), "unselect-child",
			    rendgtk_tree_item_ondeselect_callback, node);

	EDEBUG (("tree-renderer", "Creating top level tree"));
    } else {
	/* we are a child tree item, and need to build the appropriate deal
	 * to connect to our parent appropriately */
	tree_item = gtk_tree_item_new ();

	/* save the XML node for use in the callbacks */
	gtk_object_set_data (GTK_OBJECT (tree_item), "xml-node", node);

	enode_set_kv (node, "top-widget", tree_item);
	enode_set_kv (node, "tree-item-widget", tree_item);

	/* This is where labels and buttons and thigs get added. */
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (tree_item), hbox);
	enode_set_kv (node, "bottom-widget", hbox);

	EDEBUG (("tree-renderer", "Created new tree item and packed in box"));

	/* connect all signals now. */
	gtk_signal_connect (GTK_OBJECT (tree_item), "expand",
			    rendgtk_tree_item_expand_callback, node);
	gtk_signal_connect (GTK_OBJECT (tree_item), "collapse",
			    rendgtk_tree_item_collapse_callback, node);
	gtk_widget_show (hbox);
	gtk_widget_show (tree_item);
    }

    enode_attribs_sync (node);
}


static void
rendgtk_tree_parenter (ENode * parent_node, ENode * child_node)
{
    GtkWidget *parent_tree;
    GtkWidget *child_tree_item;
    GtkWidget *tree;

    /* first thing to do is to see if we're packing a tree or a misc. widget. 
     */
    if (!ebuf_equal_str (child_node->element, "tree")) {
	rendgtk_box_pack (parent_node, child_node);
	return;
    }

    /* This checks for children too, so it should now add a tree item for us
     * * to attach to if necessary */
    rendgtk_tree_expandable_attr_set (parent_node, NULL, NULL);

    /* we must be placing a tree inside a tree then. */
    parent_tree = enode_get_kv (parent_node, "tree-widget");

    child_tree_item = enode_get_kv (child_node, "tree-item-widget");

    if (!parent_tree || !child_tree_item)
	return;

    gtk_tree_append (GTK_TREE (parent_tree), child_tree_item);

    tree = enode_get_kv (child_node, "tree-widget");
    if (tree)
	gtk_tree_item_set_subtree (GTK_TREE_ITEM (child_tree_item), tree);
}

void
tree_renderer_register (void)
{
    Element *element;
    ElementAttr *e_attr;

    element = g_new0 (Element, 1);
    element->render_func = rendgtk_tree_render;
    element->destroy_func = rendgtk_tree_destroy;
    element->parent_func = rendgtk_tree_parenter;
    element->tag = "tree";
    element->description =
	"Creates a tree widget, and is also used for individual nodes in the tree.";
    element_register (element);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "selected";
    e_attr->description = "Item is highlighted or not.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "true,false";
    e_attr->set_attr_func = rendgtk_tree_select_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "expanded";
    e_attr->description = "Tree item is expanded or not.";
    e_attr->value_desc = "boolean";
    e_attr->possible_values = "false,true";
    e_attr->set_attr_func = rendgtk_tree_expanded_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "selection-type";
    e_attr->description = "The mode of the selection.";
    e_attr->value_desc = "choice";
    e_attr->possible_values = "single,multiple,browse,extended";
    e_attr->set_attr_func = rendgtk_tree_selection_type_attr_set;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onselect";
    e_attr->description =
	"Specify function to call when a sub item is 'selected'.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(selected_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "ondeselect";
    e_attr->description =
	"Specify function to call when a sub item is 'deselected'.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(deselected_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "onexpand";
    e_attr->description =
	"Specify function to call when a tree item is expanded.";
    e_attr->value_desc = "function";
    e_attr->possible_values = "(expanding_node)";
    e_attr->set_attr_func = NULL;
    element_register_attrib (element, e_attr);

    e_attr = g_new0 (ElementAttr, 1);
    e_attr->attribute = "expandable";
    e_attr->description =
	"Specify whether this tree item should be allowed to expand or not.";
    e_attr->value_desc = "boolean";
    e_attr->set_attr_func = rendgtk_tree_expandable_attr_set;
    element_register_attrib (element, e_attr);

    rendgtk_widget_attr_register (element, GTK_TYPE_TREE_ITEM);
    rendgtk_containerbox_attr_register (element);
}
