/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-msg-renderer.c,v 1.15 2004/08/22 16:27:37 hoa Exp $
 */

#include "etpan-msg-renderer.h"

#include <libgen.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/mman.h>

#include <libetpan/libetpan.h>

#include "etpan-imf-helper.h"
#include "etpan-tools.h"
#include "etpan-mime-tools.h"
#include "etpan-errors.h"
#include "etpan-cfg-common.h"
#include "etpan-msg-renderer-types.h"

#if 0
#define VERBOSE_MIME_PARTS
#endif


/* XXX - security of temp file need to be checked */

#ifdef VERBOSE_MIME_PARTS
static int print_section(FILE * f, clist * list)
{
  clistiter * cur;
  int r;

  if (clist_begin(list) == NULL) {
    r = fprintf(f, "main");
    if (r < 0)
      goto err;
  }
  else {
    for(cur = clist_begin(list) ; cur != NULL ; cur = cur->next) {
      if (cur->next == NULL) {
	r = fprintf(f, "%i", * (uint32_t *) cur->data);
        if (r < 0)
          goto err;
      }
      else {
	r = fprintf(f, "%i.", * (uint32_t *) cur->data);
        if (r < 0)
          goto err;
      }
    }
  }

  return NO_ERROR;

 err:
  return ERROR_FILE;
}
#endif

static int show_part_info(FILE * f, /* clist * section_id, */
    struct mailmime_single_fields * mime_fields,
    struct mailmime_content * content, int istext)
{
  char * description;
  char * filename;
  int col;
  int r;
  
  description = mime_fields->fld_description;
  filename = mime_fields->fld_disposition_filename;
  if (filename == NULL)
    filename = mime_fields->fld_content_name;

  if (istext && (description == NULL) && (filename == NULL))
    return NO_ERROR;

  col = 0;

#if 0
  r = fprintf(f, " [ Part ");
  if (r < 0)
    goto err;

  r = print_section(f, section_id);
  if (r != NO_ERROR)
    goto err;

  r = fprintf(f, ", ");
  if (r < 0)
    goto err;
#endif
  r = fprintf(f, " [ Part ");
  if (r < 0)
    goto err;

  if (content != NULL) {
    r = mailmime_content_type_write(f, &col, content);
    if (r != MAILIMF_NO_ERROR)
      goto err;
  }

  if (filename == NULL) {
    if (description != NULL) {
      r = fprintf(f, " : %s", description);
      if (r < 0)
        goto err;
    }
  }
  else {
    if (description == NULL) {
      r = fprintf(f, " (%s)", filename);
      if (r < 0)
        goto err;
    }
    else {
      r = fprintf(f, " (%s) : %s", filename, description);
      if (r < 0)
        goto err;
    }
  }

  r = fprintf(f, " ]\n\n");
  if (r < 0)
    goto err;

  return NO_ERROR;
  
 err:
  return ERROR_FILE;
}

static int render_process_pipe(FILE * f,
    char * data, size_t len,
    struct etpan_mime_info * mime_info)
{
  char buf[8192];
  int renderer_input_fds[2];
  int renderer_output_fds[2];
  pid_t pid_in;
  pid_t pid_out;
  size_t remaining;
  char * cur_data;
  size_t read_size;
  int r;
  int status;
  size_t write_len;
  
  r = pipe(renderer_output_fds);
  if (r < 0)
    return ERROR_COMMAND;

  /*
    flush buffer so that it is not flushed more than one time
  */
  fflush(f);

  pid_out = fork();
  switch (pid_out) {

  case -1:
    return ERROR_COMMAND;

  case 0:
    close(renderer_output_fds[0]);

    r = dup2(renderer_output_fds[1], 1);
    if (r < 0)
      exit(1);

    close(renderer_output_fds[1]);

    pipe(renderer_input_fds);

    switch (pid_in = fork()) {
    case -1:
      exit(1);
      
    case 0:
      /* child */
      close(renderer_input_fds[1]);

      r = dup2(renderer_input_fds[0], 0);
      if (r < 0)
        exit(1);

      close(renderer_input_fds[0]);
      execl("/bin/sh", "sh", "-c", mime_info->renderer, NULL);
      exit(1);
      
    default:
      /* parent */
      close(renderer_input_fds[0]);

      remaining = len;
      cur_data = data;
      while (remaining > 0) {
	size_t written;
	
	written = write(renderer_input_fds[1], cur_data, remaining);
	remaining -= written;
	cur_data += written;
      }
      close(renderer_input_fds[1]);

      r = wait4(pid_in, &status, 0, NULL);
      if (r < 0)
        return ERROR_COMMAND;

      exit(WEXITSTATUS(status));
    }
    break;

  default:

    close(renderer_output_fds[1]);
    while ((read_size = read(renderer_output_fds[0], buf, 1024)) > 0) {
      write_len = fwrite(buf, 1, read_size, f);
      if (write_len < read_size) {
        close(renderer_output_fds[0]);
        return ERROR_FILE;
      }
    }
    close(renderer_output_fds[0]);

    r = wait4(pid_out, &status, 0, NULL);
    if (r < 0)
      return ERROR_COMMAND;

    if (WEXITSTATUS(status) != 0)
      return ERROR_COMMAND;

    return NO_ERROR;
  }
}

#if 0
static int etpan_fetch_message(struct etpan_app * app,
    struct mailfolder * folder,
    mailmessage * msg_info,
    struct mailmime * mime_part,
    struct mailmime_single_fields * fields,
    char ** result, size_t * result_len)
{
  char * data;
  size_t len;
  int r;
  int encoding;
  char * decoded;
  size_t decoded_len;
  size_t cur_token;
  int res;

  r = etpan_msg_fetch_section(app,
      folder, msg_info,
      mime_part, &data, &len);
  if (r != MAIL_NO_ERROR) {
    res = ERROR_FETCH;
    goto err;
  }
  
  if (fields->encoding != NULL)
    encoding = fields->encoding->type;
  else 
    encoding = MAILMIME_MECHANISM_8BIT;
  
  cur_token = 0;
  r = mailmime_part_parse(data, len, &cur_token,
			  encoding, &decoded, &decoded_len);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_FETCH;
    goto free; 
  }

  etpan_msg_fetch_result_free(app, folder, msg_info, data);
  
  * result = decoded;
  * result_len = decoded_len;
  
  return NO_ERROR;
  
 free:
  etpan_msg_fetch_result_free(app, folder, msg_info, data);
 err:
  return res;
}
#endif

static int etpan_fetch_message(struct mailprivacy * privacy,
    mailmessage * msg_info,
    struct mailmime * mime_part,
    struct mailmime_single_fields * fields,
    char ** result, size_t * result_len)
{
  char * data;
  size_t len;
  int r;
  int encoding;
  char * decoded;
  size_t decoded_len;
  size_t cur_token;
  int res;

  r = mailprivacy_msg_fetch_section(privacy,
      msg_info, mime_part, &data, &len);
  if (r == MAIL_ERROR_STREAM) {
    res = ERROR_STREAM;
    goto err;
  }
  else if (r != MAIL_NO_ERROR) {
    res = ERROR_FETCH;
    goto err;
  }

  if (fields->fld_encoding != NULL)
    encoding = fields->fld_encoding->enc_type;
  else 
    encoding = MAILMIME_MECHANISM_8BIT;
  
  cur_token = 0;
  r = mailmime_part_parse(data, len, &cur_token,
      encoding, &decoded, &decoded_len);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_FETCH;
    goto free; 
  }

  mailprivacy_msg_fetch_result_free(privacy, msg_info, data);
  
  * result = decoded;
  * result_len = decoded_len;
  
  return NO_ERROR;
  
 free:
  mailprivacy_msg_fetch_result_free(privacy, msg_info, data);
 err:
  return res;
}

static struct mailimf_fields * fetch_fields(struct mailprivacy * privacy,
    mailmessage * msg_info,
    struct mailmime * mime)
{
  char * data;
  size_t len;
  int r;
  size_t cur_token;
  struct mailimf_fields * fields;

  r = mailprivacy_msg_fetch_section_header(privacy,
      msg_info, mime, &data, &len);
  if (r != MAIL_NO_ERROR)
    return NULL;

  cur_token = 0;
  r = mailimf_fields_parse(data, len, &cur_token, &fields);
  if (r != MAILIMF_NO_ERROR) {
    mailprivacy_msg_fetch_result_free(privacy, msg_info, data);
    return NULL;
  }

  mailprivacy_msg_fetch_result_free(privacy, msg_info, data);

  return fields;
}

static int get_command(struct etpan_app * app,
    char * command, size_t size,
    char * cmd, struct mailmime_single_fields * fields,
    char * filename,
    mailmessage * msg_info,
    struct mailmime * mime_part)
{
  MMAPString * mmapstr;
  char * s;
  int extracted;
  int res;
  char quoted_filename[PATH_MAX];

  extracted = 0;

  mmapstr = mmap_string_new("");
  if (mmapstr == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  s = cmd;
  while (* s != '\0') {
    switch (*s) {
    case '%':
      s ++;
      switch (*s) {
      case '\0':
	break;
      case '%':
	if (mmap_string_append_c(mmapstr, '%') == NULL) {
          res = ERROR_MEMORY;
          goto free;
        }
	s++;
	break;
      case 'f':
	if (!extracted) {
	  char * data;
	  size_t len;
	  int r;
	  
	  r = etpan_fetch_message(app->privacy,
              msg_info, mime_part, fields, &data, &len);
	  if (r != NO_ERROR) {
            res = r;
            goto free;
	  }

	  r = etpan_write_data(filename, data, len);
          if (r != NO_ERROR) {
            mailmime_decoded_part_free(data);
            res = r;
            goto free;
          }
	  
	  mailmime_decoded_part_free(data);
	  extracted = 1;
          
          etpan_quote_filename(quoted_filename,
              sizeof(quoted_filename), filename);
	}
        
	if (mmap_string_append(mmapstr, quoted_filename) == NULL) {
          res = ERROR_MEMORY;
          goto free;
        }
	s++;
	break;
      }
      break;

    default:
      if (mmap_string_append_c(mmapstr, * s) == NULL) {
        res = ERROR_MEMORY;
        goto free;
      }
      s++;
      break;
    }
  }
  
  fflush(stdout);
  
  strncpy(command, mmapstr->str, size);
  command[size - 1] = '\0';

  mmap_string_free(mmapstr);

  return NO_ERROR;

 free:
  mmap_string_free(mmapstr);
 err:
  return res;
}


static int render_process_command(struct etpan_app * app,
    FILE * f,
    struct mailmime_single_fields * fields,
    mailmessage * msg_info,
    struct mailmime * mime_part,
    struct etpan_mime_info * mime_info)
{
  char buf[8192];
  int renderer_output_fds[2];
  pid_t pid_out;
  size_t read_size;
  char command[PATH_MAX];
  char * filename1;
  char filename[PATH_MAX];
  char dspname[PATH_MAX];
  int res;
  int status;
  size_t write_len;
  int r;

  filename1 = fields->fld_disposition_filename;
  if (filename1 == NULL)
    filename1 = fields->fld_content_name;

  if (filename1 == NULL) {
    r = etpan_get_tmp_name(dspname, PATH_MAX);
    if (r < 0) {
      res = ERROR_FILE;
      goto err;
    }
    unlink(dspname);
  }
  else {
    strncpy(dspname, filename1, PATH_MAX);
    dspname[PATH_MAX - 1] = 0;
  }

  snprintf(filename, PATH_MAX, "%s/%s/%s",
	   getenv("HOME"), ETPAN_TMP_MIME_PATH, basename(dspname));
  
  r = get_command(app, command, PATH_MAX,
      mime_info->renderer, fields, filename,
      msg_info, mime_part);
  if (r != NO_ERROR) {
    res = r;
    goto err;
  }
  
  r = pipe(renderer_output_fds);
  if (r < 0) {
    res = ERROR_COMMAND;
    goto unlink;
  }

  /*
    flush buffer so that it is not flushed more than one time
  */
  fflush(f);

  pid_out = fork();
  switch (pid_out) {

  case -1:
    res = ERROR_COMMAND;
    goto err;

  case 0:
    close(renderer_output_fds[0]);

    r = dup2(renderer_output_fds[1], 1);
    if (r < 0)
      exit(1);

    close(renderer_output_fds[1]);

    execl("/bin/sh", "sh", "-c", command, NULL);
    exit(1);

  default:

    close(renderer_output_fds[1]);
    while ((read_size = read(renderer_output_fds[0], buf, 1024)) > 0) {
      write_len = fwrite(buf, 1, read_size, f);
      if (write_len < read_size) {
        close(renderer_output_fds[0]);
        res = ERROR_FILE;
        goto unlink;
      }
    }
    close(renderer_output_fds[0]);

    r = wait4(pid_out, &status, 0, NULL);
    if (r < 0) {
      res = ERROR_COMMAND;
      goto unlink;
    }

    if (WEXITSTATUS(status) != 0) {
      res = ERROR_COMMAND;
      goto unlink;
    }

    unlink(filename);

    break;
  }

  return NO_ERROR;

 unlink:
  unlink(filename);
 err:
  return res;
}

static int render_process(struct etpan_app * app, FILE * f,
    struct mailmime_single_fields * fields,
    mailmessage * msg_info,
    struct mailmime * mime_part,
    struct etpan_mime_info * mime_info)
{
  int r;
  char * data;
  size_t len;
  int res;

  switch (mime_info->renderer_type) {
  case COMMAND_TYPE_PIPE:
    
    r = etpan_fetch_message(app->privacy,
        msg_info, mime_part, fields, &data, &len);
    if (r != NO_ERROR) {
      res = r;
      goto err;
    }

    r = render_process_pipe(f, data, len, mime_info);
    if (r != NO_ERROR) {
      res = r;
      goto free;
    }

    mailmime_decoded_part_free(data);

    return NO_ERROR;

  case COMMAND_TYPE_PARAM:

    r = render_process_command(app, f, fields,
        msg_info, mime_part, mime_info);
    if (r != NO_ERROR) {
      res = r;
      goto err;
    }

    return NO_ERROR;
  }

  return ERROR_INVAL;
  
 free:
  mailmime_decoded_part_free(data);
 err:
  return res;
}


static int etpan_subject_write(FILE * f, int * col,
    struct etpan_app * app, char * subject)
{
  int r;
  char * decoded_subject;
  
  r = mailimf_string_write(f, col, "Subject: ", 9);
  if (r != MAILIMF_NO_ERROR) {
    return ERROR_FILE;
  }
  
  decoded_subject = etpan_decode_mime_header(app, subject);
  if (decoded_subject == NULL) {
    decoded_subject = strdup(subject);
    if (decoded_subject == NULL)
      return ERROR_MEMORY;
  }
  
  r = mailimf_header_string_write(f, col,
      decoded_subject, strlen(decoded_subject));
  if (r != MAILIMF_NO_ERROR) {
    free(decoded_subject);
    return ERROR_FILE;
  }

  free(decoded_subject);

  r = mailimf_string_write(f, col, "\r\n", 2);
  if (r != MAILIMF_NO_ERROR) {
    return ERROR_FILE;
  }
  * col = 0;

  return NO_ERROR;
}

static int fields_write(FILE * f, int * col,
    struct etpan_app * app,
    struct mailimf_fields * fields)
{
  clistiter * cur;
  int r;
  
  for(cur = clist_begin(fields->fld_list) ; cur != NULL ;
      cur = clist_next(cur)) {
    struct mailimf_field * field;

    field = clist_content(cur);

    switch (field->fld_type) {
    case MAILIMF_FIELD_FROM:
      r = mailimf_string_write(f, col, "From: ", 6);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      
      r = etpan_mailbox_list_write(f, col, app,
          field->fld_data.fld_from->frm_mb_list);
      if (r != NO_ERROR)
        goto err;

      r = mailimf_string_write(f, col, "\r\n", 2);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      * col = 0;

      break;

    case MAILIMF_FIELD_REPLY_TO:
      r = mailimf_string_write(f, col, "Reply-To: ", 10);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      
      r = etpan_address_list_write(f, col, app,
          field->fld_data.fld_reply_to->rt_addr_list);
      if (r != NO_ERROR)
        goto err;

      r = mailimf_string_write(f, col, "\r\n", 2);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      * col = 0;

      break;

    case MAILIMF_FIELD_TO:
      r = mailimf_string_write(f, col, "To: ", 4);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      
      r = etpan_address_list_write(f, col, app,
          field->fld_data.fld_to->to_addr_list);
      if (r != NO_ERROR)
        goto err;

      r = mailimf_string_write(f, col, "\r\n", 2);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      * col = 0;

      break;

    case MAILIMF_FIELD_CC:
      r = mailimf_string_write(f, col, "Cc: ", 4);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      
      r = etpan_address_list_write(f, col,
          app, field->fld_data.fld_cc->cc_addr_list);
      if (r != NO_ERROR)
        goto err;

      r = mailimf_string_write(f, col, "\r\n", 2);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      * col = 0;

      break;

    case MAILIMF_FIELD_BCC:
      r = mailimf_string_write(f, col, "Bcc: ", 5);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      
      if (field->fld_data.fld_bcc->bcc_addr_list != NULL) {
        r = etpan_address_list_write(f, col,
            app, field->fld_data.fld_bcc->bcc_addr_list);
        if (r != NO_ERROR)
          goto err;
      }

      r = mailimf_string_write(f, col, "\r\n", 2);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      * col = 0;

      break;

    case MAILIMF_FIELD_SUBJECT:
      r = etpan_subject_write(f, col, app,
          field->fld_data.fld_subject->sbj_value);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      break;

    case MAILIMF_FIELD_RESENT_FROM:
      r = mailimf_string_write(f, col, "Resent-From: ", 13);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      
      r = etpan_mailbox_list_write(f, col, app,
          field->fld_data.fld_resent_from->frm_mb_list);
      if (r != NO_ERROR)
        goto err;

      r = mailimf_string_write(f, col, "\r\n", 2);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      * col = 0;
      break;

    case MAILIMF_FIELD_RESENT_TO:
      r = mailimf_string_write(f, col, "Resent-To: ", 11);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      
      r = etpan_address_list_write(f, col, app,
          field->fld_data.fld_resent_to->to_addr_list);
      if (r != NO_ERROR)
        goto err;

      r = mailimf_string_write(f, col, "\r\n", 2);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      * col = 0;

      break;
    case MAILIMF_FIELD_RESENT_CC:
      r = mailimf_string_write(f, col, "Resent-Cc: ", 11);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      
      r = etpan_address_list_write(f, col,
          app, field->fld_data.fld_resent_cc->cc_addr_list);
      if (r != NO_ERROR)
        goto err;

      r = mailimf_string_write(f, col, "\r\n", 2);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      * col = 0;

      break;
    case MAILIMF_FIELD_RESENT_BCC:
      r = mailimf_string_write(f, col, "Resent-Bcc: ", 12);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      
      if (field->fld_data.fld_bcc->bcc_addr_list != NULL) {
        r = etpan_address_list_write(f, col,
            app, field->fld_data.fld_resent_bcc->bcc_addr_list);
        if (r != NO_ERROR)
          goto err;
      }

      r = mailimf_string_write(f, col, "\r\n", 2);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      * col = 0;

      break;

    case MAILIMF_FIELD_ORIG_DATE:
    case MAILIMF_FIELD_RESENT_DATE:
      r = mailimf_field_write(f, col, field);
      if (r != MAILIMF_NO_ERROR)
        goto err;
      break;
      
    case MAILIMF_FIELD_COMMENTS:
    case MAILIMF_FIELD_KEYWORDS:
      break;

    case MAILIMF_FIELD_OPTIONAL_FIELD:
      if ((strcasecmp(field->fld_data.fld_optional_field->fld_name,
               "X-Mailer") == 0)
          || (strcasecmp(field->fld_data.fld_optional_field->fld_name,
                  "X-Newsreader") == 0)
          || (strncasecmp(field->fld_data.fld_optional_field->fld_name,
                  "Resent-", 7) == 0)
          || (strcasecmp(field->fld_data.fld_optional_field->fld_name,
                  "Newsgroups") == 0)
          || (strcasecmp(field->fld_data.fld_optional_field->fld_name,
                  "Followup-To") == 0)
          || (strcasecmp(field->fld_data.fld_optional_field->fld_name,
                  "User-Agent") == 0)) {
        char * p;
        
        r = mailimf_string_write(f, col,
            field->fld_data.fld_optional_field->fld_name,
            strlen(field->fld_data.fld_optional_field->fld_name));
        if (r != MAILIMF_NO_ERROR)
          goto err;
        
        r = mailimf_string_write(f, col, ": ", 2);
        if (r != MAILIMF_NO_ERROR)
          goto err;
        
        /* remove all line breaks */
        for(p = field->fld_data.fld_optional_field->fld_value ;
            * p != 0 ; p ++)
          if ((* p == '\r') || (* p == '\n'))
            * p = ' ';
        
        r = mailimf_header_string_write(f, col,
            field->fld_data.fld_optional_field->fld_value,
            strlen(field->fld_data.fld_optional_field->fld_value));
        if (r != MAILIMF_NO_ERROR)
          goto err;

        r = mailimf_string_write(f, col, "\r\n", 2);
        if (r != MAILIMF_NO_ERROR)
          goto err;

#if 0
        r = mailimf_field_write(f, col, field);
        if (r != MAILIMF_NO_ERROR)
          goto err;
#endif
      }
      break;

    case MAILIMF_FIELD_MESSAGE_ID:
    case MAILIMF_FIELD_SENDER:
    case MAILIMF_FIELD_IN_REPLY_TO:
    case MAILIMF_FIELD_REFERENCES:
    default:
      break;
    }
  }

  return NO_ERROR;

 err:
  return ERROR_FILE;
}

int etpan_render_text(FILE * f, carray * attr, char * text, size_t size)
{
  size_t start_zone;
  size_t end_zone;
  int res;
  size_t write_len;
  size_t offset;
  size_t begin;
  int r;
  size_t end;
  int zone_type;

  zone_type = ETPAN_VIEWER_ZONE_NORMAL;

  start_zone = ftell(f);

  offset = 0;
  begin = offset;
  end = offset;

  while (1) {
    int do_flush;

    do_flush = 0;
    if (text[offset] == '\n') {
      end = offset + 1;
      do_flush = 1;
    }
    if (offset >= size) {
      end = size;
      do_flush = 1;
    }

    if (do_flush) {
      start_zone = ftell(f);

      write_len = fwrite(text + begin, 1, end - begin, f);
      if (write_len < end - begin) {
        res = ERROR_FILE;
        goto err;
      }

      end_zone = ftell(f);

      if (end - begin == 4) {
        if (strncmp(text + begin, "-- \n", 4) == 0)
          zone_type = ETPAN_VIEWER_ZONE_SIGNATURE;
      }
      else if (end - begin == 5) {
        if (strncmp(text + begin, "-- \r\n", 5) == 0)
          zone_type = ETPAN_VIEWER_ZONE_SIGNATURE;
      }

      r = etpan_text_prop_array_add(attr, start_zone, end_zone,
          zone_type, NULL);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
      
      begin = end;
    }

    if (offset >= size)
      break;

    offset ++;
  }

  return NO_ERROR;

 err:
  return res;
}



int etpan_render_mime(FILE * f, carray * attr,
    struct etpan_app * app,
    mailmessage * msg_info,
    struct mailmime * mime,
    int flags)
{
  int r;
  clistiter * cur;
  int col;
  int text;
  int show;
  struct etpan_mime_info * mime_info;
#if 0
  struct mailmime_section * section_id;
#endif
  struct mailmime_single_fields fields;
  size_t write_len;
  int res;
  size_t start_zone;
  size_t end_zone;

  mailmime_single_fields_init(&fields, mime->mm_mime_fields,
      mime->mm_content_type);
  col = 0;
  
  text = etpan_mime_is_text(mime);

#if 0
  r = mailmime_get_section_id(mime, &section_id);
  if (r != MAILIMF_NO_ERROR) {
    res = ERROR_MEMORY;
    goto err;
  }
#endif

  if (((flags & ETPAN_RENDER_FLAGS_HIDE_PART_INFO) == 0) &&
      (mime->mm_type == MAILMIME_SINGLE)) {
    start_zone = ftell(f);
    r = show_part_info(f, /* section_id->list, */
        &fields, mime->mm_content_type, text);
    if (r != NO_ERROR) {
#if 0
      mailmime_section_free(section_id);
#endif
      res = r;
      goto err;
    }
    end_zone = ftell(f);
  
    if (start_zone != end_zone) {
      r = etpan_text_prop_array_add(attr, start_zone, end_zone,
          ETPAN_VIEWER_ZONE_DESCRIPTION, NULL);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }
  }

#if 0
  mailmime_section_free(section_id);
#endif

  if (mime->mm_type == MAILMIME_MESSAGE) {
    if (((flags & ETPAN_RENDER_FLAGS_HIDE_HEADER) == 0) &&
        ((flags & ETPAN_RENDER_FLAGS_MAIN_HEADER) != 0) &&
        (mime->mm_data.mm_message.mm_fields != NULL)) {
      struct mailimf_fields * fields;

      start_zone = ftell(f);
      if (msg_info != NULL) {
        fields = fetch_fields(app->privacy, msg_info, mime);
        if (fields == NULL) {
          res = ERROR_FETCH;
          goto err;
        }

        col = 0;
        r = fields_write(f, &col, app, fields);
        if (r != NO_ERROR) {
          mailimf_fields_free(fields);
          res = r;
          goto err;
        }

        mailimf_fields_free(fields);
      }
      else {
        col = 0;
        r = fields_write(f, &col, app, mime->mm_data.mm_message.mm_fields);
        if (r != NO_ERROR) {
          res = r;
          goto err;
        }
      }
      end_zone = ftell(f);

      r = etpan_text_prop_array_add(attr, start_zone, end_zone,
          ETPAN_VIEWER_ZONE_HEADER, NULL);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    
      r = fprintf(f, "\r\n");
      if (r < 0) {
        res = ERROR_FILE;
        goto err;
      }
    }
  }

  mime_info = etpan_get_mime_info(app->config.mime_config, mime);

  switch (mime->mm_type) {
  case MAILMIME_SINGLE:

    show = 0;
    if (text)
      show = 1;
    
    if (mime_info) {
      if (mime_info->renderer != NULL)
        if (* mime_info->renderer != '\0')
          show = 1;
    }

#if 0
    if (fields->disposition != NULL) {
      if (fields->disposition->type == MAILMIME_DISPOSITION_TYPE_ATTACHMENT)
        show = 0;
    }
#endif

    if (show) {
      int do_render;

      do_render = 0;
      if (mime_info != NULL)
        if (mime_info->renderer != NULL)
          do_render = 1;

      if (do_render) {
        char * source_charset;
        
        source_charset = fields.fld_content_charset;
        if (source_charset == NULL) {
          source_charset = app->config.global_config->message_charset;
        }
        start_zone = ftell(f);
          
        r = render_process(app, f, &fields, msg_info, mime, mime_info);
        if (r != NO_ERROR) {
          res = r;
          goto err;
        }
          
        end_zone = ftell(f);

        r = etpan_text_prop_array_add(attr, start_zone, end_zone,
            ETPAN_VIEWER_ZONE_NORMAL, NULL);
        if (r != NO_ERROR) {
          res = r;
          goto err;
        }
          
        write_len = fwrite("\r\n\r\n", 1, 4, f);
        if (write_len < 4) {
          res = ERROR_FILE;
          goto err;
        }
      }
      else {
        char * data;
        size_t len;
        char * converted;
        size_t converted_len;
        char * source_charset;
        size_t write_len;
        int done;
          
        r = etpan_fetch_message(app->privacy,
            msg_info, mime, &fields, &data, &len);
        if (r != NO_ERROR) {
          res = r;
          goto err;
        }
          
        done = 0;
          
        /* XXX encrypted messages */
        r = ERROR_UNKNOWN_PROTOCOL;
#if 0
        r = etpan_msg_secure_clear_decode(f, attr, data, len,
            app, account_info);
#endif
        if (r == NO_ERROR) {
          done = 1;
        }
        else if (r == ERROR_UNKNOWN_PROTOCOL) {
          done = 0;
        }
        else {
          res = r;
          goto err;
        }
          
        if (!done) {
          source_charset = fields.fld_content_charset;
          if (source_charset == NULL)
            source_charset = app->config.global_config->message_charset;
          
          r = charconv_buffer(app->config.global_config->display_charset,
              source_charset,
              data, len, &converted, &converted_len);
          if (r != MAIL_CHARCONV_NO_ERROR) {
              
            start_zone = ftell(f);
              
            r = fprintf(f, "[ error converting charset from %s to %s ]\n",
                source_charset, app->config.global_config->display_charset);
            if (r < 0) {
              res = ERROR_FILE;
              goto err;
            }
              
            end_zone = ftell(f);
              
            r = etpan_text_prop_array_add(attr, start_zone, end_zone,
                ETPAN_VIEWER_ZONE_ERROR, NULL);
            if (r != NO_ERROR) {
              res = r;
              goto err;
            }

            start_zone = ftell(f);
              
            r = etpan_render_text(f, attr, data, len);
            if (r != NO_ERROR) {
              mailmime_decoded_part_free(data);
              res = r;
              goto err;
            }
              
            end_zone = ftell(f);

            r = etpan_text_prop_array_add(attr, start_zone, end_zone,
                ETPAN_VIEWER_ZONE_NORMAL, NULL);
            if (r != NO_ERROR) {
              res = r;
              goto err;
            }
          }
          else {
            start_zone = ftell(f);
              
            r = etpan_render_text(f, attr, converted, converted_len);
            if (r != NO_ERROR) {
              charconv_buffer_free(converted);
              mailmime_decoded_part_free(data);
              res = r;
              goto err;
            }
              
            charconv_buffer_free(converted);
              
            end_zone = ftell(f);
              
            r = etpan_text_prop_array_add(attr, start_zone, end_zone,
                ETPAN_VIEWER_ZONE_NORMAL, NULL);
            if (r != NO_ERROR) {
              res = r;
              goto err;
            }
          }
            
          write_len = fwrite("\r\n\r\n", 1, 4, f);
          if (write_len < 4) {
            mailmime_decoded_part_free(data);
            res = ERROR_FILE;
            goto err;
          }
        }
          
        mailmime_decoded_part_free(data);
      }
    }
    else {
      int done;

      done = 0;
      /* XXX encrypted messages */
      r = ERROR_UNKNOWN_PROTOCOL;
#if 0
      r = etpan_msg_secure_decode(f, attr, msg_info, mime,
          app, account_info);
#endif
      if (r == NO_ERROR) {
        done = 1;
      }
      else if (r == ERROR_UNKNOWN_PROTOCOL) {
        done = 0;
      }
      else {
        res = r;
        goto err;
      }

      if (!done) {
        start_zone = ftell(f);
        
        r = fprintf(f, "   (not shown)\n\n");
        if (r < 0) {
          res = ERROR_FILE;
          goto err;
        }
        
        end_zone = ftell(f);
        
        r = etpan_text_prop_array_add(attr, start_zone, end_zone,
            ETPAN_VIEWER_ZONE_DESCRIPTION, NULL);
        if (r != NO_ERROR) {
          res = r;
          goto err;
        }
      }
    }

    break;
    
  case MAILMIME_MULTIPLE:

    if (strcasecmp(mime->mm_content_type->ct_subtype, "alternative") == 0) {
      struct mailmime * prefered_body;
      int prefered_score;

      /*
	text/plain
	text/
	other
      */

      prefered_body = NULL;
      prefered_score = 0;

      for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
          cur != NULL ; cur = clist_next(cur)) {
	struct mailmime * submime;
	int score;

	score = 1;
	submime = cur->data;
        if (etpan_mime_is_text(submime))
          score = 2;

	if (submime->mm_content_type != NULL) {
          if (strcasecmp(submime->mm_content_type->ct_subtype, "plain") == 0)
            score = 3;
	}
        
        /* support of secure MIME */
        if (submime->mm_type == MAILMIME_MULTIPLE) {
          if (submime->mm_content_type != NULL) {
            if (strcasecmp(submime->mm_content_type->ct_subtype,
                    "x-verified") == 0)
              score = 4;
            if (strcasecmp(submime->mm_content_type->ct_subtype,
                    "x-decrypted") == 0)
              score = 4;
          }
	}

	if (score > prefered_score) {
	  prefered_score = score;
	  prefered_body = submime;
	}
      }

      if (prefered_body != NULL) {
	r = etpan_render_mime(f, attr, app, msg_info,
            prefered_body, flags | ETPAN_RENDER_FLAGS_MAIN_HEADER);
	if (r != NO_ERROR) {
	  res = r;
          goto err;
        }
      }
    }
    else {
      int done;

      done = 0;

      /* XXX encrypted messages */
      r = ERROR_UNKNOWN_PROTOCOL;
#if 0
      r = etpan_msg_secure_decode(f, attr, msg_info, mime,
          app, account_info);
#endif
      if (r == NO_ERROR) {
        done = 1;
      }
      else if (r == ERROR_UNKNOWN_PROTOCOL) {
        done = 0;
      }
      else {
        res = r;
        goto err;
      }

      if (!done) {
        for(cur = clist_begin(mime->mm_data.mm_multipart.mm_mp_list) ;
            cur != NULL ; cur = clist_next(cur)) {
          
          r = etpan_render_mime(f, attr, app, msg_info,
              cur->data, flags | ETPAN_RENDER_FLAGS_MAIN_HEADER);
          if (r != NO_ERROR) {
            res = r;
            goto err;
          }
        }
      }
    }

    break;
      
  case MAILMIME_MESSAGE:

    if (mime->mm_data.mm_message.mm_msg_mime != NULL) {
      r = etpan_render_mime(f, attr, app, msg_info,
          mime->mm_data.mm_message.mm_msg_mime,
          flags | ETPAN_RENDER_FLAGS_MAIN_HEADER);
      if (r != NO_ERROR) {
        res = r;
        goto err;
      }
    }

    break;
  }

  return NO_ERROR;

 err:
  return res;
}


void etpan_text_prop_array_free(carray * attr)
{
  unsigned int i;

  for(i = 0 ; i < carray_count(attr) ; i ++) {
    struct etpan_text_prop * prop;

    prop = carray_get(attr, i);
    free(prop);
  }

  carray_free(attr);
}

int etpan_text_prop_array_add(carray * attr, size_t start, size_t end,
    int zone_type, void * zone_data)
{
  struct etpan_text_prop * prop;
  int r;
  int res;

  if (attr == NULL)
    return NO_ERROR;

  prop = malloc(sizeof(struct etpan_text_prop));
  if (prop == NULL) {
    res = ERROR_MEMORY;
    goto err;
  }

  prop->start = start;
  prop->end = end;
  prop->zone_type = zone_type;
  prop->zone_data = zone_data;

  r = carray_add(attr, prop, NULL);
  if (r < 0) {
    res = ERROR_MEMORY;
    goto free;
  }

  return NO_ERROR;
  
 free:
  free(prop);
 err:
  return res;
}
