/*
 * etPan! -- a mail user agent
 *
 * Copyright (C) 2001, 2002 - DINH Viet Hoa
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the libEtPan! project nor the names of its
 *    contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/*
 * $Id: etpan-tools.c,v 1.18 2004/12/27 15:34:07 hoa Exp $
 */

#include "etpan-tools.h"

#include <stdlib.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <ncurses.h>
#include <pthread.h>
#include <regex.h>
#include <sys/types.h>
#include <ctype.h>
#include <stdio.h>
#include <fcntl.h>
#include <sys/mman.h>
#include <libgen.h>

#include <libetpan/libetpan.h>

#include "etpan-errors.h"
#include "etpan-mime-params.h"
#include "etpan-cfg-vfolder.h"

int etpan_get_tmp_name(char * filename, size_t size)
{
  int fd;
  mode_t old_mask;

  snprintf(filename, size, "%s/%s/etpan-XXXXXX",
      etpan_get_home_dir(), ETPAN_TMP_PATH);
  /* 
     we can close the file, we are in the user directory,
     in a safe environnement
  */
  old_mask = umask(0077);
  fd = mkstemp(filename);
  umask(old_mask);
  if (fd == -1)
    return -1;

  close(fd);

  return 0;
}

static FILE * get_tmp_file(char * filename)
{
  int fd;
  mode_t old_mask;
  FILE * f;
  
  old_mask = umask(0077);
  fd = mkstemp(filename);
  umask(old_mask);
  if (fd == -1)
    return NULL;
  
  f = fdopen(fd, "r+");
  if (f == NULL) {
    close(fd);
    unlink(filename);
  }
  
  return f;
}

FILE * etpan_get_tmp_file(char * filename, size_t size)
{
  snprintf(filename, size,
      "%s/%s/etpan-XXXXXX", etpan_get_home_dir(), ETPAN_TMP_PATH);
  return get_tmp_file(filename);
}

FILE * etpan_get_mime_tmp_file(char * filename, size_t size)
{
  snprintf(filename, size,
      "%s/%s/etpan-XXXXXX", etpan_get_home_dir(), ETPAN_TMP_MIME_PATH);
  return get_tmp_file(filename);
}

int etpan_write_data(char * filename, char * data, size_t len)
{
  size_t write_len;
  FILE * f;
  int res;
  mode_t old_umask;
  
  old_umask = umask(0077);
  f = fopen(filename, "w");
  umask(old_umask);
  if (f == NULL) {
    res = ERROR_FILE;
    goto err;
  }

  write_len = fwrite(data, 1, len, f);
  if (write_len < len) {
    res = ERROR_FILE;
    goto close;
  }

  fclose(f);

  return NO_ERROR;

 close:
  fclose(f);
 err:
  return res;
}

int etpan_quote_filename(char * result, size_t size, char * path)
{
  char * p;
  char * result_p;
  size_t remaining;

  result_p = result;
  remaining = size;

  for(p = path ; * p != '\0' ; p ++) {

    if (isalpha(* p) || isdigit(* p) || (* p == '/')) {
      if (remaining > 0) {
        * result_p = * p;
        result_p ++; 
        remaining --;
      }
      else {
        result[size - 1] = '\0';
        return -1;
      }
    }
    else { 
      if (remaining >= 2) {
        * result_p = '\\';
        result_p ++; 
        * result_p = * p;
        result_p ++; 
        remaining -= 2;
      }
      else {
        result[size - 1] = '\0';
        return -1;
      }
    }
  }
  if (remaining > 0) {
    * result_p = '\0';
  }
  else {
    result[size - 1] = '\0';
    return -1;
  }
  
  return 0;
}

int etpan_regexp_match(char * regexp, const char * str)
{
  regex_t regex;
  int r;
  
  r = regcomp(&regex, regexp, REG_ICASE | REG_EXTENDED);
  if (r != 0)
    return 0;
  
  r = regexec(&regex, str, 0, NULL, 0);
  if (r != 0) {
    regfree(&regex);
    return 0;
  }

  regfree(&regex);
  
  return 1;
}

struct mailfolder *
etpan_get_folder(struct etpan_app * app, char * folder_name)
{
  return etpan_vfolder_get_folder(app->config.vfolder_config, folder_name);
}

char * etpan_get_folder_name(struct etpan_app * app,
    struct mailfolder * folder)
{
  return etpan_folder_get_virtual_path(folder);
}

void etpan_strip_string(char * str)
{
  char * p;
  size_t len;

  p = strchr(str, '#');
  if (p != NULL)
    * p = 0;
  
  p = strchr(str, '\r');
  if (p != NULL)
    * p = 0;

  p = strchr(str, '\n');
  if (p != NULL)
    * p = 0;

  p = str;
  while ((* p == ' ') || (* p == '\t')) {
    p ++;
  }
  
  len = strlen(p);
  memmove(str, p, len);
  str[len] = 0;
  
  if (len == 0)
    return;
  
  p = str;
  len = len - 1;
  while ((p[len] == ' ') || (p[len] == '\t')) {
    p[len] = '\0';
    
    if (len == 0)
      break;
    
    len --;
  }
}

char * etpan_dup_file(char * source_filename)
{
  char filename[PATH_MAX];
  FILE * dest_f;
  int r;
  struct stat stat_info;
  char * dest_filename;
  char * mapping;
  size_t written;
  int fd;
  
  dest_f = etpan_get_tmp_file(filename, sizeof(filename));
  if (dest_f == NULL)
    goto err;
  
  dest_filename = strdup(filename);
  if (dest_filename == NULL)
    goto close_dest;
  
  fd = open(source_filename, O_RDONLY);
  if (fd < 0)
    goto free_dest;
  
  r = fstat(fd, &stat_info);
  if (r < 0)
    goto close_src;
  
  mapping = mmap(NULL, stat_info.st_size, PROT_READ, MAP_PRIVATE, fd, 0);
  if (mapping == MAP_FAILED)
    goto close_src;
  
  written = fwrite(mapping, 1, stat_info.st_size, dest_f);
  if ((off_t) written != stat_info.st_size)
    goto unmap;
  
  munmap(mapping, stat_info.st_size);
  close(fd);
  fclose(dest_f);
  
  return dest_filename;
  
 unmap:
  munmap(mapping, stat_info.st_size);
 close_src:
  close(fd);
 free_dest:
  free(dest_filename);
 close_dest:
  fclose(dest_f);
 err:
  return NULL;
}

#define BUF_SIZE 4096

int etpan_prepare_file_for_edition(char * source_filename)
{
  char filename[PATH_MAX];
  FILE * dest_f;
  int r;
  struct stat stat_info;
  char * dest_filename;
  char * mapping;
  size_t written;
  FILE * f;
  char buf[BUF_SIZE];
  int res;
  
  dest_f = etpan_get_tmp_file(filename, sizeof(filename));
  if (dest_f == NULL) {
    res = ERROR_FILE;
    goto err;
  }
  
  dest_filename = strdup(filename);
  if (dest_filename == NULL) {
    res = ERROR_MEMORY;
    goto close_dest;
  }
  
  f = fopen(source_filename, "r");
  if (f == NULL) {
    res = ERROR_FILE;
    goto free_dest;
  }
  
  while (fgets(buf, sizeof(buf), f)) {
    size_t len;
    
    len = strlen(buf);
    if (buf[len - 1] == '\n') {
      buf[len - 1] = '\0';
      len --;
      if (len > 0) {
        if (buf[len - 1] == '\r') {
          buf[len - 1] = '\0';
          len --;
        }
      }
    }
    
    r = fwrite(buf, 1, len, dest_f);
    if (r < 0) {
      res = ERROR_FILE;
      goto close_src;
    }
    
    r = fputs("\n", dest_f);
    if (r < 0) {
      res = ERROR_FILE;
      goto close_src;
    }
  }
  
  fclose(f);
  fclose(dest_f);
  
  rename(dest_filename, source_filename);
  
  return NO_ERROR;
  
 close_src:
  fclose(f);
 free_dest:
  free(dest_filename);
 close_dest:
  fclose(dest_f);
 err:
  return res;
}


/* extracted from libEtPan! - begin */

#define OUTPUT_SIZE 513
#define CHAR64(c)  (((c) < 0 || (c) > 127) ? -1 : index_64[(c)])

static char index_64[128] = {
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,-1,
    -1,-1,-1,-1, -1,-1,-1,-1, -1,-1,-1,62, -1,-1,-1,63,
    52,53,54,55, 56,57,58,59, 60,61,-1,-1, -1,-1,-1,-1,
    -1, 0, 1, 2,  3, 4, 5, 6,  7, 8, 9,10, 11,12,13,14,
    15,16,17,18, 19,20,21,22, 23,24,25,-1, -1,-1,-1,-1,
    -1,26,27,28, 29,30,31,32, 33,34,35,36, 37,38,39,40,
    41,42,43,44, 45,46,47,48, 49,50,51,-1, -1,-1,-1,-1
};

static char basis_64[] =
   "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";

char * etpan_encode_base64(const char * in, int len)
{
  char * output, * tmp;
  unsigned char oval;
  int out_len;

  out_len = ((len + 2) / 3 * 4) + 1;

  if ((len > 0) && (in == NULL))
    return NULL;

  output = malloc(out_len);
  if (!output)
    return NULL;
    
  tmp = output;
  while (len >= 3) {
    *tmp++ = basis_64[in[0] >> 2];
    *tmp++ = basis_64[((in[0] << 4) & 0x30) | (in[1] >> 4)];
    *tmp++ = basis_64[((in[1] << 2) & 0x3c) | (in[2] >> 6)];
    *tmp++ = basis_64[in[2] & 0x3f];
    in += 3;
    len -= 3;
  }
  if (len > 0) {
    *tmp++ = basis_64[in[0] >> 2];
    oval = (in[0] << 4) & 0x30;
    if (len > 1) oval |= in[1] >> 4;
    *tmp++ = basis_64[oval];
    *tmp++ = (len < 2) ? '=' : basis_64[(in[1] << 2) & 0x3c];
    *tmp++ = '=';
  }

  *tmp = '\0';
    
  return output;
}

char * etpan_decode_base64(const char * in, int len)
{
  char * output, * out;
  int i, c1, c2, c3, c4, out_len;

  out_len = 0;
  
  output = malloc(OUTPUT_SIZE);
  if (output == NULL)
    return NULL;
  out = output;

  if (in[0] == '+' && in[1] == ' ')
    in += 2;
  
  for (i = 0; i < (len / 4); i++) {
    c1 = in[0];
    c2 = in[1];
    c3 = in[2];
    c4 = in[3];
    if (CHAR64(c1) == -1 || CHAR64(c2) == -1 || 
        (c3 != '=' && CHAR64(c3) == -1) || 
        (c4 != '=' && CHAR64(c4) == -1))
      return NULL;

    in += 4;
    *output++ = (CHAR64(c1) << 2) | (CHAR64(c2) >> 4);
    if (++out_len >= OUTPUT_SIZE)
      return NULL;

    if (c3 != '=') {
      *output++ = ((CHAR64(c2) << 4) & 0xf0) | (CHAR64(c3) >> 2);
      if (++out_len >= OUTPUT_SIZE)
        return NULL;
      
      if (c4 != '=') {
        *output++ = ((CHAR64(c3) << 6) & 0xc0) | CHAR64(c4);  
        if (++out_len >= OUTPUT_SIZE)
          return NULL;
      }
    }
  }
  
  *output = 0;
  
  return out;
}

/* extracted from libEtPan! - end */


int etpan_mkdir(char * path)
{
  int r;
  struct stat buf;
  char base_dirname[PATH_MAX];
  char * parent;
  
  r = stat(path, &buf);
  if (r == 0)
    return 0;
  
  strncpy(base_dirname, path, sizeof(base_dirname));
  parent = dirname(base_dirname);
  r = etpan_mkdir(parent);
  if (r < 0)
    return r;
  
  return mkdir(path, 0700);
}


char * etpan_quote_mailbox(char * mb)
{
  MMAPString * gstr;
  char * str;

  gstr = mmap_string_new("");
  if (gstr == NULL)
    return NULL;
  
  while (* mb != 0) {
    char hex[3];

    if (((* mb >= 'a') && (* mb <= 'z')) ||
	((* mb >= 'A') && (* mb <= 'Z')) ||
	((* mb >= '0') && (* mb <= '9')))
      mmap_string_append_c(gstr, * mb);
    else {
      if (mmap_string_append_c(gstr, '%') == NULL)
	goto free;
      snprintf(hex, 3, "%02x", (unsigned char) (* mb));
      if (mmap_string_append(gstr, hex) == NULL)
	goto free;
    }
    mb ++;
  }

  str = strdup(gstr->str);
  if (str == NULL)
    goto free;

  mmap_string_free(gstr);
  
  return str;

 free:
  mmap_string_free(gstr);
  return NULL;
}


/* must be improved using getpwnam() or getpwuid() */

char * etpan_get_home_dir(void)
{
  char * value;
  
  value = getenv("HOME");
  if (value != NULL)
    return value;
  
  return "";
}



struct privacy_info {
  char * privacy_driver;
  char * privacy_encryption;
};

int etpan_privacy_encrypt_next(struct mailprivacy * privacy,
    char * privacy_driver, char * privacy_encryption,
    char ** pprivacy_driver, char ** pprivacy_encryption,
    int direction)
{
  unsigned int i;
  int count;
  struct privacy_info * tab;
  int k;
  carray * protocols;

  /* collects all privacy encryption in a table */
  
  protocols = mailprivacy_get_protocols(privacy);
  count = 0;
  for(i = 0 ; i < carray_count(protocols) ; i ++) {
    struct mailprivacy_protocol * protocol;
    
    protocol = carray_get(protocols, i);
    count += protocol->encryption_count;
  }
  
  tab = malloc(count * sizeof(* tab));
  if (tab == NULL)
    return MAIL_ERROR_MEMORY;
  
  k = 0;
  for(i = 0 ; i < carray_count(protocols) ; i ++) {
    struct mailprivacy_protocol * protocol;
    int j;
    
    protocol = carray_get(protocols, i);
    
    for(j = 0 ; j < protocol->encryption_count ; j ++) {
      struct mailprivacy_encryption * encryption;
      
      encryption = &protocol->encryption_tab[j];
      tab[k].privacy_driver = protocol->name;
      tab[k].privacy_encryption = encryption->name;
      k ++;
    }
  }

  if ((privacy_driver == NULL) && (privacy_encryption == NULL)) {
    if (direction > 0) {
      if (count > 0) {
        * pprivacy_driver = tab[0].privacy_driver;
        * pprivacy_encryption = tab[0].privacy_encryption;
      }
      goto exit;
    }
    else {
      if (count > 0) {
        * pprivacy_driver = tab[count - 1].privacy_driver;
        * pprivacy_encryption = tab[count - 1].privacy_encryption;
      }
      goto exit;
    }
    
    * pprivacy_driver = NULL;
    * pprivacy_encryption = NULL;
    
    goto exit;
  }

  /* get the next encryption protocol */
  
  for(k = 0 ; k < count ; k ++) {
    if (strcasecmp(tab[k].privacy_driver, privacy_driver) == 0) {
      if (strcasecmp(tab[k].privacy_encryption, privacy_encryption) == 0) {
        if (direction > 0) {
          if (k + 1 < count) {
            * pprivacy_driver = tab[k + 1].privacy_driver;
            * pprivacy_encryption = tab[k + 1].privacy_encryption;
            
            goto exit;
          }
        }
        else {
          if (k > 1) {
            * pprivacy_driver = tab[k - 1].privacy_driver;
            * pprivacy_encryption = tab[k - 1].privacy_encryption;
            
            goto exit;
          }
        }
        * pprivacy_driver = NULL;
        * pprivacy_encryption = NULL;
        
        goto exit;
      }
    }
  }
  
  * pprivacy_driver = NULL;
  * pprivacy_encryption = NULL;
  
 exit:
  free(tab);
  return NO_ERROR;
}

