/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: fullengine.h
 */

#ifndef EVMS_FULLENGINE_H_INCLUDED
#define EVMS_FULLENGINE_H_INCLUDED 1

#include <common.h>
#include <devmapper.h>
#include <options.h>
#include <enginestructs.h>
#include <appstructs.h>
#include <plugfuncs.h>
#include <appAPI.h>
#include <ece.h>

#define LOG_CRITICAL(msg, args...)              engine_write_log_entry(CRITICAL, "%s: " msg, __FUNCTION__ , ## args)
#define LOG_SERIOUS(msg, args...)               engine_write_log_entry(SERIOUS,  "%s: " msg, __FUNCTION__ , ## args)
#define LOG_ERROR(msg, args...)                 engine_write_log_entry(ERROR,    "%s: " msg, __FUNCTION__ , ## args)
#define LOG_WARNING(msg, args...)               engine_write_log_entry(WARNING,  "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DEFAULT(msg, args...)               engine_write_log_entry(DEFAULT,  "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DETAILS(msg, args...)               engine_write_log_entry(DETAILS,  "%s: " msg, __FUNCTION__ , ## args)
#define LOG_DEBUG(msg, args...)                 engine_write_log_entry(DEBUG,    "%s: " msg, __FUNCTION__ , ## args)
#define LOG_EXTRA(msg, args...)                 engine_write_log_entry(EXTRA,    "%s: " msg, __FUNCTION__ , ## args)
#define LOG(level, msg, args...)                engine_write_log_entry(level,    "%s: " msg, __FUNCTION__ , ## args)

#define LOG_PROC_ENTRY()                        engine_write_log_entry(ENTRY_EXIT, "%s: Enter.\n", __FUNCTION__);

#define LOG_PROC_EXIT_INT(rc)                   engine_write_log_entry(ENTRY_EXIT, "%s: Exit.  Return value is %d.\n", __FUNCTION__, rc);
#define LOG_PROC_EXIT_UINT(rc)                  engine_write_log_entry(ENTRY_EXIT, "%s: Exit.  Return value is %u (0x%08x).\n", __FUNCTION__, rc, rc);
#define LOG_PROC_EXIT_PTR(ptr)                  engine_write_log_entry(ENTRY_EXIT, "%s: Exit.  Returned pointer is %p.\n", __FUNCTION__, ptr);
#define LOG_PROC_EXIT_BOOLEAN(result)           engine_write_log_entry(ENTRY_EXIT, "%s: Exit.  Result is %s.\n", __FUNCTION__, result ? "TRUE" : "FALSE");
#define LOG_PROC_EXIT_BOOLEAN_INT(result, rc)   engine_write_log_entry(ENTRY_EXIT, "%s: Exit.  Result is %s.  Return value is %d.\n", __FUNCTION__, result ? "TRUE" : "FALSE",rc);
#define LOG_PROC_EXIT_VOID()                    engine_write_log_entry(ENTRY_EXIT, "%s: Exit.\n", __FUNCTION__);

/*
 * Macros for logging entry/exit using the EXTRA debug level
 * These macros are used to keep heavily used functions, such as memory
 * management and list management, from overwhelming a DEBUG level log.
 */
#define LOG_PROC_EXTRA_ENTRY()                       engine_write_log_entry(EXTRA, "%s: Enter.\n", __FUNCTION__);

#define LOG_PROC_EXTRA_EXIT_INT(rc)                  engine_write_log_entry(EXTRA, "%s: Exit.  Return value is %d.\n", __FUNCTION__, rc);
#define LOG_PROC_EXTRA_EXIT_UINT(rc)                 engine_write_log_entry(EXTRA, "%s: Exit.  Return value is %u (0x%08x).\n", __FUNCTION__, rc, rc);
#define LOG_PROC_EXTRA_EXIT_PTR(ptr)                 engine_write_log_entry(EXTRA, "%s: Exit.  Returned pointer is %p.\n", __FUNCTION__, ptr);
#define LOG_PROC_EXTRA_EXIT_BOOLEAN(result)          engine_write_log_entry(EXTRA, "%s: Exit.  Result is %s.\n", __FUNCTION__, result ? "TRUE" : "FALSE");
#define LOG_PROC_EXTRA_EXIT_BOOLEAN_INT(result, rc)  engine_write_log_entry(EXTRA, "%s: Exit.  Result is %s.  Return value is %d.\n", __FUNCTION__, result ? "TRUE" : "FALSE",rc);
#define LOG_PROC_EXTRA_EXIT_VOID()                   engine_write_log_entry(EXTRA, "%s: Exit.\n", __FUNCTION__);

#define PRINTF(msg, args...)						\
	sprintf(msg_buf, "evmsd: %s: " msg, __FUNCTION__ , ## args);	\
	write(2, msg_buf, strlen(msg_buf));

#define EVMS_VOLUME_FEATURE_ID SetPluginID(IBM_OEM_ID, EVMS_FEATURE, 0)


/* Special plug-in IDs we know */
#define REPLACE_PLUGIN_ID	SetPluginID(IBM_OEM_ID, EVMS_DEVICE_MANAGER, 12)


/* The contents of the lock file. */
typedef struct lock_file_s {
	int           daemon_lock;
	int           engine_lock;
	pid_t         pid;
	engine_mode_t mode;
	char          node[1];
} lock_file_t;


/* The name_list_entry_t structure is used to keep the registry of names. */
typedef struct name_list_entry_s {
	struct name_list_entry_s * next;
	char                     * name;
} name_list_entry_t;


/*
 * Check a required version against a given version.
 * Return
 *    0     given version satisfied required version
 *   -1     given version is not high enough for required version
 *    1     given version is too high for required version
 */
static inline int check_version(evms_version_t required_version, evms_version_t available_version) {

	int result;

	if (available_version.major == required_version.major) {
		if (available_version.minor >= required_version.minor) {
			if (available_version.minor == required_version.minor) {
				if (available_version.patchlevel >= required_version.patchlevel) {
					/*
					 * The available version has the required major and minor
					 * version.  The available patch level is equal to or
					 * greater than the required patch level.  The available
					 * version is good.
					 */
					result = 0;

				} else {
					/*
					 * The available version has the required major and minor
					 * version, but the available patch level is less than
					 * the required patch level.  The available version is
					 * too small.
					 */
					result = -1;
				}

			} else {
				/*
				 * The available version has the required major number.
				 * The available version has minor greater than required
				 * version.  That's OK.
				 */
				result = 0;
			}

		} else {
			/* The available minor number is too small. */
			result = -1;
		}

	} else {
		/* The major numbers do not match. */
		if (available_version.major > required_version.major) {
			/* The available major number is too big. */
			result = 1;
		} else {
			/* The available major number is too small. */
			result = -1;
		}
	}

	return result;
}


/*
 * Given a size in vsectors, round it down to be a multiple of the sector size
 * of a given storage object.
 */
static inline sector_count_t round_down_to_hard_sector(sector_count_t size, storage_object_t * obj) {

	if ((obj->geometry.bytes_per_sector != 0) &&
	    (obj->geometry.bytes_per_sector > EVMS_VSECTOR_SIZE)) {

		sector_count_t hard_sector_in_vsectors = ((sector_count_t) obj->geometry.bytes_per_sector) >> EVMS_VSECTOR_SIZE_SHIFT;

		return size & ~(hard_sector_in_vsectors - 1);

	} else {
		return size;
	}
}


/*
 * Given a size in vsectors, round it up to be a multiple of the sector size
 * of a given storage object.
 */
static inline sector_count_t round_up_to_hard_sector(sector_count_t size, storage_object_t * obj) {

	if ((obj->geometry.bytes_per_sector != 0) &&
	    (obj->geometry.bytes_per_sector > EVMS_VSECTOR_SIZE)) {

		sector_count_t hard_sector_in_vsectors = ((sector_count_t) obj->geometry.bytes_per_sector) >> EVMS_VSECTOR_SIZE_SHIFT;
		sector_count_t temp_size;

		temp_size = size + hard_sector_in_vsectors - 1;
		return temp_size & ~(hard_sector_in_vsectors - 1);

	} else {
		return size;
	}
}


/*
 * Convert a feature header in the CPU endian format to the disk endian
 * (little endian) format.
 */
static inline void feature_header_cpu_to_disk(evms_feature_header_t * fh) {

	fh->signature                 = CPU_TO_DISK32(fh->signature);
	fh->version.major             = CPU_TO_DISK32(fh->version.major);
	fh->version.minor             = CPU_TO_DISK32(fh->version.minor);
	fh->version.patchlevel        = CPU_TO_DISK32(fh->version.patchlevel);
	fh->engine_version.major      = CPU_TO_DISK32(fh->engine_version.major);
	fh->engine_version.minor      = CPU_TO_DISK32(fh->engine_version.minor);
	fh->engine_version.patchlevel = CPU_TO_DISK32(fh->engine_version.patchlevel);
	fh->flags                     = CPU_TO_DISK32(fh->flags);
	fh->feature_id                = CPU_TO_DISK32(fh->feature_id);
	fh->sequence_number           = CPU_TO_DISK64(fh->sequence_number);
	fh->alignment_padding         = CPU_TO_DISK64(fh->alignment_padding);
	fh->feature_data1_start_lsn   = CPU_TO_DISK64(fh->feature_data1_start_lsn);
	fh->feature_data1_size        = CPU_TO_DISK64(fh->feature_data1_size);
	fh->feature_data2_start_lsn   = CPU_TO_DISK64(fh->feature_data2_start_lsn);
	fh->feature_data2_size        = CPU_TO_DISK64(fh->feature_data2_size);
	fh->volume_serial_number      = CPU_TO_DISK64(fh->volume_serial_number);
	fh->volume_system_id          = CPU_TO_DISK32(fh->volume_system_id);
	fh->object_depth              = CPU_TO_DISK32(fh->object_depth);
}

#endif
