/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvm_info.c
 *
 * Description:
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"

/**
 * lvm_get_volume_info
 *
 * Fill in all the basic information about this volume.
 **/
int lvm_get_volume_info(lvm_logical_volume_t * volume,
			extended_info_array_t ** info_array)
{
	extended_info_array_t * info = NULL;
	int i = 0;

	LOG_ENTRY();

	/* Get memory for the info array. */
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) * 9);
	if (!info) {
		LOG_CRITICAL("Memory error creating info array\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/* Region Name */
	info->info[i].name = EngFncs->engine_strdup("LV_Name");
	info->info[i].title = EngFncs->engine_strdup(_("Region Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("Name of LVM Region (LV)"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(volume->region->name);
	i++;

	/* Container Name */
	info->info[i].name = EngFncs->engine_strdup("VG_Name");
	info->info[i].title = EngFncs->engine_strdup(_("Container Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("Name of LVM Container (VG)"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(volume->group->container->name);
	i++;

	/* LV Number */
	info->info[i].name = EngFncs->engine_strdup("LV_Number");
	info->info[i].title = EngFncs->engine_strdup(_("Region Number"));
	info->info[i].desc = EngFncs->engine_strdup(_("ID number for this region in this container"));
	info->info[i].type = EVMS_Type_Int;
	info->info[i].value.i = volume->number;
	i++;

	/* Region Size */
	info->info[i].name = EngFncs->engine_strdup("LV_Size");
	info->info[i].title = EngFncs->engine_strdup(_("Region Size"));
	info->info[i].desc = EngFncs->engine_strdup(_("Total space for this region"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = volume->region->size;
	i++;

	/* Allocated LEs */
	info->info[i].name = EngFncs->engine_strdup("Extents");
	info->info[i].title = EngFncs->engine_strdup(_("Logical Extents"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of logical extents used by this region"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = volume->lv->lv_allocated_le;
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	/* Write Permissions */
	info->info[i].name = EngFncs->engine_strdup("Permissions");
	info->info[i].title = EngFncs->engine_strdup(_("Access Permissions"));
	info->info[i].type = EVMS_Type_String;
	if (volume->lv->lv_access & LV_WRITE &&
	    volume->lv->lv_access & LV_READ ) {
		info->info[i].value.s = EngFncs->engine_strdup(_("Read-Write"));
	} else if (volume->lv->lv_access & LV_WRITE) {
		info->info[i].value.s = EngFncs->engine_strdup(_("Write-Only"));
	} else if (volume->lv->lv_access & LV_READ) {
		info->info[i].value.s = EngFncs->engine_strdup(_("Read-Only"));
	} else {
		info->info[i].value.s = EngFncs->engine_strdup(_("No-Access"));
	}
	i++;

	/* Striping Info */
	if (volume->lv->lv_stripes > 1) {
		/* Number of Stripes */
		info->info[i].name = EngFncs->engine_strdup("Stripes");
		info->info[i].title = EngFncs->engine_strdup(_("Stripes"));
		info->info[i].desc = EngFncs->engine_strdup(_("Number of objects this region is striped across"));
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].value.ui32 = volume->lv->lv_stripes;
		i++;

		/* Stripe Size */
		info->info[i].name = EngFncs->engine_strdup("Stripe_Size");
		info->info[i].title = EngFncs->engine_strdup(_("Stripe Size"));
		info->info[i].type = EVMS_Type_Unsigned_Int32;
		info->info[i].unit = EVMS_Unit_Sectors;
		info->info[i].value.ui32 = volume->lv->lv_stripesize;
		i++;
	}
	
	if (volume->flags & LVM_LV_FLAG_INCOMPLETE) {
		info->info[i].name = EngFncs->engine_strdup("Incomplete_LV");
		info->info[i].title = EngFncs->engine_strdup(_("INCOMPLETE REGION!!!"));
		info->info[i].desc = EngFncs->engine_strdup(_("This region has extents which are not mapped! One or more objects are probably missing from this container!"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("INCOMPLETE REGION!!!"));
		i++;
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * is_le_missing
 *
 * Does the specified LE not have a valid PE pointer?
 **/
inline int is_le_missing(lvm_logical_volume_t * volume, unsigned int le)
{
	return (volume->le_map[le].pe == NULL);
}

/**
 * is_next_le_consecutive
 *
 * Are the specified LE and the following LE physically coonsecutive
 * on the same PV?
 */
inline int is_next_le_consecutive(lvm_logical_volume_t * volume,
				  unsigned int this_le)
{
	int rc = TRUE;

	if (this_le + 2 > volume->lv->lv_allocated_le) {
		/* Last LE in the volume. */
		rc = FALSE;
	} else if (volume->le_map[this_le].copy_job ||
		   volume->le_map[this_le+1].copy_job) {
		/* One of the LEs is being moved. */
		rc = FALSE;
	} else if (is_le_missing(volume, this_le) ||
		   is_le_missing(volume, this_le+1)) {
		/* One of the LEs is missing. */
		rc = FALSE;
	} else if (volume->le_map[this_le].pe->pv !=
		   volume->le_map[this_le+1].pe->pv) {
		/* LEs are on different PVs. */
		rc = FALSE;
	} else if (volume->le_map[this_le].pe->number + 1 !=
		   volume->le_map[this_le+1].pe->number) {
		/* LEs are on same PV, but not consecutive. */
		rc = FALSE;
	}
	return rc;
}

/**
 * lvm_get_volume_extent_info
 *
 * Fill in the "extra" extended info about the LE-to-PE mapping for
 * this volume.
 **/
int lvm_get_volume_extent_info(lvm_logical_volume_t * volume,
			       extended_info_array_t ** info_array)
{
	extended_info_array_t * info = NULL;
	char buffer[150] = {0};
	int printed_dots = FALSE;
	int consecutive_run = FALSE;
	int info_line = 0;
	int j, i = 0;

	LOG_ENTRY();

	/* Get memory for the info array. */
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t));
	if (!info) {
		LOG_CRITICAL("Memory error creating info array\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/* Extent mappings */
	info->info[i].name = EngFncs->engine_strdup("Extent_Map");
	info->info[i].title = EngFncs->engine_strdup(_("Logical Extents"));
	info->info[i].desc = EngFncs->engine_strdup(_("LE Number : PV Name : PE Number"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].collection_type = EVMS_Collection_List;
	info->info[i].collection.list = EngFncs->engine_alloc(sizeof(value_list_t) +
							      sizeof(value_t) *
							      (volume->lv->lv_allocated_le + 1));
	snprintf(buffer, 150, "%-5s : %-15s : %-5s", "LE #", _("PV Name"), _("PE Number"));
	info->info[i].collection.list->value[info_line].s = EngFncs->engine_strdup(buffer);
	info_line++;

	/* Print info for each LE. Condense runs of consecutive
	 * LEs into the first and last, with a ... in between.
	 */
	for (j = 0; j < volume->lv->lv_allocated_le; j++) {
		if (!volume->le_map[j].pe) {
			snprintf(buffer, 150, "%-5d : %-15s : %-5s",
				 j, _("Missing PV"), "");
			consecutive_run = FALSE;
		}
		/* Are we at the end of the LV or at the end of a run
		 * of consecutive LEs?
		 */
		else if (! is_next_le_consecutive(volume, j)) {
			snprintf(buffer, 150, "%-5d : %-15s : %-5d",
				 j, volume->le_map[j].pe->pv->segment->name,
				 volume->le_map[j].pe->number);
			consecutive_run = FALSE;
		}
		/* Are we at the start of the LV or at the start of a
		 * run of consecutive LEs?
		 */
		else if (!consecutive_run) {
			snprintf(buffer, 150, "%-5d : %-15s : %-5d",
				 j, volume->le_map[j].pe->pv->segment->name,
				 volume->le_map[j].pe->number);
			consecutive_run = TRUE;
			printed_dots = FALSE;
		}
		/* Have we printed the ... yet? */
		else if (!printed_dots) {
			snprintf(buffer, 150, ".....");
			printed_dots = TRUE;
		}
		else {
			continue;
		}
		info->info[i].collection.list->value[info_line].s = EngFncs->engine_strdup(buffer);
		info_line++;
	}
	info->info[i].collection.list->count = info_line;
	i++;

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_get_group_info
 **/
int lvm_get_group_info(lvm_volume_group_t * group,
		       extended_info_array_t ** info_array)
{
	extended_info_array_t * info = NULL;
	char * buf = NULL;
	int i = 0;

	LOG_ENTRY();

	/* Get memory for the info array. */
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) * 13);
	if (!info) {
		LOG_CRITICAL("Memory error creating info array\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/* Container Name */
	info->info[i].name = EngFncs->engine_strdup("VG_Name");
	info->info[i].title = EngFncs->engine_strdup(_("Container Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("Name of LVM Container (VG)"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(group->container->name);
	i++;

	/* VG Number */
	info->info[i].name = EngFncs->engine_strdup("VG_Number");
	info->info[i].title = EngFncs->engine_strdup(_("Container Number"));
	info->info[i].desc = EngFncs->engine_strdup(_("ID number for this container in the LVM plugin"));
	info->info[i].type = EVMS_Type_Int;
	info->info[i].value.i = group->vg->vg_number;
	i++;

	/* VG Size */
	info->info[i].name = EngFncs->engine_strdup("VG_Size");
	info->info[i].title = EngFncs->engine_strdup(_("Container Size"));
	info->info[i].desc = EngFncs->engine_strdup(_("Total accumulated space in this container"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = group->container->size;
	i++;

	/* Available VG Size */
	info->info[i].name = EngFncs->engine_strdup("VG_Free_Size");
	info->info[i].title = EngFncs->engine_strdup(_("Available Size"));
	info->info[i].desc = EngFncs->engine_strdup(_("Amount of space currently available for allocating to regions"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = group->freespace->lv->lv_size;
	i++;

	/* Percent Allocated */
	info->info[i].name = EngFncs->engine_strdup("VG_Percent_Allocated");
	info->info[i].title = EngFncs->engine_strdup(_("Percent Allocated"));
	info->info[i].desc = EngFncs->engine_strdup(_("Percentage of space currently allocated to regions"));
	info->info[i].type = EVMS_Type_Real32;
	info->info[i].unit = EVMS_Unit_Percent;
	info->info[i].value.r32 = ((float)(group->vg->pe_allocated) /
				   (float)(group->vg->pe_total)) * 100.0;
	i++;

	/* PE Size */
	info->info[i].name = EngFncs->engine_strdup("PE_Size");
	info->info[i].title = EngFncs->engine_strdup(_("Extent Size"));
	info->info[i].desc = EngFncs->engine_strdup(_("Size of each extent available for allocating to regions"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = group->vg->pe_size;
	i++;

	/* Total PEs */
	info->info[i].name = EngFncs->engine_strdup("Total_PEs");
	info->info[i].title = EngFncs->engine_strdup(_("Total PEs"));
	info->info[i].desc = EngFncs->engine_strdup(_("Total number of extents in the container"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = group->vg->pe_total;
	i++;

	/* Available PEs */
	info->info[i].name = EngFncs->engine_strdup("Available_PEs");
	info->info[i].title = EngFncs->engine_strdup(_("Available PEs"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of extents available for allocating to regions"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = group->freespace->lv->lv_allocated_le;
	i++;

	/* Number of PVs */
	info->info[i].name = EngFncs->engine_strdup("Current_PVs");
	info->info[i].title = EngFncs->engine_strdup(_("Number of Objects (PVs)"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of objects consumed by this container"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = group->pv_count;
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	/* Number of LVs */
	info->info[i].name = EngFncs->engine_strdup("Current_LVs");
	info->info[i].title = EngFncs->engine_strdup(_("Number of Regions (LVs)"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of regions produced by this container"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = group->volume_count;
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	/* Max LV Size */
	info->info[i].name = EngFncs->engine_strdup("Max_LV_Size");
	info->info[i].title = EngFncs->engine_strdup(_("Maximum Region Size"));
	info->info[i].desc = EngFncs->engine_strdup(_("Maximum possible size of any region in this container for the given PE size"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = group->vg->pe_size * 64 * 1024;
	i++;

	/* VG UUID */
	info->info[i].name = EngFncs->engine_strdup("VG_UUID");
	info->info[i].title = EngFncs->engine_strdup(_("Container UUID"));
	info->info[i].type = EVMS_Type_String;
	buf = lvm_print_uuid(group->vg->vg_uuid);
	info->info[i].value.s = EngFncs->engine_strdup(buf);
	i++;

	if (group->pv_count < group->vg->pv_cur) {
		info->info[i].name = EngFncs->engine_strdup("Incomplete_VG");
		info->info[i].title = EngFncs->engine_strdup(_("INCOMPLETE CONTAINER!!!"));
		info->info[i].desc = EngFncs->engine_strdup(_("This container seems to be missing one or more objects!"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("INCOMPLETE CONTAINER!!!"));
		i++;
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_get_group_pv_list_info
 **/
int lvm_get_group_pv_list_info(lvm_volume_group_t * group,
			       extended_info_array_t ** info_array)
{
	extended_info_array_t * info = NULL;
	char buf[50] = {0};
	int j, i = 0;

	LOG_ENTRY();

	/* Get memory for the info array. */
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) *
				     group->pv_count);
	if (!info) {
		LOG_CRITICAL("Memory error creating info array\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	for (j = 1; j <= MAX_PV; j++) {
		if (group->pv_list[j]) {
			sprintf(buf, "PV%d", j);
			info->info[i].name = EngFncs->engine_strdup(buf);
			sprintf(buf, _("Object (PV) %d"), j);
			info->info[i].title = EngFncs->engine_strdup(buf);
			info->info[i].type = EVMS_Type_String;
			info->info[i].value.s =
				EngFncs->engine_strdup(group->pv_list[j]->segment->name);
			info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
			i++;
		}
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_get_group_lv_list_info
 **/
int lvm_get_group_lv_list_info(lvm_volume_group_t * group,
			       extended_info_array_t ** info_array)
{
	extended_info_array_t * info = NULL;
	char buf[50] = {0};
	int j, i = 0;

	LOG_ENTRY();

	/* Get memory for the info array. */
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) *
				     group->volume_count);
	if (!info) {
		LOG_CRITICAL("Memory error creating info array\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	for (j = 1; j <= MAX_LV; j++) {
		if (group->volume_list[j]) {
			sprintf(buf, "LV%d", j);
			info->info[i].name = EngFncs->engine_strdup(buf);
			sprintf(buf, _("Region (LV) %d"), j);
			info->info[i].title = EngFncs->engine_strdup(buf);
			info->info[i].type = EVMS_Type_String;
			info->info[i].value.s =
				EngFncs->engine_strdup(group->volume_list[j]->region->name);
			i++;
		}
	}

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * lvm_get_pv_info
 **/
int lvm_get_pv_info(lvm_physical_volume_t * pv_entry,
		    extended_info_array_t ** info_array)
{
	extended_info_array_t * info = NULL;
	char * buf1 = NULL;
	char buf2[50] = {0};
	int i = 0;

	LOG_ENTRY();

	/* Get memory for the info array. */
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) +
				     sizeof(extended_info_t) * 10);
	if (!info) {
		LOG_CRITICAL("Memory error creating info array\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/* PV Name */
	info->info[i].name = EngFncs->engine_strdup("PV_Name");
	info->info[i].title = EngFncs->engine_strdup(_("Object Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("Name of LVM Object (PV)"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(pv_entry->segment->name);
	i++;

	/* VG Name */
	info->info[i].name = EngFncs->engine_strdup("VG_Name");
	info->info[i].title = EngFncs->engine_strdup(_("Container Name"));
	info->info[i].desc = EngFncs->engine_strdup(_("Name of LVM Container (VG)"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].value.s = EngFncs->engine_strdup(pv_entry->group->container->name);
	i++;

	/* PV Number */
	info->info[i].name = EngFncs->engine_strdup("PV_Number");
	info->info[i].title = EngFncs->engine_strdup(_("Object Number"));
	info->info[i].desc = EngFncs->engine_strdup(_("ID number for this object (PV) in this container"));
	info->info[i].type = EVMS_Type_Int;
	info->info[i].value.i = pv_entry->number;
	i++;

	/* PV Size */
	info->info[i].name = EngFncs->engine_strdup("PV_Size");
	info->info[i].title = EngFncs->engine_strdup(_("Object Size"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = pv_entry->segment->size;
	i++;

	/* Usable Size */
	info->info[i].name = EngFncs->engine_strdup("Usable_PV_Size");
	info->info[i].title = EngFncs->engine_strdup(_("Usable Space"));
	info->info[i].desc = EngFncs->engine_strdup(_("PE Size * Number of PEs. Some object space is used for metadata, some is unusable due to PE size"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = pv_entry->pv->pe_size * pv_entry->pv->pe_total;
	i++;

	/* Number of Regions */
	info->info[i].name = EngFncs->engine_strdup("Current_Regions");
	info->info[i].title = EngFncs->engine_strdup(_("Current Regions"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of regions currently using space on this object"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = pv_entry->pv->lv_cur;
	i++;

	/* PE Size */
	info->info[i].name = EngFncs->engine_strdup("PE_Size");
	info->info[i].title = EngFncs->engine_strdup(_("Extent Size"));
	info->info[i].desc = EngFncs->engine_strdup(_("Size of each extent available for allocating to regions"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].unit = EVMS_Unit_Sectors;
	info->info[i].value.ui32 = pv_entry->pv->pe_size;
	i++;

	/* Total PEs */
	sprintf(buf2, "PEMapPV%ld", pv_entry->number);
	info->info[i].name = EngFncs->engine_strdup(buf2);
	info->info[i].title = EngFncs->engine_strdup(_("Total PEs"));
	info->info[i].desc = EngFncs->engine_strdup(_("Total number of extents in this object"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = pv_entry->pv->pe_total;
	info->info[i].flags = EVMS_EINFO_FLAGS_MORE_INFO_AVAILABLE;
	i++;

	/* Available PEs */
	info->info[i].name = EngFncs->engine_strdup("Available_PEs");
	info->info[i].title = EngFncs->engine_strdup(_("Available PEs"));
	info->info[i].desc = EngFncs->engine_strdup(_("Number of extents available for allocating to regions"));
	info->info[i].type = EVMS_Type_Unsigned_Int32;
	info->info[i].value.ui32 = pv_entry->pv->pe_total -
				   pv_entry->pv->pe_allocated -
				   pv_entry->move_extents;
	i++;

	/* PV UUID */
	info->info[i].name = EngFncs->engine_strdup("PV_UUID");
	info->info[i].title = EngFncs->engine_strdup(_("Object (PV) UUID"));
	info->info[i].type = EVMS_Type_String;
	buf1 = lvm_print_uuid(pv_entry->pv->pv_uuid);
	info->info[i].value.s = EngFncs->engine_strdup(buf1);
	i++;

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

/**
 * is_next_pe_consecutive
 *
 * Are the specified PE and the following PE consecutive. This can be true in
 * two cases:
 * 1) Both PEs map to the same LV, on consecutive LEs.
 * 2) Both PEs map to freespace.
 *
 * If this is the last PE on the PV, the "next" PE is not consecutive.
 **/
static inline int is_next_pe_consecutive(lvm_physical_volume_t * pv_entry,
					 int this_pe)
{
	lvm_physical_extent_t * pe_map = pv_entry->pe_map;
	int rc = TRUE;

	if (this_pe >= pv_entry->pv->pe_total - 1) {
		/* Last PE on the PV. */
		rc = FALSE;
	} else if (pe_map[this_pe].pe.lv_num !=
		   pe_map[this_pe+1].pe.lv_num) {
		/* PEs map to different LVs. */
		rc = FALSE;
	} else if (pe_map[this_pe].pe.lv_num) {
		/* PEs are mapped to real LVs. */
		if (pe_map[this_pe].pe.le_num + 1 !=
		    pe_map[this_pe+1].pe.le_num) {
			/* PEs map to non-consecutive LEs. */
			rc = FALSE;
		}
	}

	return rc;
}

/**
 * lvm_get_pv_extent_info
 **/
int lvm_get_pv_extent_info(lvm_physical_volume_t * pv_entry,
			   extended_info_array_t ** info_array)
{
	lvm_volume_group_t * group = pv_entry->group;
	lvm_physical_extent_t * pe_map = pv_entry->pe_map;
	extended_info_array_t * info = NULL;
	char buffer[150] = {0};
	int consecutive_run = FALSE;
	int printed_dots = FALSE;
	int info_line = 1;
	int j, i = 0;

	LOG_ENTRY();

	/* Get memory for the info array. */
	info = EngFncs->engine_alloc(sizeof(extended_info_array_t) + sizeof(extended_info_t));
	if (!info) {
		LOG_CRITICAL("Memory error creating info array\n");
		LOG_EXIT_INT(ENOMEM);
		return ENOMEM;
	}

	/* Extent mappings */
	info->info[i].name = EngFncs->engine_strdup("Extent_Map");
	info->info[i].title = EngFncs->engine_strdup(_("Physical Extents"));
	info->info[i].desc = EngFncs->engine_strdup(_("PE Number : LV Name : LE Number: Sector"));
	info->info[i].type = EVMS_Type_String;
	info->info[i].collection_type = EVMS_Collection_List;
	info->info[i].collection.list = EngFncs->engine_alloc(sizeof(value_list_t) +
							      sizeof(value_t) *
							      (pv_entry->pv->pe_total + 1));
	snprintf(buffer, 150, "%-5s : %-20s : %-5s : %-5s", _("PE Number"), _("LV Name"), _("LE Number"), _("Sector"));
	info->info[i].collection.list->value[0].s = EngFncs->engine_strdup(buffer);

	/* Print info for each PE. Condense runs of consecutive PEs into
	 * the first and last, with a ... in between.
	 */
	for (j = 0; j < pv_entry->pv->pe_total; j++) {
		if (pe_map[j].le) {
			if (pe_map[j].le->volume == group->freespace) {
				/* PE is unallocated. */
				if (! is_next_pe_consecutive(pv_entry, j)) {
					snprintf(buffer, 150,
						 "%-5d : %-20s : %-5s : %-5"PRIu64"",
						 j, _("none"), "n/a",
						 pv_entry->pe_map[j].sector);
					consecutive_run = FALSE;
				} else if (!consecutive_run) {
					snprintf(buffer, 150,
						 "%-5d : %-20s : %-5s : %-5"PRIu64"",
						 j, _("none"), "n/a",
						 pv_entry->pe_map[j].sector);
					consecutive_run = TRUE;
					printed_dots = FALSE;
				} else if (!printed_dots) {
					snprintf(buffer, 150, ".....");
					printed_dots = TRUE;
				} else {
					continue;
				}
			} else {
				/* PE is allocated to a volume. */
				if (! is_next_pe_consecutive(pv_entry, j)) {
					snprintf(buffer, 150, "%-5d : %-20s : %-5d : %-5"PRIu64"", j,
						 pe_map[j].le->volume->region->name,
						 pe_map[j].le->number,
						 pe_map[j].sector);
					consecutive_run = FALSE;
				} else if (!consecutive_run) {
					snprintf(buffer, 150, "%-5d : %-20s : %-5d : %-5"PRIu64"", j,
						 pe_map[j].le->volume->region->name,
						 pe_map[j].le->number,
						 pe_map[j].sector);
					consecutive_run = TRUE;
					printed_dots = FALSE;
				} else if (!printed_dots) {
					snprintf(buffer, 150, ".....");
					printed_dots = TRUE;
				} else {
					continue;
				}
			}
		} else if (pe_map[j].new_le) {
			/* PE is the target of a move. */
			snprintf(buffer, 150, "%-5d : %-20s : %-5d : %-5"PRIu64"", j,
				 pe_map[j].new_le->volume->region->name,
				 pe_map[j].new_le->number,
				 pe_map[j].sector);
		}

		info->info[i].collection.list->value[info_line].s = EngFncs->engine_strdup(buffer);
		info_line++;
	}
	info->info[i].collection.list->count = info_line;
	i++;

	info->count = i;
	*info_array = info;

	LOG_EXIT_INT(0);
	return 0;
}

