/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: mdregmgr
 * File: raid1_funcs.c
 *
 * Description: This file contains all MD RAID1's plugin functions.
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include <sys/ioctl.h>

#include "md.h"
#include "raid1_mgr.h"

#define my_plugin_record raid1_plugin

/*
 * Function: activate_spare_post_activate
 *
 * This function should be called during POST_ACTIVATE phase of commit.
 *
 */
static int activate_spare_post_ioctl(md_volume_t *vol, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	if (pkg->parm.disk_info) {
		EngFncs->engine_free(pkg->parm.disk_info);
		pkg->parm.disk_info = NULL;
	}
	vol->region_mgr_flags &= ~MD_RAID1_CONFIG_CHANGE_PENDING;
	LOG_EXIT_INT(0);
	return 0;
}


/*
 * Function: activate_spare_setup
 *
 * This function should be called during SETUP phase of commit.
 * - Retrieve the original spare disk
 * - Mark the entry faulty and removed
 * - Schedule an EVMS_MD_HOT_ADD to hot-add the spare
 * - Next phase of commit process is FIRST_METADATA_WRITE
 */
static int activate_spare_setup(md_volume_t *vol, md_setup_func_t *setup)
{
	int rc = 0;
	evms_md_disk_info_t *disk_info;
	storage_object_t *spare_disk;
	evms_md_ioctl_parm_t parm;
	md_member_t *member;

	LOG_ENTRY();

	disk_info = setup->disk_info;
	spare_disk = disk_info->object;

	if (setup->proceed == FALSE) {
		LOG_DEBUG("Cancel activate spare (%s) for region %s\n",
			  spare_disk->name, vol->name);
		rc = 0;
		goto undo_activate_spare;
	}

	member = md_volume_find_member(vol, disk_info->number);
	if (!member || member->obj != spare_disk) {
		LOG_ERROR("Could not find the original spare disk %s to activate region [%s]\n",
			  (spare_disk) ? spare_disk->name : "???", vol->name);
		rc = EINVAL;
		goto undo_activate_spare;
	}
	
	rc = md_volume_mark_faulty(member, TRUE); // mark faulty and removed
	if (!rc) {
		parm.disk_info = disk_info;
		switch (vol->sb_ver.major_version) {
		case MD_SB_VER_0:
			rc = schedule_md_ioctl_pkg(vol, EVMS_MD_HOT_ADD, &parm, activate_spare_post_ioctl);
			break;
		case MD_SB_VER_1:
			rc = schedule_md_ioctl_pkg(vol, EVMS_MD_ADD, &parm, activate_spare_post_ioctl);
			break;
		default:
			LOG_MD_BUG();
			rc = EINVAL;
			break;
		}
		if (rc) {
			goto undo_activate_spare;
		}
		vol->flags |= MD_DIRTY; //Update superblocks
	} else {
		goto undo_activate_spare;
	}

	LOG_EXIT_INT(rc);
	return rc;

undo_activate_spare:
	EngFncs->engine_free(setup->disk_info);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * raid1_activate_spare_disk
 *
 * This function changes the configuration of the RAID1 region
 * from n-way mirror to n+1-way mirror.
 */
int raid1_activate_spare_disk(md_volume_t *vol, storage_object_t * spare_disk)
{
	int rc=0;
	evms_md_disk_info_t *disk_info = NULL;
	md_member_t *member;
	list_element_t iter;
	boolean found = FALSE;

	LOG_ENTRY();

	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj && member->obj == spare_disk) {
			found = TRUE;
			break;
		}
	}
	if (found == FALSE) {
		rc = EINVAL;
		goto out;
	}

	disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
	if ( !disk_info ) {
		rc = ENOMEM;
		goto out;
	}
        disk_info->number = member->dev_number;
	disk_info->object = spare_disk;

	rc = schedule_setup_func(vol, disk_info, activate_spare_setup);
	if (rc) {
		goto out;
	}

	rc = md_volume_activate_spare(member);
	if (rc) {
		goto out;
	}

	if (md_is_region_active(vol->region)) {
		vol->region->flags |= (SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE);
	}
	vol->region_mgr_flags |= MD_RAID1_CONFIG_CHANGE_PENDING;

out:
	if (rc && disk_info) {
		EngFncs->engine_free(disk_info);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: remove_active_disk_post_activate
 *
 */
static int remove_active_disk_post_activate(md_volume_t *vol, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	vol->region_mgr_flags &= ~MD_RAID1_CONFIG_CHANGE_PENDING;
	LOG_EXIT_INT(0);
	return 0;
}

/*
 * raid1_remove_active_disk
 *
 * This function changes the  RAID1 configuration from n-way mirror
 * to n-1-way mirror.
 *
 * Note:   If the region is active, it will be marked NEEDS_DEACTIVATE.
 *         This is important because the MD array must be stopped before
 *         new superblocks are layed down.
 */
int raid1_remove_active_disk(md_volume_t * vol, storage_object_t *active_disk)
{
	int rc=0;
	md_member_t *member;
	list_element_t iter;
	boolean found = FALSE;

	LOG_ENTRY();
	if (!vol || !active_disk) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	LIST_FOR_EACH(vol->members, iter, member) {
		if (member->obj && member->obj == active_disk) {
			if (md_member_is_raid_disk(member)) {
				found = TRUE;
				break;
			} else {
				LOG_WARNING("%s is not active.\n",
					    active_disk->name);
			}
		}
	}

	if (found == FALSE) {
		LOG_MD_BUG();
		rc = EINVAL;
		goto out;
	}

	rc = schedule_md_ioctl_pkg(vol, EVMS_MD_INVOKE_CALLBACK,
				   NULL, remove_active_disk_post_activate);
	if (rc) {
		goto out;
	}

	rc = md_volume_remove_member(member, TRUE); //remove and resize
	if (rc) {
		goto out;
	}

	md_remove_region_from_object(vol->region, member->obj);
	vol->sb_func->zero_superblock(member, FALSE);
	md_free_member(member);
			
	if (md_is_region_active(vol->region)) {
		vol->region->flags |= (SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE);
	}
	vol->region_mgr_flags |= MD_RAID1_CONFIG_CHANGE_PENDING;
	vol->flags |= MD_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * Function: add_active_disk_post_ioctl
 *
 * This function should be called during POST_ACTIVATE phase of commit.
 *
 */
static int add_active_disk_post_ioctl(md_volume_t *vol, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	if (pkg->parm.disk_info) {
		EngFncs->engine_free(pkg->parm.disk_info);
		pkg->parm.disk_info = NULL;
	}
	vol->region_mgr_flags &= ~MD_RAID1_CONFIG_CHANGE_PENDING;
	vol->flags &= ~MD_ARRAY_RESIZE_PENDING;
	LOG_EXIT_INT(0);
	return 0;
}

/*
 * add_active_disk_setup
 *
 * This function should be called during SETUP phase of commit.
 *
 * - Retrieve the disk entry added in raid1_add_active_disk()
 * - Change the new entry to faulty entry. 
 * - Schedule the EVMS_MD_ADD so that on the POST_ACTIVATE phase, we will IOCTL
 *   the kernel MD driver to hot-add this new disk to fix the degraded array.  
 * - Mark volume DIRTY for superblock update.
 * - Next phase of commit process is FIRST_METADATA_WRITE, we will write 
 *   modified superblock to indicate "degraded" array.
 */
static int add_active_disk_setup(md_volume_t *vol, md_setup_func_t *setup)
{
	int rc = 0;
	evms_md_ioctl_parm_t parm;
	evms_md_disk_info_t *disk_info = NULL;
	storage_object_t *new_disk;
	md_member_t *member = NULL;

	LOG_ENTRY();

	disk_info = setup->disk_info;
	new_disk = disk_info->object;
	
	member = md_volume_find_member(vol, disk_info->number);
	if (!member || member->obj != new_disk) {
		LOG_ERROR("Could not find the new disk %s to add to region [%s]\n",
			    (new_disk) ? new_disk->name : "???", vol->name);
		rc = EINVAL;
		goto undo_add_active_disk;
	}
	
	if (setup->proceed == FALSE) {
		LOG_WARNING("Cancel add new active disk (%s) to region %s\n",
			    new_disk->name, vol->name);
		rc = 0;
		goto undo_add_active_disk;
	}

	rc = md_volume_mark_faulty(member, TRUE); // mark faulty & removed
	if (rc) {
		goto undo_add_active_disk;
	}

	parm.disk_info = disk_info;
	switch (vol->sb_ver.major_version) {
	case MD_SB_VER_0:
		rc = schedule_md_ioctl_pkg(vol, EVMS_MD_HOT_ADD, &parm, add_active_disk_post_ioctl);
		break;
	case MD_SB_VER_1:
		rc = schedule_md_ioctl_pkg(vol, EVMS_MD_ADD, &parm, add_active_disk_post_ioctl);
		break;
	default:
		LOG_MD_BUG();
		rc = EINVAL;
		break;
	}

	if (rc) {
		goto undo_add_active_disk;
	}

	vol->flags |= MD_DIRTY;

	LOG_EXIT_INT(rc);
	return rc;

undo_add_active_disk:
	if (member) {
		md_remove_region_from_object(vol->region, member->obj);
		md_volume_remove_member(member, TRUE);
		md_free_member(member);
	}
	EngFncs->engine_free(setup->disk_info);
	LOG_EXIT_INT(rc);
	return rc;
}

/*
 * raid1_add_active_disk
 *
 * Schedule add_active_disk_setup function to do most of the work.
 */
int raid1_add_active_disk(md_volume_t *vol, storage_object_t * new_disk)
{
	int rc=0;
	evms_md_disk_info_t *disk_info = NULL;
	md_member_t *member = NULL;
	
	LOG_ENTRY();

	member = md_allocate_member(new_disk);
	if (!member) {
		rc = ENOMEM;
		goto out;
	}

	member->data_size = vol->region->size;
	member->flags |= (MD_MEMBER_NEW | MD_MEMBER_DISK_ACTIVE | MD_MEMBER_DISK_PENDING);
	rc = md_volume_add_new_member(vol, member);
	if (rc) {
		goto out;
	}

	disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
	if ( !disk_info ) {
		rc = ENOMEM;
		goto out;
	}

	disk_info->number = member->dev_number;
	disk_info->object = new_disk;

	rc = schedule_setup_func(vol, disk_info, add_active_disk_setup);
	if (rc) {
		LOG_ERROR("Error adding object %s to region: Can not shedule setup function\n",
			  new_disk->name);
		goto out;
	}
	
	md_append_region_to_object(vol->region, new_disk);

	if (md_is_region_active(vol->region)) {
		vol->region->flags |= (SOFLAG_NEEDS_DEACTIVATE | SOFLAG_NEEDS_ACTIVATE);
	}
	vol->region_mgr_flags |= MD_RAID1_CONFIG_CHANGE_PENDING;
	vol->flags |= MD_ARRAY_RESIZE_PENDING;

out:
	if (rc) {
		if (disk_info) {
			EngFncs->engine_free(disk_info);
		}
		if (member) {
			md_volume_remove_member(member, TRUE);
			md_free_member(member);
		}
	}
	LOG_EXIT_INT(rc);
	return rc;
}


int raid1_add_spare_disk(md_volume_t *vol, storage_object_t * spare)
{
	int rc = 0;
	LOG_ENTRY();

	if (!vol || !spare) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (md_is_region_active(vol->region)) {
		rc = md_volume_add_spare_to_active_region(vol, spare);
	} else {
		rc = md_volume_add_spare_to_inactive_region(vol, spare);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int raid1_remove_spare_disk(md_volume_t *vol, storage_object_t *spare)
{
	int rc=0;
	LOG_ENTRY();

	if (!vol || !spare) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	if (md_is_region_active(vol->region)) {
		rc = md_volume_remove_spare_from_active_region(vol, spare);
	} else {
		rc = md_volume_remove_spare_from_inactive_region(vol, spare);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int raid1_remove_stale_disk(md_volume_t *vol, storage_object_t * stale_disk)
{
	int rc=0;
	
	LOG_ENTRY();

	if (!vol || !stale_disk) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	if (md_is_region_active(vol->region)) {
		rc = md_volume_remove_stale_from_active_region(vol, stale_disk);
	} else {
		rc = md_volume_remove_stale_from_inactive_region(vol, stale_disk);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

int raid1_remove_faulty_disk(md_volume_t * vol, storage_object_t * faulty)
{
	int rc=0;
	LOG_ENTRY();

	if (!vol || !faulty) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}
	
	if (md_is_region_active(vol->region)) {
		rc = md_volume_remove_faulty_from_active_region(vol, faulty);
	} else {
		rc = md_volume_remove_faulty_from_inactive_region(vol, faulty);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

static int mark_disk_faulty_post_ioctl(md_volume_t *volume, md_ioctl_pkg_t *pkg)
{
	LOG_ENTRY();
	if (pkg->parm.disk_info) {
		EngFncs->engine_free(pkg->parm.disk_info);
		pkg->parm.disk_info = NULL;
	}
	volume->region_mgr_flags &= ~MD_RAID1_CONFIG_CHANGE_PENDING;
	LOG_EXIT_INT(0);
	return 0;
}

int raid1_mark_disk_faulty(md_volume_t *vol, storage_object_t * active)
{
	int rc=0;
	evms_md_disk_info_t *disk_info = NULL;
	evms_md_ioctl_parm_t parm;
	mdu_disk_info_t info;
	md_member_t *member;

	LOG_ENTRY();

	if (!vol || !active) {
		LOG_EXIT_INT(EFAULT);
		return EFAULT;
	}

	member = md_volume_find_object(vol, active);
	if (!member) {
		LOG_WARNING("%s is not in region %s.\n", active->name, vol->name);
		rc = EINVAL;
		goto out;
	}
	if (!md_member_is_raid_disk(member)) {
		LOG_WARNING("%s is not active.\n", active->name);
		rc = EINVAL;
		goto out;
	}

	info.number = member->dev_number;
	rc = md_ioctl_get_disk_info(vol->region, &info);
	if (rc) {
		LOG_MD_BUG();
		rc = ENODEV;
		goto out;
	} else {
		if ((info.major != active->dev_major) || (info.minor != active->dev_minor)) {
			LOG_WARNING("(%s) mismatch major/minor, kernel(%d:%d), EVMS(%d:%d)\n",
				    active->name, info.major, info.minor, 
				    active->dev_major, active->dev_minor);
		}
	}

	disk_info = EngFncs->engine_alloc(sizeof(evms_md_disk_info_t));
	if ( !disk_info ) {
		rc = ENOMEM;
		goto out;
	}
	
	rc = md_volume_mark_faulty(member, FALSE);
	if (rc) {
		goto out;
	}

	disk_info->number = member->dev_number;
	disk_info->major = info.major;
	disk_info->minor = info.minor;
	disk_info->object = active;
	parm.disk_info = disk_info;
	schedule_md_ioctl_pkg(vol, EVMS_MD_DEACTIVATE, &parm, mark_disk_faulty_post_ioctl);

	vol->region_mgr_flags |= MD_RAID1_CONFIG_CHANGE_PENDING;
out:
	if (rc && disk_info) {
		EngFncs->engine_free(disk_info);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

