/*
 *
 *    Copyright 1995-2000 Regents of the University of Minnesota
 *    Portions Copyright 1999-2001 Sistina Software, Inc.
 *
 *    This is free software released under the GNU General Public License.
 *    There is no warranty for this software.  See the file COPYING for
 *    details.
 *
 *    See the file AUTHORS for a list of contributors.
 *
 */

/*
 * The OGFS on-disk structures and the prototypes for the routines
 * to read and write them.
 *
 * NOTE:
 *      If you add 8 byte fields to these structures, they must be 8 byte
 *      aligned.  4 byte field must be 4 byte aligned, etc...
 *
 *      All structures must be a multiple of 8 bytes long.
 *
 * GRIPES:
 *      We should have forgetten about supporting 512B FS block sizes 
 *      and made the di_reserved field in the ogfs_dinode_t structure
 *      much bigger.
 *
 *      de_rec_len in ogfs_dirent_t should really have been a 32-bit value
 *      as it now limits us to a 64k FS block size (with the current code
 *      in dir.c).
 */
#ifndef _OGFS_ONDISK_H
#define _OGFS_ONDISK_H

#include "global.h"

#define OGFS_MAGIC               (0x01161970)
#define OGFS_BASIC_BLOCK         (512)

/*  Lock numbers of the LM_TYPE_MISC type  */

#define OGFS_MOUNT_LOCK          (0)
#define OGFS_LIVE_LOCK           (1)
#define OGFS_TRANS_LOCK          (2)
#define OGFS_RENAME_LOCK         (3)

/*  Format numbers for various metadata types  */

#define OGFS_FORMAT_SB       	(100)
#define OGFS_FORMAT_RG       	(200)
#define OGFS_FORMAT_RB           (300)
#define OGFS_FORMAT_DI       	(400)
#define OGFS_FORMAT_IN           (500)
#define OGFS_FORMAT_LF           (600)
#define OGFS_FORMAT_JD           (700)
#define OGFS_FORMAT_LH           (800)
#define OGFS_FORMAT_LD           (900)
/*  These don't have actual ogfs_meta_header_t structures to go with them  */
#define OGFS_FORMAT_JI           (1000)
#define OGFS_FORMAT_RI           (1100)
#define OGFS_FORMAT_DE           (1200)
/*  These are part of the superblock  */
#define OGFS_FORMAT_FS           (1305)
#define OGFS_FORMAT_MULTI        (1401)

/*
 *  An on-disk inode number
 */

#define OGFS_INUM_EQUAL(ino1, ino2) \
(((ino1)->no_formal_ino == (ino2)->no_formal_ino) && \
 ((ino1)->no_addr == (ino2)->no_addr))

struct ogfs_inum {
	uint64 no_formal_ino;
	uint64 no_addr;
};
typedef struct ogfs_inum ogfs_inum_t;

/*
 *  Generic metadata head structure
 *
 *  Every inplace buffer logged in the journal must start with this.
 */

#define OGFS_METATYPE_NONE       (0)
#define OGFS_METATYPE_SB       	(1)
#define OGFS_METATYPE_RG       	(2)
#define OGFS_METATYPE_RB         (3)
#define OGFS_METATYPE_DI       	(4)
#define OGFS_METATYPE_IN         (5)
#define OGFS_METATYPE_LF         (6)
#define OGFS_METATYPE_JD         (7)
#define OGFS_METATYPE_LH         (8)
#define OGFS_METATYPE_LD         (9)

#define OGFS_META_CLUMP          (64)

struct ogfs_meta_header {
	uint32 mh_magic;	/* Magic number */
	uint32 mh_type;		/* OGFS_METATYPE_XX */
	uint64 mh_generation;	/* Generation number */
	uint32 mh_format;	/* OGFS_FORMAT_XX */
	uint32 mh_pad;
};
typedef struct ogfs_meta_header ogfs_meta_header_t;

typedef struct ogfs_meta_header ogfs_rgrpbits_t;
typedef struct ogfs_meta_header ogfs_jdata_t;

/*
 *  super-block structure
 *
 *  It's probably good if SIZEOF_SB <= OGFS_BASIC_BLOCK
 */

/*  Address of SuperBlock in OGFS basic blocks  */
#define OGFS_SB_ADDR     	(128)
/*  The lock number for the superblock (must be zero)  */
#define OGFS_SB_LOCK             (0)

#define OGFS_SBF_SUBPOOL_ALIGNED (0x00000001)

/*  Requirement:  OGFS_LOCKNAME_LEN % 8 == 0
    Includes: the fencing zero at the end  */
#define OGFS_LOCKNAME_LEN        (64)

struct ogfs_sb {
	/*  Order is important  */
	ogfs_meta_header_t sb_header;

	uint32 sb_fs_format;
	uint32 sb_multihost_format;
	uint32 sb_flags;

	/*  Important information  */
	uint32 sb_bsize;	/* fundamental fs block size in bytes */
	uint32 sb_bsize_shift;	/* log2(sb_bsize) */
	uint32 sb_seg_size;	/* Journal segment size in FS blocks */

	ogfs_inum_t sb_jindex_di;	/* journal index inode number (OGFS_SB_LOCK) */
	ogfs_inum_t sb_rindex_di;	/* resource index inode number (OGFS_SB_LOCK) */
	ogfs_inum_t sb_root_di;	/* root directory inode number (OGFS_ROOT_LOCK) */

	char sb_lockproto[OGFS_LOCKNAME_LEN];	/* Type of locking this FS uses */
	char sb_locktable[OGFS_LOCKNAME_LEN];	/* Name of lock table for this FS */

	char sb_reserved[128];
};
typedef struct ogfs_sb ogfs_sb_t;

/*
 *  journal index structure 
 */

#define OGFS_DEVNAME_LEN	48

struct ogfs_jindex {
	uint64 ji_addr;		/* starting block of the journal */
	uint32 ji_nsegment;	/* number of segments in journal */
	uint32 ji_pad;

	char ji_device[OGFS_DEVNAME_LEN];	/* external journal dev name */
	char ji_reserved[64 - OGFS_DEVNAME_LEN];
};
typedef struct ogfs_jindex ogfs_jindex_t;

/*
 *  resource index structure 
 */

struct ogfs_rindex {
	uint64 ri_addr;		/* rgrp block disk address */
	uint32 ri_length;	/* length of rgrp header in fs blocks */
	uint32 ri_pad;

	uint64 ri_data1;	/* first data location */
	uint32 ri_data;		/* num of data blocks in rgrp */

	uint32 ri_bitbytes;	/* number of bytes in data bitmaps */

	char ri_reserved[64];
};
typedef struct ogfs_rindex ogfs_rindex_t;

/*
 *  resource group header structure
 *
 */

/* Number of blocks per byte in rgrp */
#define OGFS_NBBY		(4)
#define OGFS_BIT_SIZE            (2)
#define OGFS_BIT_MASK            (0x00000003)

#define OGFS_BLKST_FREE          (0)
#define OGFS_BLKST_USED          (1)
#define OGFS_BLKST_FREEMETA      (2)
#define OGFS_BLKST_USEDMETA      (3)
#define OGFS_BLKST_INVALID	 (0xff)

struct ogfs_rgrp {
	ogfs_meta_header_t rg_header;

	uint32 rg_flags;	/* flags */

	uint32 rg_free;		/* number of free data blocks */

	uint32 rg_useddi;	/* number of dinodes */
	uint32 rg_freedi;	/* number of unused dinodes */
	ogfs_inum_t rg_freedi_list;	/* list of free dinodes */

	uint32 rg_usedmeta;	/* number of used metadata blocks (not including dinodes) */
	uint32 rg_freemeta;	/* number of unused metadata blocks */

	char rg_reserved[64];
};
typedef struct ogfs_rgrp ogfs_rgrp_t;

/*
 *  dinode structure
 */

#define OGFS_MAX_META_HEIGHT     (10)
#define OGFS_DIR_MAX_DEPTH       (17)

/*  Dinode types  */
#define FILE_NON                (0)
#define FILE_REG                (1)
#define FILE_DIR                (2)
#define FILE_LNK                (5)
#define FILE_BLK                (7)
#define FILE_CHR                (8)
#define FILE_FIFO               (101)
#define FILE_SOCK               (102)

/*  Dinode flags  */
#define OGFS_DIF_JDATA           (0x00000001)
#define OGFS_DIF_EXHASH          (0x00000002)
#define OGFS_DIF_UNUSED          (0x00000004)

struct ogfs_dinode {
	ogfs_meta_header_t di_header;

	ogfs_inum_t di_num;

	uint32 di_mode;		/* mode of file */
	uint32 di_uid;		/* owner's user id */
	uint32 di_gid;		/* owner's group id */
	uint32 di_nlink;	/* number of links to this file */
	uint64 di_size;		/* number of bytes in file */
	uint64 di_blocks;	/* number of blocks in file */
	int64 di_atime;		/* time last accessed */
	int64 di_mtime;		/* time last modified */
	int64 di_ctime;		/* time last changed */
	uint32 di_major;	/* device major number */
	uint32 di_minor;	/* device minor number */

	uint64 di_rgrp;		/* dinode rgrp block number */
	uint64 di_goal_rgrp;	/* rgrp to alloc from next */
	uint32 di_goal_dblk;	/* data block goal */
	uint32 di_goal_mblk;	/* metadata block goal */
	uint32 di_flags;	/* flags */
	uint32 di_payload_format;	/* ogfs_rindex_t, ogfs_jindex_t, or ogfs_dirent_t */
	uint16 di_type;		/* type of file */
	uint16 di_height;	/* height of metadata */
	uint32 di_incarn;	/* incarnation number */
	uint16 di_pad;

	/*  These only apply to directories  */
	uint16 di_depth;	/* Number of bits in the table */
	uint32 di_entries;	/* The number of entries in the directory */

	/*  This only applies to unused inodes  */
	ogfs_inum_t di_next_unused;

	char di_reserved[64];
};
typedef struct ogfs_dinode ogfs_dinode_t;

/*
 *  indirect block header
 */

struct ogfs_indirect {
	ogfs_meta_header_t in_header;

	char in_reserved[64];
};
typedef struct ogfs_indirect ogfs_indirect_t;

/*
 *  directory structure - many of these per directory file
 */

#define OGFS_FNAMESIZE       	(256)

#define OGFS_DIR_PAD (8)
#define OGFS_DIR_ROUND (OGFS_DIR_PAD - 1)
#define OGFS_DIR_REC_LEN(name_len) ((sizeof(ogfs_dirent_t) + (name_len) + OGFS_DIR_ROUND) & \
				   (~OGFS_DIR_ROUND))

struct ogfs_dirent {
	ogfs_inum_t de_inum;	/* Inode number */
	uint32 de_hash;		/* hash of the filename */
	uint16 de_rec_len;	/* the length of the dirent */
	uint16 de_name_len;	/* the length of the name */
	uint16 de_type;		/* type of dinode this points to */

	char de_reserved[14];
};
typedef struct ogfs_dirent ogfs_dirent_t;

/*
 *  Header of leaf directory nodes
 */

struct ogfs_leaf {
	ogfs_meta_header_t lf_header;

	uint16 lf_depth;	/* Depth of leaf */
	uint16 lf_entries;	/* Number of dirents in leaf */
	uint32 lf_dirent_format;	/* Format of the dirents */
	uint64 lf_next;		/* Next leaf, if overflow */

	char lf_reserved[64];
};
typedef struct ogfs_leaf ogfs_leaf_t;

/*
 *  Log header structure
 */

#define OGFS_LOG_HEAD_UNMOUNT    (0x00000001)

struct ogfs_log_header {
	ogfs_meta_header_t lh_header;

	uint32 lh_flags;	/* Flags */
	uint32 lh_pad;

	uint64 lh_first;	/* Block number of first header in this trans */
	uint64 lh_sequence;	/* Sequence number of this transaction */

	uint64 lh_tail;		/* Block number of log tail */
	uint64 lh_last_nopd;	/* block number of last nopen dump */

	char lh_reserved[64];
};
typedef struct ogfs_log_header ogfs_log_header_t;

/*
 *  Log type descriptor
 */

#define OGFS_LOG_DESC_METADATA	(300)
/*  ld_data1 is the number of metadata blocks in the descriptor,
    ld_data2 is the transaction type.
    */

#define OGFS_NOP_DUMPS_PER_LOG   (4)
#define OGFS_LOG_DESC_NOPOPEN    (400)
/*  ld_data1 is TRUE if this is a dump.
    ld_data2 is unused.
    */

#define OGFS_LOG_DESC_NOPCLOSE   (401)
/*  ld_data1 is unused.
    ld_data2 is unused.
    */

#define OGFS_LOG_DESC_LAST	(500)
/*  ld_data1 is unused.
    ld_data2 is unused.
    */

struct ogfs_log_descriptor {
	ogfs_meta_header_t ld_header;

	uint32 ld_type;		/* Type of data in this log chunk */
	uint32 ld_length;	/* Number of buffers in this chunk */
	uint32 ld_data1;	/* descriptor specific field */
	uint32 ld_data2;	/* descriptor specific field */

	char ld_reserved[64];
};
typedef struct ogfs_log_descriptor ogfs_log_descriptor_t;

/*
 *  Metadata block tags
 */

struct ogfs_block_tag {
	uint64 bt_blkno;	/* inplace block number */
	uint32 bt_flags;	/* flags */
	uint32 bt_pad;
};
typedef struct ogfs_block_tag ogfs_block_tag_t;

/*  Endian functions  */

#define OGFS_ENDIAN_BIG

#ifdef OGFS_ENDIAN_BIG

#define ogfs16_to_cpu be16_to_cpu
#define ogfs32_to_cpu be32_to_cpu
#define ogfs64_to_cpu be64_to_cpu

#define cpu_to_ogfs16 cpu_to_be16
#define cpu_to_ogfs32 cpu_to_be32
#define cpu_to_ogfs64 cpu_to_be64

#else				/*  OGFS_ENDIAN_BIG  */

#define ogfs16_to_cpu le16_to_cpu
#define ogfs32_to_cpu le32_to_cpu
#define ogfs64_to_cpu le64_to_cpu

#define cpu_to_ogfs16 cpu_to_le16
#define cpu_to_ogfs32 cpu_to_le32
#define cpu_to_ogfs64 cpu_to_le64

#endif				/*  OGFS_ENDIAN_BIG  */

/*  Translation functions  */

void ogfs_inum_in(ogfs_inum_t * no, char *buf);
void ogfs_inum_out(ogfs_inum_t * no, char *buf);
void ogfs_meta_header_in(ogfs_meta_header_t * mh, char *buf);
void ogfs_meta_header_out(ogfs_meta_header_t * mh, char *buf);
void ogfs_sb_in(ogfs_sb_t * sb, char *buf);
void ogfs_sb_out(ogfs_sb_t * sb, char *buf);
void ogfs_jindex_in(ogfs_jindex_t * jindex, char *buf);
void ogfs_jindex_out(ogfs_jindex_t * jindex, char *buf);
void ogfs_rindex_in(ogfs_rindex_t * rindex, char *buf);
void ogfs_rindex_out(ogfs_rindex_t * rindex, char *buf);
void ogfs_rgrp_in(ogfs_rgrp_t * rgrp, char *buf);
void ogfs_rgrp_out(ogfs_rgrp_t * rgrp, char *buf);
void ogfs_dinode_in(ogfs_dinode_t * dinode, char *buf);
void ogfs_dinode_out(ogfs_dinode_t * dinode, char *buf);
void ogfs_indirect_in(ogfs_indirect_t * indirect, char *buf);
void ogfs_indirect_out(ogfs_indirect_t * indirect, char *buf);
void ogfs_dirent_in(ogfs_dirent_t * dirent, char *buf);
void ogfs_dirent_out(ogfs_dirent_t * dirent, char *buf);
void ogfs_leaf_in(ogfs_leaf_t * leaf, char *buf);
void ogfs_leaf_out(ogfs_leaf_t * leaf, char *buf);
void ogfs_log_header_in(ogfs_log_header_t * head, char *buf);
void ogfs_log_header_out(ogfs_log_header_t * head, char *buf);
void ogfs_desc_in(ogfs_log_descriptor_t * desc, char *buf);
void ogfs_desc_out(ogfs_log_descriptor_t * desc, char *buf);
void ogfs_tag_in(ogfs_block_tag_t * tag, char *buf);
void ogfs_tag_out(ogfs_block_tag_t * tag, char *buf);

/*  Printing functions  */

void ogfs_inum_print(ogfs_inum_t * no, int console);
void ogfs_meta_header_print(ogfs_meta_header_t * mh, int console);
void ogfs_sb_print(ogfs_sb_t * sb, int console);
void ogfs_jindex_print(ogfs_jindex_t * jindex, int console);
void ogfs_rindex_print(ogfs_rindex_t * rindex, int console);
void ogfs_rgrp_print(ogfs_rgrp_t * rgrp, int console);
void ogfs_dinode_print(ogfs_dinode_t * dinode, int console);
void ogfs_indirect_print(ogfs_indirect_t * indirect, int console);
void ogfs_dirent_print(ogfs_dirent_t * dirent, char *name, int console);
void ogfs_leaf_print(ogfs_leaf_t * leaf, int console);
void ogfs_log_header_print(ogfs_log_header_t * head, int console);
void ogfs_desc_print(ogfs_log_descriptor_t * desc, int console);
void ogfs_tag_print(ogfs_block_tag_t * tag, int console);

#endif /* _OGFS_ONDISK_H */
