/*
 *
 *    Copyright 1995-2000 Regents of the University of Minnesota
 *    Portions Copyright 1999-2001 Sistina Software, Inc.
 *    Portions Copyright 2001 The OpenGFS Project
 *
 *    This is free software released under the GNU General Public License.
 *    There is no warranty for this software.  See the file COPYING for
 *    details.
 *
 *    See the file AUTHORS for a list of contributors.
 *
 */

/*
 * This module converts all OGFS structures that exist on
 * disk to internal processor formats.
 */
#include <stdio.h>
#include <string.h>
#define __STDC_FORMAT_MACROS 1
#include <inttypes.h>

#include "osi_endian.h"
#include "ogfs_ondisk.h"

#define con_printf printf
#define db_printf printf

#define pv(struct, member, fmt) printf("  "#member" = "fmt"\n", struct->member);

#define CPIN_08(s1, s2, member, count) {memcpy((s1->member), (s2->member), (count));}
#define CPOUT_08(s1, s2, member, count) {memcpy((s2->member), (s1->member), (count));}
#define CPIN_16(s1, s2, member) {(s1->member) = ogfs16_to_cpu((s2->member));}
#define CPOUT_16(s1, s2, member) {(s2->member) = cpu_to_ogfs16((s1->member));}
#define CPIN_32(s1, s2, member) {(s1->member) = ogfs32_to_cpu((s2->member));}
#define CPOUT_32(s1, s2, member) {(s2->member) = cpu_to_ogfs32((s1->member));}
#define CPIN_64(s1, s2, member) {(s1->member) = ogfs64_to_cpu((s2->member));}
#define CPOUT_64(s1, s2, member) {(s2->member) = cpu_to_ogfs64((s1->member));}

#define pa(struct, member, count) print_array(#member, struct->member, count, console);

/**
 * print_array - Print out an array of bytes
 * @title: what to print before the array
 * @buf: the array
 * @count: the number of bytes
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
static void
print_array(char *title, char *buf, int count, int console)
{
	int x;

	if (console) {
		con_printf("  %s =\n", title);
		for (x = 0; x < count; x++) {
			con_printf("%.2X ", (unsigned char) buf[x]);
			if (x % 16 == 15)
				con_printf("\n");
		}
		if (x % 16)
			con_printf("\n");
	} else {
		db_printf("  %s =\n", title);
		for (x = 0; x < count; x++) {
			db_printf("%.2X ", (unsigned char) buf[x]);
			if (x % 16 == 15)
				db_printf("\n");
		}
		if (x % 16)
			db_printf("\n");
	}
}

/**
 * ogfs_inum_in - Read in an inode number
 * @no: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_inum_in(ogfs_inum_t * no, char *buf)
{
	ogfs_inum_t *str = (ogfs_inum_t *) buf;

	CPIN_64(no, str, no_formal_ino);
	CPIN_64(no, str, no_addr);
}

/**
 * ogfs_inum_out - Write out an inode number
 * @no: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_inum_out(ogfs_inum_t * no, char *buf)
{
	ogfs_inum_t *str = (ogfs_inum_t *) buf;

	CPOUT_64(no, str, no_formal_ino);
	CPOUT_64(no, str, no_addr);
}

/**
 * ogfs_inum_print - Print out a inode number
 * @no: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_inum_print(ogfs_inum_t * no, int console)
{
	pv(no, no_formal_ino, "%"PRIu64);
	pv(no, no_addr, "%"PRIu64);
}

/**
 * ogfs_meta_header_in - Read in a metadata header
 * @mh: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_meta_header_in(ogfs_meta_header_t * mh, char *buf)
{
	ogfs_meta_header_t *str = (ogfs_meta_header_t *) buf;

	CPIN_32(mh, str, mh_magic);
	CPIN_32(mh, str, mh_type);
	CPIN_64(mh, str, mh_generation);
	CPIN_32(mh, str, mh_format);
	CPIN_32(mh, str, mh_pad);
}

/**
 * ogfs_meta_header_in - Write out a metadata header
 * @mh: the cpu-order structure
 * @buf: the disk-order buffer
 *
 * Don't ever change the generation number in this routine.
 * It's done manually in increment_generation().
 */
void
ogfs_meta_header_out(ogfs_meta_header_t * mh, char *buf)
{
	ogfs_meta_header_t *str = (ogfs_meta_header_t *) buf;

	CPOUT_32(mh, str, mh_magic);
	CPOUT_32(mh, str, mh_type);
	/*CPOUT_64(mh, str, mh_generation); */
	CPOUT_32(mh, str, mh_format);
	CPOUT_32(mh, str, mh_pad);
}

/**
 * ogfs_meta_header_print - Print out a metadata header
 * @mh: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_meta_header_print(ogfs_meta_header_t * mh, int console)
{
	pv(mh, mh_magic, "0x%.8X");
	pv(mh, mh_type, "%u");
	pv(mh, mh_generation, "%"PRIu64);
	pv(mh, mh_format, "%u");
	pv(mh, mh_pad, "%u");
}

/**
 * ogfs_sb_in - Read in a superblock
 * @sb: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_sb_in(ogfs_sb_t * sb, char *buf)
{
	ogfs_sb_t *str = (ogfs_sb_t *) buf;

	ogfs_meta_header_in(&sb->sb_header, buf);

	CPIN_32(sb, str, sb_fs_format);
	CPIN_32(sb, str, sb_multihost_format);
	CPIN_32(sb, str, sb_flags);

	CPIN_32(sb, str, sb_bsize);
	CPIN_32(sb, str, sb_bsize_shift);
	CPIN_32(sb, str, sb_seg_size);

	ogfs_inum_in(&sb->sb_jindex_di, (char *) &str->sb_jindex_di);
	ogfs_inum_in(&sb->sb_rindex_di, (char *) &str->sb_rindex_di);
	ogfs_inum_in(&sb->sb_root_di, (char *) &str->sb_root_di);

	CPIN_08(sb, str, sb_lockproto, OGFS_LOCKNAME_LEN);
	CPIN_08(sb, str, sb_locktable, OGFS_LOCKNAME_LEN);

	CPIN_08(sb, str, sb_reserved, 128);
}

/**
 * ogfs_sb_out - Write out a superblock
 * @sb: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_sb_out(ogfs_sb_t * sb, char *buf)
{
	ogfs_sb_t *str = (ogfs_sb_t *) buf;

	ogfs_meta_header_out(&sb->sb_header, buf);

	CPOUT_32(sb, str, sb_fs_format);
	CPOUT_32(sb, str, sb_multihost_format);
	CPOUT_32(sb, str, sb_flags);

	CPOUT_32(sb, str, sb_bsize);
	CPOUT_32(sb, str, sb_bsize_shift);
	CPOUT_32(sb, str, sb_seg_size);

	ogfs_inum_out(&sb->sb_jindex_di, (char *) &str->sb_jindex_di);
	ogfs_inum_out(&sb->sb_rindex_di, (char *) &str->sb_rindex_di);
	ogfs_inum_out(&sb->sb_root_di, (char *) &str->sb_root_di);

	CPOUT_08(sb, str, sb_lockproto, OGFS_LOCKNAME_LEN);
	CPOUT_08(sb, str, sb_locktable, OGFS_LOCKNAME_LEN);

	CPOUT_08(sb, str, sb_reserved, 128);
}

/**
 * ogfs_sb_print - Print out a superblock
 * @sb: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */

void
ogfs_sb_print(ogfs_sb_t * sb, int console)
{
	ogfs_meta_header_print(&sb->sb_header, console);

	pv(sb, sb_fs_format, "%u");
	pv(sb, sb_multihost_format, "%u");
	pv(sb, sb_flags, "%u");

	pv(sb, sb_bsize, "%u");
	pv(sb, sb_bsize_shift, "%u");
	pv(sb, sb_seg_size, "%u");

	ogfs_inum_print(&sb->sb_jindex_di, console);
	ogfs_inum_print(&sb->sb_rindex_di, console);
	ogfs_inum_print(&sb->sb_root_di, console);

	pv(sb, sb_lockproto, "%s");
	pv(sb, sb_locktable, "%s");

	pa(sb, sb_reserved, 128);
}

/**
 * ogfs_jindex_in - Read in a journal index structure
 * @jindex: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */

void
ogfs_jindex_in(ogfs_jindex_t * jindex, char *buf)
{
	ogfs_jindex_t *str = (ogfs_jindex_t *) buf;

	CPIN_64(jindex, str, ji_addr);
	CPIN_32(jindex, str, ji_nsegment);
	CPIN_32(jindex, str, ji_pad);

	CPIN_08(jindex, str, ji_device, OGFS_DEVNAME_LEN);
	CPIN_08(jindex, str, ji_reserved, 64 - OGFS_DEVNAME_LEN);
}

/**
 * ogfs_jindex_out - Write out a journal index structure
 * @jindex: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_jindex_out(ogfs_jindex_t * jindex, char *buf)
{
	ogfs_jindex_t *str = (ogfs_jindex_t *) buf;

	CPOUT_64(jindex, str, ji_addr);
	CPOUT_32(jindex, str, ji_nsegment);
	CPOUT_32(jindex, str, ji_pad);

	CPOUT_08(jindex, str, ji_device, OGFS_DEVNAME_LEN);
	CPOUT_08(jindex, str, ji_reserved, 64 - OGFS_DEVNAME_LEN);
}

/**
 * ogfs_jindex_print - Print out a journal index structure
 * @ji: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_jindex_print(ogfs_jindex_t * ji, int console)
{
	pv(ji, ji_addr, "%"PRIu64);
	pv(ji, ji_nsegment, "%u");
	pv(ji, ji_pad, "%u");

	pa(ji, ji_device, OGFS_DEVNAME_LEN);
	pa(ji, ji_reserved, 64 - OGFS_DEVNAME_LEN);
}

/**
 * ogfs_rindex_in - Read in a resource index structure
 * @rindex: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_rindex_in(ogfs_rindex_t * rindex, char *buf)
{
	ogfs_rindex_t *str = (ogfs_rindex_t *) buf;

	CPIN_64(rindex, str, ri_addr);
	CPIN_32(rindex, str, ri_length);
	CPIN_32(rindex, str, ri_pad);

	CPIN_64(rindex, str, ri_data1);
	CPIN_32(rindex, str, ri_data);

	CPIN_32(rindex, str, ri_bitbytes);

	CPIN_08(rindex, str, ri_reserved, 64);
}

/**
 * ogfs_rindex_out - Write out a resource index structure
 * @rindex: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_rindex_out(ogfs_rindex_t * rindex, char *buf)
{
	ogfs_rindex_t *str = (ogfs_rindex_t *) buf;

	CPOUT_64(rindex, str, ri_addr);
	CPOUT_32(rindex, str, ri_length);
	CPOUT_32(rindex, str, ri_pad);

	CPOUT_64(rindex, str, ri_data1);
	CPOUT_32(rindex, str, ri_data);

	CPOUT_32(rindex, str, ri_bitbytes);

	CPOUT_08(rindex, str, ri_reserved, 64);
}

/**
 * ogfs_rindex_print - Print out a resource index structure
 * @ri: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_rindex_print(ogfs_rindex_t * ri, int console)
{
	pv(ri, ri_addr, "%"PRIu64);
	pv(ri, ri_length, "%u");
	pv(ri, ri_pad, "%u");

	pv(ri, ri_data1, "%"PRIu64);
	pv(ri, ri_data, "%u");

	pv(ri, ri_bitbytes, "%u");

	pa(ri, ri_reserved, 64);
}

/**
 * ogfs_rgrp_in - Read in a resource group header
 * @rgrp: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_rgrp_in(ogfs_rgrp_t * rgrp, char *buf)
{
	ogfs_rgrp_t *str = (ogfs_rgrp_t *) buf;

	ogfs_meta_header_in(&rgrp->rg_header, buf);

	CPIN_32(rgrp, str, rg_flags);

	CPIN_32(rgrp, str, rg_free);

	CPIN_32(rgrp, str, rg_useddi);
	CPIN_32(rgrp, str, rg_freedi);
	ogfs_inum_in(&rgrp->rg_freedi_list, (char *) &str->rg_freedi_list);

	CPIN_32(rgrp, str, rg_usedmeta);
	CPIN_32(rgrp, str, rg_freemeta);

	CPIN_08(rgrp, str, rg_reserved, 64);
}

/**
 * ogfs_rgrp_out - Write out a resource group header
 * @rgrp: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_rgrp_out(ogfs_rgrp_t * rgrp, char *buf)
{
	ogfs_rgrp_t *str = (ogfs_rgrp_t *) buf;

	ogfs_meta_header_out(&rgrp->rg_header, buf);

	CPOUT_32(rgrp, str, rg_flags);

	CPOUT_32(rgrp, str, rg_free);

	CPOUT_32(rgrp, str, rg_useddi);
	CPOUT_32(rgrp, str, rg_freedi);
	ogfs_inum_out(&rgrp->rg_freedi_list, (char *) &str->rg_freedi_list);

	CPOUT_32(rgrp, str, rg_usedmeta);
	CPOUT_32(rgrp, str, rg_freemeta);

	CPOUT_08(rgrp, str, rg_reserved, 64);
}

/**
 * ogfs_rgrp_print - Print out a resource group header
 * @rg: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_rgrp_print(ogfs_rgrp_t * rg, int console)
{
	ogfs_meta_header_print(&rg->rg_header, console);

	pv(rg, rg_flags, "%u");

	pv(rg, rg_free, "%u");

	pv(rg, rg_useddi, "%u");
	pv(rg, rg_freedi, "%u");
	ogfs_inum_print(&rg->rg_freedi_list, console);

	pv(rg, rg_usedmeta, "%u");
	pv(rg, rg_freemeta, "%u");

	pa(rg, rg_reserved, 64);
}

/**
 * ogfs_dinode_in - Read in a dinode
 * @dinode: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_dinode_in(ogfs_dinode_t * dinode, char *buf)
{
	ogfs_dinode_t *str = (ogfs_dinode_t *) buf;

	ogfs_meta_header_in(&dinode->di_header, buf);

	ogfs_inum_in(&dinode->di_num, (char *) &str->di_num);

	CPIN_32(dinode, str, di_mode);
	CPIN_32(dinode, str, di_uid);
	CPIN_32(dinode, str, di_gid);
	CPIN_32(dinode, str, di_nlink);
	CPIN_64(dinode, str, di_size);
	CPIN_64(dinode, str, di_blocks);
	CPIN_64(dinode, str, di_atime);
	CPIN_64(dinode, str, di_mtime);
	CPIN_64(dinode, str, di_ctime);
	CPIN_32(dinode, str, di_major);
	CPIN_32(dinode, str, di_minor);

	CPIN_64(dinode, str, di_rgrp);
	CPIN_64(dinode, str, di_goal_rgrp);
	CPIN_32(dinode, str, di_goal_dblk);
	CPIN_32(dinode, str, di_goal_mblk);
	CPIN_32(dinode, str, di_flags);
	CPIN_32(dinode, str, di_payload_format);
	CPIN_16(dinode, str, di_type);
	CPIN_16(dinode, str, di_height);
	CPIN_32(dinode, str, di_incarn);
	CPIN_16(dinode, str, di_pad);

	CPIN_16(dinode, str, di_depth);
	CPIN_32(dinode, str, di_entries);

	ogfs_inum_in(&dinode->di_next_unused, (char *) &str->di_next_unused);

	CPIN_08(dinode, str, di_reserved, 64);
}

/**
 * ogfs_dinode_out - Write out a dinode
 * @dinode: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_dinode_out(ogfs_dinode_t * dinode, char *buf)
{
	ogfs_dinode_t *str = (ogfs_dinode_t *) buf;

	ogfs_meta_header_out(&dinode->di_header, buf);

	ogfs_inum_out(&dinode->di_num, (char *) &str->di_num);

	CPOUT_32(dinode, str, di_mode);
	CPOUT_32(dinode, str, di_uid);
	CPOUT_32(dinode, str, di_gid);
	CPOUT_32(dinode, str, di_nlink);
	CPOUT_64(dinode, str, di_size);
	CPOUT_64(dinode, str, di_blocks);
	CPOUT_64(dinode, str, di_atime);
	CPOUT_64(dinode, str, di_mtime);
	CPOUT_64(dinode, str, di_ctime);
	CPOUT_32(dinode, str, di_major);
	CPOUT_32(dinode, str, di_minor);

	CPOUT_64(dinode, str, di_rgrp);
	CPOUT_64(dinode, str, di_goal_rgrp);
	CPOUT_32(dinode, str, di_goal_dblk);
	CPOUT_32(dinode, str, di_goal_mblk);
	CPOUT_32(dinode, str, di_flags);
	CPOUT_32(dinode, str, di_payload_format);
	CPOUT_16(dinode, str, di_type);
	CPOUT_16(dinode, str, di_height);
	CPOUT_32(dinode, str, di_incarn);
	CPOUT_16(dinode, str, di_pad);

	CPOUT_16(dinode, str, di_depth);
	CPOUT_32(dinode, str, di_entries);

	ogfs_inum_out(&dinode->di_next_unused, (char *) &str->di_next_unused);

	CPOUT_08(dinode, str, di_reserved, 64);
}

/**
 * ogfs_dinode_print - Print out a dinode
 * @di: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_dinode_print(ogfs_dinode_t * di, int console)
{
	ogfs_meta_header_print(&di->di_header, console);

	ogfs_inum_print(&di->di_num, console);

	pv(di, di_mode, "0%o");
	pv(di, di_uid, "%u");
	pv(di, di_gid, "%u");
	pv(di, di_nlink, "%u");
	pv(di, di_size, "%"PRIu64);
	pv(di, di_blocks, "%"PRIu64);
	pv(di, di_atime, "%"PRId64);
	pv(di, di_mtime, "%"PRId64);
	pv(di, di_ctime, "%"PRId64);
	pv(di, di_major, "%u");
	pv(di, di_minor, "%u");

	pv(di, di_rgrp, "%"PRIu64);
	pv(di, di_goal_rgrp, "%"PRIu64);
	pv(di, di_goal_dblk, "%u");
	pv(di, di_goal_mblk, "%u");
	pv(di, di_flags, "0x%.8X");
	pv(di, di_payload_format, "%u");
	pv(di, di_type, "%u");
	pv(di, di_height, "%u");
	pv(di, di_incarn, "%u");
	pv(di, di_pad, "%u");

	pv(di, di_depth, "%u");
	pv(di, di_entries, "%u");

	ogfs_inum_print(&di->di_next_unused, console);

	pa(di, di_reserved, 64);
}

/**
 * ogfs_indirect_in - copy in the header of an indirect block
 * @indirect: the in memory copy
 * @buf: the buffer copy
 *
 */
void
ogfs_indirect_in(ogfs_indirect_t * indirect, char *buf)
{
	ogfs_indirect_t *str = (ogfs_indirect_t *) buf;

	ogfs_meta_header_in(&indirect->in_header, buf);

	CPIN_08(indirect, str, in_reserved, 64);
}

/**
 * ogfs_indirect_out - copy out the header of an indirect block
 * @indirect: the in memory copy
 * @buf: the buffer copy
 *
 */
void
ogfs_indirect_out(ogfs_indirect_t * indirect, char *buf)
{
	ogfs_indirect_t *str = (ogfs_indirect_t *) buf;

	ogfs_meta_header_out(&indirect->in_header, buf);

	CPOUT_08(indirect, str, in_reserved, 64);
}

/**
 * ogfs_indirect_print - Print out a indirect block header
 * @indirect: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_indirect_print(ogfs_indirect_t * indirect, int console)
{
	ogfs_meta_header_print(&indirect->in_header, console);

	pa(indirect, in_reserved, 64);
}

/**
 * ogfs_dirent_in - Read in a directory entry
 * @dirent: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_dirent_in(ogfs_dirent_t * dirent, char *buf)
{
	ogfs_dirent_t *str = (ogfs_dirent_t *) buf;

	ogfs_inum_in(&dirent->de_inum, (char *) &str->de_inum);
	CPIN_32(dirent, str, de_hash);
	CPIN_16(dirent, str, de_rec_len);
	CPIN_16(dirent, str, de_name_len);
	CPIN_16(dirent, str, de_type);

	CPIN_08(dirent, str, de_reserved, 14);
}

/**
 * ogfs_dirent_out - Write out a directory entry
 * @dirent: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_dirent_out(ogfs_dirent_t * dirent, char *buf)
{
	ogfs_dirent_t *str = (ogfs_dirent_t *) buf;

	ogfs_inum_out(&dirent->de_inum, (char *) &str->de_inum);
	CPOUT_32(dirent, str, de_hash);
	CPOUT_16(dirent, str, de_rec_len);
	CPOUT_16(dirent, str, de_name_len);
	CPOUT_16(dirent, str, de_type);

	CPOUT_08(dirent, str, de_reserved, 14);
}

/**
 * ogfs_dirent_print - Print out a directory entry
 * @de: the cpu-order buffer
 * @name: the filename
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_dirent_print(ogfs_dirent_t * de, char *name, int console)
{
	char buf[OGFS_FNAMESIZE + 1];

	ogfs_inum_print(&de->de_inum, console);
	pv(de, de_hash, "0x%.8X");
	pv(de, de_rec_len, "%u");
	pv(de, de_name_len, "%u");
	pv(de, de_type, "%u");

	pa(de, de_reserved, 14);

	memset(buf, 0, OGFS_FNAMESIZE + 1);
	memcpy(buf, name, de->de_name_len);
	if (console)
		con_printf("  name = %s\n", buf);
	else
		db_printf("  name = %s\n", buf);
}

/**
 * ogfs_leaf_in - Read in a directory leaf header
 * @leaf: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_leaf_in(ogfs_leaf_t * leaf, char *buf)
{
	ogfs_leaf_t *str = (ogfs_leaf_t *) buf;

	ogfs_meta_header_in(&leaf->lf_header, buf);

	CPIN_16(leaf, str, lf_depth);
	CPIN_16(leaf, str, lf_entries);
	CPIN_32(leaf, str, lf_dirent_format);
	CPIN_64(leaf, str, lf_next);

	CPIN_08(leaf, str, lf_reserved, 64);
}

/**
 * ogfs_leaf_out - Write out a directory leaf header
 * @leaf: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_leaf_out(ogfs_leaf_t * leaf, char *buf)
{
	ogfs_leaf_t *str = (ogfs_leaf_t *) buf;

	ogfs_meta_header_out(&leaf->lf_header, buf);

	CPOUT_16(leaf, str, lf_depth);
	CPOUT_16(leaf, str, lf_entries);
	CPOUT_32(leaf, str, lf_dirent_format);
	CPOUT_64(leaf, str, lf_next);

	CPOUT_08(leaf, str, lf_reserved, 64);
}

/**
 * ogfs_leaf_print - Print out a directory leaf header
 * @lf: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_leaf_print(ogfs_leaf_t * lf, int console)
{
	ogfs_meta_header_print(&lf->lf_header, console);

	pv(lf, lf_depth, "%u");
	pv(lf, lf_entries, "%u");
	pv(lf, lf_dirent_format, "%u");
	pv(lf, lf_next, "%"PRIu64);

	pa(lf, lf_reserved, 64);
}

/**
 * ogfs_log_header_in - Read in a log header
 * @head: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_log_header_in(ogfs_log_header_t * head, char *buf)
{
	ogfs_log_header_t *str = (ogfs_log_header_t *) buf;

	ogfs_meta_header_in(&head->lh_header, buf);

	CPIN_32(head, str, lh_flags);
	CPIN_32(head, str, lh_pad);

	CPIN_64(head, str, lh_first);
	CPIN_64(head, str, lh_sequence);

	CPIN_64(head, str, lh_tail);
	CPIN_64(head, str, lh_last_nopd);

	CPIN_08(head, str, lh_reserved, 64);
}

/**
 * ogfs_log_header_out - Write out a log header
 * @head: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_log_header_out(ogfs_log_header_t * head, char *buf)
{
	ogfs_log_header_t *str = (ogfs_log_header_t *) buf;

	ogfs_meta_header_out(&head->lh_header, buf);

	CPOUT_32(head, str, lh_flags);
	CPOUT_32(head, str, lh_pad);

	CPOUT_64(head, str, lh_first);
	CPOUT_64(head, str, lh_sequence);

	CPOUT_64(head, str, lh_tail);
	CPOUT_64(head, str, lh_last_nopd);

	CPOUT_08(head, str, lh_reserved, 64);
}

/**
 * ogfs_log_header_print - Print out a log header
 * @head: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_log_header_print(ogfs_log_header_t * lh, int console)
{
	ogfs_meta_header_print(&lh->lh_header, console);

	pv(lh, lh_flags, "0x%.8X");
	pv(lh, lh_pad, "%u");

	pv(lh, lh_first, "%"PRIu64);
	pv(lh, lh_sequence, "%"PRIu64);

	pv(lh, lh_tail, "%"PRIu64);
	pv(lh, lh_last_nopd, "%"PRIu64);

	pa(lh, lh_reserved, 64);
}

/**
 * ogfs_desc_in - Read in a log descriptor
 * @desc: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_desc_in(ogfs_log_descriptor_t * desc, char *buf)
{
	ogfs_log_descriptor_t *str = (ogfs_log_descriptor_t *) buf;

	ogfs_meta_header_in(&desc->ld_header, buf);

	CPIN_32(desc, str, ld_type);
	CPIN_32(desc, str, ld_length);
	CPIN_32(desc, str, ld_data1);
	CPIN_32(desc, str, ld_data2);

	CPIN_08(desc, str, ld_reserved, 64);
}

/**
 * ogfs_desc_out - Write out a log descriptor
 * @desc: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_desc_out(ogfs_log_descriptor_t * desc, char *buf)
{
	ogfs_log_descriptor_t *str = (ogfs_log_descriptor_t *) buf;

	ogfs_meta_header_out(&desc->ld_header, buf);

	CPOUT_32(desc, str, ld_type);
	CPOUT_32(desc, str, ld_length);
	CPOUT_32(desc, str, ld_data1);
	CPOUT_32(desc, str, ld_data2);

	CPOUT_08(desc, str, ld_reserved, 64);
}

/**
 * ogfs_desc_print - Print out a log descriptor
 * @ld: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_desc_print(ogfs_log_descriptor_t * ld, int console)
{
	ogfs_meta_header_print(&ld->ld_header, console);

	pv(ld, ld_type, "%u");
	pv(ld, ld_length, "%u");
	pv(ld, ld_data1, "%u");
	pv(ld, ld_data2, "%u");

	pa(ld, ld_reserved, 64);
}

/**
 * ogfs_tag_in - Read in a block tag
 * @tag: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_tag_in(ogfs_block_tag_t * tag, char *buf)
{
	ogfs_block_tag_t *str = (ogfs_block_tag_t *) buf;

	CPIN_64(tag, str, bt_blkno);
	CPIN_32(tag, str, bt_flags);
	CPIN_32(tag, str, bt_pad);
}

/**
 * ogfs_tag_out - Write out a block tag
 * @tag: the cpu-order structure
 * @buf: the disk-order buffer
 *
 */
void
ogfs_tag_out(ogfs_block_tag_t * tag, char *buf)
{
	ogfs_block_tag_t *str = (ogfs_block_tag_t *) buf;

	CPOUT_64(tag, str, bt_blkno);
	CPOUT_32(tag, str, bt_flags);
	CPOUT_32(tag, str, bt_pad);
}

/**
 * ogfs_tag_print - Print out a block tag
 * @tg: the cpu-order buffer
 * @console - TRUE if this should be printed to the console,
 *            FALSE if it should be just printed to the incore debug
 *            buffer
 *
 */
void
ogfs_tag_print(ogfs_block_tag_t * tg, int console)
{
	pv(tg, bt_blkno, "%"PRIu64);
	pv(tg, bt_flags, "%u");
	pv(tg, bt_pad, "%u");
}
