/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libs390.so
 *
 *   File: options.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>

#include "vtoc.h"
#include "390segmgr.h"
#include "helpers.h"
#include "options.h"
#include "format.h"
#include "move.h"

typedef struct disk_type_info_s {
        char *name;
} disk_type_info_t;


static const disk_type_info_t disk_types[] =
{
        {"CDL"},
        {"LDL"},
        {NULL}                    /* end-of-list */
};


/*
 *  Called by init_task() to allocate the create option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_move_option_descriptors( task_context_t * context )
{
        int rc;

        LOG_ENTRY();
        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to initialize the MOVE options in the task context
 *  for the specified LOGICALDISK.
 */
static int initialize_move_option_descriptors( DISKSEG *seg, task_context_t * context)
{
        int rc;

        LOG_ENTRY();
        rc=0;
        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Called to set and validate an option for a MOVE task.
 */
static int set_move_option( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            task_effect_t  * effect )
{
        int               rc=EINVAL;
        DISKSEG          *seg=NULL;
        LOGICALDISK      *ld=NULL;


        LOG_ENTRY();

        if (index == SEG_MOVE_OPTION_INDEX ) {

                seg = EngFncs->first_thing( context->selected_objects, NULL );
                if ( seg ) {

                        ld = get_logical_disk( seg );

                        if ( ld != NULL &&
                             seg->object_type == SEGMENT &&
                             seg->data_type   == DATA_TYPE ) {

                                if ( strlen(value->s) > 0 ||
                                     strlen(value->s) <= EVMS_NAME_SIZE ) {

                                        strncpy( context->option_descriptors->option[index].value.s,
                                                 value->s,
                                                 EVMS_NAME_SIZE );
                                        rc = 0;

                                }

                        }
                        else {
                                LOG_ERROR("error, segment cannot be moved.\n");
                        }

                }
                else {
                        LOG_ERROR("error, no object in selected object list.\n");
                }
        }
        else {
                LOG_ERROR("error, invalid move option index.\n");
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function: set move object
 *
 *  Called by set_object() to validate the task selected
 *  object.
 */
static int set_move_object( task_context_t * context,
                            list_anchor_t          declined_objects,
                            task_effect_t  * effect )
{
        int  rc=EINVAL;
        DISKSEG           *seg;
        boolean            found_good_object = FALSE;
        declined_object_t *declined_object=NULL;
        list_element_t     iter,e;

        LOG_ENTRY();

        LIST_FOR_EACH( context->selected_objects, iter, seg ) {

                if ( found_good_object == FALSE ) {

                        found_good_object = TRUE;

                        rc = initialize_move_option_descriptors(seg, context);

                        if (rc == 0) {
                                *effect |=  EVMS_Effect_Reload_Options;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = seg;
                                declined_object->reason = EINVAL;

                                e = EngFncs->insert_thing( declined_objects,
                                                           declined_object,
                                                           INSERT_AFTER,
                                                           NULL );
                                if (e!=NULL) {
                                        rc=0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: prune unmoveable segs
 *
 *  Called by get_acceptable_move_objects() to prune
 *  any segment that cannot be moved from a list of
 *  segments.
 */
static void prune_invalid_move_targets( DISKSEG *src, list_anchor_t list )
{
        LOGICALDISK *ld1=NULL;
        LOGICALDISK *ld2=NULL;
        list_element_t iter, iter2;
        int rc=0;
        boolean prune;
        DISKSEG *trg;

        ld1 = get_logical_disk(src);

        LIST_FOR_EACH_SAFE( list, iter, iter2, trg ) {

                prune = TRUE;

                ld2 = get_logical_disk(trg);

                if ( ld1 != NULL && ld1 == ld2 ) {
                        rc = s390_validate_move_target(src,trg);
                        if (rc == 0) {
                                prune=FALSE;
                        }
                }

                if (prune==TRUE) EngFncs->delete_element(iter);
        }

}


/*
 *  Function: get acceptable move objects
 *
 *  Called when setting up a move task to go out and find all the
 *  data segments that we could move on their disks.
 */
static int get_acceptable_move_objects( task_context_t * context )
{
        int   rc = EINVAL;

        LOG_ENTRY();

        if ( (context) &&
             (context->object) &&
             (context->acceptable_objects) ) {

                if ( EngFncs->list_count(context->acceptable_objects) == 0) {

                        rc = EngFncs->get_object_list( SEGMENT,
                                                            FREE_SPACE_TYPE,
                                                            NULL,
                                                            NULL,
                                                            TOPMOST,
                                                            &context->acceptable_objects );

                        if (rc == 0) {
                                prune_invalid_move_targets( context->object, context->acceptable_objects );
                        }

                }
                else {
                        LOG_ERROR("error, context already has acceptable objects\n");
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called by init_task() to allocate the assign option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_format_option_descriptors( task_context_t * context )
{
        int rc = EINVAL;


        LOG_ENTRY();

        if (context) {

                context->option_descriptors->count = SEG_FORMAT_OPTION_COUNT ;      // must set count

                // Disk Type
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].flags = 0; // EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].constraint.list =
                EngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t) * (sizeof(disk_types)/sizeof(struct disk_type_info_s)));

                if (context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].constraint.list) {
                        int i=0;

                        while (disk_types[i].name) {
                                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].constraint.list->value[i].s = EngFncs->engine_strdup(disk_types[i].name);
                                i++;
                        }

                        context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].constraint.list->count = i;
                }
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].constraint_type = EVMS_Collection_List;
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].help = NULL;
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].name = EngFncs->engine_strdup(SEG_FORMAT_OPTION_TYPENAME_NAME);
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].tip = EngFncs->engine_strdup( _("Choose between LDL (Linux Disk Layout) and CDL (Compatibility Disk Layout). CDL will allow you to create and resize partitions on the disk.  LDL provides a single default partition on the disk.") );
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].title = EngFncs->engine_strdup( _("Disk Layout") );
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].type = EVMS_Type_String;
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].min_len = 1;
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].max_len = MAX_TYPENAME_SIZE;
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].value.s = EngFncs->engine_alloc (MAX_TYPENAME_SIZE+1);

                if (context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].value.s)
                        strcpy (context->option_descriptors->option[SEG_FORMAT_OPTION_TYPENAME_INDEX].value.s, "CDL");
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }



                // Block Size
                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].flags = 0; //EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].constraint.list =
                EngFncs->engine_alloc(sizeof(value_list_t) + (sizeof(value_t) * 4) );

                if (context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].constraint.list) {
                        context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].constraint.list->count = 4;

                        context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].constraint.list->value[0].ui32 = 512;
                        context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].constraint.list->value[1].ui32 = 1024;
                        context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].constraint.list->value[2].ui32 = 2048;
                        context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].constraint.list->value[3].ui32 = 4096;
                }
                else {
                        rc = ENOMEM;
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].constraint_type = EVMS_Collection_List;
                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].help = NULL;
                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].name = EngFncs->engine_strdup( SEG_FORMAT_OPTION_BLKSIZE_NAME );
                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].tip = EngFncs->engine_strdup( _("Choose between the allowable block sizes (hard sector size) for the disk you are formatting.") );
                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].title = EngFncs->engine_strdup( _("Block Size") );
                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].type = EVMS_Type_Unsigned_Int32;
                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].unit = EVMS_Unit_None;
                context->option_descriptors->option[SEG_FORMAT_OPTION_BLKSIZE_INDEX].value.ui32 = 4096;


                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Called to set and validate an option for a FORMAT task.
 */
static int set_format_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int        rc        =        EINVAL;


        LOG_ENTRY();

        switch (index) {

        case SEG_FORMAT_OPTION_TYPENAME_INDEX:

                if ( strlen(value->s) > 0 && strlen(value->s) <= MAX_TYPENAME_SIZE) {

                        strcpy( context->option_descriptors->option[index].value.s, value->s);
                        rc = 0;

                }
                break;

        case SEG_FORMAT_OPTION_BLKSIZE_INDEX:

                if ( value->ui32 == 512 ||
                     value->ui32 == 1024 ||
                     value->ui32 == 2048 ||
                     value->ui32 == 4096 ) {

                        context->option_descriptors->option[index].value.ui32 = value->ui32;
                        rc = 0;

                }
                break;

        default:
                LOG_ERROR("index is unknown or unsupported\n");
                rc = EINVAL;
                break;
        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to initialize the CREATE segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_create_option_descriptors( LOGICALDISK *ld, DISKSEG *freespace, task_context_t * context)
{
        int                 rc=EINVAL;
        lba_t               Start;
        lba_t               End;
        sector_count_t      MaxSize;
        sector_count_t      CylinderSize=0;
        DISK_PRIVATE_DATA  *disk_pdata;


        LOG_ENTRY();

        disk_pdata = get_s390_disk_private_data( ld );
        if (disk_pdata) {
                if (disk_pdata->signature == S390_SEG_MGR_PDATA_SIGNATURE) {

                        // we should be able to get the cylinder size if there are no problems with disk geometry
                        CylinderSize = get_cylinder_size(ld);
                        if (CylinderSize ) {

                                // there should be exactly SEG_CREATE_OPTION_COUNT options
                                if ( context->option_descriptors->count == SEG_CREATE_OPTION_COUNT) {
                                        rc = 0;
                                }
                                else {
                                        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                                }

                        }
                        else {
                                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
                        }

                }
        }

        if (rc == 0) {

                // determine defaults based on geometry
                Start    = freespace->start;
                End      = rounddown_to_cylinder_boundary( ld, freespace->start + freespace->size ) - 1;
                MaxSize  = End - Start + 1;

                if ( MaxSize >= CylinderSize ) {

                        if (rc == 0) {

                                // Segment Size
                                if ( MaxSize >= CylinderSize ) {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = 0;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = CylinderSize;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = CylinderSize;
                                        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = CylinderSize;
                                }

                                // Sector Offset to start of Partition
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].flags = EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->min.ui64       = 0;
                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64       = MaxSize-CylinderSize;

                                rc = 0;
                        }

                }
                else {
                        rc = EINVAL;
                        LOG_ERROR("error, trying to create a segment out of something smaller than a cylinder.\n");
                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to initialize the EXPAND segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_expand_option_descriptors( LOGICALDISK *ld, DISKSEG *freespace, task_context_t * context)
{
        int                rc=EINVAL;
        int64_t            MaxSize=0;
        int64_t            MinSize=0;
        sector_count_t     cylinder_size=0;
        DISK_PRIVATE_DATA *disk_pdata;
        lba_t              freespace_end_lba;


        LOG_ENTRY();

        disk_pdata = get_s390_disk_private_data( ld );
        if (disk_pdata) {

                if (disk_pdata->signature == S390_SEG_MGR_PDATA_SIGNATURE) {

                        // we should be able to get the cylinder size if there are no problems with disk geometry
                        cylinder_size = get_cylinder_size(ld);
                        if (cylinder_size ) {

                                // there should be exactly SEG_EXPAND_OPTION_COUNT options
                                if ( context->option_descriptors->count == SEG_EXPAND_OPTION_COUNT) {

                                        rc = 0;

                                }
                                else {
                                        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                                }

                        }
                        else {
                                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
                        }

                }
        }

        if (rc == 0) {

                // partitions end on a cylinder boundary. if freespace doesnt end on
                // a cylinder bdy then round it down to a cyl bdy before testing if
                // freespace can handle the ExpandBy.
                if (ends_on_cylinder_boundary(ld,freespace->start+freespace->size-1)) {
                        freespace_end_lba = freespace->start+freespace->size-1;
                }
                else {
                        freespace_end_lba = rounddown_to_cylinder_boundary(ld, freespace->start+freespace->size-1) - 1;
                }

                // make sure we can actually expand the DATA segment by at least 1 cylinder
                if ( (freespace_end_lba - freespace->start + 1) >= cylinder_size ) {

                        LOG_DEBUG("freespace= %s   size= %"PRIu64"\n", freespace->name, freespace->size );

                        // get minimum and maximum sizes for expanding this DATA segment
                        MaxSize  = freespace_end_lba - freespace->start + 1;
                        MinSize  = cylinder_size;

                        // ask engine how much we can expand by
                        while (MaxSize > 0) {

                                sector_count_t delta_size = MaxSize;

                                rc = EngFncs->can_expand_by(context->object, &delta_size);
                                if (rc == 0) {
                                        break;
                                }
                                else {
                                        MaxSize -= cylinder_size;
                                }
                        }

                        if (MaxSize > 0) {

                                context->option_descriptors->count = SEG_EXPAND_OPTION_COUNT;
                                LOG_DEBUG("Setting constraints ... Max= %"PRIu64"   Min= %"PRIu64"\n", MaxSize, MinSize );
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->min.ui64       = MinSize;
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = cylinder_size;
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].value.ui64 = MinSize;
                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].flags = 0;

                                rc = 0;
                        }
                        else {
                                rc = EINVAL;
                                LOG_ERROR("error, engine said segment could not be expanded\n");
                        }

                }
                else {
                        rc = EINVAL;
                        LOG_ERROR("error trying to expand segment into freespace < 1 cylinder.\n");
                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  Called to initialize the SHRINK segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_shrink_option_descriptors( LOGICALDISK *ld, DISKSEG *seg, task_context_t * context)
{
        int                rc=EINVAL;
        sector_count_t     cylinder_size=0;
        DISK_PRIVATE_DATA *disk_pdata;
        sector_count_t     biggest_delta=0;
        sector_count_t     smallest_delta=0;

        LOG_ENTRY();

        disk_pdata = get_s390_disk_private_data( ld );
        if (disk_pdata) {

                if (disk_pdata->signature == S390_SEG_MGR_PDATA_SIGNATURE) {

                        // we should be able to get the cylinder size if there are no problems with disk geometry
                        cylinder_size = get_cylinder_size(ld);
                        if (cylinder_size ) {

                                // there should be exactly SEG_SHRINK_OPTION_COUNT options
                                if ( context->option_descriptors->count == SEG_SHRINK_OPTION_COUNT ) {
                                        rc = 0;
                                }
                                else {
                                        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                                }

                        }
                        else {
                                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
                        }

                }

        }

        if (rc == 0) {

                // get minimum and maximum sizes for shrinking this DATA segment
                if ( seg->size > cylinder_size ) {

                        biggest_delta  = seg->size - cylinder_size;
                        smallest_delta = cylinder_size;

                        // ask engine how much we can shrink by ... when the
                        // engine says stop ... check if the engine will allow
                        // any shrink at all.
                        do {
                                sector_count_t delta_size = biggest_delta;

                                rc = EngFncs->can_shrink_by( seg, &delta_size );

                                if (rc == 0) {
                                        break;   // engine said Ok
                                }

                                biggest_delta -= cylinder_size;

                        } while ( biggest_delta >= smallest_delta ); // continue till engine says Ok or we cant shrink at all

                }
                else {  // cant shrink this segment !
                        rc = EINVAL;
                }


                // make sure we can actually shrink the DATA segment
                if ( rc == 0 ) {

                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->min.ui64       = smallest_delta;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->max.ui64       = biggest_delta;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = cylinder_size;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].flags = 0;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].value.ui64 = smallest_delta;

                        rc = 0;

                }


        }


        LOG_EXIT_INT(rc);
        return rc;
}




/*
 *  Called by init_task() to allocate the assign option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_assign_option_descriptors( task_context_t * context )
{
        int rc = ENOSYS;
        LOG_ENTRY();
        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called by init_task() to allocate the create option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_create_option_descriptors( task_context_t * context )
{
        int rc = EINVAL;


        LOG_ENTRY();

        if (context) {

                context->option_descriptors->count = SEG_CREATE_OPTION_COUNT ;      // must set count

                // Segment Size
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.list = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                if (context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range==NULL) {
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint_type                  = EVMS_Collection_Range;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = 0;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = 0xffffffff;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 1;

                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_SIZE_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].tip = EngFncs->engine_strdup( _("This option allows you to set the size of the segment that you are creating.") );
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].title = EngFncs->engine_strdup( _("Size") );
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;
                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = 0;

                // Sector Offset to start of Partition
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                if (context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range==NULL) {
                        LOG_EXIT_INT(ENOMEM);
                        return ENOMEM;
                }
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint_type                  = EVMS_Collection_Range;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->min.ui64       = 0;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64       = 1;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->increment.ui64 = 1;


                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE | EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].help = NULL;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].name = EngFncs->engine_strdup( SEG_CREATE_OPTION_OFFSET_NAME );
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].tip =
			EngFncs->engine_strdup( _("Use this option if you don't want the segment you are creating to start at the beginning of freespace. "
						  "It allows you to specifiy the number of sectors to skip before creating the segment.") );
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].title = EngFncs->engine_strdup( _("Offset"));
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].type = EVMS_Type_Unsigned_Int64;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].unit = EVMS_Unit_Sectors;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = 0;

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Called by init_task() to allocate the expand option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int  allocate_expand_option_descriptors( task_context_t * context )
{
        int rc = EINVAL;
        storage_object_t *obj;
        LOGICALDISK      *ld=NULL;
        DISKSEG          *freespace=NULL;
        sector_count_t    cylinder_size;


        LOG_ENTRY();

        context->option_descriptors->count = 0;

        obj = context->object;
        if ( obj ) {

                if (obj->object_type == SEGMENT) {

                        if (obj->data_type == DATA_TYPE) {

                                ld = get_logical_disk(obj);
                                if (ld) {

                                        // we increment partition sizes ... 1 cylinder at a time ... due to forced cyl allignment
                                        cylinder_size = get_cylinder_size(ld);

                                        // DATA segment better have freespace following it.
                                        freespace = get_freespace_following_s390_segment( obj );
                                        if (freespace == NULL) {
                                                rc = EINVAL;
                                                LOG_EXIT_INT(rc);
                                                return rc;
                                        }

                                        // make sure we can actually expand the DATA segment by at least 1 cylinder
                                        if ( freespace->size >= cylinder_size ) {

                                                context->option_descriptors->count = SEG_EXPAND_OPTION_COUNT;

                                                // Expanded segment size delta
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.list = NULL;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                                                if (context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range==NULL) {
                                                        LOG_EXIT_INT(ENOMEM);
                                                        return ENOMEM;
                                                }
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].flags = 0;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].help = NULL;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].name = EngFncs->engine_strdup( SEG_EXPAND_OPTION_SIZE_NAME );
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].tip =
							EngFncs->engine_strdup( _("Use this option to specify how many sectors to add to the segment. "
										  "Remember that segments grow by cylinder size amounts.") );
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].title = EngFncs->engine_strdup( _("Size") );
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                                                context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;

                                                rc = 0;
                                        }

                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;

}


/*
 *  Called by init_task() to allocate the shrink option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_shrink_option_descriptors( task_context_t * context )
{
        int                rc = EINVAL;
        storage_object_t  *obj;
        LOGICALDISK       *ld;
        sector_count_t     cylinder_size;
        DISK_PRIVATE_DATA *disk_pdata;

        LOG_ENTRY();

        context->option_descriptors->count = 0;

        obj = context->object;

        if ( obj ) {

                if (obj->object_type == SEGMENT) {

                        if (obj->data_type == DATA_TYPE) {

                                ld = get_logical_disk( (DISKSEG *) obj );

                                disk_pdata = get_s390_disk_private_data(ld);

                                if (disk_pdata) {

                                        // we shrink partition sizes ... 1 cylinder at a time ... due to forced cyl allignment
                                        cylinder_size = get_cylinder_size(ld);

                                        // make sure we can actually shrink the DATA segment while falling
                                        // on a cylinder boundary.
                                        if ( obj->size > cylinder_size ) {

                                                context->option_descriptors->count = SEG_SHRINK_OPTION_COUNT ;

                                                // Shrunken segment size
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.list = NULL;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range = EngFncs->engine_alloc( sizeof(value_range_t) );
                                                if (context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range==NULL) {
                                                        LOG_EXIT_INT(ENOMEM);
                                                        return ENOMEM;
                                                }
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].flags = 0;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].help = NULL;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].name = EngFncs->engine_strdup( SEG_SHRINK_OPTION_SIZE_NAME );
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].tip =
							EngFncs->engine_strdup( _("Use this option to specify how many sectors to remove from the segment. "
										  "Remember that segments shrink by cylinder size amounts.") );
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].title = EngFncs->engine_strdup( _("Size") );
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                                                context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;

                                                rc = 0;
                                        }

                                }

                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to set and validate an option for a CREATE task.
 */
static int set_assign_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int rc = ENOSYS;
        LOG_ENTRY();
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to set and validate an option for a CREATE task.
 */
static int set_create_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int                rc = EINVAL;
        storage_object_t  *obj=NULL;
        LOGICALDISK       *ld=NULL;
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        sector_count_t     MaxSize = 0;
        sector_count_t     size = 0;
        lba_t              start_lba = 0;
        lba_t              end_lba = 0;
        lba_t              rounded_end_lba = 0;
        sector_count_t     CylinderSize=0;


        LOG_ENTRY();

        obj = EngFncs->first_thing( context->selected_objects, NULL );
        if (obj) {

                if (obj->data_type == META_DATA_TYPE) {
                        rc = ENODEV;
                        LOG_ERROR("error, i do not create segs from metadata type segments\n");
                        LOG_EXIT_INT(rc);
                        return rc;
                }

                // get disk info
                ld = get_logical_disk(obj);
                if (ld == NULL) {
                        rc = ENODEV;
                        LOG_ERROR("error, no disk info available\n");
                        LOG_EXIT_INT(rc);
                        return rc;
                }
                else {
                        disk_pdata = get_s390_disk_private_data(ld);
                        if (disk_pdata == NULL) {
                                rc = ENODATA;
                                LOG_ERROR("error, no disk private data available\n");
                                LOG_EXIT_INT(rc);
                                return rc;
                        }
                }


                CylinderSize = get_cylinder_size(ld);

                start_lba    = obj->start;
                end_lba      = obj->start + obj->size - 1;
                size         = obj->size;

                rounded_end_lba = rounddown_to_cylinder_boundary( ld, end_lba+1 ) - 1;

                MaxSize = rounded_end_lba - start_lba + 1;

        }
        else {
                rc = ENODEV;
                LOG_ERROR("error, no disk segment found in the selected_objects list\n");
                LOG_EXIT_INT(rc);
                return rc;
        }

        switch (index) {

        case SEG_CREATE_OPTION_SIZE_INDEX:
                {

                        sector_count_t segoffset = context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64;

                        // SIZE == 0 ... ERROR
                        if ( value->ui64 == 0) {
                                *effect |= EVMS_Effect_Inexact ;
                                rc = EINVAL;
                        }
                        // if OFFSET + SIZE < MAX_USEABLE_FREESPACE_SIZE ... then SIZE is OK
                        else if ( value->ui64+segoffset <= MaxSize ) {

                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                                *effect |= EVMS_Effect_Reload_Options;
                                context->option_descriptors->option[index].value.ui64 = value->ui64;
                                rc = 0;

                        }
                        // user is trying to create something that runs off end
                        // of the useable area of the freespace segment.
                        else {

                                if (segoffset > 0) { // OFFSET and SIZE both need to be examined
                                                     // to determine how to fix things up.

                                        if (segoffset > obj->size) {     // OFFSET is toooo big ... ERROR

                                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                                                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = MaxSize - value->ui64;
                                                context->option_descriptors->option[index].value.ui64 = value->ui64;

                                                *effect |= EVMS_Effect_Reload_Options ;
                                                rc = 0;

                                        }
                                        else {   // OFFSET is OK ... perhaps big ... so reduce SIZE
                                                 // and let the user figure things out.

                                                sector_count_t newsize = ((obj->size - segoffset)/CylinderSize)*CylinderSize;

                                                context->option_descriptors->option[index].constraint.range->max.ui64 = newsize;
                                                context->option_descriptors->option[index].value.ui64 = newsize;
                                                *effect |= EVMS_Effect_Reload_Options ;
                                                rc = 0;
                                        }

                                }
                                else {  // OFFSET == 0 ... so SIZE is the only option we need
                                        // to adjust downward for user.
                                        context->option_descriptors->option[index].value.ui64 = MaxSize;
                                        *effect |= EVMS_Effect_Inexact;
                                        rc = 0;
                                }

                        }


                }

                break;

        case SEG_CREATE_OPTION_OFFSET_INDEX:

                {
                        sector_count_t segsize = context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64;

                        // if OFFSET + SIZE < MAX_USEABLE_FREESPACE_SIZE ... then OFFSET is OK
                        if ( value->ui64+segsize <= MaxSize ) {

                                context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                                context->option_descriptors->option[index].value.ui64 = value->ui64;
                                *effect |= EVMS_Effect_Reload_Options ;
                                rc = 0;
                        }
                        else {

                                context->option_descriptors->option[index].constraint.range->max.ui64 = MaxSize - segsize;
                                context->option_descriptors->option[index].value.ui64 = MaxSize - segsize;
                                *effect |= EVMS_Effect_Reload_Options  ;
                                rc = 0;
                        }

                }

                break;


        default:
                LOG_ERROR("error, index is unknown or unsupported\n");
                rc = EINVAL;
                break;
        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Called to set and validate an option for a EXPAND task.
 */
static int set_expand_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int               rc=EINVAL;
        storage_object_t *obj=NULL;
        LOGICALDISK      *ld=NULL;
        DISKSEG          *freespace=NULL;
        DISKSEG          *seg=NULL;
        sector_count_t    cylinder_size;
        lba_t             end_lba = 0;
        sector_count_t    expand_sectors=0;

        LOG_ENTRY();

        if (index == SEG_EXPAND_OPTION_SIZE_INDEX ) {

                obj = EngFncs->first_thing( context->selected_objects, NULL );
                if ( obj ) {

                        if (obj->object_type == SEGMENT) {

                                if (obj->data_type == FREE_SPACE_TYPE) {

                                        seg = context->object;

                                        ld = get_logical_disk( (DISKSEG *) seg );

                                        if (ld) {

                                                freespace = get_freespace_following_s390_segment( seg );

                                                if ( freespace == obj ) {

                                                        // calc the minimum number of sectors we need for a partition
                                                        cylinder_size = get_cylinder_size(ld);

                                                        // get expand sector count
                                                        expand_sectors = value->ui64;

                                                        LOG_DEBUG("Info ...\n");
                                                        LOG_DEBUG("     segment  = %s\n", obj->name );
                                                        LOG_DEBUG("     start    = %"PRIu64"\n", obj->start );
                                                        LOG_DEBUG("     end      = %"PRIu64"\n", obj->start+obj->size-1);
                                                        LOG_DEBUG("     size     = %"PRIu64"\n", obj->size );
                                                        LOG_DEBUG("     expand sectors = %"PRIu64"\n", expand_sectors );
                                                        LOG_DEBUG("     cylinder size  = %"PRIu64"\n", cylinder_size );

                                                        // we can only expand if in cylinder size amounts so the freespace
                                                        // following the expanding segment must be at least 1 cylinder.
                                                        // and caller should be asking for at least a cylinder
                                                        if ( ( freespace->size >= cylinder_size ) &&
                                                             ( expand_sectors >= cylinder_size ) &&
                                                             ( expand_sectors <= freespace->size ) ) {

                                                                // do cylinder allignment
                                                                end_lba = seg->start + seg->size + expand_sectors - 1;
                                                                if (ends_on_cylinder_boundary(ld, end_lba) == FALSE) {
                                                                        end_lba = roundup_to_cylinder_boundary( ld, end_lba );
                                                                }

                                                                // adjust downwards if too big ...
                                                                if ( end_lba > (freespace->start + freespace->size - 1) ) {
                                                                        end_lba = roundup_to_cylinder_boundary(ld, end_lba - cylinder_size );
                                                                }

                                                                // NOW ... test if we can expand the data seg into the freespace area
                                                                if ( ( end_lba > freespace->start ) &&
                                                                     ( end_lba <= freespace->start+freespace->size - 1) ) {

                                                                        // calc actual expand sector count
                                                                        expand_sectors = end_lba - freespace->start + 1;

                                                                        if (value->ui64 != expand_sectors) {
                                                                                *effect |= EVMS_Effect_Inexact;
                                                                        }

                                                                        context->option_descriptors->option[index].value.ui64 = expand_sectors;
                                                                        rc = 0;
                                                                }
                                                                else {
                                                                        LOG_ERROR("error, cannot expand segment ... still passes end of freespace\n");
                                                                }
                                                        }
                                                        else {
                                                                LOG_ERROR("error, invalid expand sector count\n");
                                                        }
                                                }
                                                else {
                                                        LOG_ERROR("error, selected freespace does not immediately follow data segment\n");
                                                }
                                        }
                                        else {
                                                LOG_ERROR("error, no logical drive found for segment\n");
                                        }
                                }
                                else {
                                        LOG_ERROR("error, selected freespace is NOT a freespace segment\n");
                                }
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called to set and validate an option for a SHRINK task.
 */
static int set_shrink_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
        int               rc=EINVAL;
        storage_object_t *obj=NULL;
        LOGICALDISK      *ld=NULL;
        sector_count_t    cylinder_size;
        sector_count_t    shrink_sector_count=0;
        lba_t             end_lba=0;


        LOG_ENTRY();

        if (index == SEG_SHRINK_OPTION_SIZE_INDEX ) {

                obj = EngFncs->first_thing( context->selected_objects, NULL );
                if ( obj ) {

                        if (obj->object_type == SEGMENT) {

                                if (obj->data_type == DATA_TYPE) {

                                        ld = get_logical_disk( (DISKSEG *) obj );

                                        if ( ld ) {

                                                // get option value
                                                shrink_sector_count = value->ui64;

                                                // calc the minimum number of sectors we need for a partition
                                                cylinder_size = get_cylinder_size(ld);

                                                LOG_DEBUG("Info ...\n");
                                                LOG_DEBUG("     segment  = %s\n", obj->name );
                                                LOG_DEBUG("     start    = %"PRIu64"\n", obj->start );
                                                LOG_DEBUG("     end      = %"PRIu64"\n", obj->start+obj->size-1);
                                                LOG_DEBUG("     size     = %"PRIu64"\n", obj->size );
                                                LOG_DEBUG("     shrink sectors = %"PRIu64"\n", shrink_sector_count );
                                                LOG_DEBUG("     cylinder size  = %"PRIu64"\n", cylinder_size );

                                                // we can only shrink in cylinder size amounts so the shrink
                                                // sectors must be >= 1 cylinder.
                                                if ( ( shrink_sector_count < obj->size ) &&
                                                     ( shrink_sector_count >= cylinder_size ) ) {

                                                        // do cylinder allignment
                                                        end_lba = obj->start + (obj->size - shrink_sector_count) - 1;
                                                        if (ends_on_cylinder_boundary(ld, end_lba) == FALSE) {
                                                                end_lba = rounddown_to_cylinder_boundary( ld, end_lba-1 ) - 1;
                                                        }

                                                        // adjust upwards if we shrunk too small ...
                                                        if ( end_lba <= obj->start ) {
                                                                end_lba = roundup_to_cylinder_boundary(ld, end_lba + 1 );
                                                        }

                                                        // final test if we can shrink the segment
                                                        if (  ( end_lba > obj->start ) &&
                                                              ( end_lba < (obj->start+obj->size-1)) ) {

                                                                shrink_sector_count = obj->size - (end_lba - obj->start + 1);

                                                                if ( value->ui64 != shrink_sector_count ) {
                                                                        value->ui64 = shrink_sector_count;
                                                                        *effect |= EVMS_Effect_Inexact;
                                                                }

                                                                context->option_descriptors->option[index].value.ui64 = shrink_sector_count;
                                                                rc = 0;
                                                        }
                                                        else {
                                                                LOG_ERROR("error, invalid shrink sector count after cylinder rounding\n");
                                                        }
                                                }
                                                else {
                                                        LOG_ERROR("error, invalid shrink sector count\n");
                                                }
                                        }
                                        else {
                                                LOG_ERROR("error, logical drive is missing for this segment\n");
                                        }
                                }
                                else {
                                        LOG_ERROR("error, segment is NOT a data segment\n");
                                }

                        }
                        else {
                                LOG_ERROR("error, not a SEGMENT object type\n");
                        }

                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Get the user selected object from the assign context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the boolean flag found_good_object.
 */
static int set_assign_object( task_context_t * context,
                              list_anchor_t          declined_objects,
                              task_effect_t  * effect )
{
        int rc=ENOSYS;
        LOG_ENTRY();
        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Get the user selected object from the create context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the boolean flag found_good_object.
 */
static int set_create_object( task_context_t * context,
                              list_anchor_t          declined_objects,
                              task_effect_t  * effect )
{
        int                 rc=EINVAL;
        storage_object_t   *obj;
        LOGICALDISK        *ld;
        int                 found_good_object = FALSE;
        declined_object_t  *declined_object=NULL;
        list_element_t      iter, e;

        LOG_ENTRY();

        LIST_FOR_EACH( context->selected_objects, iter, obj ) {

                if ( ( obj->data_type == FREE_SPACE_TYPE ) &&
                     ( obj->plugin    == s390_plugin_rec) &&
                     ( found_good_object == FALSE ) ) {

                        ld = get_logical_disk(obj);

                        if (ld) {

                                found_good_object = TRUE;
                                rc = initialize_create_option_descriptors(ld, obj, context);

                                if (rc == 0) {
                                        *effect |=  EVMS_Effect_Reload_Options;
                                }

                        }
                        else {
                                rc = ENODEV;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                e = EngFncs->insert_thing( declined_objects,
                                                           declined_object,
                                                           INSERT_AFTER,
                                                           NULL );
                                if (e!=NULL) {
                                        rc=0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 *
 * Context->object == expansion point
 * Context->selected_objects->object == freespace segment
 * Context->acceptable_objects->object == freespace segment chosen during inittask time
 *
 */
static int set_expand_object( task_context_t * context,
                              list_anchor_t          declined_objects,
                              task_effect_t  * effect )
{
        int  rc=EINVAL;
        storage_object_t  *obj;
        storage_object_t  *freespace;
        LOGICALDISK       *ld;
        boolean            found_good_object = FALSE;
        declined_object_t *declined_object=NULL;
        list_element_t   iter,e;

        LOG_ENTRY();

        LIST_FOR_EACH( context->selected_objects, iter, obj ) {

                if ( ( obj->data_type == FREE_SPACE_TYPE ) &&
                     ( obj->plugin    == s390_plugin_rec) &&
                     ( found_good_object == FALSE ) ) {

                        freespace = get_freespace_following_s390_segment( context->object );

                        ld = get_logical_disk(obj);

                        if (ld && freespace) {

                                found_good_object = TRUE;

                                // is selected object correct?
                                if (freespace == obj) {

                                        rc = initialize_expand_option_descriptors(ld, obj, context);

                                        if (rc == 0) {
                                                *effect |= EVMS_Effect_Reload_Options;
                                        }

                                }
                                else {
                                        LOG_ERROR("error, selected object is not freespace that follows the data segment\n");
                                        rc = EINVAL;
                                }

                        }
                        else {
                                rc = ENODEV;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                e = EngFncs->insert_thing( declined_objects,
                                                           declined_object,
                                                           INSERT_AFTER,
                                                           NULL );
                                if (e!=NULL) {
                                        rc=0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("error, unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

        }



        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int set_shrink_object( task_context_t * context,
                              list_anchor_t          declined_objects,
                              task_effect_t  * effect )
{
        int  rc=EINVAL;
        storage_object_t  *obj;
        LOGICALDISK       *ld;
        boolean            found_good_object = FALSE;
        declined_object_t *declined_object=NULL;
        list_element_t   iter,e;

        LOG_ENTRY();

        LIST_FOR_EACH( context->selected_objects, iter, obj ) {

                if ( ( obj->data_type == DATA_TYPE ) &&
                     ( obj->plugin    == s390_plugin_rec) &&
                     ( found_good_object == FALSE ) ) {

                        ld = get_logical_disk(obj);

                        if (ld) {

                                found_good_object = TRUE;

                                rc = initialize_shrink_option_descriptors(ld, obj, context);

                                if (rc == 0) {
                                        *effect |=  EVMS_Effect_Reload_Options;
                                }

                        }
                        else {
                                rc = ENODEV;
                        }

                }
                else {

                        declined_object = EngFncs->engine_alloc( sizeof(declined_object_t));

                        if (declined_object) {

                                declined_object->object = obj;
                                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                                e = EngFncs->insert_thing( declined_objects,
                                                           declined_object,
                                                           INSERT_AFTER,
                                                           NULL );
                                if (e!=NULL) {
                                        rc=0;
                                        *effect |=  EVMS_Effect_Reload_Objects;
                                }
                                else {
                                        rc = EPERM;
                                        EngFncs->engine_free(declined_object);
                                }

                        }
                        else {
                                LOG_ERROR("unable to malloc a declined object struct\n");
                                rc = ENOMEM;
                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}




/*
 * Validate the objects in the selected_objects list in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects list.
 * Modify the accepatble_objects list in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int S390_SetObjects( task_context_t * context,
                     list_anchor_t          declined_objects,
                     task_effect_t  * effect )
{

        int rc = EINVAL;

        LOG_ENTRY();

        if (context) {

                switch (context->action) {

                case EVMS_Task_Create:

                        rc = set_create_object( context, declined_objects, effect );
                        break;

                case  EVMS_Task_Assign_Plugin:

                        rc = set_assign_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_Expand:

                        rc = set_expand_object( context, declined_objects, effect );
                        break;

                case EVMS_Task_Shrink:

                        rc = set_shrink_object( context, declined_objects, effect );
                        break;

                case  EVMS_Task_Format_Disk:
                        /* A "format" task doesn't need any selected objects.
                         * The disk to be formatted is always the child of the
                         * context object.
                         */
                        break;

                case EVMS_Task_Move_Segment:

                        rc = set_move_object( context, declined_objects, effect );
                        break;

                default:

                        LOG_ERROR("context->action is unknown or unsupported\n");
                        break;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_assign_objects( task_context_t * context )
{
        int   rc = 0;

        LOG_ENTRY();

        // no objects are acceptable to the 390 seg mgr ... it
        // doesn't support assign or unassign.
        EngFncs->delete_all_elements( context->acceptable_objects );

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Called from get_acceptable_create_objects, once for each freespace
 *  segment we got back from the kernel. The purpose is to weed out
 *  any FREESPACE segment that we cannot create a DATA segment from.
 *  This is so we only present FREESPACE segments that are truly acceptable
 *  for CREATE.
 */
static void prune_bad_freespace_segs( list_anchor_t list )
{
        LOGICALDISK       *ld;
        DISKSEG *seg;
        DISK_PRIVATE_DATA *disk_pdata;
        sector_count_t     CylinderSize;
        lba_t              Start;
        lba_t              End;
        sector_count_t     MaxSize;
        list_element_t     iter1, iter2;
        boolean            prune;

	LOG_ENTRY();

        LIST_FOR_EACH_SAFE( list, iter1, iter2, seg ) {

                prune = TRUE;

                ld = get_logical_disk(seg);
                if (ld) {

                        disk_pdata = get_s390_disk_private_data(ld);

                        if ( !( ld->flags & DISK_HAS_FORMAT_PENDING )  &&         // disk doesnt need formatting
                             ( i_can_add_partition_to_vtoc( ld ) == TRUE )) {    // vtoc has room for another partition

                                CylinderSize = get_cylinder_size(ld);

                                Start = seg->start;

                                if ( ends_on_cylinder_boundary( ld, seg->start + seg->size - 1 ) == TRUE ) {
                                        End = seg->start + seg->size - 1;
                                }
                                else if (seg->size > CylinderSize) {
                                        End = rounddown_to_cylinder_boundary( ld, seg->start + seg->size - 1 ) - 1;
                                }
                                else {
                                        End = Start; // force failure ... cuz we cant round down this seg to cyl bdy
                                }

                                if (Start < End) {

                                        MaxSize  = End - Start + 1;

                                        // if this seg is at least 1 cylinder (minus a track) in size then it is Ok
                                        // because they can start on track boundaries.
                                        if ( MaxSize >= CylinderSize ) {

                                                // make sure the disk will allow another partition on it.
                                                if (disk_pdata->partition_count < disk_pdata->max_partitions_allowed) {

                                                        prune = FALSE;

                                                }

                                        }
                                        else {
                                                LOG_DEBUG("max size < cyl size\n");
                                        }

                                }

                        }

                }

                if (prune) EngFncs->delete_element(iter1);
        }

	LOG_EXIT_VOID();
}




/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_create_objects( task_context_t * context )
{
        int   rc = EINVAL;

        LOG_ENTRY();

        if (EngFncs->list_count(context->acceptable_objects) == 0) {

                rc = EngFncs->get_object_list( SEGMENT,
                                                    FREE_SPACE_TYPE,
                                                    s390_plugin_rec,
                                                    NULL,
                                                    VALID_INPUT_OBJECT,
                                                    &context->acceptable_objects );

                if (rc == 0) {
                        prune_bad_freespace_segs(context->acceptable_objects);
                }

        }
        else {
                LOG_ERROR("context already has acceptable objects\n");
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_expand_objects( task_context_t * context )
{
        int   rc = EINVAL;
        storage_object_t  *freespace;
        LOGICALDISK       *ld;
        sector_count_t     cylinder_size;
        lba_t              end_lba;
        DISK_PRIVATE_DATA *disk_pdata;
        list_element_t     e;

        LOG_ENTRY();

        if ( ( context ) &&
             ( context->acceptable_objects ) &&
             ( context->selected_objects ) &&
             ( context->object ) ) {

                if ( EngFncs->list_count(context->acceptable_objects) == 0 ) {   // cant start with any

                        freespace = get_freespace_following_s390_segment( context->object );
                        if ( freespace ) {

                                ld = get_logical_disk(context->object);
                                if (ld) {

                                        disk_pdata = get_s390_disk_private_data(ld);

                                        if ( disk_pdata->disk_layout == ibm_cdl ) {  // must be compatibility disk

                                                cylinder_size = get_cylinder_size(ld);

                                                if (freespace->size >= cylinder_size) {

                                                        end_lba = roundup_to_cylinder_boundary(ld, freespace->start + 1);

                                                        if ( end_lba <= freespace->start + freespace->size - 1) {

                                                                e=EngFncs->insert_thing( context->acceptable_objects,
                                                                                         freespace,
                                                                                         INSERT_AFTER | EXCLUSIVE_INSERT,
                                                                                         NULL );
                                                                if (e!=NULL) {
                                                                        rc = 0;
                                                                }
                                                                else {
                                                                        rc = EPERM;
                                                                }

                                                        }

                                                }

                                        }

                                }

                        }

                }

        }


        LOG_EXIT_INT(rc);
        return rc;
}



/*
 * Validate the objects in the source and target lists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * lists.  Also, for any object which is removed from the selected lists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects list.
 */
static int get_acceptable_shrink_objects( task_context_t * context )
{
        int   rc = EINVAL;
        list_element_t e;

        LOG_ENTRY();

        if ( (context) &&
             (context->object) &&
             (context->acceptable_objects) ) {

                if ( EngFncs->list_count(context->acceptable_objects) == 0 ) {

                        // only acceptable object is the focus object for the shrink
                        // task we are initializing.
                        e = EngFncs->insert_thing( context->acceptable_objects,
                                                   context->object,
                                                   INSERT_AFTER,
                                                   NULL );
                        if (e!=NULL) {
                                rc = 0;
                        }
                        else {
                                rc = EPERM;
                        }

                }
                else {
                        LOG_ERROR("error, context already has acceptable objects\n");
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Initialize a new task context by allocating the option descriptor
 *  array and by getting acceptable objects, from the engine, for the
 *  task context, i.e. create, shrink, expand, assign ...
 */
int S390_InitTask(task_context_t * context)
{
        int rc = EINVAL;

        LOG_ENTRY();

        if (context) {

                switch (context->action) {

                case EVMS_Task_Create:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_create_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_create_objects( context );
                        }

                        break;

                case EVMS_Task_Assign_Plugin:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_assign_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_assign_objects( context );
                        }

                        break;


                case EVMS_Task_Expand:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_expand_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_expand_objects( context );
                        }

                        break;

                case EVMS_Task_Shrink:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_shrink_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_shrink_objects( context );
                        }


                        break;

                case EVMS_Task_Format_Disk:

                        /* No selected objects for a "format" task. The selected
                         * object is implicitly the segment's child disk.
                         */
                        context->min_selected_objects = 0;
                        context->max_selected_objects = 0;

                        rc = allocate_format_option_descriptors( context );

                        break;

                case EVMS_Task_Move_Segment:

                        context->min_selected_objects = 1;
                        context->max_selected_objects = 1;

                        rc = allocate_move_option_descriptors( context );
                        if (rc == 0) {
                                rc = get_acceptable_move_objects( context );
                        }
                        break;

                default:
                        LOG_ERROR("error, context->action is unknown or unsupported\n");
                        break;
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}



/*
 *  Returns count of options for specified task
 */
int S390_GetOptionCount( task_context_t * task )
{
        int count;

        LOG_ENTRY();

        switch (task->action) {

        case EVMS_Task_Create:
                count = SEG_CREATE_OPTION_COUNT;
                break;

        case EVMS_Task_Assign_Plugin:
                count = SEG_ASSIGN_OPTION_COUNT;
                break;

        case EVMS_Task_Expand:
                count = SEG_EXPAND_OPTION_COUNT;
                break;

        case EVMS_Task_Shrink:
                count = SEG_SHRINK_OPTION_COUNT;
                break;

        case EVMS_Task_Format_Disk:
                count = SEG_FORMAT_OPTION_COUNT;
                break;

        case EVMS_Task_Move_Segment:
                count = SEG_MOVE_OPTION_COUNT;
                break;
        default:
                count = -1;
                break;
        }

        LOG_EXIT_INT(count);
        return count;
}



/*
 *
 */
int S390_SetOption( task_context_t * context,
                    u_int32_t        index,
                    value_t        * value,
                    task_effect_t  * effect )
{
        int rc=EINVAL;

        LOG_ENTRY();


        if (context ) {

                switch (context->action) {

                case EVMS_Task_Create:
                        rc = set_create_option( context, index, value, effect );
                        break;

                case EVMS_Task_Assign_Plugin:
                        rc = set_assign_option( context, index, value, effect );
                        break;

                case EVMS_Task_Expand:
                        rc = set_expand_option( context, index, value, effect );
                        break;

                case EVMS_Task_Shrink:
                        rc = set_shrink_option( context, index, value, effect );
                        break;

                case EVMS_Task_Format_Disk:
                        rc = set_format_option(context,index,value,effect);
                        break;

                case EVMS_Task_Move_Segment:
                        rc = set_move_option(context, index, value, effect );
                        break;

                default:
                        LOG_ERROR("error, context->action is unknown or unsupported\n");
                        break;
                }
        }

        LOG_EXIT_INT(rc);

        return rc;
}



/*
 * Returns segment specific information
 */
int S390_GetInfo( storage_object_t  * object, char * name, extended_info_array_t * * info)
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        LOGICALDISK             *ld = get_logical_disk(object);
        DISK_PRIVATE_DATA *disk_pdata=NULL;
        char   format_type[24];

        LOG_ENTRY();

        // a measure of protection ...
        if (info == NULL || ld == NULL) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        disk_pdata = get_s390_disk_private_data(ld);

        rc    = 0;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( 4 * sizeof(extended_info_t) ) );
        if (Info) {
                *info = Info;
                Info->count = 4;

                Info->info[0].name = EngFncs->engine_strdup( "Name" );
                Info->info[0].title = EngFncs->engine_strdup( _("Name") );
                Info->info[0].desc = EngFncs->engine_strdup( _("This is the partition name. It must be unique on the system.") );
                Info->info[0].type               = EVMS_Type_String;
                Info->info[0].unit               = EVMS_Unit_None;
                Info->info[0].value.s = EngFncs->engine_strdup( object->name );
                Info->info[0].collection_type    = EVMS_Collection_None;
                memset( &Info->info[0].group, 0, sizeof(group_info_t));

                Info->info[1].name = EngFncs->engine_strdup( "Size" );
                Info->info[1].title = EngFncs->engine_strdup( _("Size") );
                Info->info[1].desc = EngFncs->engine_strdup( _("This is the size of the partition in 512 byte sectors.  EVMS uses a common sector size of 512 bytes, making adjustments when the disk hardsector size is greater than 512 bytes.") );
                Info->info[1].type               = EVMS_Type_Unsigned_Int64;
                Info->info[1].unit               = EVMS_Unit_Sectors;
                Info->info[1].value.ui64         = object->size;
                Info->info[1].format             = EVMS_Format_Normal;
                Info->info[1].collection_type    = EVMS_Collection_None;
                memset( &Info->info[1].group, 0, sizeof(group_info_t));
                Info->info[1].flags |= EVMS_EINFO_FLAGS_NO_UNIT_CONVERSION;

                Info->info[2].name = EngFncs->engine_strdup( "Start" );
                Info->info[2].title = EngFncs->engine_strdup( _("Start Logical Block Aaddress") );
                Info->info[2].desc = EngFncs->engine_strdup( _("This is the sector offset of the partition on the disk, i.e., the logical block address of the first sector of the partition.") );
                Info->info[2].type               = EVMS_Type_Unsigned_Int64;
                Info->info[2].unit               = EVMS_Unit_None;
                Info->info[2].value.ui64         = object->start;
                Info->info[2].format             = EVMS_Format_Normal;
                Info->info[2].collection_type    = EVMS_Collection_None;
                memset( &Info->info[2].group, 0, sizeof(group_info_t));

                Info->info[3].name = EngFncs->engine_strdup( "Layout" );
                Info->info[3].title = EngFncs->engine_strdup( _("Disk Layout") );
                Info->info[3].desc = EngFncs->engine_strdup( _("This is the layout of the disk on which this partition is found. A CDL layout allows you to create/delete/resize partitions. Other disk layouts do not allow changes to partitions.") );
                Info->info[3].type               = EVMS_Type_String;
                Info->info[3].unit               = EVMS_Unit_None;
                Info->info[3].collection_type    = EVMS_Collection_None;
                memset( &Info->info[3].group, 0, sizeof(group_info_t));

                switch (disk_pdata->disk_layout) {
                case ibm_ldl:
                        strcpy(format_type, "LDL");
                        break;
                case ibm_cdl:
                        strcpy(format_type, "CDL");
                        break;
                case ibm_cms:
                        strcpy(format_type, "CMS");
                        break;
                case ibm_none:
                        strcpy(format_type, _("UNLABELED"));
                        break;
                default:
                        strcpy(format_type, "n/a");
                        break;
                }
                Info->info[3].value.s = EngFncs->engine_strdup( format_type );

        }
        else {
                rc = ENOMEM;
        }
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *
 */
int S390_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
        int rc = EINVAL;
        extended_info_array_t   *Info;
        char                     version_string[64];
        char                     required_engine_api_version_string[64];
        char                     required_plugin_api_version_string[64];



        LOG_ENTRY();

        // a measure of protection ...
        if (info == NULL) {
                LOG_EXIT_INT(rc);
                return rc;
        }

        rc    = ENOMEM;  // init to failed calloc
        *info = NULL;     // init to no info returned

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + (6*sizeof(extended_info_t))  );
        if (Info) {

                Info->count = 6;

                sprintf(version_string, "%d.%d.%d",
                        MAJOR_VERSION,
                        MINOR_VERSION,
                        PATCH_LEVEL );

                sprintf(required_engine_api_version_string, "%d.%d.%d",
                        s390_plugin_rec->required_engine_api_version.major,
                        s390_plugin_rec->required_engine_api_version.minor,
                        s390_plugin_rec->required_engine_api_version.patchlevel );

                sprintf(required_plugin_api_version_string, "%d.%d.%d",
                        s390_plugin_rec->required_plugin_api_version.plugin.major,
                        s390_plugin_rec->required_plugin_api_version.plugin.minor,
                        s390_plugin_rec->required_plugin_api_version.plugin.patchlevel );


                Info->info[0].name = EngFncs->engine_strdup( "ShortName" );
                Info->info[0].title = EngFncs->engine_strdup( _("Short Name") );
                Info->info[0].desc = EngFncs->engine_strdup( _("A short name given to this plug-in") );
                Info->info[0].type               = EVMS_Type_String;
                Info->info[0].unit               = EVMS_Unit_None;
                Info->info[0].value.s = EngFncs->engine_strdup( s390_plugin_rec->short_name );
                Info->info[0].collection_type    = EVMS_Collection_None;
                memset( &Info->info[0].group, 0, sizeof(group_info_t));

                Info->info[1].name = EngFncs->engine_strdup( "LongName" );
                Info->info[1].title = EngFncs->engine_strdup( _("Long Name") );
                Info->info[1].desc = EngFncs->engine_strdup( _("A longer, more descriptive name for this plug-in") );
                Info->info[1].type               = EVMS_Type_String;
                Info->info[1].unit               = EVMS_Unit_None;
                Info->info[1].value.s = EngFncs->engine_strdup( s390_plugin_rec->long_name );
                Info->info[1].collection_type    = EVMS_Collection_None;
                memset( &Info->info[1].group, 0, sizeof(group_info_t));

                Info->info[2].name = EngFncs->engine_strdup( "Type" );
                Info->info[2].title = EngFncs->engine_strdup( _("Plug-in Type") );
                Info->info[2].desc = EngFncs->engine_strdup( _("There are various types of plug-ins, each responsible for some kind of storage object or logical volume.") );
                Info->info[2].type               = EVMS_Type_String;
                Info->info[2].unit               = EVMS_Unit_None;
                Info->info[2].value.s = EngFncs->engine_strdup( _("Segment Manager") );
                Info->info[2].collection_type    = EVMS_Collection_None;
                memset( &Info->info[2].group, 0, sizeof(group_info_t));

                Info->info[3].name = EngFncs->engine_strdup( "Version" );
                Info->info[3].title = EngFncs->engine_strdup( _("Plug-in Version") );
                Info->info[3].desc = EngFncs->engine_strdup( _("This is the version number of the plugin.") );
                Info->info[3].type               = EVMS_Type_String;
                Info->info[3].unit               = EVMS_Unit_None;
                Info->info[3].value.s = EngFncs->engine_strdup( version_string );
                Info->info[3].collection_type    = EVMS_Collection_None;
                memset( &Info->info[3].group, 0, sizeof(group_info_t));

                Info->info[4].name = EngFncs->engine_strdup( "Required Version" );
                Info->info[4].title = EngFncs->engine_strdup( _("Required Engine Services Version") );
                Info->info[4].desc = EngFncs->engine_strdup( _("This is the version of the Engine services that the plug-in requires.  "
							       "It will not run on older versions of the Engine services.") );
                Info->info[4].type               = EVMS_Type_String;
                Info->info[4].unit               = EVMS_Unit_None;
                Info->info[4].value.s = EngFncs->engine_strdup( required_engine_api_version_string );
                Info->info[4].collection_type    = EVMS_Collection_None;
                memset( &Info->info[4].group, 0, sizeof(group_info_t));

                Info->info[5].name = EngFncs->engine_strdup( "Required Version" );
                Info->info[5].title = EngFncs->engine_strdup( _("Required Plugin API Version") );
                Info->info[5].desc = EngFncs->engine_strdup( _("This is the version of the Engine plug-in API that the plug-in requires.  "
							       "It will not run on older versions of the Engine plug-in API.") );
                Info->info[5].type               = EVMS_Type_String;
                Info->info[5].unit               = EVMS_Unit_None;
                Info->info[5].value.s = EngFncs->engine_strdup( required_plugin_api_version_string );
                Info->info[5].collection_type    = EVMS_Collection_None;
                memset( &Info->info[5].group, 0, sizeof(group_info_t));


                *info = Info;

                rc = 0;
        }


        LOG_EXIT_INT(rc);
        return rc;
}

