/*
 *
 *   (C) Copyright IBM Corp. 2001, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: activation.c
 */

#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "thing.h"
#include "activation.h"
#include "readable.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   void on_ac_deac_window_clist_select_row (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is selected.
 *      All we do is ensure that the action button is sensitive
 *      to allow the operation to be initiated.
 *
 *   Entry:
 *      clist     - address of GtkCList object that received select-row signal
 *      row       - index of row selected
 *      column    - index of column selected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - address of user data bound to clist (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_ac_deac_window_clist_select_row(GtkCList * clist, gint row, gint column,
					 GdkEventButton * event, gpointer user_data)
{
	GtkWidget *ac_deac_button = lookup_widget(GTK_WIDGET(clist), "selection_window_next_button");

	if (!(GTK_WIDGET_IS_SENSITIVE(ac_deac_button))) {
		gtk_widget_set_sensitive(ac_deac_button, TRUE);
		gtk_widget_grab_default(ac_deac_button);
	}

}

/*
 *
 *   void on_ac_deac_window_clist_unselect_row (GtkCList *, gint, gint, GdkEventButton *, gpointer)
 *
 *   Description:
 *      This routine is called whenever a row is unselected.
 *      All we do is ensure that the action button is not sensitive
 *      to disallow the operation from being initiated until a choice is made.
 *
 *   Entry:
 *      clist     - address of GtkCList object that received unselect-row signal
 *      row       - index of row deselected
 *      column    - index of column deselected on current row
 *      event     - address of structure to obtain additional context information
 *      user_data - address of user data bound to clist (not used)
 *
 *   Exit:
 *      Returns nothing.
 *
 */
void on_ac_deac_window_clist_unselect_row(GtkCList * clist, gint row, gint column,
					   GdkEventButton * event, gpointer user_data)
{
	GtkWidget *ac_deac_button = lookup_widget(GTK_WIDGET(clist), "selection_window_next_button");

	if (g_list_length(clist->selection) == 0 && GTK_WIDGET_IS_SENSITIVE(ac_deac_button))
		gtk_widget_set_sensitive(ac_deac_button, FALSE);
}

/*
 *
 *   void on_ac_deac_button_clicked (GtkButton *, gboolean)
 *
 *   Description:
 *      This routine initiates an activate/deactivate using the handle of the
 *      objects associated with rows selected. It then creates and
 *      displays the results popup which destroys both windows when dismissed.
 *
 *   Entry:
 *      button   - address of the GtkButton widget
 *      activate - activate / deactivate
 *
 *   Exit:
 *      evms_activate() / evms_deactivate() is invoked and results window
 *      is displayed.
 *
 */
void on_ac_deac_button_clicked(GtkButton * button, gboolean activate)
{
	gchar *error_msg;
	gchar *success_msg;
	GtkCList *clist;
	GSList *slist;
	object_handle_t handle;

	clist = GTK_CLIST(lookup_widget(GTK_WIDGET(button), "selection_window_clist"));

	slist = get_clist_selection_data(clist);

	if (slist != NULL) {
		gint count = g_slist_length(slist);

		if (count > 0) {
			gint rc;
			gint row = 0;
			gboolean errors = FALSE;
			GSList *item = slist;
			GtkCList *rclist;
			GtkWidget *window = NULL;

			/*
			 * If we have multiple objects, create the multi-results window
			 * and populate it with object info. We must do it now as opposed
			 * to later due the operation potentially deleting the object
			 * or altering its name (such as with name shifts when deleting
			 * segments).
			 */

			if (count > 1) {
				if (activate) {
					window = create_multi_results_window(GTK_WIDGET(button),
									     _("Results of Multiple Activate Operations"),
									     &rclist);
				} else {
					window = create_multi_results_window(GTK_WIDGET(button),
									     _("Results of Multiple Deactivate Operations"),
									     &rclist);
				}

				while (item != NULL) {
					handle = GPOINTER_TO_UINT(item->data);
					append_row_to_results_clist(rclist, handle);
					item = g_slist_next(item);
				}
				item = slist;
			}

			while (item != NULL) {
				handle = GPOINTER_TO_UINT(item->data);

				if (activate) {
					rc = evms_activate(handle);
				} else {
					rc = evms_deactivate(handle);
				}

				if (activate) {
					success_msg = g_strdup_printf(_("%s is scheduled to be activated."),
								      get_object_name(handle));
				} else {
					success_msg = g_strdup_printf(_("%s is scheduled to be deactivated."),
								      get_object_name(handle));
				}

				if (rc != SUCCESS)
					errors = TRUE;

				if (count == 1) {
					if (activate) {
						error_msg = g_strdup_printf(_("An error was encountered attempting to activate %s."),
									    get_object_name(handle));
					} else {
						error_msg = g_strdup_printf(_("An error was encountered attempting to deactivate %s."),
									    get_object_name(handle));
					}

					display_selection_window_results(GTK_WIDGET(button), rc,
									 error_msg, success_msg);
					g_free(error_msg);
				}
				else {
					set_results_clist_row(rclist, row, rc, success_msg);
				}
				g_free(success_msg);

				row++;
				item = g_slist_next(item);
			}

			if (count > 1) {
				gchar *status;

				if (errors) {
					if (activate) {
						status = _("One or more Activate operations failed.");
					} else {
						status = _("One or more Deactivate operations failed.");
					}
				} else {
					if (activate) {
						status = _("All Activate operations completed successfully.");
					} else {
						status = _("All Deactivate operations completed successfully.");
					}
				}

				display_multi_results_window(window, status);
			}
		}

		g_slist_free(slist);
	}
}

/*
 *
 *   void on_ac_deac_clist_realize (GtkWidget *, gboolean)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of things that can be activated.
 *
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      activate  - activate / deactivate
 *
 *   Exit:
 *      Selection list populated with acceptable things
 *
 */
void on_ac_deac_clist_realize(GtkWidget * widget, gboolean activate)
{
	gint rc = 0;
	gchar *col_title;
	GtkCList *clist = GTK_CLIST(widget);
	handle_array_t *things = NULL;
	guint i;
	int (*can_func)(object_handle_t handle);

	if (activate) {
		can_func = evms_can_activate;
	} else {
		can_func = evms_can_deactivate;
	}

	col_title = _("Volumes and/or Objects");
		
	set_selection_window_clist_column_titles(clist, _("Size"), col_title, NULL);

	rc = evms_get_volume_list(0, 0, 0, &things);

	if (rc != SUCCESS) {
		log_error("%s: evms_get_volume_list() returned error code %d.\n", __FUNCTION__, rc);
	} else {
		if (things->count > 0) {
			for (i = 0; i < things->count; i++) {
				if (can_func(things->handle[i]) == 0)
					add_thing_to_selection_list(clist, things->handle[i], FALSE);
			}
		}
		evms_free(things);
	}
	
	rc = evms_get_object_list(EVMS_OBJECT, DATA_TYPE, 0, 0, 0, &things);
	
	if (rc != SUCCESS) {
		log_error("%s: evms_get_object_list() for EVMS objects returned error code %d.\n", __FUNCTION__, rc);
	} else {
		if (things->count > 0) {
			for (i = 0; i < things->count; i++) {
				if (can_func(things->handle[i]) == 0)
					add_thing_to_selection_list(clist, things->handle[i], FALSE);
			}
		}
		evms_free(things);
	}

	rc = evms_get_object_list(REGION, DATA_TYPE, 0, 0, 0, &things);
	
	if (rc != SUCCESS) {
		log_error("%s: evms_get_object_list() for regions returned error code %d.\n", __FUNCTION__, rc);
	} else {
		if (things->count > 0) {
			for (i = 0; i < things->count; i++) {
				if (can_func(things->handle[i]) == 0)
					add_thing_to_selection_list(clist, things->handle[i], FALSE);
			}
		}
		evms_free(things);
	}

	rc = evms_get_object_list(SEGMENT, DATA_TYPE, 0, 0, 0, &things);
	
	if (rc != SUCCESS) {
		log_error("%s: evms_get_object_list() for segments returned error code %d.\n", __FUNCTION__, rc);
	} else {
		if (things->count > 0) {
			for (i = 0; i < things->count; i++) {
				if (can_func(things->handle[i]) == 0)
					add_thing_to_selection_list(clist, things->handle[i], FALSE);
			}
		}
		evms_free(things);
	}

	gtk_clist_set_selection_mode(clist, GTK_SELECTION_MULTIPLE);

	if (clist->rows == 1)
		gtk_clist_select_row(clist, 0, 0);
}

/*
 *
 *   void on_activate_button_clicked (GtkButton *, gchar *, gchar *, gboolean)
 *
 *   Description:
 *      This routine initiates an activate using the handle of the
 *      objects associated with rows selected. It then creates and
 *      displays the results popup which destroys both windows when dismissed.
 *
 *   Entry:
 *      button        - address of the GtkButton widget
 *
 *   Exit:
 *      evms_activate() is invoked and results window is displayed.
 *
 */
void on_activate_button_clicked(GtkButton * button, gpointer user_data)
{
	on_ac_deac_button_clicked(button, TRUE);
}

/*
 *
 *   GtkWidget *create_activate_window (gchar *, gchar *, GtkSignalFunc,
 *                                      GtkSignalFunc, gpointer)
 *
 *   Description:
 *      This creates the window used for activating "things".
 *
 *   Entry:
 *      title - the text for the window title
 *      clist_realize_handler - the handler to populate the selection list
 *      delete_button_handler - the handler to act upon clicking "Delete"
 *      user_data - whatever is needed by the signal handlers
 *
 *   Exit:
 *      Returns the id of the deletion window.
 *
 */
GtkWidget *create_activate_window(GtkSignalFunc clist_realize_handler,
				  gpointer user_data)
{
	GtkWidget *window;
	GtkWidget *tooltips;
	GtkWidget *activate_button;
	GtkWidget *instruction_label;

	window = create_standard_selection_window(_("Activate"), _("Activate"),
						  get_activate_help_text(),
						  clist_realize_handler,
						  on_activate_button_clicked,
						  NULL,
						  NULL,
						  on_ac_deac_window_clist_select_row,
						  on_ac_deac_window_clist_unselect_row, user_data);

	tooltips = gtk_object_get_data(GTK_OBJECT(window), "tooltips");
	instruction_label =
	    gtk_object_get_data(GTK_OBJECT(window), "selection_window_instruction_label");
	activate_button = gtk_object_get_data(GTK_OBJECT(window), "selection_window_next_button");

	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), activate_button,
			     _("Activate the selected objects"), NULL);

	gtk_label_set_text(GTK_LABEL(instruction_label),
			   _("Make a selection from the list above and then press the Activate button"));

	return window;
}

/*
 *
 *   void on_activate_clist_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of things that can be activated.
 *
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *
 *   Exit:
 *      Selection list populated with acceptable things
 *
 */
void on_activate_clist_realize(GtkWidget * widget, gpointer user_data)
{
	on_ac_deac_clist_realize(widget, TRUE);
}

/*
 *
 *   void on_activate_popup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine initiates the activation for one specific volume
 *      or object.
 *
 *   Entry:
 *      menuitem  - the popup menuitem that was selected
 *      user_data - the user data which is the object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_activate_popup_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_activate_window(add_thing_as_selected_list_item, user_data);

	gtk_widget_show(window);
}

/*
 *
 *   void on_deactivate_button_clicked (GtkButton *, gchar *, gchar *, gboolean)
 *
 *   Description:
 *      This routine initiates the deactivate
 *      API call using the handle of the object associated with the last row
 *      selected.
 *
 *   Entry:
 *      button        - address of the GtkButton widget
 *
 *   Exit:
 *      evms_deactivate() is invoked and results window is displayed.
 *
 */
void on_deactivate_button_clicked(GtkButton * button, gpointer user_data)
{
	on_ac_deac_button_clicked(button, FALSE);
}

/*
 *
 *   GtkWidget *create_deactivate_window (gchar *, gchar *, GtkSignalFunc,
 *                                      GtkSignalFunc, gpointer)
 *
 *   Description:
 *      This creates the window used for deactivating "things".
 *
 *   Entry:
 *      title - the text for the window title
 *      clist_realize_handler - the handler to populate the selection list
 *      delete_button_handler - the handler to act upon clicking "Delete"
 *      user_data - whatever is needed by the signal handlers
 *
 *   Exit:
 *      Returns the id of the deletion window.
 *
 */
GtkWidget *create_deactivate_window(GtkSignalFunc clist_realize_handler,
				    gpointer user_data)
{
	GtkWidget *window;
	GtkWidget *tooltips;
	GtkWidget *activate_button;
	GtkWidget *instruction_label;

	window = create_standard_selection_window(_("Deactivate"), _("Deactivate"),
						  get_deactivate_help_text(),
						  clist_realize_handler,
						  on_deactivate_button_clicked,
						  NULL,
						  NULL,
						  on_ac_deac_window_clist_select_row,
						  on_ac_deac_window_clist_unselect_row, user_data);

	tooltips = gtk_object_get_data(GTK_OBJECT(window), "tooltips");
	instruction_label =
	    gtk_object_get_data(GTK_OBJECT(window), "selection_window_instruction_label");
	activate_button = gtk_object_get_data(GTK_OBJECT(window), "selection_window_next_button");

	gtk_tooltips_set_tip(GTK_TOOLTIPS(tooltips), activate_button,
			     _("Deactivate the selected objects"), NULL);

	gtk_label_set_text(GTK_LABEL(instruction_label),
			   _("Make a selection from the list above and then press the Deactivate button"));

	return window;
}

/*
 *
 *   void on_deactivate_clist_realize (GtkWidget *, gpointer)
 *
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of things that can be deactivated.
 *
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *
 *   Exit:
 *      Selection list populated with acceptable things
 *
 */
void on_deactivate_clist_realize(GtkWidget * widget, gpointer user_data)
{
	on_ac_deac_clist_realize(widget, FALSE);
}

/*
 *
 *   void on_deactivate_popup_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine initiates the deactivation for one specific volume
 *      or object.
 *
 *   Entry:
 *      menuitem  - the popup menuitem that was selected
 *      user_data - the user data which is the object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_deactivate_popup_menu_item_activate(GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *window;

	window = create_deactivate_window(add_thing_as_selected_list_item, user_data);

	gtk_widget_show(window);
}

