/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 */ 

#include <glib.h>
#include <ncurses.h>
#include <panel.h>
#include <frontend.h>

#include "common.h"
#include "window.h"
#include "menu.h"
#include "clist.h"
#include "dialog.h"
#include "selwin.h"
#include "enum.h"
#include "task.h"
#include "readable.h"
#include "modify.h"
#include "volume.h"
#include "views.h"
#include "logging.h"

/**
 *	context_modify_container_menuitem_activated - modify the properties of a container
 *	@item: the menu item that initiated this action
 *
 *	This routine is invoked when a menu item is activated to allow modifying
 *	a container's properties. 
 */
int context_modify_container_menuitem_activated(struct menu_item *item)
{
	struct dialog_window *dialog;

	dialog = create_task_dialog(NULL, GPOINTER_TO_UINT(item->user_data),
				EVMS_Task_Set_Container_Info,
				_("Modify Container Properties"),
				_("Modify"),
				NULL);
	if (dialog != NULL) {
		process_dialog_list_events(dialog->list);
		delete_dialog_list(dialog->list->list);
	}
	return 0; 
}

/**
 *	filter_modify - filter things that can have properties changed
 *	@handle: the thing handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given thing can have it attributes/properties changed.
 */
int filter_modify(engine_handle_t handle, void *user_data)
{
	return evms_can_set_info(handle);
}

/**
 *	modify_selection_button_activated - callback invoked when thing to modify has been selected
 *	@item: the menu item/button that was activated
 *	@action: either EVMS_Task_Set_Info or EVMS_Task_Set_Container_Info
 *
 *	This routine is invoked when the Next button is activated after selecting
 *	a container or object to set/change info. We then redisplay the next task
 *	dialog if the selection did not change or create a new one if it did.
 */
int modify_selection_button_activated(struct menu_item *item, task_action_t action)
{
	struct selwin *selwin = item->user_data;
	object_handle_t curr_selection, prev_selection;
	struct dialog_window *dialog = item->user_data;

	curr_selection = get_selected_handle(selwin->clist);
	prev_selection = GPOINTER_TO_UINT(dialog->user_data);

	if (curr_selection != prev_selection) {
		char *title;
		GList *next_dialog;
		struct dialog_window *new_dialog;
		/*
		 * Since the selection has changed we need to delete any
		 * dialogs that we may have built for the given task.
		 */
		next_dialog = get_next_dialog(dialog);
		if (next_dialog != NULL)
			delete_dialog_list(next_dialog);

		if (action == EVMS_Task_Set_Container_Info)
			title = _("Modify Container Properties");
		else
			title = _("Modify Object Properties");

		new_dialog = create_task_dialog(dialog->list, curr_selection,
					action, title,
					_("Modify"), NULL);

		if (new_dialog != NULL) {
			dialog->user_data = GUINT_TO_POINTER(curr_selection);
			dialog->list->current = get_next_dialog(dialog);
		}
	} else {
		dialog->list->current = get_next_dialog(dialog);
	}

	return 0;
}

/**
 *	modify_container_button_activated - callback invoked when container to modify has been selected
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Next button is activated after selecting
 *	a container to modify its attributes/properties.
 */
int modify_container_button_activated(struct menu_item *item)
{
	return modify_selection_button_activated(item, EVMS_Task_Set_Container_Info);
}

/**
 *	actions_modify_object_menuitem_activated - modify the properties of a storage object
 *	@item: the menu item that initiated this action
 *
 *	This routine is invoked when the "Actions->Modify-Storage Object" menu item is
 *	activated to allow modifying a storage object's properties. 
 */
int actions_modify_container_menuitem_activated(struct menu_item *item)
{
	struct selwin *selwin;
	struct dialog_list dialogs;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Modify Container Properties"),
					NULL, NULL,
					NULL,
					(menuitem_activate_cb)modify_container_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Storage Container"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	clist_populate(selwin->clist, enum_containers,
			filter_modify,
			format_standard_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);

	return 0;
}

/**
 *	context_modify_object_menuitem_activated - modify the properties of a storage object
 *	@item: the menu item that initiated this action
 *
 *	This routine is invoked when a menu item is activated to allow modifying
 *	a storage object's properties. 
 */
int context_modify_object_menuitem_activated(struct menu_item *item)
{
	struct dialog_window *dialog;

	dialog = create_task_dialog(NULL, GPOINTER_TO_UINT(item->user_data),
				EVMS_Task_Set_Info,
				_("Modify Object Properties"),
				_("Modify"),
				NULL);
	if (dialog != NULL) {
		process_dialog_list_events(dialog->list);
		delete_dialog_list(dialog->list->list);
	}
	return 0; 
}

/**
 *	modify_object_button_activated - callback invoked when storage object to modify has been selected
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Next button is activated after selecting
 *	a storage object to modify its attributes/properties.
 */
int modify_object_button_activated(struct menu_item *item)
{
	return modify_selection_button_activated(item, EVMS_Task_Set_Info);
}

/**
 *	actions_modify_object_menuitem_activated - modify the properties of a storage object
 *	@item: the menu item that initiated this action
 *
 *	This routine is invoked when the "Actions->Modify-Storage Object" menu item is
 *	activated to allow modifying a storage object's properties. 
 */
int actions_modify_object_menuitem_activated(struct menu_item *item)
{
	struct selwin *selwin;
	struct dialog_list dialogs;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Modify Object Properties"),
					NULL, NULL,
					NULL,
					(menuitem_activate_cb)modify_object_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;

	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Storage Object"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	clist_populate(selwin->clist, enum_data_objects,
			filter_modify,
			format_standard_item, NULL, NULL);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	init_dialog_list(&dialogs);
	append_dialog_to_list(dialog, &dialogs);
	process_dialog_list_events(&dialogs);
	delete_dialog_list(dialogs.list);

	return 0;
}

/**
 *	modify_volume_button_activated - callback to modify an EVMS volume name
 *	@item: the menu item/button that was activated
 *
 *	This routine is invoked when the Modify button in the Modify Volume Name
 *	dialog window is activated.
 */
int modify_volume_button_activated(struct menu_item *item)
{
	int rc;
	char *name;
	struct selwin *selwin = item->user_data;
	struct dialog_window *dialog = item->user_data;

	name = dialog->user_data;
	rc = evms_set_volume_name(get_selected_handle(selwin->clist), name);
	update_status(_("Modify"), rc);

	if (rc == 0) {
		dialog->status = DLG_STATUS_CLOSING;
		refresh_views();
	}
	return 0;
}

/**
 *	volume_to_modify_selected - callback activated when EVMS volume to modify is selected
 *	@clist: the clist for the row item selected
 *	@item: the clist item that was selected
 *
 *	This routine is invoked when a volume in the clist is selected to allow 
 *	changing it's EVMS volume. We prompt for a volume name and store the name in the
 *	dialog->user_data replacing any previous name.
 */
int volume_to_modify_selected(struct clist *clist, struct clist_item *item)
{
	char *prev_name;
	struct dialog_window *dialog = clist->user_data;

	prev_name = dialog->user_data;
	dialog->user_data = get_volume_name(prev_name);
	if (dialog->user_data != NULL) {
		row_selected(clist, item);
		g_free(prev_name);
	}

	return (dialog->user_data != NULL ? 0 : ENODATA);
}

/**
 *	filter_set_volume_name - filter volumes that can have name changed
 *	@handle: the volume handle
 *	@user_data: not used
 *
 *	This routine is a standard clist_filter_func function type that checks to see
 *	if the given volume can have its name changed.
 */
int filter_set_volume_name(engine_handle_t handle, void *user_data)
{
	return evms_can_set_volume_name(handle);
}

/**
 *	on_modify_volume_dialog_delete - callback invoked when dialog used to modify a volume is deleted
 *	@dialog: the dialog window
 *
 *	This routine handles cleanup for the dialog used to modify EVMS volume names.
 *	We free any string that may be associated with the dialog containing 
 *	a volume name.
 */
void on_modify_volume_dialog_delete(struct dialog_window *dialog)
{
	g_free(dialog->user_data);
}

/**
 *	show_modify_volume_dialog - display dialog to change volume name
 *	@handle: non-zero when invoked from context popup menu
 *
 *	This routine displays a selection window to allow the user to change the
 *	name of an EVMS volume.
 **/
int show_modify_volume_dialog(object_handle_t handle)
{
	struct selwin *selwin;
	struct dialog_window *dialog;

	selwin = create_selection_window(_("Modify EVMS Volume Name"),
					NULL, NULL,
					_("_Modify"),
					(menuitem_activate_cb)modify_volume_button_activated,
					NULL,
					(menuitem_activate_cb)NULL,
					NULL);

	dialog = (struct dialog_window *)selwin;
	print_clist_column_title(selwin->clist, 0, " ");
	print_clist_column_title(selwin->clist, 1, _("Volume"));
	print_clist_column_title(selwin->clist, 2, _("Size"));

	if (handle == 0)
		clist_populate(selwin->clist, enum_volumes, filter_set_volume_name,
				format_standard_item, NULL, NULL);
	else
		clist_populate_handle(selwin->clist, handle, format_standard_item, NULL, NULL);

	set_clist_select_item_cb(selwin->clist, (clist_select_item_cb)volume_to_modify_selected);
	set_dialog_delete_cb(dialog, (dialog_delete_cb)on_modify_volume_dialog_delete);

	if (g_list_length(selwin->clist->choices) == 1)
		select_item(selwin->clist, selwin->clist->choices->data);

	set_menu_item_visibility(dialog->prev_button, FALSE);

	process_modal_dialog(dialog);
	return 0;
}

/**
 *	context_modify_vol_menuitem_activated - change the name of an EVMS volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is called to change the name of an EVMS volume.
 */
int context_modify_vol_menuitem_activated(struct menu_item *item)
{
	return show_modify_volume_dialog(GPOINTER_TO_UINT(item->user_data));
}

/**
 *	actions_modify_vol_menuitem_activated - change the name of an EVMS volume
 *	@item: the menu item that caused this function to get invoked
 *
 *	This routine is invoked by the "Modify->Volume" Actions pulldown menu item.
 */
int actions_modify_vol_menuitem_activated(struct menu_item *item)
{
	return show_modify_volume_dialog(0);
}
