/******************************************************************** 
   Copyright (C) 2000 Bassoukos Tassos <abas@aix.meng.auth.gr>
   
   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License
   as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later
   version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*********************************************************************/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <errno.h>

#include <gtk/gtk.h>
#include <gnome.h>

#include "server.h"
#include "protocol.h"
#include "guiprefs.h"
#include "smalltrans.h"
#include "userlist.h"
#include "login.h"
#include "main.h"
#include "users.h"
#include "hldat.h"

/* ================================= */
/* ================================= */
/* ================================= */

#define AGREEMENT "Agreement"

typedef struct {
  char *text;
  Connection *c;
  gboolean shown;
  GtkWidget *window;
} Agreement;

static void agreement_destroy(Connection *c,gpointer p){
  Agreement *a=(Agreement *)p;
  hooks_set_data(c,AGREEMENT,NULL);
  if(a==NULL)
    return;
  if(a->text!=NULL)
    free(a->text);
  if(a->window!=NULL)
    gtk_widget_destroy(a->window);
  free(a);
}

static gboolean close_agreement(GtkWidget *b,gpointer p){
  Agreement *a=(Agreement *)p;

  if(a->window!=NULL)
    gtk_widget_destroy(a->window);
  a->window=NULL;
  return FALSE;
}

static gint close_agreement_window(GtkWidget *widget,gpointer p){
  Agreement *a=(Agreement *)p;
  a->window=NULL;
  return FALSE;
}

static void show_agreement_dialog(Agreement *a,int all_buttons){
  GtkWidget *w,*text,*scrol,*parent=NULL;

  if(a->c->gui!=NULL)
    parent=a->c->gui->main_window;
  if(all_buttons==TRUE){
    w=gnome_dialog_new(_("Agreement"),_("Decline"),_("Accept"),NULL);
    gnome_dialog_button_connect(GNOME_DIALOG(w),1,
				GTK_SIGNAL_FUNC(close_agreement),
				(gpointer)a);
  } else {
    w=gnome_dialog_new(_("Agreement"),GNOME_STOCK_BUTTON_CLOSE,NULL);
  }
  a->window=w;
  gnome_dialog_button_connect(GNOME_DIALOG(w),0,
			      GTK_SIGNAL_FUNC(close_agreement),
			      (gpointer)a);
  if(parent!=NULL)
    gnome_dialog_set_parent(GNOME_DIALOG(w),GTK_WINDOW(parent));
  scrol=gtk_scrolled_window_new(NULL,NULL);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(w)->vbox),scrol,TRUE,TRUE,0);

  text=gtk_text_new(NULL,NULL);
  gtk_container_add(GTK_CONTAINER(scrol),text);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrol),
				 GTK_POLICY_NEVER,
				 GTK_POLICY_ALWAYS);
  gtk_text_set_editable(GTK_TEXT(text),FALSE);
  gtk_text_set_word_wrap(GTK_TEXT(text),TRUE);
  gtk_text_insert(GTK_TEXT(text),NULL,NULL,NULL,a->text,-1);
  guiprefs_widget_set_font(text,&agreement_font);
  
  gtk_widget_set_usize(w,350,450);
  gnome_dialog_close_hides(GNOME_DIALOG(w),FALSE);
  gtk_signal_connect(GTK_OBJECT(w),"destroy",
		     GTK_SIGNAL_FUNC(close_agreement_window),(gpointer)a);
  guiprefs_add_window(GTK_WINDOW(w),"Main/Agreement/Size");
  gtk_widget_show_all(w);  
}

void show_agreement(Connection *c){
  Agreement *a=(Agreement *)hooks_get_data(c,AGREEMENT);
  if(a==NULL || a->window!=NULL)
    return;
  if(a->c->gui!=NULL && a->text!=NULL)
    gtk_widget_set_sensitive(a->c->gui->toolbar_info[0].widget,TRUE);
  if(!(a->shown==FALSE && dont_show_agreements==TRUE) &&
     a->text!=NULL)
    show_agreement_dialog(a,a->shown?FALSE:TRUE);
  a->shown=TRUE;
}

static gboolean attach_agreement(gpointer data){
  Connection *c=(Connection *)data;
  Agreement *a=(Agreement *)hooks_get_data(c,AGREEMENT);

  if(dont_show_agreements==FALSE && a->text!=NULL)
    show_agreement_dialog(a,TRUE);
  return FALSE;
}

void handle_server_agreement(Connection *c,HLTransaction *t,gpointer data){
  HLObject *o;
  Agreement *a=(Agreement *)malloc(sizeof(Agreement));

  a->c=c;
  a->window=NULL;
  a->shown=FALSE;
  a->text=NULL;
  transaction_read_objects(c,t);
  o=transaction_find_object(t,HLO_MESSAGE);
  if(o!=NULL){
    string_net_to_unix(o);
    a->text=strdup(o->data.string);
  }
  transaction_destroy(t);  
  hooks_create(c,AGREEMENT,a,NULL,agreement_destroy);
  if(c->version==0)
    gtk_idle_add(attach_agreement,(gpointer)c);
  else
    gtk_idle_add(login_done_after_agreement,(gpointer)c);
}

/* ================================= */
/* ================================= */
/* ================================= */

#define MESSAGES "Messages"

typedef struct {
  Connection *c;
  GtkWidget *w;
  char *message;
  gboolean is_error;
} UnkMessage;

typedef struct {
  Connection *c;
  GList *windows;
} MessageData;

typedef struct {
  MessageData *md;
  GtkWidget *toplevel;
  GtkWidget *text_widget;
  gpointer data;
  int socket;
  char *text;
  char *quote;
} MessageWindow;

static void message_destroy_event(GtkWidget *w,gpointer data){
  MessageWindow *m=(MessageWindow *)data;
  m->md->windows=g_list_remove(m->md->windows,m);

  if(m->quote!=NULL)
    free(m->quote);
  if(m->text!=NULL)
    free(m->text);
  free(m);
}

static void message_destroy_all(Connection *c,gpointer data){
  MessageData *md=(MessageData *)hooks_get_data(c,MESSAGES);
  hooks_set_data(c,MESSAGES,NULL);
  while(md->windows!=NULL)
    gtk_widget_destroy(((MessageWindow *)md->windows->data)->toplevel);
}

void check_messages(Connection *c){
  MessageData *md;
  if(hooks_get_data(c,MESSAGES)!=NULL)
    return;
  md=(MessageData *)calloc(sizeof(MessageData),1);
  md->c=c;
  md->windows=NULL;
  hooks_create(c,MESSAGES,md,NULL,message_destroy_all);
}

static MessageWindow *message_create_messagewindow(Connection *c,int socket){
  MessageWindow *m=(MessageWindow *)calloc(sizeof(MessageWindow),1);
  MessageData *md=(MessageData *)hooks_get_data(c,MESSAGES);

  m->md=md;
  m->toplevel=NULL;
  m->socket=socket;
  m->toplevel=NULL;
  m->text_widget=NULL;
  m->text=NULL;
  m->quote=NULL;
  md->windows=g_list_prepend(md->windows,m);
  return m;
}

void message_display_text(Connection *c,char *string){
  MessageWindow *m=message_create_messagewindow(c,0);
  GtkWidget *label,*d;

  label=gtk_label_new(string);
  d=gnome_dialog_new(_("User Info"),GNOME_STOCK_BUTTON_CLOSE,NULL);
  if(c!=NULL && c->gui!=NULL && c->gui->main_window!=NULL)
    gnome_dialog_set_parent(GNOME_DIALOG(d),GTK_WINDOW(c->gui->main_window));
  gtk_label_set_justify(GTK_LABEL(label),GTK_JUSTIFY_LEFT);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(d)->vbox),label,TRUE,TRUE,0);
  m->toplevel=d;
  gtk_signal_connect(GTK_OBJECT(d),"destroy",
		     GTK_SIGNAL_FUNC(message_destroy_event),
		     (gpointer)m);
  gnome_dialog_button_connect_object(GNOME_DIALOG(d),0,
			      GTK_SIGNAL_FUNC(gtk_widget_destroy),
			      GTK_OBJECT(m->toplevel));
  gtk_widget_show_all(GTK_WIDGET(d));
}

static gboolean message_transaction_send_done(gpointer data){
  gtk_widget_destroy(((MessageWindow *)data)->toplevel);
  return FALSE;
}

static void message_transaction_send(Connection *c,gpointer data){
  MessageWindow *m=(MessageWindow *)data;
  HLTransaction *t=transaction_new(HLCT_SENDPRIVM,NULL,FALSE);
  HLObject *o;

  transaction_add_object(t,create_number(HLO_SOCKET,m->socket));
  o=create_string(HLO_MESSAGE,m->text);
  string_unix_to_net(o);
  transaction_add_object(t,o);
  if(m->quote!=NULL){
    o=create_string(HLO_QUOTE,m->quote);
    string_unix_to_net(o);
    transaction_add_object(t,o);
  }
  server_transaction_reply_set(m->md->c,t,server_reply_default,
			       strdup(_("Could not send Message:")));
  transaction_send(t,m->md->c);
  gtk_idle_add(message_transaction_send_done,m);
}

static void message_send(GtkButton *b,gpointer data){
  MessageWindow *m=(MessageWindow *)data;
  if(m->toplevel==NULL || m->text_widget==NULL || m->text!=NULL)
    return;
  m->text=gtk_editable_get_chars(GTK_EDITABLE(m->text_widget),0,-1);
  if(m->toplevel!=NULL)
    gtk_widget_hide(m->toplevel);
  server_transaction_start(m->md->c,message_transaction_send,m);
}

static GtkWidget *get_text_with_label(char *label,char *text,GtkWidget **textwidget){
  GtkWidget *vbox,*labelw,*textw,*sl;

  vbox=gtk_vbox_new(FALSE,0);
  if(label!=NULL){
    labelw=gtk_label_new(label);
    gtk_box_pack_start(GTK_BOX(vbox),labelw,FALSE,FALSE,0);
    gtk_misc_set_alignment(GTK_MISC(labelw),0,0.5);
  }
  textw=gtk_text_new(NULL,NULL);
  if(text!=NULL)
    gtk_text_insert(GTK_TEXT(textw),NULL,NULL,NULL,text,-1);
  gtk_text_set_editable(GTK_TEXT(textw),FALSE);
  sl=gtk_scrolled_window_new(NULL,NULL);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sl),
				 GTK_POLICY_NEVER,
				 GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(sl),textw);
  gtk_box_pack_start(GTK_BOX(vbox),sl,TRUE,TRUE,0);
  if(textwidget!=NULL)
    *textwidget=textw;
  return vbox;
}

static GtkWidget *make_textboxes(MessageWindow *m,GtkWidget **msg){
  char buf[256];
  GtkWidget *hdr=NULL;
  UserlistEntry *ue;
  char *uname=NULL;

  if(m==NULL){
    if(msg!=NULL) *msg=NULL;
    return NULL;
  }
  ue=get_user_by_socket(m->md->c,m->socket);
  if(ue!=NULL) uname=ue->name;
  if(m->quote!=NULL){
    hdr=get_text_with_label(_("You wrote:"),m->quote,NULL);
    if(uname!=NULL)
      sprintf(buf,_("%s replied:"),uname);
    else
      sprintf(buf,_("Socket %d replied:"),m->socket);
  } else {
    if(uname!=NULL)
      sprintf(buf,_("%s writes:"),uname);
    else
      sprintf(buf,_("Socket %d writes:"),m->socket);
  }
  *msg=get_text_with_label(buf,m->text,NULL);
  return hdr;
}

gpointer message_create_simple(Connection *c, char *message, int socket, void *dummy){
  MessageWindow *m=message_create_messagewindow(c,socket);
  MessageWindow *orig=dummy;
  GtkWidget *d,*text,*vbox,*label;
  UserlistEntry *ue=get_user_by_socket(c,socket);
  char buf[256];

  if(orig==NULL){
    if(ue==NULL)
      strcpy(buf,_("Message"));
    else 
      sprintf(buf,_("Message to %s"),ue->name);
  } else {
    if(orig->text!=NULL)
      m->quote=strdup(orig->text);
    if(ue==NULL)
      strcpy(buf,_("Reply"));
    else 
      sprintf(buf,_("Reply to %s"),ue->name);
  }

  d=gnome_dialog_new(buf,GNOME_STOCK_BUTTON_CANCEL,"Send",NULL);
  m->toplevel=d;
  gnome_dialog_button_connect_object(GNOME_DIALOG(d),0,
				     GTK_SIGNAL_FUNC(gtk_widget_destroy),
				     GTK_OBJECT(m->toplevel));
  gnome_dialog_button_connect(GNOME_DIALOG(d),1,
			      GTK_SIGNAL_FUNC(message_send),
			      (gpointer)m);
  vbox=gtk_vbox_new(FALSE,0);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(d)->vbox),vbox,TRUE,TRUE,0);

  label=make_textboxes(orig,&text);
  if(label!=NULL)
    gtk_box_pack_start(GTK_BOX(vbox),label,TRUE,TRUE,0);
  if(text!=NULL)
    gtk_box_pack_start(GTK_BOX(vbox),text,TRUE,TRUE,0);

  label=get_text_with_label(orig==NULL?NULL:_("Your reply:"),message,&text);
  gtk_text_set_editable(GTK_TEXT(text),TRUE);
  gtk_box_pack_start(GTK_BOX(vbox),label,TRUE,TRUE,0);
  if(message!=NULL)
    gtk_text_insert(GTK_TEXT(text),NULL,NULL,NULL,"\n\n",1);
  m->text_widget=text;
  if(m->md->c->gui->main_window!=NULL)
    gnome_dialog_set_parent(GNOME_DIALOG(d),
			    GTK_WINDOW(m->md->c->gui->main_window));
  gtk_window_set_focus(GTK_WINDOW(d),m->text_widget);
  gtk_window_set_modal(GTK_WINDOW(d),FALSE);
  gtk_signal_connect(GTK_OBJECT(text),"activate",
		     GTK_SIGNAL_FUNC(message_send),
		     (gpointer)m);
  gtk_signal_connect(GTK_OBJECT(d),"destroy",
		     GTK_SIGNAL_FUNC(message_destroy_event),
		     (gpointer)m);
  guiprefs_add_window(GTK_WINDOW(d),"Server/Messages/ReplyDialogSize");
  gtk_window_set_policy(GTK_WINDOW(d), FALSE, TRUE, FALSE);
  gtk_widget_show_all(d);
  return m;
}

static void message_clear_window(GtkButton *b,gpointer data){
  MessageWindow *m=(MessageWindow *)data;
  if(m->text_widget!=NULL)
    gtk_editable_delete_text(GTK_EDITABLE(m->text_widget),0,-1);
}

static void message_open_reply(GtkButton *b,gpointer data){
  MessageWindow *m=(MessageWindow *)data;
  message_create_simple(m->md->c,NULL,m->socket,m);
  gtk_widget_destroy(m->toplevel);
}

static void message_quote_and_fill(char *msg, GtkWidget *text){
  char **v=g_strsplit(msg,"\n",0);
  int i;

  gtk_text_freeze(GTK_TEXT(text));
  if(v!=NULL)
    for(i=0;v[i]!=NULL;i++){
      gtk_text_insert(GTK_TEXT(text),NULL,NULL,NULL,"> ",2);
      gtk_text_insert(GTK_TEXT(text),NULL,NULL,NULL,v[i],-1);
      gtk_text_insert(GTK_TEXT(text),NULL,NULL,NULL,"\n",1);
    }
  gtk_text_insert(GTK_TEXT(text),NULL,NULL,NULL,"\n",1);
  g_strfreev(v);
  gtk_text_thaw(GTK_TEXT(text));
}

static void message_open_reply_quoted(GtkButton *t,gpointer data){
  MessageWindow *m=(MessageWindow *)data,*b;

  b=(MessageWindow *)message_create_simple(m->md->c,NULL,m->socket,m);
  message_quote_and_fill(m->text,b->text_widget);
  gtk_widget_destroy(m->toplevel);
}

static MessageWindow *message_handle_privmsg(Connection *c,HLTransaction *t,
					     int socket,char *nick,
					     char *origmsg,char *quote){
  MessageWindow *m;
  char buf[256];
  GtkWidget *vbox,*label,*msg;
  GtkWidget *text_message=NULL,*paned=NULL;
  int i=0;
  UserlistEntry *ue;
  char *uname;

  if(userlist_get_ignored(c,socket)==TRUE) return NULL;
  m=message_create_messagewindow(c,socket);
  if(quote!=NULL)
    m->quote=strdup(quote);
  if(origmsg!=NULL)
    m->text=strdup(origmsg);
  ue=get_user_by_socket(c,socket);
  uname=(ue!=NULL && ue->name!=NULL)?ue->name:nick;
  if(uname!=NULL)
    sprintf(buf,_("Message from %s"),uname);
  else
    sprintf(buf,_("Message from socket %d ???"),socket);
  if(prefs_messages_show_with_reply==TRUE){
    i=1;
    m->toplevel=gnome_dialog_new(buf,_("Clear"),_("Send"),
				 GNOME_STOCK_BUTTON_CLOSE,NULL);
    vbox=GNOME_DIALOG(m->toplevel)->vbox;
    paned=gtk_vpaned_new();
    gtk_box_pack_end(GTK_BOX(vbox),paned,TRUE,TRUE,2);

    label=get_text_with_label(_("Your reply:"),NULL,&text_message);
    gtk_text_set_editable(GTK_TEXT(text_message),TRUE);
    gtk_paned_pack2(GTK_PANED(paned),label,TRUE,FALSE);
    if(prefs_messages_reply_quoted==TRUE)
      message_quote_and_fill(origmsg,text_message);
    gnome_dialog_button_connect(GNOME_DIALOG(m->toplevel),0,
				GTK_SIGNAL_FUNC(message_clear_window),
				(gpointer)m);
    gnome_dialog_button_connect_object(GNOME_DIALOG(m->toplevel),2,
				GTK_SIGNAL_FUNC(gtk_widget_destroy),
				GTK_OBJECT(m->toplevel));
    gnome_dialog_button_connect(GNOME_DIALOG(m->toplevel),1,
				GTK_SIGNAL_FUNC(message_send),
				(gpointer)m);
    gtk_window_set_focus(GTK_WINDOW(m->toplevel),text_message);  
    guiprefs_add_window(GTK_WINDOW(m->toplevel),"Server/Messages/MessageReplySize");
  } else {
    if(prefs_messages_reply_quoted==TRUE){
      i=0;
      m->toplevel=gnome_dialog_new(buf,_("Reply"),
				   GNOME_STOCK_BUTTON_CLOSE,NULL);
    } else {
      i=1;
      m->toplevel=gnome_dialog_new(buf,_("Reply Quoted"),_("Reply"),
				   GNOME_STOCK_BUTTON_CLOSE,NULL);
      gnome_dialog_button_connect(GNOME_DIALOG(m->toplevel),0,
				  GTK_SIGNAL_FUNC(message_open_reply_quoted),
				  (gpointer)m);
    }
    vbox=GNOME_DIALOG(m->toplevel)->vbox;
    gnome_dialog_button_connect_object(GNOME_DIALOG(m->toplevel),1+i,
				       GTK_SIGNAL_FUNC(gtk_widget_destroy),
				       GTK_OBJECT(m->toplevel));
    gnome_dialog_button_connect(GNOME_DIALOG(m->toplevel),i,
				GTK_SIGNAL_FUNC(message_open_reply),
				(gpointer)m);
    guiprefs_add_window(GTK_WINDOW(m->toplevel),"Server/Messages/MessageDialogSize");
  }
  
  label=make_textboxes(m,&msg);
  if(label!=NULL)
    gtk_box_pack_start(GTK_BOX(vbox),label,TRUE,TRUE,0);
  if(paned!=NULL){
    gtk_paned_pack1(GTK_PANED(paned),msg,TRUE,FALSE);
  } else {
    gtk_box_pack_start(GTK_BOX(vbox),msg,TRUE,TRUE,0);
  }

  if(prefs_messages_show_with_reply==TRUE){
    free(m->quote);
    m->quote=m->text;
    m->text=NULL;
  }

  if(m->text_widget==NULL)
    m->text_widget=text_message;
  if(m->text_widget!=NULL)
    gtk_signal_connect(GTK_OBJECT(m->text_widget),"activate",
		       GTK_SIGNAL_FUNC(message_send),
		       (gpointer)m);
  gtk_window_set_policy(GTK_WINDOW(m->toplevel), FALSE, TRUE, FALSE);
  gtk_window_set_modal(GTK_WINDOW(m->toplevel),FALSE);

  return m;
}

static MessageWindow *message_handle_error(Connection *c,char *message){
  MessageWindow *m=message_create_messagewindow(c,0);
  GtkWidget *parent=main_app;
  if(c!=NULL && c->gui!=NULL && c->gui->main_window!=NULL)
    parent=c->gui->main_window;
  m->toplevel=gnome_error_dialog_parented(message,GTK_WINDOW(parent));
  play_sound(HL_SOUND_ERROR);
  return m;
}

static MessageWindow *message_handle_broadcast(Connection *c,char *message){
  MessageWindow *m=message_create_messagewindow(c,0);
  GtkWidget *parent=main_app;
  if(c!=NULL && c->gui!=NULL && c->gui->main_window!=NULL)
    parent=c->gui->main_window;
  m->toplevel=gnome_ok_dialog_parented(message,GTK_WINDOW(parent));
  play_sound(HL_SOUND_SERVMSG);
  return m;
}

void show_message_error(Connection *c,char *text){
  MessageWindow *m=message_handle_error(c,text);
  gtk_signal_connect(GTK_OBJECT(m->toplevel),"destroy",
		     GTK_SIGNAL_FUNC(message_destroy_event),
		     (gpointer)m);
  gtk_widget_show_all(m->toplevel);
}

static gboolean show_error_idle(gpointer data){
  UnkMessage *um=data;;

  if(um==NULL) return FALSE;
  if(um->c!=NULL && um->c->gui!=NULL && um->c->gui->main_window!=NULL){
    show_message_error(um->c,um->message);
  } else {
    GtkWidget *dialog,*parent;
    /* FIXME: is this neccesary? */
    parent=um->w;
    if(um->w==NULL)
      parent=main_app;
    dialog=gnome_error_dialog_parented(um->message,GTK_WINDOW(parent));
    gnome_dialog_close_hides(GNOME_DIALOG(dialog),FALSE);
    gtk_widget_show_all(dialog);
  }
  free(um->message);
  free(um);
  return FALSE;
}

static void show_error_ap(Connection *c, GtkWidget *w, const char *fmt, va_list ap){
  UnkMessage *um = malloc(sizeof(UnkMessage));
  um->c = c;
  um->w = w;
  um->message=g_strdup_vprintf(fmt, ap);
  gtk_idle_add(show_error_idle, um);
}
void show_error(Connection *c, GtkWidget *w, const char *fmt, ...){
  va_list ap;
  va_start(ap, fmt);
  show_error_ap(c, w, fmt, ap);
  va_end(ap);
}
void show_error_errno(Connection *c, GtkWidget *w, const char *fmt, ...){
  va_list ap;
  char *fmt2;
  
  va_start(ap, fmt);
  fmt2=g_strdup_printf("%s\n%s", fmt, strerror(errno));
  show_error_ap(c, w, fmt2, ap);
  free(fmt2);
  va_end(ap);
}

static gboolean handle_display_message(gpointer p){
  HLTransaction *t=(HLTransaction *)p;
  Connection *c=(Connection *)t->data;
  HLObject *message=transaction_find_object(t,HLO_MESSAGE);
  HLObject *sock=transaction_find_object(t,HLO_SOCKET);
  HLObject *nick=transaction_find_object(t,HLO_NICK);
  HLObject *quote=transaction_find_object(t,HLO_QUOTE);

  MessageWindow *m=NULL;

  if(message==NULL){
    transaction_destroy(t);
    return FALSE;
  }
  string_net_to_unix(message);
  if(quote!=NULL)
    string_net_to_unix(quote);
  if(transaction_find_object(t,HLO_PARAMETER)!=NULL)
    m=message_handle_error(c,message->data.string);
  else {
    if(sock==NULL)
      m=message_handle_broadcast(c,message->data.string);
    else
      m=message_handle_privmsg(c,t,sock->data.number,
			       nick==NULL?NULL:nick->data.string,
			       message->data.string,
			       quote==NULL?NULL:quote->data.string);
  }
  if(m!=NULL && m->toplevel!=NULL){
    gtk_signal_connect(GTK_OBJECT(m->toplevel),"destroy",
    		       GTK_SIGNAL_FUNC(message_destroy_event),
		       (gpointer)m);
    gtk_widget_show_all(m->toplevel);
  }
  transaction_destroy(t);
  return FALSE;
}

void handle_server_message(Connection *c,HLTransaction *t,gpointer data){
  transaction_read_objects(c,t);
  t->data=c;
  gtk_idle_add(handle_display_message,t);
}

