//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: Shape.hpp,v 1.7 2004/02/08 18:44:19 delpinux Exp $


#ifndef SHAPE_HPP
#define SHAPE_HPP

#include <ReferenceCounting.hpp>

#include <Transform.hpp>

#include <string>
#include <vector>

/*!
  \class Shape
  This virtual class allow to manipulate shapes (primitives and set operations).

  \author Stphane Del Pino.
*/
class Shape
{
public:
  /*!
    \enum ShapeType
    This is the list of identifiers used for shapes.
    Identifiers are mean to be explicit ...
    \li union_ is used instead of 'union' because 'union' is a C++ keyword.
    \li not_ is used instead of 'not' because 'not' is a C++ keyword.
  */
  enum ShapeType { 
    sphere,   // primitives
    cylinder,
    cone,
    cube,
    plane,
    torus,
    union_,    
    difference,
    intersection,
    not_,
    infiniteCylinder,
    infiniteCone
  };

  typedef std::vector<ConstReferenceCounting<Transform> > TransformationsList;

protected:

  //! This member contains the Shape type information
  const ShapeType __stype;

  /*!
      This is a vector of pointers on Transformation. Many transformations
      can be applied on one shape.
   */
  TransformationsList __trans;

  //! Returns true if TinyVector<3> v is IN the PRIMITIVE shape.
  virtual bool inShape(const TinyVector<3, real_t>& v) const = 0;

  /*! Return overloaded output.
    \todo remove this method and make operator << virtual.
  */
  virtual std::ostream& put(std::ostream& s) const = 0;

public:
  /** 
   * read only Access to the transformation lists
   * 
   * @return __trans
   */
  const TransformationsList& transformationsList() const
  {
    return __trans;
  }

  /** 
   * sets the transformation list
   * 
   */
  void setTransformationsList(const TransformationsList& l)
  {
    __trans = l;
  }

  //! Returns \p true if the point \a p is inside the Shape.
  const inline bool inside(const TinyVector<3>& v) const
  {
    TinyVector<3> w = v;
    for (size_t i = nbTransform()-1; i<nbTransform(); --i)
      w = inverseTransform(i, w);
    return (inShape(w));
  }

  /*! Prints the type name of the Shape into a std::string.
    \warning keep or remove this method?
   */
  const std::string typeName() const;

  //! Returns the type of the shape.
  inline const ShapeType& type() const;

  //! returns the number of Transform applied to the Shape.
  inline size_t nbTransform() const;

  //! Adds the transformation \a T to the Shape.
  void setTransform(const Transform& T);

  //! Applies invert of transformation \a i to the point \a P.
  inline TinyVector<3>& inverseTransform(const size_t& i,
					 TinyVector<3>& P) const;

  //! Outputs the Shape \a S to the stream \a os.
  friend std::ostream& operator << (std::ostream& os,
				    const Shape& s);

  //! Builds the transformations vector using informations stored in \a trans.
  void parseTransform(const parsetrans& trans);

  //! Constructs a Shape for a given type \a T.
  Shape(const ShapeType& T)
    : __stype(T)
  {
    ;
  }

  //! Copy constructor
  Shape(const Shape& S)
    : __stype(S.__stype),
      __trans(S.__trans)
  {
    ;
  }

  //! Destructor.
  virtual ~Shape()
  {
    ;
  }
};

//! Returns the type of the shape.
const Shape::ShapeType& Shape::type() const
{ 
  return __stype;
}

//! returns the number of Transform applied to the Shape.
size_t Shape::nbTransform() const
{
  return __trans.size();
}


//! Applies invert of transformation \a i to the point \a P.
inline TinyVector<3>& Shape::inverseTransform(const size_t& i,
					      TinyVector<3>& P) const
{
  return (*__trans[i]).inverse(P);
}


#endif // SHAPE_HPP

