//  This file is part of ff3d - http://www.freefem.org/ff3d
//  Copyright (C) 2001, 2002, 2003 Stphane Del Pino

//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2, or (at your option)
//  any later version.

//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.

//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

//  $Id: FirstOrderOperator.hpp,v 1.3 2003/09/20 21:31:09 delpinux Exp $


#ifndef _FIRST_ORDER_OP_
#define _FIRST_ORDER_OP_

#include <PDEOperator.hpp>
/*!
  \class FirstOrderOperator

  The class FirstOrderOp describes first order partial differencial
  operators: \f$ \nu\cdot\nabla \f$ where \f$ \nu \f$ is a dimension 3
  function vector.
  
  \par example:
  if \f$ \nu = (1,1,1) \f$, the operator is the divergence.

  \author Stphane Del Pino
 */
class FirstOrderOperator
  : public PDEOperator
{
public:
  typedef TinyVector<3, ReferenceCounting<UserFunction> >  Vector;
  /*!
    Coeficients of \f$ \nu \f$: \f$ \nu_i \f$.
  */
private:
  ReferenceCounting<FirstOrderOperator::Vector> __nu;

public:
  const UserFunction& coefficient(const size_t i)
  {
    assert (i<3);
    
    return (*(*__nu)[i]);
  }

  std::string TypeName() const
  {
    return std::string("FirstOrderOperator");
  }

  //! Read-only access to operator coefs.
  const UserFunction& nu(const int&i) const
  {
    return *(*__nu)[i];
  }

  //! Read-only access to operator coefs.
  const FirstOrderOperator::Vector& nu() const
  {
    return *__nu;
  }

  //! Returns a pointer on the "multiplied" by u FirstOrderOperator
  ReferenceCounting<PDEOperator> operator * (ConstReferenceCounting<UserFunction> u) const
  {
    ReferenceCounting<FirstOrderOperator::Vector> __nu2
      = new FirstOrderOperator::Vector;

    for (size_t i=0; i<3; ++i)
      (*__nu2)[i] = new MulUserFunction((*__nu)[i], u);

    return (new FirstOrderOperator(__nu2));
  }

  //! Returns a pointer on the opposed FirstOrderOperator operator.
  ReferenceCounting<PDEOperator> operator - () const
  {
    ReferenceCounting<FirstOrderOperator::Vector> __nu2
      = new FirstOrderOperator::Vector;

    for (size_t i=0; i<3; ++i)
      (*__nu2)[i] = new UnaryMinusUserFunction((*__nu)[i]);

    return (new FirstOrderOperator(__nu2));
  }

  //! Constructs the operator using the \a Mesh and a given function matrix \a initA.
  FirstOrderOperator(ReferenceCounting<FirstOrderOperator::Vector> initNu)
    : PDEOperator(PDEOperator::firstorderop,
		  3),
      __nu(initNu)
  {
    ;
  }

  //! Copy constructor.
  FirstOrderOperator(const FirstOrderOperator& D)
    : PDEOperator(D),
      __nu(D.__nu)
  {
    ;
  }

  //! Destrutor.
  ~FirstOrderOperator()
  {
    ;
  }
};

#endif // _FIRST_ORDER_OP_

