# Copyright (C) 2005 by Imendio AB
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.

from xml.sax.saxutils import escape

from gazpacho import util
from gazpacho.widget import Widget, get_widget_from_gtk_widget
from gazpacho.widgetregistry import widget_registry

import gobject
try:
    import gazpacho_widgets

    class CContext(gobject.GObject, gazpacho_widgets.Context):
        
        def __init__(self, real_context):
            gobject.GObject.__init__(self)
            
            self.real_context = real_context
            
        def do_create_placeholder(self):
            return self.real_context.create_placeholder()

        def do_setup_internal_child(self, parent, child_name, child):
            gparent = get_widget_from_gtk_widget(parent)

            if not gparent: return

            type_name = gobject.type_name(child.__gtype__)
            child_class = widget_registry.get_by_name(type_name)
            if not child_class: return

            gchild = Widget(child_class, self.real_context.get_project())

            gchild.setup_internal_widget(child, child_name, gparent.name)
            
    gobject.type_register(CContext)

    class CProxy(gobject.GObject, gazpacho_widgets.EditorProxy):
        def __init__(self, real_proxy):
            gobject.GObject.__init__(self)
            
            self.real_proxy = real_proxy

        def do_set_value(self, value):
            self.real_proxy.set_value(value)
            
        def do_set_property(self, property_name, value):
            self.real_proxy.set_property(property_name, value)
        
    gobject.type_register(CProxy)
    
except ImportError:
    gazpacho_widgets = None

class CPropertyAdaptor:
    def __init__(self, widget_class, property_name, property_type):
        self.widget_class = widget_class
        self.property_name = property_name
        self.property_type = property_type

    def create_editor(self, context):
        klass = self.widget_class
        return klass.invoke_create_property_editor(CContext(context),
                                                   self.property_name)

    def update_editor(self, context, editor, widget, proxy):
        klass = self.widget_class

        klass.invoke_update_property_editor(CContext(context),
                                            self.property_name,
                                            editor, widget, CProxy(proxy))
        
    def get(self, context, object):
        klass = self.widget_class
        return klass.invoke_get_property(CContext(context),
                                         object, self.property_name,
                                         self.property_type)

    def set(self, context, object, value):
        klass = self.widget_class
        klass.invoke_set_property(CContext(context),
                                  object, self.property_name,
                                  self.property_type, value)

    def load(self, context, object, value):
        klass = self.widget_class
        value = klass.invoke_load_property(CContext(context),
                                           object, self.property_name, value)
        
    def save(self, context, object):
        klass = self.widget_class
        value = klass.invoke_save_property(CContext(context),
                                           object, self.property_name)
        return escape(value)
    
class CWidgetAdaptor:
    def __init__(self, widget_class):
        self.widget_class = widget_class

    def pre_create(self, context, widget, interactive=True):
        # We don't support pre-create for C widgets
        pass

    def post_create(self, context, widget, interactive=True):
        self.widget_class.invoke_post_create(CContext(context),
                                             widget, interactive)

    def fill_empty(self, context, widget):
        self.widget_class.invoke_fill_empty(CContext(context),
                                            widget)

    def replace_child(self, context, old_child, new_child, widget):
        self.widget_class.invoke_replace_child(CContext(context),
                                               widget, old_child, new_child)

    def get_internal_child(self, context, widget, name):
        return self.widget_class.invoke_get_internal_child(CContext(context),
                                                           widget, name)

    def child_property_applies(self, context, ancestor, widget, property_id):
        klass = self.widget_class
        return klass.invoke_child_property_applies(CContext(context),
                                                   widget, ancestor,
                                                   property_id)
        
class Library:
    def register_type(self, type_name, type_function_name):
        pass
    
class CLibrary(Library):
    def __init__(self, library_name, libglade_module):
        self.plugin = gazpacho_widgets.plugin_load(library_name)
        self.libglade_module = libglade_module
        
    def register_type(self, type_name, type_function_name):
        if not type_function_name:
            type_function_name = ''
            for c in type_name:
                if c.isupper():
                    if type_function_name != '':
                        type_function_name += '_'
                    type_function_name += c.lower()
                else:
                    type_function_name += c

            type_function_name += '_get_type'
            
        self.plugin.call_get_type_function(type_function_name)
        
    def create_widget(self, type):
        widget = gobject.new (type)
        
        return widget

    def get_widget_adaptor_class(self, type_name, class_name):
        type = gobject.type_from_name(type_name)

        wc = gazpacho_widgets.widget_class_get(type)

        return CWidgetAdaptor(wc)

    def get_property_adaptor_class(self, type_name, property_name,
                                   property_type, class_name):
        type = gobject.type_from_name(type_name)

        wc = gazpacho_widgets.widget_class_get(type)

        return CPropertyAdaptor(wc, property_name, property_type)

class PythonLibrary(Library):
    def __init__(self, library_name, libglade_module):
        self.library_name = library_name
        self.libglade_module = libglade_module
        
        self.module = __import__(self.library_name, globals(), locals ())

        # look in possible submodules
        self.root_library = __import__(self.module.root_library, globals(),
                                       locals())
        submodules = self.module.root_library.split('.')
        for module in submodules[1:]:
            self.root_library = getattr(self.root_library, module)
        
    def get_function(self, func_name):
        # XXX: Maybe have error handling
        return getattr(self.module, func_name)

    def _get_adaptor_class(self, type_name, class_name):
        if class_name:
            adaptor_class = self.get_class(class_name)

            if not adaptor_class:
                print "Adaptor class %s not found" % class_name
                
            return adaptor_class()

    def get_property_adaptor_class(self, type_name, property_name,
                                   property_type, class_name):
        return self._get_adaptor_class(type_name, class_name)

    def get_widget_adaptor_class(self, type_name, class_name):
        return self._get_adaptor_class(type_name, class_name)
        
    def get_class(self, class_name):
        module = self.module
        
        for name in class_name.split('.'):
            if not hasattr(module, name):
                return None
            else:
                module = getattr(module, name)
        
        return module

    def create_widget(self, type):
        klass = util.class_from_gtype(type, [self.root_library])

        return klass()
        
def load_library(library_name, library_type, libglade_module):
    if library_type == 'python':
        return PythonLibrary(library_name, libglade_module)
    elif library_type == 'c':
        if gazpacho_widgets:
            return CLibrary(library_name, libglade_module)
        else:
            msg = ("Library type %s not supported. The appropriate module "
                   "can't be found")
            raise ValueError, msg % library_type
    else:
        raise ValueError, "Uknown library type %s" % library_type

