
/*
  
   1998 Roberto Alameda 
  You may modify and distribute this file under the terms of the GNU
  General Public License, version 2, or any later version, at your
  convenience. See the file COPYING for details. 
  

*/ 


#include <config.h>

#if HAVE_UNISTD_H
# include <unistd.h>
# include <sys/types.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <time.h>
#include <ctype.h>

#if HAVE_SYS_WAIT_H
# include <sys/wait.h>
#endif
#ifndef WEXITSTATUS
# define WEXITSTATUS(stat_val) ((unsigned)(stat_val) >> 8)
#endif
#ifndef WIFEXITED
# define WIFEXITED(stat_val) (((stat_val) & 255) == 0)
#endif


extern "C" {
#include <gif_lib.h>
}

#include "gfont.h"
#include "error.xpm"
#include "font.xpm"
#include "openhand.xpm"

#include "ps.h"


extern GtkWidget *mainwindow, *fontclistw;
extern GtkStyle* stylebold;
extern GtkWidget *samplet, *stbar, *prbar;
extern GdkPixmap *emptypixmap;

extern GdkColor graypalette[];
extern GSList *fontlist;
extern GdkGC *bwGC, *defGC, *redGC, *blueGC;
extern GdkColorContext *mCC;

extern bool antialiasing, kerning;
extern struct EntryDialog *fontsizeD, *characterD, *stringD;
extern int xresolution, yresolution;

extern GtkTargetEntry dnd_target_table[];
extern guint dnd_targets;

static GtkWidget *errwindow;  // The window where error messages come
static GtkWidget *tree;       // The root tree in errwindow


// ************** Function prototypes (internally used in this module)
static bool delete_yes_no(GtkWidget *widget, GdkEvent *event, gpointer data);
static void set_variable(GtkWidget *button, gpointer data);

static Callback do_print, selectprintfile;
static void okselprintfile(gpointer diag);
static bool make_gif(GtkWidget *parent, GdkImage* image, char* filename);




// ********************* Begin of program functions



void out_of_memory(int value, const char *file, int line)
{
  fprintf(stderr, _("Unable to reserve %d bytes of memory\n"), value);
  fprintf(stderr, _("File %s, Line %d\n"), file, line);
  fprintf(stderr, _("Exiting ...\n"));
  exit(1);
}




/* Note: Dealing with trees, we cannot call gtk_window_show_all as we
usually do, because that would also show the asts of the tree even in
collapsed state.  
We solve it by calling gtk_widget_show for each element.  */

// Adds an error entry for the given font
void add_error(FontData* fd, const char *msg, ...)
{
  va_list args;
  int size = 64;
  char* message = NULL;
  GtkTreeItem *font = NULL;

  // Construct a string (message) with the arguments  
  va_start(args, msg);
  while(1) {  
    if (message == NULL) message = (char*)g_malloc(size);
    else message = (char*)g_realloc(message, size);
    if (message == NULL) out_of_memory(size, __FILE__, __LINE__);
  
    int nchars = g_vsnprintf(message, size, msg, args);
    if (nchars < size) break;  // if enough space
    // if not enough space, enlarge size and try again
    size = nchars + 1; /* as in glibc >= 2.1 */
  }
  va_end(args);

  // Look for a tree item with the same font name
  for (GList *list=GTK_TREE(tree)->children; list; list=list->next) {
    gchar *name;
    GtkTreeItem *item = GTK_TREE_ITEM(list->data);
    GtkLabel *label = GTK_LABEL(GTK_BIN(item)->child);
    gtk_label_get(label, &name);
    if (strcmp(name, fd->fontFile) == 0) {
      font = item;
      break;
    }
  }
  // If no such font name, create a subtree with that name
  if (font == NULL) {
    font = GTK_TREE_ITEM(gtk_tree_item_new_with_label(fd->fontFile));
    gtk_tree_append(GTK_TREE(tree), GTK_WIDGET(font));
    gtk_widget_show(GTK_WIDGET(font));
    GtkWidget *subtree = gtk_tree_new();
    gtk_tree_item_set_subtree(font, subtree);
  }

  // Create a tree item in the subtree
  GtkWidget *messageitem = gtk_tree_item_new_with_label(message);
  gtk_tree_append(GTK_TREE(GTK_TREE_ITEM_SUBTREE(font)), messageitem);
  gtk_widget_show(messageitem);

  g_free(message);
  gtk_widget_show(errwindow);
  gdk_window_raise(errwindow->window);
}




void show_message_window(GtkWidget*, gpointer)
{
  gtk_widget_show(errwindow);
}



// Initialize the error messages window
void make_message_window(void)
{  
#ifdef USE_GNOME
  errwindow = gnome_dialog_new(_("Font Viewer messages"),
			       GNOME_STOCK_BUTTON_CLOSE,
			       NULL);
  gnome_dialog_close_hides(GNOME_DIALOG(errwindow), TRUE);
  gnome_dialog_set_close(GNOME_DIALOG(errwindow), TRUE);
  GtkWidget *diag_vbox = GNOME_DIALOG(errwindow)->vbox;
#else
  errwindow = gtk_window_new(GTK_WINDOW_TOPLEVEL); 
  gtk_window_set_title(GTK_WINDOW(errwindow), _("Font Viewer messages"));
  gtk_signal_connect_object(GTK_OBJECT(errwindow), "delete_event", 
			    GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			    GTK_OBJECT(errwindow));
  gtk_container_border_width(GTK_CONTAINER(errwindow), 5);

  GtkWidget *diag_vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(errwindow), diag_vbox);
  gtk_widget_show(diag_vbox);
#endif
  gtk_widget_set_name(errwindow, "message window"); 

  GtkWidget *scrolled_win = gtk_scrolled_window_new(NULL, NULL);  
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win), 
				 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);  
  gtk_widget_set_usize(scrolled_win, 400, 300);  
  gtk_box_pack_start(GTK_BOX(diag_vbox), scrolled_win, TRUE, FALSE, 0);
  gtk_widget_show(scrolled_win);

  tree = gtk_tree_new();
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled_win), 
					tree);
  gtk_tree_set_selection_mode(GTK_TREE(tree), GTK_SELECTION_SINGLE);
  gtk_tree_set_view_lines(GTK_TREE(tree), FALSE);
  gtk_tree_set_view_mode(GTK_TREE(tree), GTK_TREE_VIEW_ITEM);
  gtk_widget_show(tree);
}



// Called in ask_yes_no if window closed; equivalent to saying No
static bool delete_yes_no(GtkWidget *widget, GdkEvent *event, gpointer data)
{
  int *variable = (int*)data;
  *variable = 0;  // As if the user clicked 'No'
  return TRUE;    // Do not destroy
}


// Assign the value of the button to the variable; used in ask_yes_no
static void set_variable(GtkWidget *button, gpointer data)
{
  int *variable = (int*)data;
  int value = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), "value"));
  *variable = value;
}



/* Asks and waits till user responds; stays on top of parent
   Return 1 if yes, 0 if no; delete window acts like no
*/
bool ask_yes_no(GtkWidget *parent, const char* msg, ...)
{
  int rc = -1;
  va_list args;
  int size = 64;
  char* message = NULL;

  // Construct a string (message) with the arguments
  va_start(args, msg);
  while(1) {  
    if (message == NULL) message = (char*)g_malloc(size);
    else message = (char*)g_realloc(message, size);
    if (message == NULL) out_of_memory(size, __FILE__, __LINE__);

    int nchars = g_vsnprintf(message, size, msg, args);
    if (nchars < size) break;  // if enough space
    // if not enough space, enlarge size and try again
    size = nchars + 1; /* as in glibc >= 2.1 */
  }
  va_end(args);

#ifdef USE_GNOME
  GtkWidget *qw = gnome_message_box_new(message, 
                                GNOME_MESSAGE_BOX_QUESTION,
                                GNOME_STOCK_BUTTON_YES, 
                                GNOME_STOCK_BUTTON_NO, 
                                NULL);
  gtk_widget_set_name(qw, "question window");
  gnome_dialog_set_parent(GNOME_DIALOG(qw), GTK_WINDOW(parent));
  gnome_dialog_set_default(GNOME_DIALOG(qw), 1); /* Default is no */
  rc = gnome_dialog_run(GNOME_DIALOG(qw));
  g_free(message);

  switch (rc) {
  case 0: return 1;  /* Yes was clicked */
  case 1:            /* No was clicked */
  case -1: return 0; /* Window was closed */
  }
  return 0;
#else
  GtkWidget* qw = gtk_dialog_new();
  gtk_widget_set_name(qw, "question window");
  gtk_window_set_title(GTK_WINDOW(qw), _("Font Viewer question"));
  // Delete event (closing the window) is equivalent to 'No'
  gtk_signal_connect(GTK_OBJECT(qw), "delete_event", 
		     GTK_SIGNAL_FUNC(delete_yes_no), (gpointer)&rc);
  gtk_widget_realize(qw);

  // Buttons in action_area
  GtkWidget* okb = gtk_button_new(); 
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(qw)->action_area), 
		     okb, FALSE, FALSE, 0);
  gtk_object_set_data(GTK_OBJECT(okb), "value", GINT_TO_POINTER(1));  // Return 1 if pressed
  gtk_signal_connect(GTK_OBJECT(okb), "clicked", 
		     GTK_SIGNAL_FUNC(set_variable), (gpointer)&rc);
  GtkWidget* label = gtk_label_new(_("Yes"));
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(okb), label);

  GtkWidget* cancelb = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(qw)->action_area), cancelb, 
		     FALSE, FALSE, 0);
  gtk_object_set_data(GTK_OBJECT(cancelb), "value", GINT_TO_POINTER(0));  // Return 0 if pressed
  gtk_signal_connect(GTK_OBJECT(cancelb), "clicked", 
		     GTK_SIGNAL_FUNC(set_variable), (gpointer)&rc);
  label = gtk_label_new(_("No"));
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(cancelb), label);

  // Message area (vbox)
  GtkWidget* hboxm = gtk_hbox_new(FALSE, 10);
  gtk_container_border_width(GTK_CONTAINER(hboxm), 20);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(qw)->vbox), hboxm, TRUE, FALSE, 0);

  GtkWidget* qicon = makeicon(qw, HandOpen_xpm);
  gtk_box_pack_start(GTK_BOX(hboxm), qicon, FALSE, FALSE, 0);
  label = gtk_label_new(message);
  //gtk_label_set_line_wrap(GTK_LABEL(label), TRUE);
  gtk_box_pack_start(GTK_BOX(hboxm), label, FALSE, FALSE, 10);

  gtk_window_set_transient_for(GTK_WINDOW(qw), GTK_WINDOW(parent));
  gtk_window_position(GTK_WINDOW(qw), GTK_WIN_POS_CENTER);
  gtk_window_set_modal(GTK_WINDOW(qw), TRUE);
  gtk_widget_show_all(qw);

  gdk_beep();
  g_free(message);

  // Loop until user presses Yes or No
  while (rc == -1) gtk_main_iteration();
  // Buf, user pressed something at last; return its value
  gtk_widget_destroy(qw);
  return rc;
#endif
}




// Create an error message window with a OK button
void errormsg(GtkWidget *parent, const char* msg, ...)
{
  va_list args;
  int size = 64;
  char* message = NULL;

  // Construct a string (message) with the arguments  
  va_start(args, msg);
  while(1) {  
    if (message == NULL) message = (char*)g_malloc(size);
    else message = (char*)g_realloc(message, size);
    if (message == NULL) out_of_memory(size, __FILE__, __LINE__);

    int nchars = vsnprintf(message, size, msg, args);
    if (nchars < size) break;  // if enough space
    // if not enough space, enlarge size and try again
    size = nchars + 1; /* as in glibc >= 2.1 */
  }
  va_end(args);

#ifdef USE_GNOME
  GtkWidget *errw = gnome_message_box_new(message, 
                                GNOME_MESSAGE_BOX_ERROR,
                                GNOME_STOCK_BUTTON_CLOSE, 
                                NULL);
  gtk_widget_set_name(errw, "error window");
  gnome_dialog_set_parent(GNOME_DIALOG(errw), GTK_WINDOW(parent));
  gnome_dialog_set_default(GNOME_DIALOG(errw), 0);
  gnome_dialog_run(GNOME_DIALOG(errw));
#else
  // Now give 'message' as error message
  GtkWidget* errw = gtk_dialog_new();
  gtk_widget_set_name(errw, "error window");
  gtk_window_set_title(GTK_WINDOW(errw), _("Font Viewer error"));
  gtk_widget_realize(errw);

  // Buttons in action_area
  GtkWidget* okb = gtk_button_new();  
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(errw)->action_area), okb, 
		     FALSE, FALSE, 0);
  gtk_signal_connect_object(GTK_OBJECT(okb), "clicked", 
			    GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			    GTK_OBJECT(errw));
  GtkWidget* label = gtk_label_new(_("OK"));
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(okb), label);

  // Message area (vbox)
  GtkWidget* hboxm = gtk_hbox_new(FALSE, 10);
  gtk_container_border_width(GTK_CONTAINER(hboxm), 20);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(errw)->vbox), hboxm, TRUE, FALSE, 0);

  GtkWidget* erricon = makeicon(errw, errorxpm);
  gtk_box_pack_start(GTK_BOX(hboxm), erricon, FALSE, FALSE, 0);
  label = gtk_label_new(message);
  gtk_box_pack_start(GTK_BOX(hboxm), label, FALSE, FALSE, 10);

  gtk_window_set_transient_for(GTK_WINDOW(errw), GTK_WINDOW(parent));
  gtk_window_position(GTK_WINDOW(errw), GTK_WIN_POS_CENTER);
  gdk_window_set_group(errw->window, parent->window);
  gtk_window_set_modal(GTK_WINDOW(errw), TRUE);
  gtk_widget_show_all(errw);

  GTK_WIDGET_SET_FLAGS(okb, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(okb);
  gdk_beep();
#endif
  g_free(message);
}



// Make a GtkPixmap out of the given xpm data; uses style of toplevel
// for transparent color of xpm
GtkWidget* makeicon(GtkWidget* toplevel, char** xpm)
{
  GdkBitmap* mask;

  GtkStyle* style = gtk_widget_get_style(toplevel);
  GdkPixmap* pixmap = gdk_pixmap_create_from_xpm_d(toplevel->window, &mask,
                                      &style->bg[GTK_STATE_NORMAL], xpm);
  GtkWidget* pixmapwid = gtk_pixmap_new(pixmap, mask);
  return pixmapwid;
}



// The name says it all
void makeabout(GtkWidget*, gpointer)
{
  const gchar *title = _("Font Viewer");
  const gchar *comments = _("Font Viewer for PostScript Type 1 and TrueType fonts\nhttp://gfontview.sourceforge.net");
  const gchar *authors[] = {
    "Roberto Alameda <jroberto@linuxfan.com>",
      NULL
      };
  
#ifdef USE_GNOME
  GtkWidget *aboutw = gnome_about_new(title, VERSION, NULL, 
				      authors, comments, NULL);
  gtk_window_set_modal(GTK_WINDOW(aboutw), TRUE);
  gtk_widget_show(aboutw);
#else
  GtkWidget* aboutw = gtk_dialog_new();
  gtk_widget_set_name(aboutw, "about window");
  gtk_window_set_title(GTK_WINDOW(aboutw), _("About Font Viewer"));
  gtk_widget_realize(aboutw);
  
  GtkWidget* okb = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(aboutw)->action_area), okb, 
		     FALSE, FALSE, 0);
  gtk_signal_connect_object(GTK_OBJECT(okb), "clicked", 
			    GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(aboutw));
  GtkWidget* hbox = gtk_hbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(okb), hbox);
  GtkWidget* label = gtk_label_new(_("OK"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 25);

  // Message area
  GtkWidget* hboxm = gtk_hbox_new(FALSE, 10);
  gtk_container_border_width(GTK_CONTAINER(hboxm), 20);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(aboutw)->vbox), hboxm, TRUE, FALSE, 0);

  GtkWidget* icon = makeicon(aboutw, font_xpm);
  gtk_box_pack_start(GTK_BOX(hboxm), icon, FALSE, FALSE, 0);

  GString *msg = g_string_new("");
  g_string_sprintf(msg, "%s\nAuthor: %s\nVersion %s\n", 
		   comments, authors[0], VERSION);

  label = gtk_label_new(msg->str);
  gtk_box_pack_start(GTK_BOX(hboxm), label, FALSE, FALSE, 10);
  g_string_free(msg, TRUE);

  gtk_window_set_position(GTK_WINDOW(aboutw), GTK_WIN_POS_MOUSE);
  gtk_window_set_modal(GTK_WINDOW(aboutw), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(aboutw), GTK_WINDOW(mainwindow));
  GTK_WIDGET_SET_FLAGS(okb, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(okb);
  gtk_widget_show_all(aboutw);
#endif
}



struct EntryDialog*
make_entry_dialog(const char* labeltext, const char* entrytext)
{
  struct EntryDialog *temp = g_new(struct EntryDialog, 1);

  temp->hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER(temp->hbox), 0);

  temp->label = gtk_label_new(labeltext);
  gtk_misc_set_alignment(GTK_MISC(temp->label), 1.0, 0.5);
  gtk_widget_set_style(temp->label, stylebold);

  temp->entry = gtk_entry_new();
  gtk_widget_set_usize(temp->entry, 50, -1);
  gtk_entry_set_text(GTK_ENTRY(temp->entry), entrytext);
  gtk_box_pack_start(GTK_BOX(temp->hbox), temp->label, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(temp->hbox), temp->entry, FALSE, FALSE, 0);
  return temp;
}



// **** Printer dialog

static struct PrintDialog {
  GtkWidget *window, *okb, *cancelb;
  GtkWidget *printerentry, *fileentry, *browsebutton;
  GtkWidget *printerbutton, *filebutton, *subsetbutton;
  GtkWidget *fontbutton, *charsetbutton, *textbutton, *phrasebutton;
  GtkWidget *phraseentry, *sizeentry, *spacingentry;
} *prtdiag;



/* Output the used fonts to the PS trailer and free list structures */
static void write_usedfonts(FILE *ofp, GSList *fontlist)
{
  guint pos = 0; 
  for (GSList *p=fontlist; p; p=p->next) {
    if (pos == 0) fputs("%%+ font ", ofp);
    size_t len = strlen((const char*)p->data);
    if (pos+len > 120) {
      fputs("\n%%+ font ", ofp);
      pos = 0;
    }
    fputs((const char*)p->data, ofp);
    fputc(' ', ofp);
    pos += len+1;
    g_free(p->data);
  }
  fputc('\n', ofp);
  g_slist_free(fontlist);
}


/* 
   Write PostScript code with a size sample of all fonts in
   the selection to the given file
 */
static void do_sizesample(FILE* ofp, GList *selection)
{
  int rc;
  const char *psfontname;
  int subset = GTK_TOGGLE_BUTTON(prtdiag->subsetbutton)->active; // Wether to subset a font

  guint pageno = 1;  /* Keep track of written page numbers */
  GSList *flist = NULL; /* list of downloaded fonts */
  static const char *charlist = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789<=>:;?@ !\"#$%&'( )*+,-./ [ \\ ]^_ `{|}~";

  fputs("%%BeginProlog\n", ofp);
  fputs("%%BeginResource: procset SizeSamplerDict 0 0\n", ofp);
  fputs(size_samples, ofp);
  fputs("%%EndResource\n", ofp);
  fputs("%%EndProlog\n", ofp);
  fputs("%%BeginSetup\n", ofp);
  fputs("%%IncludeResource: font Times-Roman\n", ofp);
  fputs("%%IncludeResource: font Helvetica\n", ofp);
  fputs("/languagelevel where {pop} {/languagelevel 1 def} ifelse\n", ofp);

  /* If any of the fonts is TrueType, download code to check
     that the printer handles Type 42 fonts correctly */
  for (GList *p=selection; p; p=p->next) {
    int row = GPOINTER_TO_INT(p->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
    if (fd->fontType == TTFONT) {
      fputs(check_type42, ofp);  /* check_type42 is defined in ps.h */
      break;
    }
  }
  fputs("%%EndSetup\n", ofp);
  
  for (GList *sel=selection; sel; sel=sel->next) {
    while (gtk_events_pending()) gtk_main_iteration(); 
    int row = GPOINTER_TO_INT(sel->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
  
    /* 
       Put all needed glyphs in the 'codes' array

       charlist contains the list of chars to embed.
       Get the glyph name (using the ISOLatin1Encoding vector, which
       will be the encoding vector of the downloaded font) in order 
       to get the index of the glyph in the original font. 
       Then add the index to the list of indices to embed.

       Why not use a Unicode charmap from the font? Because although
       IsoLatin1 is the first page of Unicode by definition, most
       Unicode charmaps actually map that first page to Windows ANSI, 
       not to IsoLatin1. If you cannot beat them, join them.
       So we have to use the same encoding here and in the actual font
       (which will be ISOLatin1Encoding).
    */
    GArray *codes = g_array_new(FALSE, FALSE, sizeof(TT_UShort));
    array_insert(codes, 0);  // Index 0 (the default glyph) must be present
    if (subset && fd->fontType == TTFONT)
      for (TT_UShort i=0; i<strlen(charlist); i++) {
	TT_UShort index;
	extern char *ISOLatin1Encoding[];
	guint code = charlist[i]&0xFF;
	const char *psname = ISOLatin1Encoding[code];
	rc = tt_lookup(fd, psname, &index);
	if (rc) array_insert(codes, index);
      }
    long f_pos = ftell(ofp);
    fprintf(ofp, "%%%%Page: %u %u\n", pageno, pageno);
    /* Page description is surrounded by save - restore pair,
       to preserve VM of the PS interpreter */
    fputs("save SizeSamplerDict begin\n", ofp);
    if (fd->fontType == T1FONT)
      rc = t1_downloadfont(ofp, fd);  /* Write the font to the file*/
    else
      /* Download TT font as Type 42 font */
      rc = tt_downloadfont(ofp, fd, subset?codes:NULL);  
    if (rc == FALSE) {
      /* errors already shown via add_error */
      fseek(ofp, f_pos, SEEK_SET); /* Go back to before downloading the font */
      continue;
    }
    
    if (fd->fontType == T1FONT) 
      psfontname = fd->fontName;
    else 
      psfontname = getfontname(fd, TT_NAME_ID_PS_NAME);
    fprintf(ofp, "(%s) DoTitle\n", fd->fontName);
    fprintf(ofp, "(%s) DoSamples\n", psfontname);
    fputs("end restore showpage\n", ofp);
    g_array_free(codes, TRUE);
    /* if error writing to file, give up; error message comes in do_print */
    if (ferror(ofp)) return; 
    flist = g_slist_append(flist, g_strdup(psfontname));
    pageno++;
  }  

  fputs("%%Trailer\n", ofp);
  fprintf(ofp, "%%%%Pages: %d\n", pageno-1);
  fputs("%%DocumentSuppliedResources: procset SizeSamplerDict 0 0\n", ofp);
  write_usedfonts(ofp, flist);
}


/*
  make a font catalog (one sample sentence of each font)
  code based on fontsampler tool, by Johan Vromans
  (http://www.squirrel.nl/people/jvromans/software.html)
 */
static void do_phrasesample(FILE* ofp, GList *selection)
{
  int rc;
  const char *psfontname;
  int subset = GTK_TOGGLE_BUTTON(prtdiag->subsetbutton)->active; // Wether to subset a font
  guint pageno = 1;     /* Keep track of written page numbers */
  double fsize = 14.0;  /* Font sample size in points */
  GSList *flist = NULL; /* List of downloaded fonts */
  gint yposmax = 780 - MAX((int)fsize,10);  /* Maximum height in points of writable page */
  gint ypos = yposmax;

  char *message = gtk_entry_get_text(GTK_ENTRY(prtdiag->phraseentry));

  fputs("%%BeginProlog\n", ofp);
  fputs("%%BeginResource: procset PhraseSamplerDict 0 0\n", ofp);
  fputs(phrase_samples, ofp);
  fputs("%%EndResource\n", ofp);
  fputs("%%EndProlog\n", ofp);
  fputs("%%BeginSetup\n", ofp);
  fputs("%%IncludeResource: font Times-Roman\n", ofp);
  fputs("%%IncludeResource: font Helvetica\n", ofp);
  fputs("/languagelevel where {pop} {/languagelevel 1 def} ifelse\n", ofp);
  fputs("/StringSample (", ofp);
  for (char *p=message; *p; p++) {
    /* Escape PostScript delimiters */
    if (strchr("()<>[]{}/%", *p)) fputc('\\', ofp);
    fputc(*p, ofp);
  }
  fputs(") def\n", ofp);
  /* If any of the fonts is TrueType, download code to check
     that the printer handles Type 42 fonts correctly */
  for (GList *p=selection; p; p=p->next) {
    int row = GPOINTER_TO_INT(p->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
    if (fd->fontType == TTFONT) {
      fputs(check_type42, ofp);  /* check_type42 is defined in ps.h */
      break;
    }
  }
  fputs("%%EndSetup\n", ofp);

  for (GList *sel=selection; sel; sel=sel->next) {
    while (gtk_events_pending()) gtk_main_iteration(); 
    int row = GPOINTER_TO_INT(sel->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);

    GArray *codes = g_array_new(FALSE, FALSE, sizeof(TT_UShort));
    array_insert(codes, 0);  // Index 0 (the default glyph) must be present
    if (subset && fd->fontType == TTFONT)
      for (TT_UShort i=0; i<strlen(message); i++) {
	TT_UShort index;
	extern char *ISOLatin1Encoding[];
	guint code = message[i]&0xFF;
	const char *psname = ISOLatin1Encoding[code];
	rc = tt_lookup(fd, psname, &index);
	if (rc) array_insert(codes, index);
      }

    if (ypos < 40) {
      fputs("end restore showpage\n", ofp);
      ypos = yposmax;
      pageno++;
    }
    long f_pos = ftell(ofp);
    if (ypos == yposmax) { // Begin of new page
      fprintf(ofp, "%%%%Page: %u %u\n", pageno, pageno);
      /* Page description is surrounded by save - restore pair,
	 to preserve VM of the PS interpreter */
      fputs("save PhraseSamplerDict begin\n", ofp);
      fprintf(ofp, "(Page %u) 780 Header\n", pageno);
    }

    /* Each font download, that is, each line of the page,
       is surrounded by an additional save-restore pair */
    fputs("save\n", ofp);
    if (fd->fontType == T1FONT)
      rc = t1_downloadfont(ofp, fd);  /* Write the font to the file*/
    else
      /* Download TT font as Type 42 font */
      rc = tt_downloadfont(ofp, fd, subset?codes:NULL);  
    if (rc == FALSE) {
      /* errors already shown via add_error */
      fseek(ofp, f_pos, SEEK_SET); /* Go back to before downloading the font */
      continue;
    }

    if (fd->fontType == T1FONT) 
      psfontname = fd->fontName;
    else 
      psfontname = getfontname(fd, TT_NAME_ID_PS_NAME);
    /* Inquiry actual locale for decimal dot, modify it to use an english
       decimal dot, do it and set back to original locale */
    char *loc = setlocale(LC_NUMERIC, NULL);
    setlocale(LC_NUMERIC, "C");
    fprintf(ofp, "(%s) (%s) %d FontName", fd->fontName, fd->fontFile, ypos);
    fprintf(ofp, "(%s) %.1f %d FontSample\n", psfontname, fsize, ypos);
    fputs("restore\n", ofp);
    setlocale(LC_NUMERIC, loc);

    ypos -= (gint)(1.4*MAX(10,fsize));
    g_array_free(codes, TRUE);
    /* if error writing to file, give up; error message comes in do_print */
    if (ferror(ofp)) return; 
    flist = g_slist_append(flist, g_strdup(psfontname));
  }

  fputs("end restore showpage\n", ofp);
  fputs("%%Trailer\n", ofp);
  fprintf(ofp, "%%%%Pages: %d\n", pageno);
  fputs("%%DocumentSuppliedResources: procset PhraseSamplerDict 0 0\n", ofp);
  write_usedfonts(ofp, flist);
}




/*
  Write PostScript code to the given file with a sample text, 
  for all fonts in the selection
 */
static void do_textsample(FILE* ofp, GList *selection)
{
  int rc;
  guint pageno = 1;  /* Keep track of written page numbers */
  int subset = GTK_TOGGLE_BUTTON(prtdiag->subsetbutton)->active; // Whether to subset a font
  const char *psfontname;
  GSList *flist = NULL; /* list of downloaded fonts */
  static const char *charlist = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789.,;: []";

  double fontsize = g_strtod(gtk_entry_get_text
			     (GTK_ENTRY(prtdiag->sizeentry)), NULL);
  double spacing = gtk_spin_button_get_value_as_float
    (GTK_SPIN_BUTTON(prtdiag->spacingentry));

  fputs("%%BeginProlog\n", ofp);
  fputs("%%BeginResource: procset TextSamplerDict 0 0\n", ofp);
  fputs(text_samples, ofp);
  fputs("\n/textstring (", ofp);
  fputs(sample_text, ofp);
  fputs(") def\n", ofp);
  fputs("%%EndResource\n", ofp);
  fputs("%%EndProlog\n", ofp);
  fputs("%%BeginSetup\n", ofp);
  fputs("%%IncludeResource: font Times-Roman\n", ofp);
  fputs("%%IncludeResource: font Helvetica\n", ofp);
  fputs("/languagelevel where {pop} {/languagelevel 1 def} ifelse\n", ofp);

  /* If any of the fonts is TrueType, download code to check
     that the printer handles Type 42 fonts correctly */
  for (GList *p=selection; p; p=p->next) {
    int row = GPOINTER_TO_INT(p->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
    if (fd->fontType == TTFONT) {
      fputs(check_type42, ofp);  /* check_type42 is defined in ps.h */
      break;
    }
  }
  fputs("%%EndSetup\n", ofp);

  for (GList *sel=selection; sel; sel=sel->next) {
    while (gtk_events_pending()) gtk_main_iteration(); 
    int row = GPOINTER_TO_INT(sel->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
    GArray *codes = g_array_new(FALSE, FALSE, sizeof(TT_UShort));
    array_insert(codes, 0);  // Index 0 (the default glyph) must be present

    if (fd->fontType == TTFONT)
      for (TT_UShort i=0; i<strlen(charlist); i++) {
	TT_UShort index;
	extern char *ISOLatin1Encoding[];
	guint code = charlist[i]&0xFF;
	const char *psname = ISOLatin1Encoding[code];
	rc = tt_lookup(fd, psname, &index);
	if (rc) array_insert(codes, index);
      }

    long f_pos = ftell(ofp);
    fprintf(ofp, "%%%%Page: %u %u\n", pageno, pageno);
  
    /* Page description is surrounded by save - restore pair,
       to preserve VM of the PS interpreter */
    fputs("save TextSamplerDict begin\n", ofp);
    if (fd->fontType == T1FONT)
      rc = t1_downloadfont(ofp, fd);  // Write the font to the file
    else
      /* Download TT font as Type 42 font */
      rc = tt_downloadfont(ofp, fd, subset?codes:NULL);
    if (rc == FALSE) {
      /* errors already shown via add_error */
      fseek(ofp, f_pos, SEEK_SET); /* Go back to before downloading the font */
      continue;
    }

    if (fd->fontType == T1FONT) 
      psfontname = fd->fontName;
    else 
      psfontname = getfontname(fd, TT_NAME_ID_PS_NAME);
    
    /* Inquiry actual locale for decimal dot, modify it to use an english
       decimal dot, do it and set back to original locale */
    char *loc = setlocale(LC_NUMERIC, NULL);
    setlocale(LC_NUMERIC, "C");
    fprintf(ofp, "(%s) %.1f %.1f DoSamples\n", psfontname, fontsize, spacing);
    fprintf(ofp, "(%s) DoTitle\n", fd->fontName);
    setlocale(LC_NUMERIC, loc);
    
    fputs("end restore showpage\n", ofp);
    g_array_free(codes, TRUE);
    /* if error writing to file, give up; error message comes in do_print */
    if (ferror(ofp)) return; 
    flist = g_slist_append(flist, g_strdup(psfontname));
    pageno++;
  }

  fputs("%%Trailer\n", ofp);
  fprintf(ofp, "%%%%Pages: %d\n", pageno-1);
  fputs("%%DocumentSuppliedResources: procset TextSamplerDict 0 0\n", ofp);
  write_usedfonts(ofp, flist);
}



/* 
  Write PostScript code to the given file with a map of all characters
  in the fonts in the selection.
  mapsample code based on fontsampler tool, by Johan Vromans
  (http://www.squirrel.nl/people/jvromans/software.html)
*/
static void do_mapsample(FILE* ofp, GList *selection)
{
  int rc;
  guint pageno = 1;  /* Keep track of written page numbers */
  char *psfontname;
  GSList *flist = NULL; /* list of downloaded fonts */
  int subset = GTK_TOGGLE_BUTTON(prtdiag->subsetbutton)->active; // Whether to subset a font

  fputs("%%BeginProlog\n", ofp);
  fputs("%%BeginResource: procset MapSamplerDict 0 0\n", ofp);
  fputs(map_samples, ofp);
  fputs("%%EndResource\n", ofp);
  fputs("%%EndProlog\n", ofp);
  fputs("%%BeginSetup\n", ofp);
  fputs("%%IncludeResource: font Times-Roman\n", ofp);
  fputs("%%IncludeResource: font Helvetica\n", ofp);
  fputs("/languagelevel where {pop} {/languagelevel 1 def} ifelse\n", ofp);

  /* If any of the fonts is TrueType, download code to check
     that the printer handles Type 42 fonts correctly */
  for (GList *p=selection; p; p=p->next) {
    int row = GPOINTER_TO_INT(p->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
    if (fd->fontType == TTFONT) {
      fputs(check_type42, ofp);  /* check_type42 is defined in ps.h */
      break;
    }
  }
  fputs("%%EndSetup\n", ofp);
  
  for (GList *sel=selection; sel; sel=sel->next) {
    char **glyphnames;
    int num_glyphs = -1;

    while (gtk_events_pending()) gtk_main_iteration(); 
    int row = GPOINTER_TO_INT(sel->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
    if (fd->fontType == T1FONT) {
      if (T1_LoadFont(fd->t1data.fontID)==-1) {
	add_error(fd, _("Cannot load font"));
	continue;
      }
      psfontname = fd->fontName;
      glyphnames = T1_GetAllCharNames(fd->t1data.fontID);
      if (!glyphnames) {
	add_error(fd, _("Cannot get character names"));
	continue;
      }
      while (glyphnames[++num_glyphs]);  // Calculate size of ptr array (last element is NULL)
    }
    else {
      /* tt_downloadfont will overwrite fontname, so strdup it */
      psfontname = g_strdup(getfontname(fd, TT_NAME_ID_PS_NAME));
      num_glyphs = fd->num_glyphs;
    }

    guint n_pages = 1 + (num_glyphs-1)/256; // Number of pages to print
    for (guint page=0; page<n_pages; page++) {
      guint maxnum = MIN(256, num_glyphs - page*256);  // glyphs on page
      GArray *codes = g_array_new(FALSE, FALSE, sizeof(TT_UShort));
      array_insert(codes, 0);  // Index 0 (the default glyph) must be present
      if (n_pages > 1) {  // Subset only if more than 256 characters in font
	for (TT_UShort i=0; i<maxnum; i++)
	  array_insert(codes, page*256+i);
      }
      long f_pos = ftell(ofp);
      fprintf(ofp, "%%%%Page: %u %u\n", pageno, pageno);
      fputs("save MapSamplerDict begin\n", ofp);
      if (fd->fontType == T1FONT)
	rc = t1_downloadfont(ofp, fd);  // Write the font to the file
      else
	rc = tt_downloadfont(ofp, fd, (subset&&n_pages>1)?codes:NULL);  
      if (rc == FALSE) {
	/* errors already shown via add_error */
	/* Go back to before downloading the font */
	fseek(ofp, f_pos, SEEK_SET); 
	continue;
      }
      /* 
	 If only one page (up to 256 glyphs in font), use FontShow0
	 If more, for each page make encoding and use FontShowV
      */
      fprintf(ofp, "(%u/%u) Header\n", page+1, n_pages);
      fprintf(ofp, "(%s) DoTitle\n", fd->fontName);
      if (n_pages == 1)
	fprintf(ofp, "(%s) FontShow0\n", psfontname);
      else {
	/* Write encoding vector */
	fputs("[", ofp);
	for (guint i=0, pos=1; i<maxnum; i++) {
	  char *cname;
	  if (fd->fontType == T1FONT)
	    cname = glyphnames[page*256 + i];
	  else {
	    int error = TT_Get_PS_Name(fd->ttdata.face, page*256 + i, &cname);
	    if (error) {
	      add_error(fd, _("Cannot get PostScript name for glyph %u: %s"), 
			page*256 + i, TT_ErrToString18(error));
	      cname = ".notdef";
	    }
	  }
	  size_t len = strlen(cname);
	  if (pos + len + 2 > 128) {
	    fputc('\n', ofp);
	    pos = 1;
	  }
	  fprintf(ofp, "/%s ", cname);
	  pos += len+2;
	}
	fputs("]\n", ofp);
	fprintf(ofp, "(%s) FontShowV\n", psfontname);
      }
      fputs("end restore showpage\n", ofp);
      g_array_free(codes, TRUE);
      /* if error writing to file, give up; error message comes in do_print */
      if (ferror(ofp)) return; 
      pageno++;
    }  // End page loop
    flist = g_slist_append(flist, g_strdup(psfontname));
    if (fd->fontType==TTFONT) g_free(psfontname);
  }  // End font loop

  fputs("%%Trailer\n", ofp);
  fprintf(ofp, "%%%%Pages: %d\n", pageno-1);
  fputs("%%DocumentSuppliedResources: procset MapSamplerDict 0 0\n", ofp);
  write_usedfonts(ofp, flist);
}




/*
  Actually print the font sample according to the settings in the options
  window
*/
static void do_print(GtkWidget*, gpointer ptr)
{
  int rc;
  time_t ti;
  static GString *command = NULL;
  const char *psfile;
  int printtoprinter = GTK_TOGGLE_BUTTON(prtdiag->printerbutton)->active;

  GList *selection = GTK_CLIST(fontclistw)->selection;

  if (command==NULL) command = g_string_new("");
  const char *printername = gtk_entry_get_text(GTK_ENTRY(prtdiag->printerentry));
  if (*printername == '\0') printername = "lp";
  const char *filename = gtk_entry_get_text(GTK_ENTRY(prtdiag->fileentry));
  double fontsize = g_strtod(gtk_entry_get_text(GTK_ENTRY(prtdiag->sizeentry)), NULL);

  if (GTK_TOGGLE_BUTTON(prtdiag->textbutton)->active && 
      (fontsize<1.0 || fontsize>50.0)) {
    errormsg(prtdiag->window, _("Invalid size"));
    return;
  }
  if (printtoprinter) {     // if printing to printer
    psfile = tmpnam(NULL);  // generate temporary file to print it later
    if (psfile == NULL) {
      errormsg(prtdiag->window, _("Cannot generate temporary file: %s"), 
	       g_strerror(errno));
      gtk_widget_hide(prtdiag->window);
      return;
    }
  }
  else {  // if printing to file
    struct stat buf;
    if (stat(filename, &buf)==0) {
      if (S_ISDIR(buf.st_mode)) {
	errormsg(prtdiag->window, _("Cannot print to a directory"));
	return;
      }
      int rc = ask_yes_no(prtdiag->window, 
			  _("File %s already exists\nDo you want to overwrite it?"), 
			  filename);
      if (rc == 0) return;  // if no
    }
    psfile = filename;  // if yes
  }

  FILE *ofp = fopen(psfile, "w");
  if (ofp == NULL) {
    errormsg(prtdiag->window, _("Cannot open file %s: %s"), 
	     psfile, g_strerror(errno));
    return;
  }
  set_window_busy(prtdiag->window, TRUE);
  ti = time(NULL);
  fputs("%!PS-Adobe-3.0\n", ofp);
  fprintf(ofp, "%%%%Creator: gfontview %s\n", VERSION);
  fputs("%%Title: Font Samples\n", ofp);
  fprintf(ofp, "%%%%CreationDate: %s", ctime(&ti));  // No \n!
  fputs("%%Pages: (atend)\n", ofp);
  fputs("%%DocumentNeededResources: font Times-Roman Helvetica\n", ofp);
  fputs("%%DocumentSuppliedResources: (atend)\n", ofp);
  fputs("%%EndComments\n", ofp);

  // PostScript strings and commands defined in ps.h
  if (GTK_TOGGLE_BUTTON(prtdiag->fontbutton)->active) { 
    do_sizesample(ofp, selection);
  }
  else if (GTK_TOGGLE_BUTTON(prtdiag->textbutton)->active) {
    do_textsample(ofp, selection);
  }
  else if (GTK_TOGGLE_BUTTON(prtdiag->charsetbutton)->active) {
    do_mapsample(ofp, selection);
  }
  else if (GTK_TOGGLE_BUTTON(prtdiag->phrasebutton)->active) {
    do_phrasesample(ofp, selection);
  }
  

  fputs("%%EOF\n", ofp);
  rc = ferror(ofp); 
  set_window_busy(prtdiag->window, FALSE);
  if (fclose(ofp) || rc) {  // if error writing to file
    errormsg(prtdiag->window, _("Error writing to file %s: %s"), 
	     psfile, g_strerror(errno));
    return;
  }

  if (!printtoprinter) {  // If printing to file, we are done
    gtk_widget_hide(prtdiag->window);
    return;
  }

  // Print to printer here
  char buf[256];
  g_string_sprintf(command, "%s%s %s 2>&1", PRINTER_SPOOLER, 
		   printername, psfile);
  FILE *stream = popen(command->str, "r");
  if (stream == NULL) {
    errormsg(prtdiag->window, _("Cannot start printer spooler: %s"), 
	     g_strerror(errno));
    unlink(psfile);
    return;
  }
  char *err = fgets(buf, sizeof(buf), stream); // Not NULL if lpr generates output
  rc = pclose(stream);
  unlink(psfile);
  if (rc == -1) {
    errormsg(prtdiag->window, _("System error in printer spooler command: %s"), 
	     g_strerror(errno));
    return;
  }
  if (!WIFEXITED(rc)) {  // Abnormal termination of spooler
    errormsg(prtdiag->window, _("Printer spooler terminated abnormally"));
    return;
  }
  if (WEXITSTATUS(rc) != 0) {
    errormsg(prtdiag->window, _("Error in printer spooler\n%s"), err?err:"");
    return;
  }
  gtk_widget_hide(prtdiag->window);  // Hide only if successful
}



// The user selected a file to print to; put it in the corresponding
// entry field and destroy the file select window
static void okselprintfile(gpointer diag)
{
  GtkWidget *filesel = GTK_WIDGET(diag);
  const char *newfile = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  gtk_entry_set_text(GTK_ENTRY(prtdiag->fileentry), newfile);
  gtk_widget_destroy(filesel);
}


// Called when user wants to select the file to print to
static void selectprintfile(GtkWidget*, gpointer)
{
  set_window_busy(mainwindow, TRUE);
  set_window_busy(prtdiag->window, TRUE);

  GtkWidget* filesel = gtk_file_selection_new(_("Font Viewer file selection"));
  gtk_file_selection_hide_fileop_buttons(GTK_FILE_SELECTION(filesel));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			    GTK_OBJECT(filesel));
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(okselprintfile), 
			    GTK_OBJECT(filesel));
  gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), 
				  gtk_entry_get_text(GTK_ENTRY(prtdiag->fileentry)));
  gtk_window_position(GTK_WINDOW(filesel), GTK_WIN_POS_CENTER);
  gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);
  gtk_widget_show(filesel);
  gtk_window_set_transient_for(GTK_WINDOW(filesel), GTK_WINDOW(mainwindow));
  set_window_busy(mainwindow, FALSE);
  set_window_busy(prtdiag->window, FALSE);
}



static GList* get_printer_list(void)
{
  GList *list = NULL;
  char buf[256];

  FILE *fp = fopen("/etc/printcap", "r");
  if (fp == NULL) return list;

  while (fgets(buf, sizeof(buf), fp)) {
    char *e, *p = buf;
    while (isspace(*p)) p++;  // Skip leading blanks
    if (*p == '#') continue;  // Skip comment line
    if (*p == ':') continue;  // Skip line other than the first
    // Found valid line: name1|name2|name3|....
    // Take the first name
    e = p;
    while (*e && *e != '|') e++;
    *e = '\0';
    list = g_list_append(list, g_strdup(p));
  }
  fclose(fp);
  return list;
}





/* 
   Callback called when user wants to print the selected fonts in the font list
   It shows a print options dialog. The dialog is not destroyed when finished,
   just hidden. This is done to maintain the settings from the user 
   for the next font to print.
*/
void printfonts(GtkWidget*, gpointer)
{
  GtkWidget *label, *hbox, *diag_vbox;
  int rc;

  GList *selection = GTK_CLIST(fontclistw)->selection;
  if (selection == NULL) {
    errormsg(mainwindow, _("No row selected"));
    return;
  }

  // If print dialog already exists, show it
  if (prtdiag != NULL) {
    gtk_widget_show(prtdiag->window);
    return;
  }

  // Otherwise make print dialog
  prtdiag = g_new(struct PrintDialog, 1);

#ifdef USE_GNOME
  prtdiag->window = gnome_dialog_new(_("Font Viewer print dialog"),
				     GNOME_STOCK_BUTTON_OK,
				     GNOME_STOCK_BUTTON_CANCEL,
				     NULL);
  /* Closing the dialog just hides it, not destroy */
  gnome_dialog_close_hides(GNOME_DIALOG(prtdiag->window), TRUE);
  /* OK prints */
  gnome_dialog_button_connect(GNOME_DIALOG(prtdiag->window), 0,
			      GTK_SIGNAL_FUNC(do_print), 
			      NULL);
  /* Cancel closes window */
  gnome_dialog_button_connect_object(GNOME_DIALOG(prtdiag->window), 1, 
				     GTK_SIGNAL_FUNC(gnome_dialog_close),
				     GTK_OBJECT(prtdiag->window));
  gnome_dialog_set_default(GNOME_DIALOG(prtdiag->window), 1);
  gnome_dialog_set_parent(GNOME_DIALOG(prtdiag->window), 
			  GTK_WINDOW(mainwindow));
  gtk_window_set_modal(GTK_WINDOW(prtdiag->window), TRUE);
  diag_vbox = GNOME_DIALOG(prtdiag->window)->vbox;

#else
  prtdiag->window = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(prtdiag->window), 
		       _("Font Viewer print dialog"));
  gtk_signal_connect_object(GTK_OBJECT(prtdiag->window), "delete_event", 
			    GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			    GTK_OBJECT(prtdiag->window));

  // Button area (action_area)
  prtdiag->okb = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(prtdiag->window)->action_area), 
		     prtdiag->okb, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(prtdiag->okb), "clicked", 
		     GTK_SIGNAL_FUNC(do_print), 
		     (gpointer)NULL);
  label = gtk_label_new(_("OK"));
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(prtdiag->okb), label);
  
  prtdiag->cancelb = gtk_button_new();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(prtdiag->window)->action_area), 
		     prtdiag->cancelb, FALSE, FALSE, 0);
  gtk_signal_connect_object(GTK_OBJECT(prtdiag->cancelb), "clicked", 
			    GTK_SIGNAL_FUNC(gtk_widget_hide), 
			    GTK_OBJECT(prtdiag->window));
  label = gtk_label_new(_("Cancel"));
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(prtdiag->cancelb), label);

  gtk_window_position(GTK_WINDOW(prtdiag->window), GTK_WIN_POS_CENTER);
  gtk_window_set_modal(GTK_WINDOW(prtdiag->window), TRUE);
  gtk_window_set_transient_for(GTK_WINDOW(prtdiag->window), 
			       GTK_WINDOW(mainwindow));
  diag_vbox = GTK_DIALOG(prtdiag->window)->vbox;
#endif

  gtk_widget_set_name(prtdiag->window, "print window");

  // **** Dialog area (vbox)
  
  // 'Print to' area
  GtkWidget *printframe = gtk_frame_new(_("Print to:"));
  gtk_container_border_width(GTK_CONTAINER(printframe), 5);
  gtk_box_pack_start(GTK_BOX(diag_vbox), printframe, 
		     TRUE, FALSE, 0);
  
  GtkWidget *table = gtk_table_new(2, 4, FALSE); // A table with 2 rows, 4 columns
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(printframe), table);
  
  prtdiag->printerbutton = gtk_radio_button_new_with_label(NULL, 
							   _("Printer:"));
  gtk_table_attach(GTK_TABLE(table), prtdiag->printerbutton, 0, 1, 0, 1, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  prtdiag->filebutton = gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(prtdiag->printerbutton)), 
     _("File:"));
  gtk_table_attach(GTK_TABLE(table), prtdiag->filebutton, 0, 1, 1, 2, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(prtdiag->printerbutton), 
			      TRUE);  // Set default pressed
  
  GtkWidget *pcombo = gtk_combo_new();
  prtdiag->printerentry = GTK_COMBO(pcombo)->entry;
  // Put printers into the list
  GList *ilist = get_printer_list();
  if (ilist) {
    gtk_combo_set_popdown_strings(GTK_COMBO(pcombo), ilist);
    g_list_free(ilist);
  }

  gtk_entry_set_text(GTK_ENTRY(prtdiag->printerentry), "lp");
  gtk_table_attach(GTK_TABLE(table), pcombo, 1, 2, 0, 1, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  
  prtdiag->fileentry = gtk_entry_new();
  GString *file = g_string_new("");
  g_string_sprintf(file, "%s/sample.ps", getenv("HOME"));
  gtk_entry_set_text(GTK_ENTRY(prtdiag->fileentry), file->str);
  g_string_free(file, TRUE);
  gtk_table_attach(GTK_TABLE(table), prtdiag->fileentry, 1, 3, 1, 2, 
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND), GTK_EXPAND, 0, 0);
  /* Entry field is DnD target */
  gtk_drag_dest_set(prtdiag->fileentry, GTK_DEST_DEFAULT_ALL,
		    dnd_target_table, dnd_targets-1,
		    (enum GdkDragAction)(GDK_ACTION_COPY | GDK_ACTION_MOVE)); 
  gtk_signal_connect(GTK_OBJECT(prtdiag->fileentry), "drag_data_received",
		     GTK_SIGNAL_FUNC(combo_drag_data_received), NULL);

  prtdiag->browsebutton = gtk_button_new();
  label = gtk_label_new(_("Browse..."));
  gtk_misc_set_padding(GTK_MISC(label), 3, 3);
  gtk_container_add(GTK_CONTAINER(prtdiag->browsebutton), label);
  gtk_signal_connect(GTK_OBJECT(prtdiag->browsebutton), "clicked", 
		     GTK_SIGNAL_FUNC(selectprintfile), NULL);
  gtk_table_attach(GTK_TABLE(table), prtdiag->browsebutton, 3, 4, 1, 2, 
		   GTK_FILL, GTK_EXPAND, 10, 10);

  
  // 'Options' area
  GtkWidget *optionsframe = gtk_frame_new(_("Options:"));
  gtk_container_border_width(GTK_CONTAINER(optionsframe), 5);
  gtk_box_pack_start(GTK_BOX(diag_vbox), optionsframe, 
		     TRUE, FALSE, 0);

  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_add(GTK_CONTAINER(optionsframe), hbox);
  prtdiag->subsetbutton = gtk_check_button_new_with_label(_("Subset fonts"));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(prtdiag->subsetbutton), TRUE);
  gtk_box_pack_start(GTK_BOX(hbox), prtdiag->subsetbutton, FALSE, FALSE, 0);


  // 'Select sample' area
  GtkWidget *sampleframe = gtk_frame_new(_("Select sample:"));
  gtk_container_border_width(GTK_CONTAINER(sampleframe), 5);
  gtk_box_pack_start(GTK_BOX(diag_vbox), sampleframe, 
		     TRUE, FALSE, 0);
  
  table = gtk_table_new(4, 3, FALSE); // A table with 4 rows, 3 columns
  gtk_table_set_row_spacings(GTK_TABLE(table), 5);
  gtk_container_add(GTK_CONTAINER(sampleframe), table);
  
  prtdiag->fontbutton = gtk_radio_button_new_with_label(NULL, 
							_("Size samples"));
  gtk_table_attach(GTK_TABLE(table), prtdiag->fontbutton, 0, 1, 0, 1, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  prtdiag->charsetbutton = gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(prtdiag->fontbutton)), 
     _("Font map"));
  gtk_table_attach(GTK_TABLE(table), prtdiag->charsetbutton, 0, 1, 1, 2, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  prtdiag->phrasebutton = gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(prtdiag->fontbutton)), 
     _("Font catalog"));
  gtk_table_attach(GTK_TABLE(table), prtdiag->phrasebutton, 0, 1, 2, 3, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  prtdiag->textbutton = gtk_radio_button_new_with_label
    (gtk_radio_button_group(GTK_RADIO_BUTTON(prtdiag->fontbutton)), 
     _("Text samples"));
  gtk_table_attach(GTK_TABLE(table), prtdiag->textbutton, 0, 1, 3, 4, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  gtk_toggle_button_set_state(GTK_TOGGLE_BUTTON(prtdiag->fontbutton), TRUE);  // Set default pressed
  
  /* Extra entry fields in Text Sample */
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER(hbox), 0);
  gtk_table_attach(GTK_TABLE(table), hbox, 1, 2, 3, 4, 
		   GTK_FILL, GTK_EXPAND, 0, 0);
  prtdiag->sizeentry = gtk_entry_new();
  gtk_widget_set_usize(prtdiag->sizeentry, 40, -1);
  gtk_entry_set_text(GTK_ENTRY(prtdiag->sizeentry), "12");
  gtk_box_pack_start(GTK_BOX(hbox), prtdiag->sizeentry, FALSE, FALSE, 0);
  label = gtk_label_new(_("points"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);
  
  hbox = gtk_hbox_new(FALSE, 5);
  gtk_container_border_width(GTK_CONTAINER(hbox), 0);
  gtk_table_attach(GTK_TABLE(table), hbox, 2, 3, 3, 4, 
		   GTK_FILL, GTK_EXPAND, 10, 10);
  GtkAdjustment *adj = (GtkAdjustment*)gtk_adjustment_new(1.0, 1.0, 2.0, 
							  0.1, 0.2, 0.0);
  prtdiag->spacingentry = gtk_spin_button_new(adj, 0, 1);
  gtk_box_pack_start(GTK_BOX(hbox), prtdiag->spacingentry, FALSE, FALSE, 0);
  label = gtk_label_new(_("spacing"));
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 0);

  /* Extra entry fields in Phrase Sample */
  prtdiag->phraseentry = gtk_entry_new();
  gtk_table_attach(GTK_TABLE(table), prtdiag->phraseentry, 1, 3, 2, 3, 
		   (GtkAttachOptions)(GTK_FILL|GTK_EXPAND), GTK_EXPAND, 10, 0);
  gtk_entry_set_text(GTK_ENTRY(prtdiag->phraseentry), "ABCDEFGHIJKLMabcdefghijkml0123456789");
  gtk_entry_set_position(GTK_ENTRY(prtdiag->phraseentry), 0);


  /* End */
  gtk_widget_show_all(prtdiag->window);
}



// *** Save names


static void do_savenames(GtkWidget *button, gpointer)
{
  struct stat buf;
  int rc;

  GtkWidget *filesel = gtk_widget_get_toplevel(button);
  char *file = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  if (strchr(file, '*')) return;

  if (stat(file, &buf)==0) {
    if (S_ISDIR(buf.st_mode)) {
      errormsg(filesel, _("Selected file is a directory"));
      return;
    }
    rc = ask_yes_no(filesel, 
		    _("File %s already exists\nDo you want to overwrite it?"), 
		    g_basename(file));
    if (rc == 0) return; // if no
  }
  FILE *fp = fopen(file, "w");
  if (fp == NULL) {
    errormsg(filesel, _("Cannot open file %s: %s"), file, g_strerror(errno));
    return;
  }
  for (GSList* p=fontlist; p; p=p->next) {
    FontData* fd = (FontData *)p->data;
    fprintf(fp, "%s\t\t%s\n", g_basename(fd->fontFile), fd->fontName);
  }
  rc = ferror(fp); 
  if (fclose(fp) || rc) {  // if error writing to file
    errormsg(filesel, _("Error writing to file %s: %s"), 
	     file, g_strerror(errno));
    return;
  }
  gtk_widget_hide(filesel);
}


// Save a list of all font files in the directory to a text file
void save_names(GtkWidget*, gpointer)
{
  static GtkWidget *filesel = NULL;
  
  if (filesel == NULL) {
    filesel = gtk_file_selection_new(_("Save font names"));
    gtk_window_position(GTK_WINDOW(filesel), GTK_WIN_POS_CENTER);
    gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);
    //gtk_file_selection_complete(GTK_FILE_SELECTION(filesel), "*.txt");
    GtkWidget *cancelb = GTK_FILE_SELECTION(filesel)->cancel_button;
    GtkWidget *okb = GTK_FILE_SELECTION(filesel)->ok_button;
    
    gtk_signal_connect_object(GTK_OBJECT(cancelb), "clicked", 
			      GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			      GTK_OBJECT(filesel));
    gtk_signal_connect(GTK_OBJECT(okb), "clicked",
		       GTK_SIGNAL_FUNC(do_savenames), NULL);
    gtk_signal_connect_object(GTK_OBJECT(filesel), "delete_event", 
			      GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			      GTK_OBJECT(filesel));
  }

  gtk_window_set_transient_for(GTK_WINDOW(filesel), GTK_WINDOW(mainwindow));
  gtk_widget_show(filesel);
}



// **** Show properties


// Show the properties of all selected fonts
void show_properties(GtkWidget*, gpointer)
{
  GList *selection = GTK_CLIST(fontclistw)->selection;
  if (selection == NULL) {
    errormsg(mainwindow, _("No row selected"));
    return;
  }

  set_window_busy(mainwindow, TRUE);
  while (selection) {
    int row = GPOINTER_TO_INT(selection->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
    GtkWidget *window = show_font_properties(fd);
    gtk_widget_show(window);
    selection = selection->next;
  }
  set_window_busy(mainwindow, FALSE);
}



// Show the properties of the selected font
GtkWidget* show_font_properties(FontData *fd)
{
  static GString *str = NULL;
  if (str == NULL) str = g_string_new(NULL);
  g_string_sprintf(str, _("Font Properties: %s"), fd->fontName);

#ifdef USE_GNOME
  GtkWidget *window = gnome_dialog_new(str->str, 
				       GNOME_STOCK_BUTTON_CLOSE, NULL);
  GtkWidget *body = GNOME_DIALOG(window)->vbox;
#else
  GtkWidget *window = gtk_dialog_new();
  GtkWidget *body = GTK_DIALOG(window)->vbox;
  gtk_window_set_title(GTK_WINDOW(window), str->str);
#endif

  gtk_widget_set_name(window, "properties window");
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE); /* shrink, grow, auto_shrink */
  gtk_signal_connect(GTK_OBJECT(window), "delete_event", 
		     GTK_SIGNAL_FUNC(gtk_false), NULL);
  GtkWidget *frame = gtk_frame_new(NULL);
  gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_NONE);
  gtk_container_border_width(GTK_CONTAINER(frame), 10);
  gtk_box_pack_start(GTK_BOX(body), frame, TRUE, FALSE, 0);

#ifdef USE_GNOME
  gnome_dialog_button_connect_object(GNOME_DIALOG(window), 0, 
				     GTK_SIGNAL_FUNC(gnome_dialog_close),
				     GTK_OBJECT(window));
  gnome_dialog_set_default(GNOME_DIALOG(window), 0);
#else
  GtkWidget* okb = gtk_button_new();  // Button in action_area
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(window)->action_area), okb, 
		     FALSE, FALSE, 0);
  gtk_signal_connect_object(GTK_OBJECT(okb), "clicked", 
			    GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			    GTK_OBJECT(window));
  GtkWidget* label = gtk_label_new(_("OK"));
  gtk_widget_show(label);
  gtk_misc_set_padding(GTK_MISC(label), 10, 5);
  gtk_container_add(GTK_CONTAINER(okb), label);
  gtk_widget_show(okb);
  GTK_WIDGET_SET_FLAGS(okb, GTK_CAN_DEFAULT);
  gtk_widget_grab_default(okb);
#endif

  switch (fd->fontType) {
  case T1FONT:
    t1_showproperties(fd, frame);
    break;
  case TTFONT: 
    tt_showproperties(fd, frame);
    break;
  }
  gtk_widget_show(frame);
  return window;
}


/* Called when user clicked on a font name label */
void make_props_window(GtkWidget *label)
{
  FontData* fd = (FontData*)gtk_object_get_data(GTK_OBJECT(label), 
						"font");
  GtkWidget *imagewindow = gtk_widget_get_toplevel(label);
  GtkWidget *propwindow = show_font_properties(fd);
  gtk_window_set_position(GTK_WINDOW(propwindow), GTK_WIN_POS_MOUSE);
  gtk_window_set_transient_for(GTK_WINDOW(propwindow), 
			       GTK_WINDOW(imagewindow));
  gtk_widget_show(propwindow);
}


void make_save_gif(GtkWidget *label)
{
  GdkImage *image = (GdkImage*)gtk_object_get_data(GTK_OBJECT(label), 
						   "image");
  GtkWidget *imagewindow = gtk_widget_get_toplevel(label);
  GtkWidget *savewindow = save_asgif(image);
  gtk_window_set_position(GTK_WINDOW(savewindow), GTK_WIN_POS_MOUSE);
  gtk_window_set_transient_for(GTK_WINDOW(savewindow), 
			       GTK_WINDOW(imagewindow));
  gtk_widget_show(savewindow);
}


/*
  Called when a event regarding the event box of a image (in the
  character or string window) happens
*/
gint fname_event(GtkWidget* widget, GdkEvent *event, gpointer data)
{
  switch (event->type) {
  case GDK_ENTER_NOTIFY:
    gtk_widget_set_state(widget, GTK_STATE_ACTIVE);
    break;
  case GDK_LEAVE_NOTIFY:
    gtk_widget_set_state(widget, GTK_STATE_NORMAL);
    break;
  case GDK_BUTTON_PRESS:
    if (event->button.button == 1) {
      make_props_window(widget);
    }
    
    if (event->button.button == 3) {
      GtkWidget *menu, *item; 
      menu = gtk_menu_new();

      item = gtk_menu_item_new_with_label(_("Properties"));
      gtk_menu_append(GTK_MENU(menu), item);
      gtk_signal_connect_object(GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(make_props_window), 
				GTK_OBJECT(widget));
      gtk_widget_show(item);

      item = gtk_menu_item_new_with_label(_("Save as GIF"));
      gtk_menu_append(GTK_MENU(menu), item);
      gtk_signal_connect_object(GTK_OBJECT(item), "activate",
				GTK_SIGNAL_FUNC(make_save_gif), 
				GTK_OBJECT(widget));
      gtk_widget_show(item);


      gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		     event->button.button, event->button.time);
    }    
    break;

  default: break;
  }
  return FALSE;  /* Not handled, go further */
}



void show_char_or_string(GtkWidget *button, gpointer list)
{
  GtkWidget *gtkimage;
  GdkImage  *gdkimage;
  GtkRequisition req;
  char *invl;

  GList *selection = GTK_CLIST(list)->selection;
  if (selection == NULL) {
    errormsg(mainwindow, _("No row selected"));
    return;
  }

  double size = g_strtod(gtk_entry_get_text(GTK_ENTRY(fontsizeD->entry)), 
			 NULL);
  if (size<1.0 || size>1000.0) {
    errormsg(mainwindow, _("Invalid size"));
    return;
  }
  const char *chr = gtk_entry_get_text(GTK_ENTRY(characterD->entry));
  long rc = strtol(chr, &invl, 0);
  if (*chr=='\0' || *invl!='\0' || rc==LONG_MIN || rc==LONG_MAX || 
      rc<0 || rc>0xFFFF) {
    errormsg(mainwindow, _("Invalid character"));
    return;
  }
  guint32 code = (guint32)rc;

  const char* string = gtk_entry_get_text(GTK_ENTRY(stringD->entry));
  if (string==NULL || *string=='\0') string = _("Test");

  // Are we making a char or a string?
  int makechar = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(button), 
						     "makechar")); 

  /* If character code > 255, check that no Type 1 fonts are selected */
  if (makechar && code > 255) {
    for (GList *sel=selection; sel; sel=sel->next) {
      int row = GPOINTER_TO_INT(sel->data);
      FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(list), row);
      if (fd->fontType==T1FONT) {
	errormsg(mainwindow, _("Type 1 fonts do not allow character codes greater than 255"));
	return;
      } 
    }
  }

  set_window_busy(mainwindow, TRUE);
#ifdef USE_GNOME
  GtkWidget *window = gnome_dialog_new(_("Font Viewer Samples"),
				       GNOME_STOCK_BUTTON_CLOSE,
				       NULL);
  gnome_dialog_set_close(GNOME_DIALOG(window), TRUE);
  GtkWidget *diag_vbox = GNOME_DIALOG(window)->vbox;
#else
  GtkWidget* window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title(GTK_WINDOW(window), _("Font Viewer Samples"));
  GtkWidget *diag_vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(window), diag_vbox);
#endif
  gtk_widget_set_name(window, "string window");
  gtk_widget_ensure_style(window); 
  gtk_container_border_width(GTK_CONTAINER(window), 10);
  gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);

  GtkWidget* scwindow = gtk_scrolled_window_new(NULL, NULL);
  gtk_widget_show(scwindow);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scwindow), 
				 GTK_POLICY_AUTOMATIC, 
				 GTK_POLICY_AUTOMATIC);
  gtk_box_pack_start(GTK_BOX(diag_vbox), scwindow, TRUE, FALSE, 0);

  GtkWidget *vbox = gtk_vbox_new(FALSE, 10); 
  gtk_widget_show(vbox);
  gtk_container_border_width(GTK_CONTAINER(vbox), 3); 
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scwindow), vbox);

  GtkWidget *hbox = NULL;
  int hwidth = 0;
  int n_items = g_list_length(selection);
  int cur_item = 0;
  STATUSBAR_PUSH(stbar, _("Making samples..."));
  for (GList *sel=selection; sel; sel=sel->next) {
    gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 
			    double(cur_item++)/n_items);
    while (gtk_events_pending()) gtk_main_iteration();

    int row = GPOINTER_TO_INT(sel->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(list), row);
    GtkWidget *samplebox = gtk_vbox_new(FALSE, 0);
    gtk_widget_show(samplebox);

    GtkWidget *evb = gtk_event_box_new();
    gtk_widget_show(evb);
    gtk_widget_set_name(evb, "fontname label");
    gtk_box_pack_start(GTK_BOX(samplebox), evb, FALSE, FALSE, 0);
    GtkWidget *label = gtk_label_new(fd->fontName);
    gtk_widget_show(label);
    gtk_widget_set_name(label, "fontname label");
    gtk_container_add(GTK_CONTAINER(evb), label);
    gtk_signal_connect(GTK_OBJECT(evb), "button_press_event",
		       GTK_SIGNAL_FUNC(fname_event), (gpointer)fd);
    gtk_signal_connect(GTK_OBJECT(evb), "enter_notify_event",
		       GTK_SIGNAL_FUNC(fname_event), (gpointer)fd);
    gtk_signal_connect(GTK_OBJECT(evb), "leave_notify_event",
		       GTK_SIGNAL_FUNC(fname_event), (gpointer)fd);

    switch(fd->fontType) {
    case T1FONT:
      if (makechar) 
	gdkimage = t1_makechar(fd, (char)code, size, antialiasing);
      else
	gdkimage = t1_makestring(fd, string, size, kerning, antialiasing);
      break;
    case TTFONT: 
      if (makechar) {  // Making a char
	TT_CharMap charmap;
	int error;
	
	error = tt_findcharmap(fd, &charmap);
	if (error) {
	  add_error(fd, _("Could not find proper charmap"));
	  continue;
	}
	TT_UShort index = TT_Char_Index(charmap, code);
	gdkimage = tt_makechar(fd, index, size, antialiasing);
      }
      else  // Making a string
	gdkimage = tt_makestring(fd, string, size, kerning, antialiasing);
      break;
    }
    if (gdkimage == NULL) continue;  // For whatever reason
    gtk_object_set_data(GTK_OBJECT(evb), "font", (gpointer)fd);
    gtk_object_set_data(GTK_OBJECT(evb), "image", (gpointer)gdkimage);

    gtkimage = gtk_image_new(gdkimage, NULL);
    gtk_widget_show(gtkimage);
    gtk_box_pack_start(GTK_BOX(samplebox), gtkimage, FALSE, FALSE, 0);
    gtk_signal_connect(GTK_OBJECT(gtkimage), "destroy", 
		       GTK_SIGNAL_FUNC(delete_image), (gpointer)gdkimage);
    gtk_signal_connect(GTK_OBJECT(gtkimage), "destroy", 
		       GTK_SIGNAL_FUNC(delete_stringsw), (gpointer)fd);

    fd->refcount++;

    /* Compute the size requested by the widgets in the samplebox;
     if it does not fit on the screen in that row, make another row */
    gtk_widget_size_request(samplebox, &req);
    if (hbox==NULL || (hwidth+req.width)>gdk_screen_width()-60) {
      if (hbox) {
	GtkWidget *sep = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), sep, FALSE, FALSE, 0); 
	gtk_widget_show(sep);
      }
      hbox = gtk_hbox_new(FALSE, 8);
      gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0); 
      gtk_widget_show(hbox);
      hwidth = 0;
    }

    gtk_box_pack_start(GTK_BOX(hbox), samplebox, FALSE, FALSE, 0); 
    hwidth += 8+req.width;
  } // For all in selection
  STATUSBAR_POP(stbar);

  gtk_widget_size_request(vbox, &req);
  gtk_widget_set_usize(GTK_BIN(scwindow)->child,
		       MIN(req.width+10,  gdk_screen_width()-50),
  		       MIN(req.height+10, gdk_screen_height()-50));
  gtk_widget_show_all(window);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(prbar), 0);
  set_window_busy(mainwindow, FALSE);
}





/* See function below; keeps track of resize events of window to cover */
static gint busy_configure_event(GtkWidget *widget, 
				 GdkEventConfigure *event, gpointer data)
{
  GdkWindow *window = (GdkWindow*)data;
  gdk_window_resize(window, event->width, event->height);
  return FALSE;
}



/*
  Avoid any interaction with the window: the window will not respond to
  any events, and it will have a wait cursor over it if state TRUE
  The function can be called nested: if called 2 times with TRUE, 
  it has to be called 2 times with FALSE
  It works by superposing an input window of the same size of the window
  to cover (resizes tracked by the configure events) which discards
  any mouse or key events.
 */
void set_window_busy(GtkWidget *window, bool state)
{
  GdkWindowAttr attr; 
  GdkWindow *busy;
  int busy_count;

  g_return_if_fail(window->window != NULL);

  busy = (GdkWindow*)gtk_object_get_data(GTK_OBJECT(window), "busy");
  if (busy == NULL) {
    attr.window_type = GDK_WINDOW_CHILD;
    //attr.wclass = GDK_INPUT_OUTPUT;  // For testing
    attr.wclass = GDK_INPUT_ONLY;
    attr.cursor = gdk_cursor_new(GDK_WATCH);
    attr.width = 0;
    attr.height = 0;
    attr.event_mask = GDK_KEY_PRESS_MASK;
    busy = gdk_window_new(window->window, &attr, GDK_WA_CURSOR);
    gtk_object_set_data(GTK_OBJECT(window), "busy", (gpointer)busy);
    gtk_object_set_data(GTK_OBJECT(window), "busycount", GINT_TO_POINTER(0));
    gtk_signal_connect(GTK_OBJECT(window), "configure_event", 
		       (GtkSignalFunc)busy_configure_event, (gpointer)busy);
    gtk_signal_handler_block_by_func(GTK_OBJECT(window), 
				     (GtkSignalFunc)busy_configure_event, 
				     (gpointer)busy);
    gdk_cursor_destroy(attr.cursor);
  }

  busy_count = GPOINTER_TO_INT(gtk_object_get_data(GTK_OBJECT(window), 
						   "busycount"));
  if (state) busy_count++;
  else busy_count--;

  if (state && busy_count==1) { // Transition 0->1
    GdkWindowPrivate *priv = (GdkWindowPrivate*)window->window; 
    gdk_window_resize(busy, priv->width, priv->height);
    gdk_window_show(busy);
    gtk_signal_handler_unblock_by_func(GTK_OBJECT(window), 
				       (GtkSignalFunc)busy_configure_event, 
				       (gpointer)busy);
  }
  if (busy_count==0) {  // Transition 1->0
    gdk_window_hide(busy);
    gtk_signal_handler_block_by_func(GTK_OBJECT(window), 
				     (GtkSignalFunc)busy_configure_event, 
				     (gpointer)busy);
  }
  if (busy_count < 0) busy_count = 0;
  gtk_object_set_data(GTK_OBJECT(window), "busycount", 
		      GINT_TO_POINTER(busy_count));
  while (gtk_events_pending()) gtk_main_iteration();
}




/*    
      Handling of the 'Save as GIF' option for images

*/

// Makes the popup menu for the given window (containing an image, character or string)
GtkWidget* make_imagepopupmenu(GtkWidget *window)
{
  GtkWidget *menu = gtk_menu_new();
  GtkWidget *item = gtk_menu_item_new_with_label(_("Save as GIF"));
  gtk_menu_append(GTK_MENU(menu), item);
  gtk_signal_connect_object(GTK_OBJECT(item), "activate",
                GTK_SIGNAL_FUNC(save_asgif), GTK_OBJECT(window));
  gtk_widget_show(item);
  return menu;
}




static void do_savegif(GtkWidget *button, GdkImage *image)
{
  struct stat buf;

  GtkWidget *filesel = gtk_widget_get_toplevel(button);
  char *file = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  if (strchr(file, '*')) return;

  size_t len = strlen(file);
  if (len<4 || strcmp(file+len-4, ".gif")!=0) {
    int rc = ask_yes_no(filesel, 
			_("File extension is not gif\nProceed anyway?"));
    if (rc == 0) return;  // if no
  }

  if (stat(file, &buf)==0) {
    if (S_ISDIR(buf.st_mode)) {
      errormsg(filesel, _("Selected file is a directory"));
      return;
    }
    int rc = ask_yes_no(filesel, 
			_("File %s already exists\nDo you want to overwrite it?"), 
			g_basename(file));
    if (rc == 0) return; // if no
  }

  bool ret = make_gif(filesel, image, file);
  if (ret == FALSE) return;  // sth went wrong
  gtk_widget_hide(filesel);
}




/* The user selected to save the image as a GIF file
   Called from popup menu
*/
GtkWidget* save_asgif(GdkImage *image)
{
  static GtkWidget *filesel = NULL;

  if (filesel == NULL) {
    filesel = gtk_file_selection_new(_("Save GIF file"));
    gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);
    gtk_file_selection_complete(GTK_FILE_SELECTION(filesel), "*.gif");
    GtkWidget *cancelb = GTK_FILE_SELECTION(filesel)->cancel_button;
    gtk_signal_connect_object(GTK_OBJECT(cancelb), "clicked", 
			      GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			      GTK_OBJECT(filesel));
    // Delete events just hide the window
    gtk_signal_connect_object(GTK_OBJECT(filesel), "delete_event", 
			      GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			      GTK_OBJECT(filesel));
  }
  GtkWidget *okb = GTK_FILE_SELECTION(filesel)->ok_button;
  gtk_signal_handlers_destroy(GTK_OBJECT(okb));
  gtk_signal_connect(GTK_OBJECT(okb), "clicked",
		     GTK_SIGNAL_FUNC(do_savegif), (gpointer)image);
  return filesel;
}




/*
  Save the TT Fonts in the selection as Type 42
  If only one font is selected, ask for a file name to save
  If several are selected, ask for a directory, and save them
  with the original name and extension pfa
 */

static void do_save42(GtkWidget *button, GList *selection)
{
  struct stat buf;
  size_t len;
  int n_fonts = g_list_length(selection);

  GtkWidget *filesel = gtk_widget_get_toplevel(button);
  char *filedg = gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel));
  if (strchr(filedg, '*')) return;

  switch (n_fonts) {
  case 1: /* One font selected */
    len = strlen(filedg);
    if (len<4 || strcmp(filedg+len-4, ".pfa")!=0) {
      int rc = ask_yes_no(filesel, 
			  _("File extension is not pfa\nProceed anyway?"));
      if (rc == 0) return;  // if no
    }
    
    if (stat(filedg, &buf)==0) {
      if (S_ISDIR(buf.st_mode)) {
	errormsg(filesel, _("Selected file is a directory"));
	return;
      }
      int rc = ask_yes_no(filesel, 
			  _("File %s already exists\nDo you want to overwrite it?"), 
			  g_basename(filedg));
      if (rc == 0) return; // if no
    }
    break;
  default: /* More than one font selected */
    if (stat(filedg, &buf)==0) {
      if (!S_ISDIR(buf.st_mode)) {
	errormsg(filesel, _("Selection is not a directory"));
	return;
      }
    }
    else { /* filedg does not exist */
      errormsg(filesel, _("Directory does not exist"));
      return;
    }
    break;
  }

  for (GList *sel=selection; sel; sel=sel->next) {
    int row = GPOINTER_TO_INT(sel->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
    // Take care: maybe not all selected fonts are TrueType
    if (fd->fontType != TTFONT) continue;
    GString *fname = g_string_new(filedg);

    if (n_fonts > 1) { /* Take font name with extension pfa */
      fname = g_string_append(fname, g_basename(fd->fontFile));
      fname = g_string_truncate(fname, fname->len-3);
      fname = g_string_append(fname, "pfa"); 
    }
    FILE *fp = fopen(fname->str, "w");
    if (fp == NULL) {
      add_error(fd, _("Cannot open file %s: %s"), 
		fname->str, g_strerror(errno));
      g_string_free(fname, TRUE);
      continue;
    }
    bool rc = tt_download_as_t42(fp, fd, NULL);
    if (rc == FALSE) { 
      fclose(fp);
      unlink(fname->str);
      g_string_free(fname, TRUE);
      continue;
    }

    rc = ferror(fp); 
    if (fclose(fp) || rc) {  // if error writing to file
      if (n_fonts == 1) {
	errormsg(filesel, _("Error writing to file %s: %s"), fname->str, 
		 g_strerror(errno));
	return;
      }
      else {
	int ans = ask_yes_no(filesel, 
			     _("Error writing to file %s: %s\nContinue?"), 
			     fname->str, g_strerror(errno));
	if (ans == 0) return;  // if no
      }
    }

    g_string_free(fname, TRUE);
  }
  gtk_widget_hide(filesel); 
}


// The user selected to save the font as type 42
// Called from popup menu
static void save_astype42(GtkWidget *, gpointer data)
{
  static GtkWidget *filesel = NULL;

  GList *selection = (GList*)data;
  int n_fonts = g_list_length(selection);

  if (filesel == NULL) {
    filesel = gtk_file_selection_new(_("Save as Type42"));
    gtk_window_position(GTK_WINDOW(filesel), GTK_WIN_POS_CENTER);
    gtk_window_set_modal(GTK_WINDOW(filesel), TRUE);
    gtk_signal_connect_object(GTK_OBJECT(filesel), "delete_event", 
			      GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			      GTK_OBJECT(filesel));
    GtkWidget *cancelb = GTK_FILE_SELECTION(filesel)->cancel_button;
    gtk_signal_connect_object(GTK_OBJECT(cancelb), "clicked", 
			      GTK_SIGNAL_FUNC(gtk_widget_hide_on_delete), 
			      GTK_OBJECT(filesel));
  }
  GtkWidget *okb = GTK_FILE_SELECTION(filesel)->ok_button;
  gtk_signal_handlers_destroy(GTK_OBJECT(okb));
  gtk_signal_connect(GTK_OBJECT(okb), "clicked",
		     GTK_SIGNAL_FUNC(do_save42), selection);
  
  /* if several fonts selected, we just want the directory, to put the original
     file name with the extension pfa */
  if (g_list_length(selection)>1) {
    gtk_widget_set_sensitive(GTK_FILE_SELECTION(filesel)->file_list, FALSE);
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), "");
  }
  else {
    int row = GPOINTER_TO_INT(selection->data);
    FontData* fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(fontclistw), 
						     row);
    gtk_widget_set_sensitive(GTK_FILE_SELECTION(filesel)->file_list, TRUE);
    char *newname = g_strdup(g_basename(fd->fontFile));
    strcpy(newname+strlen(newname)-3, "pfa");
    gtk_file_selection_set_filename(GTK_FILE_SELECTION(filesel), newname);
    g_free(newname);
  }
  gtk_window_set_transient_for(GTK_WINDOW(filesel), GTK_WINDOW(mainwindow));
  gtk_widget_show(filesel);
}



void drawsample(GtkWidget *list, int row)
{
  int ysize;
  const gchar *msg = _("the quick brown fox jumps over the lazy dog");
  double size = 18.0;

  GdkImage *image;
  FontData *fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(list), row);

  double yscale = yresolution/72.0;
  int ypixels = (int)(1.4*size*yscale);
  gtk_widget_set_usize(samplet->parent, 1, ypixels);

  /* Image with no kerning, no antialiasing */
  if (fd->fontType == T1FONT)
    image = t1_makestring(fd, msg, size, FALSE, FALSE);
  else
    image = tt_makestring(fd, msg, size, FALSE, FALSE);

  if (image == NULL) {
    gtk_pixmap_set(GTK_PIXMAP(samplet), emptypixmap, NULL);  
    return;
  }
  GdkPixmap* pixmap = gdk_pixmap_new(mainwindow->window, image->width, 
				     image->height, -1);
  gdk_draw_image(pixmap, defGC, image, 0, 0, 0, 0, -1, -1);
  gdk_image_destroy(image);

  gtk_pixmap_set(GTK_PIXMAP(samplet), pixmap, NULL);  /* Deletes old pixmap too */
}



void unselect_fontlist(GtkWidget* list, gint row, gint column, 
		       GdkEventButton *event)
{
  GList *selection = GTK_CLIST(list)->selection;
  gint n_sel = g_list_length(selection);
  if (n_sel == 0) {
    gtk_pixmap_set(GTK_PIXMAP(samplet), emptypixmap, NULL);
    gdk_window_clear(samplet->window);  // Clear old font sample
  }
  if (n_sel == 1) {
    int selected_row = GPOINTER_TO_INT(selection->data);
    drawsample(list, selected_row);
  }

}


void select_fontlist(GtkWidget* list, gint row, gint column, 
		     GdkEventButton *event)
{
  //printf("GtkCList Selection: row %d column %d button %d\n", row, column, event?event->button:0);
  
  /* 
     If more than one font is selected, we get a selection event for each row.
     For multiple selection, this is too much.
     So in this case do not show any sample
  */
  GList *selection = GTK_CLIST(list)->selection;
  gint n_sel = g_list_length(selection);
  if (n_sel > 1) {
    gtk_pixmap_set(GTK_PIXMAP(samplet), emptypixmap, NULL);
    gdk_window_clear(samplet->window);  // Clear old font sample
    return;
  }
  drawsample(list, row);
}



// Called when user clicks with the mouse on the font list
gint fontlist_event(GtkWidget* list, GdkEvent *event, gpointer)
{
  GtkWidget *menu, *item;
  FontData *fd;
  gint row, rc;
  GList *selection;
  bool found = FALSE;

  selection = GTK_CLIST(list)->selection;

  switch (event->type) {
  case GDK_BUTTON_PRESS:
    if (event->button.window != GTK_CLIST(list)->clist_window) return FALSE;
    switch (event->button.button) {
    case 3:
      /* Only post menu if some font is TrueType */
      for (GList *sel=selection; sel; sel=sel->next) {
	row = GPOINTER_TO_INT(sel->data);
	fd = (FontData*)gtk_clist_get_row_data(GTK_CLIST(list), row);
	if (fd->fontType == TTFONT) {
	  found = TRUE;
	  break;
	}
      }
      if (!found) return FALSE;
      menu = gtk_menu_new();
      item = gtk_menu_item_new_with_label(_("Save as Type42"));
      gtk_widget_show(item);
      gtk_menu_append(GTK_MENU(menu), item);
      gtk_signal_connect(GTK_OBJECT(item), "activate",
			 GTK_SIGNAL_FUNC(save_astype42), selection);
      gtk_menu_popup(GTK_MENU(menu), NULL, NULL, NULL, NULL,
		     event->button.button, event->button.time);
      return TRUE;   // Handled
      break;
    }
    break;

  case GDK_BUTTON_RELEASE:
    break;
    
  default: break;
  }
  return FALSE;
}






// ************** Handling of GIF files


// Make an error message
static const char* getGIFError(int error)
{
  static GString *msg = NULL;
  if (msg == NULL) msg = g_string_new(NULL);

  switch(error) {
  case D_GIF_ERR_OPEN_FAILED:
  case E_GIF_ERR_OPEN_FAILED:
    g_string_sprintf(msg, _("Failed to open given file (%s)"), 
		     g_strerror(errno));
    break;
  case E_GIF_ERR_WRITE_FAILED:
    g_string_sprintf(msg, _("Failed to Write to given file (%s)"), 
		     g_strerror(errno));
    break;
  case E_GIF_ERR_HAS_SCRN_DSCR:
    g_string_assign(msg, _("Screen Descriptor already been set"));
    break;
  case E_GIF_ERR_HAS_IMAG_DSCR:
    g_string_assign(msg, _("Image Descriptor is still active"));
    break;
  case E_GIF_ERR_NO_COLOR_MAP:
    g_string_assign(msg, _("Neither Global Nor Local color map"));
    break;
  case E_GIF_ERR_DATA_TOO_BIG:
    g_string_assign(msg, _("#Pixels bigger than Width * Height"));
    break;
  case E_GIF_ERR_NOT_ENOUGH_MEM:
    g_string_assign(msg, _("Fail to allocate required memory"));
    break;
  case E_GIF_ERR_DISK_IS_FULL:
    g_string_sprintf(msg, _("Write failed (%s)"), g_strerror(errno));
    break;
  case E_GIF_ERR_CLOSE_FAILED:
    g_string_sprintf(msg, _("Failed to close given file (%s)"), 
		     g_strerror(errno));
    break;
  case E_GIF_ERR_NOT_WRITEABLE:
    g_string_assign(msg, _("Given file was not opened for write"));
    break;

  case D_GIF_ERR_READ_FAILED:
    g_string_sprintf(msg, _("Failed to Read from given file (%s)"), 
		     g_strerror(errno));
    break;
  case D_GIF_ERR_NOT_GIF_FILE:
    g_string_assign(msg, _("Given file is not a GIF file"));
    break;
  case D_GIF_ERR_NO_SCRN_DSCR:
    g_string_assign(msg, _("No Screen Descriptor detected"));
    break;
  case D_GIF_ERR_NO_IMAG_DSCR:
    g_string_assign(msg, _("No Image Descriptor detected"));
    break;
  case D_GIF_ERR_NO_COLOR_MAP:
    g_string_assign(msg, _("Neither Global Nor Local color map"));
    break;
  case D_GIF_ERR_WRONG_RECORD:
    g_string_assign(msg, _("Wrong record type detected"));
    break;
  case D_GIF_ERR_DATA_TOO_BIG:
    g_string_assign(msg, _("#Pixels bigger than Width * Height"));
    break;
  case D_GIF_ERR_NOT_ENOUGH_MEM:
    g_string_assign(msg, _("Fail to allocate required memory"));
    break;
  case D_GIF_ERR_CLOSE_FAILED:
    g_string_sprintf(msg, _("Failed to close given file (%s)"), g_strerror(errno));
    break;
  case D_GIF_ERR_NOT_READABLE:
    g_string_assign(msg, _("Given file was not opened for read"));
    break;
  case D_GIF_ERR_IMAGE_DEFECT:
    g_string_assign(msg, _("Image is defective, decoding aborted"));
    break;
  case D_GIF_ERR_EOF_TOO_SOON:
    g_string_assign(msg, _("Image EOF detected, before image was complete"));
    break;
  default:
    g_string_assign(msg, _("Unknown GIF error"));
    break;
  }
  return msg->str;
}




// Make a GIF (89a) file out of the given image
static bool make_gif(GtkWidget *parent, GdkImage* image, char* filename)
{
  GifFileType *giffile;
  ColorMapObject *gifColorMap;
  GifByteType *GlblGifBuffer, *GlblGifBufferPtr;
  GdkGCValues gcval;
  GdkColor c1, c2;
  int i, rc, error;

  GlblGifBufferPtr = GlblGifBuffer = g_new(GifByteType, 
					   image->width * image->height);
  switch (image->depth) {
  case 1:   // Bitmap image
    gifColorMap = MakeMapObject(2, NULL);  // 2 colors (black, white)
    gifColorMap->Colors[0].Red = gifColorMap->Colors[0].Green = 
      gifColorMap->Colors[0].Blue = 0xFF;  // White
    gifColorMap->Colors[1].Red = gifColorMap->Colors[1].Green = 
      gifColorMap->Colors[1].Blue = 0x0;   // Black
    break;
  default:  // Pixmap (antialiased image)
    /* 8 colors: 5 in graypalette, 1 in c1, 1 in c2, 1 unused */
    gifColorMap = MakeMapObject(8, NULL);  
    for (i=0; i<5; i++) { // 5 colors in graypalette[]
      gifColorMap->Colors[i].Red   = graypalette[i].red / 256;
      gifColorMap->Colors[i].Green = graypalette[i].green / 256;
      gifColorMap->Colors[i].Blue  = graypalette[i].blue / 256;
    }
    /* redGC is color index 5 */
    gdk_gc_get_values(redGC, &gcval);
    c1.pixel = gcval.foreground.pixel;
    gdk_color_context_query_color(mCC, &c1); 
    gifColorMap->Colors[5].Red = c1.red / 256;
    gifColorMap->Colors[5].Green = c1.green / 256;
    gifColorMap->Colors[5].Blue = c1.blue / 256;

    /* blueGC is color index 6 */
    gdk_gc_get_values(blueGC, &gcval);
    c2.pixel = gcval.foreground.pixel;
    gdk_color_context_query_color(mCC, &c2); 
    gifColorMap->Colors[6].Red   = c2.red / 256;
    gifColorMap->Colors[6].Green = c2.green / 256;
    gifColorMap->Colors[6].Blue  = c2.blue / 256;

    break;
  }
  // EGifSetGifVersion("89a"); // Triggers giflib bug (%%$##!!)
  giffile = EGifOpenFileName(filename, FALSE);
  if (giffile == NULL) {
    error = GifLastError();
    errormsg(parent, getGIFError(error));
    return FALSE;
  }
  rc = EGifPutScreenDesc(giffile, image->width, image->height, 
			    8, 0, gifColorMap);
  if (rc==GIF_ERROR) {
    errormsg(parent, getGIFError(GifLastError()));
    g_free(GlblGifBuffer);
    EGifCloseFile(giffile);
    return FALSE;
  }

  char buf[] = {0x1, 0, 0, 0};  // Set transparent color to index 0 (white)
  EGifPutExtension(giffile, GRAPHICS_EXT_FUNC_CODE, 4, buf);
  rc = EGifPutImageDesc(giffile, 0, 0, image->width, image->height, 
			     FALSE, NULL);
  if (rc==GIF_ERROR) {
    errormsg(parent, getGIFError(GifLastError()));
    g_free(GlblGifBuffer);
    EGifCloseFile(giffile);
    return FALSE;
  }

  // Scan image pixel by pixel (slow?)
  for (int y=0; y<image->height; y++) 
    for (int x=0; x<image->width; x++) {
      guint32 pixel = gdk_image_get_pixel(image, x, y);
      *GlblGifBufferPtr = 0;
      // Find pixel value in graypalette and put it into gif buffer
      switch (image->depth) {
      case 1:
	*GlblGifBufferPtr = pixel;  // 0 or 1
	break;
      default:
	if (pixel == c1.pixel) {
	  *GlblGifBufferPtr = 5;
	  break;
	}
	if (pixel == c2.pixel) {
	  *GlblGifBufferPtr = 6;
	  break;
	}
	for (int col=0;col<5;col++) 
	  if (pixel == graypalette[col].pixel) {
	    *GlblGifBufferPtr = col;
	    break;
	  } 
	break;
      }
      GlblGifBufferPtr++;
    }

  // GlblGifBuffer contains now the gif colormap values bytewise
  rc = EGifPutLine(giffile, GlblGifBuffer, image->width * image->height);
  if (rc==GIF_ERROR) {
    errormsg(parent, getGIFError(GifLastError()));
    g_free(GlblGifBuffer);
    EGifCloseFile(giffile);
    return FALSE;
  }
 
  g_free(GlblGifBuffer);
  rc = EGifCloseFile(giffile);
  if (rc==GIF_ERROR) {
    errormsg(parent, getGIFError(GifLastError()));
    return FALSE;
  }
  return TRUE;
}

