/* gmoo - a gtk+ based graphical MOO/MUD/MUSH/... client
 * Copyright (C) 1999-2000 Gert Scholten
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "tclscript.h"
#ifdef TCL
#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <tcl.h>

#include "script.h"

Tcl_Interp *tcl_interp;

void tcl_register_functions();

void gm_tcl_init(const char *programname) {
    if(debug) printf("\tIniting tcl interpreter...\n");

    tcl_interp = Tcl_CreateInterp();
    Tcl_Init(tcl_interp);
    Tcl_Preserve(tcl_interp);

    tcl_register_functions();
}

void gm_tcl_exit() {
    if(debug) printf("Exiting the Tcl interpreter\n");
    Tcl_Release(tcl_interp);
    Tcl_DeleteInterp(tcl_interp);
}

int gm_tcl_is_tcl_file(const char *filename) {
    char *s;
    if((s = strrchr(filename, '.'))) {
	if(g_strcasecmp(s + 1, TCL_EXTENTION) == 0) {
	    return TRUE;
	}
    }
    return FALSE;
}

void gm_tcl_load_file(const char *filename) {
    if(debug) printf("\tLoading Tcl file: %s\n", filename);
    if(Tcl_EvalFile(tcl_interp, (char *) filename) != TCL_OK) {
        if(debug) printf("\e[31mTcl Error evalling file: \"%s\"\e[0m\n",
                         Tcl_GetStringResult(tcl_interp));
    }
}

void gm_tcl_exec(const char *command) {
    if(Tcl_Eval(tcl_interp, (char *) command) != TCL_OK) {
        if(debug) printf("\e[31mTcl Error evalling \"%s\": \"%s\"\e[0m\n",
                         command, Tcl_GetStringResult(tcl_interp));
    }
}

char *escape_curly(const char *args) {
    char *ret = g_malloc(strlen(args) * 2 + 1);
    int i, j = 0;

    for(i = 0; args[i]; i++) {
	if(args[i] == '{' ||args[i] == '}' || args[i] == '\\')
            ret[j++] = '\\';
	ret[j] = args[i];
	j++;
    }
    ret[j] = '\0';

    return ret;
}

void gm_tcl_do_macro(int id, const char *macro_name, const char *argstr) {
    char *args = escape_curly(argstr);
    char *line = g_strdup_printf("%s %d {%s}", macro_name, id, args);

    if(debug) printf("Tcl: %s %d {%s}\n", macro_name, id, argstr);
    if(Tcl_Eval(tcl_interp, line) != TCL_OK) {
        if(debug) printf("\e[31mTcl Error evalling \"%s\": \"%s\"\e[0m\n",
                         line, Tcl_GetStringResult(tcl_interp));
    }
    g_free(line);
    g_free(args);
}

/*****************************************************************************/
/*****************************************************************************/
/*****************************************************************************/
int c_tcl_print(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_println(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_msg(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_write(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_writeln(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_name(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_hostname(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_port(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_connected(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_width(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_height(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_register_open(ClientData cd, Tcl_Interp *interp,
			int argc, char *argv[]);
int c_tcl_register_close(ClientData cd, Tcl_Interp *interp,
			 int argc, char *argv[]);
int c_tcl_register_connect(ClientData cd, Tcl_Interp *interp,
			   int argc, char *argv[]);
int c_tcl_register_disconnect(ClientData cd, Tcl_Interp *interp,
			      int argc, char *argv[]);

int c_tcl_register_macro(ClientData cd, Tcl_Interp *interp,
			 int argc, char *argv[]);
int c_tcl_register_start(ClientData cd, Tcl_Interp *interp,
			 int argc, char *argv[]);
int c_tcl_register_end(ClientData cd, Tcl_Interp *interp,
		       int argc, char *argv[]);
int c_tcl_version(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_msgbox(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);
int c_tcl_input(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]);

#define register(nspace, name, fname) Tcl_CreateCommand(tcl_interp, nspace "::" name, fname, NULL, NULL);
void tcl_register_functions() {
    register("WORLD", "print",     c_tcl_print);
    register("WORLD", "println",   c_tcl_println);
    register("WORLD", "tell",      c_tcl_println);
    register("WORLD", "msg",       c_tcl_msg);
    register("WORLD", "write",     c_tcl_write);
    register("WORLD", "writeln",   c_tcl_writeln);
    register("WORLD", "send",      c_tcl_writeln);
    register("WORLD", "name",      c_tcl_name);
    register("WORLD", "hostname",  c_tcl_hostname);
    register("WORLD", "port",      c_tcl_port);
    register("WORLD", "connected", c_tcl_connected);
    register("WORLD", "width",     c_tcl_width);
    register("WORLD", "height",    c_tcl_height);
    register("WORLD", "register_open",       c_tcl_register_open);
    register("WORLD", "register_close",      c_tcl_register_close);
    register("WORLD", "register_connect",    c_tcl_register_connect);
    register("WORLD", "register_disconnect", c_tcl_register_disconnect);

    register("CLIENT", "register_macro", c_tcl_register_macro);
    register("CLIENT", "register_start", c_tcl_register_start);
    register("CLIENT", "register_end",   c_tcl_register_end);
    register("CLIENT", "version",        c_tcl_version);
    register("CLIENT", "msgbox",         c_tcl_msgbox);
    register("CLIENT", "input",          c_tcl_input);
}

#define return_int(i) Tcl_SetObjResult(interp, Tcl_NewIntObj(i)); return TCL_OK
#define return_str(s) Tcl_SetResult(interp, s, NULL); return TCL_OK

int c_tcl_print(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    if(argc != 3) return TCL_ERROR;

    return_int(gm_script_do_print(atoi(argv[1]), argv[2]));
}

int c_tcl_println(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    if(argc != 3) return TCL_ERROR;

    return_int(gm_script_do_println(atoi(argv[1]), argv[2]));
}

int c_tcl_msg(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    if(argc != 3) return TCL_ERROR;

    return_int(gm_script_do_msg(atoi(argv[1]), argv[2]));
}

int c_tcl_write(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    if(argc != 3) return TCL_ERROR;

    return_int(gm_script_do_write(atoi(argv[1]), argv[2]));
}

int c_tcl_writeln(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    if(argc != 3) return TCL_ERROR;

    return_int(gm_script_do_writeln(atoi(argv[1]), argv[2]));
}

int c_tcl_name(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    char *s;

    if(argc != 2) return TCL_ERROR;

    if(gm_script_do_name(atoi(argv[1]), &s)) {
	return_str(s);
    }
    return TCL_ERROR;
}

int c_tcl_hostname(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    char *s;

    if(argc != 2) return TCL_ERROR;

    if(gm_script_do_hostname(atoi(argv[1]), &s)) {
	return_str(s);
    }
    return TCL_ERROR;
}

int c_tcl_port(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    int i;

    if(argc != 2) return TCL_ERROR;

    if(gm_script_do_port(atoi(argv[1]), &i)) {
	return_int(i);
    }
    return TCL_ERROR;
}

int c_tcl_connected(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]){
    int i;

    if(argc != 2) return TCL_ERROR;

    if(gm_script_do_connected(atoi(argv[1]), &i)) {
	return_int(i);
    }
    return TCL_ERROR;
}

int c_tcl_width(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    int i;

    if(argc != 2) return TCL_ERROR;

    if(gm_script_do_width(atoi(argv[1]), &i)) {
	return_int(i);
    }
    return TCL_ERROR;
}

int c_tcl_height(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    int i;

    if(argc != 2) return TCL_ERROR;

    if(gm_script_do_height(atoi(argv[1]), &i)) {
	return_int(i);
    }
    return TCL_ERROR;
}

int c_tcl_register_open(ClientData cd, Tcl_Interp *interp, 
			int argc, char *argv[]) {
    if(argc != 2) return TCL_ERROR;
    return_int(gm_script_do_register_open(TCL_TYPE, argv[1]));
}
int c_tcl_register_close(ClientData cd, Tcl_Interp *interp, 
			int argc, char *argv[]) {
    if(argc != 2) return TCL_ERROR;
    return_int(gm_script_do_register_close(TCL_TYPE, argv[1]));
}
int c_tcl_register_connect(ClientData cd, Tcl_Interp *interp, 
			int argc, char *argv[]) {
    if(argc != 2) return TCL_ERROR;
    return_int(gm_script_do_register_connect(TCL_TYPE, argv[1]));
}
int c_tcl_register_disconnect(ClientData cd, Tcl_Interp *interp, 
			int argc, char *argv[]) {
    if(argc != 2) return TCL_ERROR;
    return_int(gm_script_do_register_disconnect(TCL_TYPE, argv[1]));
}



int c_tcl_register_macro(ClientData cd, Tcl_Interp *interp,
			 int argc, char *argv[]) {
    char *mname, *fname, *desc = "";

    if(argc != 3 && argc != 4) return TCL_ERROR;

    mname = argv[1];
    fname = argv[2];
    if(argc == 4) desc = argv[3];

    return_int(gm_script_do_register_macro(TCL_TYPE, mname, fname, desc));
}

int c_tcl_register_start(ClientData cd, Tcl_Interp *interp,
			 int argc, char *argv[]) {
    char *name;

    if(argc != 2) return TCL_ERROR;

    name = argv[1];

    return_int(gm_script_do_register_start(TCL_TYPE, name));
}

int c_tcl_register_end(ClientData cd, Tcl_Interp *interp,
			 int argc, char *argv[]) {
    char *name;

    if(argc != 2) return TCL_ERROR;

    name = argv[1];

    return_int(gm_script_do_register_end(TCL_TYPE, name));
}

int c_tcl_version(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    if(argc != 1) return TCL_ERROR;
    return_str(VERSION);
}

int c_tcl_msgbox(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    char *title, *type, *buttons, *text; int i;
    int def = -1;
    if(argc != 5 && argc != 6) return TCL_ERROR;

    title = argv[1];
    type = argv[2];
    buttons = argv[3];
    text = argv[4];
    if(argc == 6) def = atoi(argv[5]);

    if(gm_script_do_msgbox(title, type, buttons, text, def, &i)) {
	return_int(i);
    }
    return TCL_ERROR;
}

int c_tcl_input(ClientData cd, Tcl_Interp *interp, int argc, char *argv[]) {
    char *title, *type, *desc, *def = "";
    int type_int;
    static char *ret_str = NULL; int ret_int;
    if(argc != 4 && argc != 5) return TCL_ERROR;

    title = argv[1];
    type = argv[2];
    desc = argv[3];
    if(argc == 5) def = argv[4];

    g_free(ret_str); ret_str = NULL; /* static variable agains memleaks :( */

    if(gm_script_do_input(title, desc, type, &type_int, def,
			  &ret_str, &ret_int)) {
	if(type_int == TYPE_STR) {
	    return_str(ret_str);
	} else if(type_int == TYPE_INT) {
	    return_int(ret_int);
	}
    }
    return TCL_ERROR;
}

#endif /* TCL */
