/*  GnomeKiss - A KiSS viewer for the GNOME desktop
    Copyright (C) 2000-2002  Nick Lamb <njl195@zepler.org.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <string.h>
#include <stdio.h>
#include <ctype.h>
#include <sys/types.h>
#include <dirent.h>
#include "kiss.h"

int global_lock= KISS_DONE;

static GMemChunk *event_chunk= NULL;

void free_action_list(GSList *list) {
  /* FIXME small leak, debug() notify() strings lost */
  g_slist_free(list);
}

static void free_actions(gpointer key, gpointer value, gpointer data) {
  KissActionList *list = (KissActionList *) value;
  FREE_ACTION_LIST(*list);
}

static void reset_events() {
  if (config.alarms) {
    g_hash_table_foreach(config.alarms, free_actions, NULL);
    g_hash_table_destroy(config.alarms);
    config.alarms= NULL;
  }
  if (config.labels) {
    g_hash_table_foreach(config.labels, free_actions, NULL);
    g_hash_table_destroy(config.labels);
    config.labels= NULL;
  }

  /* Now reset memchunk for the KissActionList placeholders */
  if (event_chunk) {
    g_mem_chunk_destroy(event_chunk);
  }
  event_chunk= g_mem_chunk_create(KissActionList, 100, G_ALLOC_ONLY);
}

void render_object(KissObject *object) {
  guchar *start, *end;

  start= config.rgb_buf + (config.row_stride * object->y[view]);
  end= start + (config.row_stride * object->height);
  if (start < global_start) global_start= start;
  if (end > global_end) global_end= end;
  global_lock= KISS_PENDING;
  gtk_widget_queue_draw_area(area, object->x[view], object->y[view],
                                   object->width, object->height);
}

void render_cell(KissCell *cell) {
  guchar *start, *end;

  start= config.rgb_buf + (config.row_stride * cell->y[view]);
  end= start + (config.row_stride * cell->height);
  if (start < global_start) global_start= start;
  if (end > global_end) global_end= end;
  global_lock= KISS_PENDING;
  gtk_widget_queue_draw_area(area, cell->x[view], cell->y[view],
                                   cell->width, cell->height);
}


void render_all() {
  GdkColor color;
  GdkColormap *cmap;
  gint set;
  static long int last = -1;

  if (area == NULL) return;

  global_start= config.rgb_buf;
  global_end= config.rgb_buf + (config.row_stride * config.height);
  global_lock= KISS_PENDING;
  gtk_widget_queue_draw(area); /* redraw everything */

  /* render_all gets called just in time to fix the bg color */
  cmap= gtk_widget_get_colormap(area);
  set= config.pal_set[view];

  if (set * 256 + config.bg_color != last) {
    last = set * 256 + config.bg_color;
    color.red= 257 * palettes[set][config.bg_color * 3];
    color.green= 257 * palettes[set][config.bg_color * 3 + 1];
    color.blue= 257 * palettes[set][config.bg_color * 3 + 2];

    /* I have no idea if this is portable, removing it will just cause
       the unused border to be grey instead of colored */

    if (gdk_colormap_alloc_color(cmap, &color, FALSE, TRUE)) {
      gdk_window_set_background(area->window, &color);
      gdk_window_clear(area->window);
    }
  }

}

/* Only works for filename in the current directory, deliberately */
int case_fixup(char *filename) {
  DIR *dir;
  struct dirent *dirent;
  
  dir= opendir(".");
  if (dir == NULL) return 0;

  while ((dirent= readdir(dir))) {
    if (!g_strcasecmp(filename, dirent->d_name)) {
      strcpy(filename, dirent->d_name);
      closedir(dir);
      return 1;
    }
  }

  closedir(dir);
  return 0;
}

static int length_fixup(char *filename) {
  int offset, j, k;

  for (k=0; k < 8; ++k) {
    if (filename[k] == '.' || filename[k] == 0) break;
  }

  if (filename[k] == 0) return 0;

  /* Remove additional characters before '.' separator */
  offset= 0;
  while (filename[k] != '.' && filename[k] != 0) {
    offset++;
    filename[k]= filename[k+offset];
  }

  if (filename[k] == 0) return offset;

  /* Count through first three characters after '.' separator */
  for (j= 1; j <= 3; ++j) {
    filename[k + j]= filename[k+offset + j];
    if (filename[k + j] == 0) break;
  }

  if (filename[k + j] == 0) return offset;

  /* truncating trailing characters after e.g. .CEL */
  filename[k + j] = 0;
  return 1;
}

FILE *open_any(const char *path, const char *mode) {
  FILE *fp = NULL;
  char *temp;

  temp = g_strdup(path);

  fp= fopen(temp, mode);
  if (fp == NULL && case_fixup(temp)) {
    fp= fopen(temp, mode);
  }
  if (fp == NULL && length_fixup(temp) && case_fixup(temp)) {
    log_warning(_("filename \"%s\" is too long"), path);
    fp= fopen(temp, mode);
  }

  g_free(temp);
  if (fp == NULL)
    log_error(_("file not found \"%s\""), path);
  return fp;
}

void change_cursor(GdkWindow *window, GdkCursorType type) {
  GdkCursor *cursor;

  cursor = gdk_cursor_new(type);
  gdk_window_set_cursor(window, cursor);
  gdk_cursor_destroy(cursor);
}

void clean_up() {
  int k;

  /* Timers first */
  reset_timers();

  /* French KiSS */
  reset_french();

  /* Cells and Objects */
  reset_cells();

  /* Variables */
  reset_variables();

  /* Label and Alarm events */
  reset_events();

  if (area != NULL) {
    gdk_window_clear(area->window);
  }

  /* free config filename */
  if (config.filename!= NULL) {
    g_free(config.filename);
  }

  config.filename= NULL;
  /* free config description */
  if (config.description!= NULL) {
    g_free(config.description);
  }
  config.description= NULL;

  /* free Configuration wide actions */
  FREE_ACTION_LIST(config.init);
  FREE_ACTION_LIST(config.begin);
  FREE_ACTION_LIST(config.version);
  FREE_ACTION_LIST(config.never);
  FREE_ACTION_LIST(config.end);
  FREE_ACTION_LIST(config.overflow);

  /* free Sets */
  for (k=0; k < SETS; ++k) {
    FREE_ACTION_LIST(config.sets[k]);
  }

  /* free Cols */
  for (k=0; k < COLS; ++k) {
    FREE_ACTION_LIST(config.cols[k]);
  }

  /* free Errors */
  FREE_ACTION_LIST(fkiss_errors);

  /* Remove RGB buffer */
  if (config.rgb_buf != NULL) {
    g_free(config.rgb_buf);
    config.rgb_buf= NULL;
  }
  config.width= 640;
  config.height= 480;
  config.row_stride= config.width * 3;

  /* free Palettes */
  for (k= 0; k < PALETTES * COLS; ++k) {
    if (palettes[k] != NULL) {
      g_free(palettes[k]);
    }
    palettes[k]= NULL;
  }

  /* clear BG color index */
  config.bg_color= 0;

  /* Selected and target cells */
  config.target= NULL;
  config.selected= NULL;
}

KissCell *intersect(int x, int y) {
  GSList *list;
  KissCell *final, *current;

  final= NULL;
  for (list= cells; list != NULL; list= g_slist_next(list)) {
    current= cell_intersect((KissCell *) list->data, x, y);
    if (current) final= current;
  }
  return final;
}

void open_either(const char *filename) {
  int len;
  char *temp;
  
  len= strlen(filename);
  if (len > 4 && g_strcasecmp(filename + len - 4, ".lzh") == 0) {
    if ((temp= lha_open(filename))) {
      gtk_file_selection_complete(GTK_FILE_SELECTION(config_open), temp);
      gtk_file_selection_complete(GTK_FILE_SELECTION(config_open), "*.cnf");
      gtk_widget_show (config_open);
      g_free(temp);
    }
  } else {
    parse_file(filename);
  }
  gtk_window_set_title(GTK_WINDOW(app), filename);
}

void switch_color(int col) {
  if (col < 0 || col >= COLS) return;
  if (config.pal_set[view] == col) return;
  if (config.objects == NULL) return;

  config.pal_set[view]= col; /* Should this be sticky or not? */

  /* Run events */
  events(config.cols[col]);

  render_all();
}

static void check_each_object(gpointer key, gpointer value, gpointer data) {
  KissObject *object = KissObject (value);
  check_collisions(object, 1); /* suppress event firing */
}

void switch_view(int set) {

  if (set < 0 || set >= SETS) return;
  if (set == view) return;
  if (config.objects == NULL) return;

  view= set;

  gtk_menu_item_activate(GTK_MENU_ITEM(items[view]));
  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(buttons[view]), TRUE);

  /* Check all object collisions */
  g_hash_table_foreach(config.objects, check_each_object, NULL);

  /* Run events */
  events(config.sets[view]);

  render_all();
}

long numeric(gchar *integer) {
  long value;

  value= atol(integer);
  while (*integer != '\0') {
    if (*integer != '-' && !isdigit(*integer)) return INT_MIN;
    integer++;
  }
  return value;
}

KissActionList *event_find(GHashTable *table, gint id) {

  return (KissActionList *)
    (g_hash_table_lookup(table, GINT_TO_POINTER(id)));
}

KissActionList *event_find_or_insert(GHashTable *table, gint id) {

  KissActionList *event;

  event= (KissActionList *)
    (g_hash_table_lookup(table, GINT_TO_POINTER(id)));

  if (event == NULL) {
    event= g_chunk_new0(KissActionList, event_chunk);
    g_hash_table_insert(table, GINT_TO_POINTER(id), event);
  }
  return event;
}

