#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# GridCellEditors.py
#
# DESCRIPTION:
# Provides common wxGrid cell eitors
#
# NOTES:
#

__all__ = [ 'CharCellEditor', 'RestrictedCellEditor',
            'BoolCellEditor', 'IntCellEditor']

import sys, os, string
from wxPython.wx import *
from wxPython.grid import *
from gnue.common.apps import GDebug
from gnue.common.formatting import GTypecast

#
#
#
class CharCellEditor (wxPyGridCellEditor):
  def __init__(self, grid, attributes):
    wxPyGridCellEditor.__init__(self)
    self.grid = grid
    self.attributes = attributes
    self.__created = 0


  def Create(self, parent, id, evtHandler):
    self.__created = 1
    GDebug.printMesg(10,'Creating CharCellEditor')
    self._tc = wxTextCtrl(parent, id, "")
    self._tc.SetInsertionPoint(0)
    self.SetControl(self._tc)
    if evtHandler:
      self._tc.PushEventHandler(evtHandler)

  def SetSize(self, rect):
    self._tc.SetDimensions(rect.x, rect.y, rect.width+2, rect.height+2)

  def Show(self, show, attr):
    self.base_Show(show, attr)

  def PaintBackground(self, rect, attr):
    pass

  def BeginEdit(self, row, col, grid):
    self.startValue = grid.GetTable().GetValue(row, col)
    self._tc.SetValue(self.startValue)
    self._tc.SetInsertionPointEnd()
    self._tc.SetFocus()

    # For this example, select the text
    self._tc.SetSelection(0, self._tc.GetLastPosition())


  def EndEdit(self, row, col, grid):
    changed = false

    val = self._tc.GetValue()
    if val != self.startValue:
       changed = true
       grid.GetTable().SetValue(row, col, val) # update the table

    self.startValue = ''
    self._tc.SetValue('')
    return changed


  def Reset(self):
    self._tc.SetValue(self.startValue)
    self._tc.SetInsertionPointEnd()


  def IsAcceptedKey(self, evt):
    return (not (evt.ControlDown() or evt.AltDown()) and
            evt.GetKeyCode() != WXK_SHIFT)


  def StartingKey(self, evt):
    GDebug.printMesg(10, "Starting Key; created=%s" % self.__created)
    key = evt.GetKeyCode()
    ch = None
    if key in [WXK_NUMPAD0, WXK_NUMPAD1, WXK_NUMPAD2, WXK_NUMPAD3, WXK_NUMPAD4,
               WXK_NUMPAD5, WXK_NUMPAD6, WXK_NUMPAD7, WXK_NUMPAD8, WXK_NUMPAD9]:
        ch = ch = chr(ord('0') + key - WXK_NUMPAD0)

    elif key < 256 and key >= 0 and chr(key):
      ch = chr(key)
      if not evt.ShiftDown():
        ch = string.lower(ch)

    if ch is not None:
      # For this example, replace the text.  Normally we would append it.
      #self._tc.AppendText(ch)
#      self._tc.SetValue(ch)
 #     self._tc.SetInsertionPointEnd()
      pass
    else:
      evt.Skip()


  def StartingClick(self):
    pass

  def Destroy(self):
    self.base_Destroy()

  def Clone(self):
    return CharCellEditor(self.attributes)


class IntCellEditor (CharCellEditor):
  def __init__(self, grid, attributes):
    CharCellEditor.__init__(self, grid, attributes)

  def Create(self, parent, id, evtHandler):
    CharCellEditor.Create(self, parent, id, evtHandler)
    EVT_CHAR(self._tc, self.OnKeyPressed)

  def OnKeyPressed(self, evt):
    if (ord('0') <= evt.KeyCode() <= ord('9') or \
        evt.KeyCode() in (ord('-'), ord('+')) or \
        evt.KeyCode() < 32 or evt.KeyCode() > 126):
      evt.Skip()

  def StartingKey(self, evt):
    key = evt.GetKeyCode()
    ch = None
    if key in [WXK_NUMPAD0, WXK_NUMPAD1, WXK_NUMPAD2, WXK_NUMPAD3, WXK_NUMPAD4,
               WXK_NUMPAD5, WXK_NUMPAD6, WXK_NUMPAD7, WXK_NUMPAD8, WXK_NUMPAD9]:
        ch = ch = chr(ord('0') + key - WXK_NUMPAD0)

    elif key < 256 and key >= 0 and chr(key):
      ch = chr(key)
      if not evt.ShiftDown():
        ch = string.lower(ch)

    if '0' <= ch <= '9':
      # For this example, replace the text.  Normally we would append it.
      #self._tc.AppendText(ch)
      pass
#      self._tc.SetValue(ch)
#      self._tc.SetInsertionPointEnd()
    else:
      evt.Skip()


class RestrictedCellEditor (CharCellEditor):
  def __init__(self, grid, attributes):
    CharCellEditor.__init__(self, grid, attributes)
    self.valueList = attributes['ValueSet'].keys()

    self.valueList.sort()
    self.selectionList = []
    self.valueMap = {}

    i = 0
    if not (attributes.has_key('Required') and attributes['Required'] ):
      self.valueMap[''] = 0
      self.selectionList.append(' <Not Set> ')
      i = 1

    for v in self.valueList:
      self.valueMap[v] = i
      self.selectionList.append("%s" % (attributes['ValueSet'][v] or v))
      i = i + 1

    self.valueList.insert(0, '')

  def Create(self, parent, id, evtHandler):
    GDebug.printMesg(10,'Creating CharCellEditor')
    self._tc = wxComboBox(parent, id, "", style=wxCB_READONLY,
                          choices=self.selectionList)
    self.SetControl(self._tc)
    if evtHandler:
      self._tc.PushEventHandler(evtHandler)

  def BeginEdit(self, row, col, grid):
    self.startValue = grid.GetTable().GetValue(row, col)
    self._tc.SetSelection(self.valueMap[self.startValue])
    self._tc.SetFocus()

  def EndEdit(self, row, col, grid):
    changed = false

    val = self.valueList[self._tc.GetSelection()]
    if val != self.startValue:
       changed = true
       grid.GetTable().SetValue(row, col, "%s" % val) # update the table

    self.startValue = ''
    self._tc.SetSelection(0)
    return changed


  def Reset(self):
    self._tc.SetSelection(self.valueMap[self.startValue])
    #self._tc.SetInsertionPointEnd()


  def StartingKey(self, evt):
    key = evt.GetKeyCode()
    ch = None
    if key in [WXK_NUMPAD0, WXK_NUMPAD1, WXK_NUMPAD2, WXK_NUMPAD3, WXK_NUMPAD4,
               WXK_NUMPAD5, WXK_NUMPAD6, WXK_NUMPAD7, WXK_NUMPAD8, WXK_NUMPAD9]:
        ch = ch = chr(ord('0') + key - WXK_NUMPAD0)

    elif key < 256 and key >= 0 and chr(key):
      ch = chr(key)
      if not evt.ShiftDown():
        ch = string.lower(ch)

    if ch is not None:
      # For this example, replace the text.  Normally we would append it.
      #self._tc.AppendText(ch)
      pass
#      self._tc.SetValue(ch)
#      self._tc.SetInsertionPointEnd()
    else:
      evt.Skip()


class BoolCellEditor (CharCellEditor):
  def __init__(self, grid, attributes):
    CharCellEditor.__init__(self, grid, attributes)

    self.selectionList = ['FALSE','TRUE']

  def Create(self, parent, id, evtHandler):
    self._tc = wxComboBox(parent, id, "", style=wxCB_READONLY, 
                          choices=self.selectionList)
    self.SetControl(self._tc)
    if evtHandler:
      self._tc.PushEventHandler(evtHandler)

  def BeginEdit(self, row, col, grid):
    self.startValue = grid.GetTable().GetValue(row, col)
    
    self._tc.SetValue(self.startValue)
    self._tc.SetFocus()

  def EndEdit(self, row, col, grid):
    changed = false

    if self._tc.GetSelection(): 
      val = 'TRUE' 
    else: 
      val = 'FALSE' 
    if val != self.startValue:
       changed = true
       grid.GetTable().SetValue(row, col, "%s" % val) # update the table

    self.startValue = ''
    self._tc.SetSelection(0)
    return changed


  def Reset(self):
    self._tc.SetValue(self.startValue)
    #self._tc.SetInsertionPointEnd()


  def StartingKey(self, evt):
    key = evt.GetKeyCode()
    ch = None
    if key in [WXK_NUMPAD0, WXK_NUMPAD1, WXK_NUMPAD2, WXK_NUMPAD3, WXK_NUMPAD4,
               WXK_NUMPAD5, WXK_NUMPAD6, WXK_NUMPAD7, WXK_NUMPAD8, WXK_NUMPAD9]:
        ch = ch = chr(ord('0') + key - WXK_NUMPAD0)

    elif key < 256 and key >= 0 and chr(key):
      ch = chr(key)
      if not evt.ShiftDown():
        ch = string.lower(ch)

    if ch is not None:
      # For this example, replace the text.  Normally we would append it.
      #self._tc.AppendText(ch)
      pass
#      self._tc.SetValue(ch)
#      self._tc.SetInsertionPointEnd()
    else:
      evt.Skip()

