/*
**  GNUMail+TaskManager.m
**
**  Copyright (c) 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "GNUMail+TaskManager.h"

#include "ConsoleWindowController.h"
#include "Filter.h"
#include "FilterManager.h"
#include "Constants.h"
#include "MailboxManagerController.h"
#include "MailWindowController.h"
#include "MessageViewWindowController.h"
#include "Task.h"
#include "TaskManager.h"
#include "Utilities.h"

#include <Pantomime/Constants.h>
#include <Pantomime/IMAPFolder.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/Message.h>
#include <Pantomime/URLName.h>

#include <Foundation/NSArray.h>
#include <Foundation/NSPort.h>
#include <Foundation/NSThread.h>
#include <Foundation/NSTimer.h>

#define LONG_DELAY 86400.0

//
// TaskManager related methods
//
@implementation GNUMail (TaskManager)

//
//
//
- (void) addTask: (Task *) theTask
{
  // We insert it at the beginning of our array
  if ( theTask )
    {
      NSDebugLog(@"Inserting the task, Op: %d", [theTask op]);
      [tasks insertObject: theTask
	     atIndex: 0];

      // We fire immediately our TaskManager's timer if we have
      // only one task OR if our task is an "immediate" one
      // but NOT if there's already a task that is running.
      if ( ([tasks count] == 1 || [theTask immediate] == YES) &&
	   [[ConsoleWindowController singleInstance] animation] == nil &&
	   [theTask origin] != ORIGIN_STARTUP )
	{
	  NSDebugLog(@"Immediately starting the task...");
	  [(TaskManager *)[connection rootProxy] fire];
	  [[ConsoleWindowController singleInstance] startAnimation];
	}

      [[[ConsoleWindowController singleInstance] tasksTableView] reloadData];
    }
}


//
//
//
- (void) removeTask: (Task *) theTask
{
  [tasks removeObject: theTask];
  [[[ConsoleWindowController singleInstance] tasksTableView] reloadData];
}


//
//
//
- (NSArray *) allTasks
{
  return [NSArray arrayWithArray: tasks];
}


//
//
//
- (void) authenticationFailed: (NSDictionary *) theDictionary
{
  NSString *aString;
  int theType;

  theType = [[theDictionary objectForKey: @"Type"] intValue];

  switch ( theType )
    {
    case POP3:
      aString = _(@"POP3");
      break;

    case IMAP:
      aString = _(@"IMAP");
      break;

    default:
      aString = _(@"SMTP");
    }

  NSRunAlertPanel(_(@"Error!"),
		  _(@"%@ authentication failed for %@."),
		  _(@"OK"),
		  NULL,
		  NULL,
		  aString,
		  [NSString stringWithFormat: @"%@ @ %@", [theDictionary objectForKey: @"Username"],
			    [theDictionary objectForKey: @"Server"]]);
  
  // We now invalidate our password
  [[Utilities passwordCache] removeObjectForKey: [NSString stringWithFormat: @"%@ @ %@", [theDictionary objectForKey: @"Username"],
							   [theDictionary objectForKey: @"Server"]]];
}


//
// This method checks for new mail on the specified account name. 
//
// If the account type is POP3 or UNIX, this method adds a new task to 
// immediately check for new mails on this account.
//
// If the account type is IMAP, it NOOPs the folder or the store if
// and only if it is open.
//
- (void) checkMailForAccount: (NSString *) theAccountName
		      origin: (int) theOrigin
		       owner: (id) theOwner
{
  NSDictionary *allValues;
  Task *aTask;
  
  int op, subOp;
 
  // If the account is disabled or is set to never check mail we do nothing.
  if ( [[[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] objectForKey: theAccountName]
	  objectForKey: @"ENABLED"] boolValue] == NO ||
       [[[[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] objectForKey: theAccountName]
	   objectForKey: @"RECEIVE"] objectForKey: @"RETRIEVEMETHOD"] intValue] == NEVER )
    {
      return;
    }

  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		 objectForKey: theAccountName] objectForKey: @"RECEIVE"];
  subOp = 0;
  
  if ( ![allValues objectForKey: @"SERVERTYPE"] || [[allValues objectForKey: @"SERVERTYPE"] intValue] == POP3 )
    {
      op = RECEIVE_POP3;
    }
  else if ([[allValues objectForKey: @"SERVERTYPE"] intValue] == IMAP)
    {
      IMAPStore *aStore;
      
      aStore = [[MailboxManagerController singleInstance] storeForName: [allValues objectForKey: @"SERVERNAME"]
							  username: [allValues objectForKey: @"USERNAME"]];
      
      // We noop the store, if one is initialized.
      if ( aStore )
	{  
	  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: (@"NOOPing IMAP server %@."), 
										 [aStore name]]];
	  [aStore noop];

	  if ( IMAPSTORE_IS_DISCONNECTED(aStore) )
	    {
	      [[MailboxManagerController singleInstance] setStore: nil
							 name: [aStore name]
							 username: [aStore username]];
	    }
	}
      else
	{
	  // The IMAP store is closed, no need to check for new mail on that one.
	  return;
	}
      
      op = RECEIVE_IMAP;
      subOp = IMAP_STATUS;
    }
  else
    {
      op = RECEIVE_UNIX;
    }
  
  aTask = [[Task alloc] init];
  [aTask setOp: op];
  [aTask setSubOp: subOp];
  [aTask setMessage: nil];
  [aTask setKey: theAccountName];
  [aTask setImmediate: YES];
  [aTask setOrigin: theOrigin];
  [aTask setOwner: theOwner];
  [self addTask: aTask];
  RELEASE(aTask);
}


//
//
//
- (void) checkForNewMail: (id) theSender
	      controller: (MailWindowController *) theMailWindowController
{
  NSArray *allKeys;
  int i, origin;

  // First of all, we get the 'origin' of the action. That is, if it was 
  // initiated from the user, from the timer or from self, when verifying
  // mail on statup.
  if ( theSender == theMailWindowController ||
       theSender == [NSApp delegate] )
    {
      origin = ORIGIN_STARTUP;
    }
  else
    {
      // We clicked on the get button or on the menu item.
      origin = ORIGIN_USER;
    }
  
  //
  // If it's our menu item that called this method but, it's NOT the get "All"
  // that has been clicked. So, we must verify only one account.
  //
  if ( theSender && 
       [theSender isKindOfClass: [NSMenuItem class]] &&
       [theSender tag] >= 0 )
    {
      allKeys = [NSArray arrayWithObject: [theSender title]];
    }
  //
  // If the user has clicked on the Get button OR 
  // It is our "All" menu item that was clicked on.
  //
  else if ( (theMailWindowController && theSender == theMailWindowController->get) ||
	    (theSender && [theSender isKindOfClass: [NSMenuItem class]] && [theSender tag] < 0) )
    {
      // We get all accounts here but we'll verify later that we only use POP3 accounts.
      allKeys = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] allKeys]
		  sortedArrayUsingSelector: @selector(compare:)];
    }
  //
  // Our sender is the app's delegate. That means that we were asked to verify mails 
  // on startup. Let's only get the POP3 and UNIX accounts that we must verify on startup.
  // We skip over IMAP accounts since we don't want to establish the required connections.
  //
  else if ( theSender == self )
    {
      NSMutableArray *aMutableArray;
      
      aMutableArray = [[NSMutableArray alloc] initWithArray: [[[NSUserDefaults standardUserDefaults]
								objectForKey: @"ACCOUNTS"] allKeys]];      
      for (i = ([aMutableArray count]-1); i >= 0; i--)
	{
	  NSDictionary *allValues;
	  NSString *aKey;
	  
	  aKey = [aMutableArray objectAtIndex: i];

	  // If the account is disabled or it's set to never check mail, we skip over it
	  if ( [[[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] objectForKey: aKey]
		  objectForKey: @"ENABLED"] boolValue] == NO ||
	       [[[[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] objectForKey: aKey]
		   objectForKey: @"RECEIVE"] objectForKey: @"RETRIEVEMETHOD"] intValue] == NEVER )
	    {
	      continue;
	    }

	  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
			 objectForKey: aKey] objectForKey: @"RECEIVE"];
	  
	  if ( ![allValues objectForKey: @"CHECKONSTARTUP"] ||
	       [[allValues objectForKey: @"CHECKONSTARTUP"] intValue] == NSOffState ||
	       ([allValues objectForKey: @"SERVERTYPE"] && [[allValues objectForKey: @"SERVERTYPE"] intValue] == IMAP) )
	    {
	      [aMutableArray removeObject: aKey];
	    }
	  else
	    {
	      NSDebugLog(@"Will verify for new mail on startup for %@", aKey);
	    }
	}

      allKeys = AUTORELEASE(aMutableArray);
    }
  else
    {
      NSDebugLog(@"GNUMail+TaskManager: -checkForNewMail: controller: called w/o being handled.");
      return;
    }
  
  
  // If we got no keys, we immediately stop our animation
  if ( [allKeys count] == 0 && theMailWindowController )
    {
      [theMailWindowController stopAnimation];
    }

  // We send all our tasks!
  for (i = 0; i < [allKeys count]; i++)
    {
      [self checkMailForAccount: [allKeys objectAtIndex: i]
	    origin: origin
	    owner: theMailWindowController];
    }
}


//
//
//
- (void) consoleWindowNeedsDisplay
{
  [[[ConsoleWindowController singleInstance] tasksTableView] setNeedsDisplay: YES];
}


//
//
//
- (Task *) currentTask
{
  return currentTask;
}


//
//
//
- (void) runAlertPanel: (NSDictionary *) theDictionary
{
  NSRunAlertPanel([theDictionary objectForKey: @"Title"],
		  [theDictionary objectForKey: @"Message"],
		  _(@"OK"),
		  NULL,
		  NULL,
		  NULL);
}


//
// This method is the ONLY one that assigns the current running task.
// If there's already one set, it returns nil.
// 
- (Task *) nextTask
{
  NSDate *aDate;
  Task *aTask;
  int i;

  NSDebugLog(@"Getting next task");

  // If there's already a running task, let's finish that one before
  // starting a new one. That must task MUST exist and MUST be runnning.
  // We verify that it MUST be running since we could have (for example),
  // a stopped task for a mail being sent in SMTP.
  if ( currentTask && [currentTask running])
    {
      return nil;
    }

  // We search for a task that should be run immediately
  for (i = ([tasks count] - 1); i >= 0; i--)
    {
      aTask = [tasks objectAtIndex: i];

      if ( [aTask running] == NO )
	{
	  // We found one, let's return it.
	  if ( [aTask immediate] == YES )
	    {
	      ASSIGN(currentTask, aTask);
	      [aTask setRunning: YES];
	      [[ConsoleWindowController singleInstance] startAnimation];
	      [self consoleWindowNeedsDisplay];
	      return aTask; 
	    }
	}
    }  
  
  // We haven't found an immediate task. We loop from the oldest task
  // to the newest one to find a task that -date meets the requirements of
  // the current date/time.
  aDate = [NSDate date];

  for (i = ([tasks count] - 1); i >= 0; i--)
    {
      aTask = [tasks objectAtIndex: i];

      if ( [aTask running] == NO )
	{
	  if ( [[aTask date] compare: aDate] == NSOrderedAscending )
	    {
	      ASSIGN(currentTask, aTask);
	      [aTask setRunning: YES];
	      [[ConsoleWindowController singleInstance] startAnimation];
	      [self consoleWindowNeedsDisplay];
	      return aTask;
	    }
	}
    }
  
  NSDebugLog(@"No task found meeting requirements.");

  return nil;
}


//
//
//
- (NSString *) passwordForKey: (id) theKey
                         type: (int) theType
{
  return [Utilities passwordForKey: theKey
		    type: theType
		    prompt: YES];
}


//
//
//
- (void) startTaskManager
{ 
  TaskManager *aTaskManager;
  NSPort *port1, *port2;
  NSArray *ports;
  
  tasks = [[NSMutableArray alloc] init];

  aTaskManager = [TaskManager singleInstance];
  
  // We initialize our NSConnection object and our ports
  port1 = [[NSPort alloc] init];
  port2 = [[NSPort alloc] init];
  
  connection = [[NSConnection alloc] initWithReceivePort: port1
				     sendPort: port2];
  [connection setRootObject: self];
  [connection setRequestTimeout: LONG_DELAY];
  [connection setReplyTimeout: LONG_DELAY];

  ports = [NSArray arrayWithObjects: port2, port1, nil];
  
  RELEASE(port1);
  RELEASE(port2);
  
  NS_DURING
    {
      [NSThread detachNewThreadSelector: @selector(run:)
		toTarget: aTaskManager
		withObject: ports];
    }
  NS_HANDLER
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"A fatal error occured while detaching the thread.\nRecompile your runtime with threads support."),
		      _(@"OK"),
		      NULL,
		      NULL);
      DESTROY(connection); 
    }
  NS_ENDHANDLER
}


//
//
//
- (void) startTimer
{  
  timer = [NSTimer scheduledTimerWithTimeInterval: 60
		   target: self
		   selector: @selector(tick:)
		   userInfo: nil
		   repeats: YES];
  counter = 0;
  RETAIN(timer);
}


//
//
//
- (void) stopTaskManager
{
  if ( connection )
    {
      NSDebugLog(@"Stopping the TaskManager...");
      [(TaskManager *)[connection rootProxy] stop];
      [connection invalidate];
      DESTROY(connection);
      RELEASE(tasks);
      NSDebugLog(@"Done.");
    }
}


//
//
//
- (void) stopTimer
{
  if ( timer )
    {
      [timer invalidate];
      DESTROY(timer);
    }
}


//
//
//
- (void) taskCompleted
{
  int i;

  if ( currentTask )
    {
      // If it was a RECEIVE task and we transferred messages to other mailboxes,
      // we warn the user about this.
      if ( [currentTask op] == RECEIVE_POP3 || [currentTask op] == RECEIVE_UNIX )
	{
	  if ( [currentTask receivedMessagesCount] > 0 &&
	       ([currentTask origin] == ORIGIN_STARTUP || [currentTask origin] == ORIGIN_TIMER) )
	    {
	      // We verify if we must play a sound. If the sound isn't found, we simply use NSBeep()
	      if ( [[NSUserDefaults standardUserDefaults] boolForKey: @"PLAY_SOUND"] )
		{
		  NSString *aString;

		  aString = [[NSUserDefaults standardUserDefaults] stringForKey: @"PATH_TO_SOUND"];
		  if ( [[NSFileManager defaultManager] fileExistsAtPath: aString] )
		    {
		      NSSound *aSound;
		      
		      aSound = [[NSSound alloc] initWithContentsOfFile: aString  byReference: YES];
		      [aSound play];
		      RELEASE(aSound);
		    }
		  else
		    {
		      NSBeep();
		    }
		}
	    }
	  
	  if ( [[currentTask filteredMessagesFolders] count] > 0 )
	    {
	      
	      if ( [[NSUserDefaults standardUserDefaults] boolForKey: @"SHOW_FILTER_PANEL"] )
		{
		  NSRunInformationalAlertPanel(_(@"Filtered messages..."),
					       _(@"%d messages have been transfered to the following folders:\n%@"),
					       _(@"OK"),
					       NULL,
					       NULL,
					       [currentTask filteredMessagesCount],
					       [[currentTask filteredMessagesFolders] componentsJoinedByString: @"\n"]);
		}

	      if ( [[NSUserDefaults standardUserDefaults] boolForKey: @"OPEN_MAILBOX_AFTER_TRANSFER"] )
		{
		  NSString *aString, *aStoreName, *aFolderName;
		  URLName *theURLName;
		  NSRange aRange;
 
		  for (i = 0; i < [[currentTask filteredMessagesFolders] count]; i++)
		    {
		      aString = [[currentTask filteredMessagesFolders] objectAtIndex: i];
		      aRange = [aString rangeOfString: @" - "];
		      
		      aFolderName = [aString substringFromIndex: NSMaxRange(aRange)];
		      aStoreName = [aString substringToIndex: aRange.location];

		      if ( [aStoreName isEqualToString: _(@"Local")] )
			{
			  NSString *thePathToLocalMailDir;

			  thePathToLocalMailDir = [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"];
			  theURLName = [[URLName alloc] initWithString: [NSString stringWithFormat: @"local://%@/%@", 
										  thePathToLocalMailDir, aFolderName]
							path: thePathToLocalMailDir];
			  
			}
		      else
			{
			  // We skip the "IMAP " part
			  aRange = [aStoreName rangeOfString: _(@"IMAP ")];
			  aStoreName = [aStoreName substringFromIndex: NSMaxRange(aRange)];
			  theURLName = [[URLName alloc] initWithString: [NSString stringWithFormat: @"imap://%@/%@",
										  aStoreName, aFolderName]];
			}
		      
		      // We finally opens the folder and release our URLName object.
		      [[MailboxManagerController singleInstance] openFolderWithURLName: theURLName
								 sender: [NSApp delegate]];
		      RELEASE(theURLName);
		    }    
		}
	    }
	}
      
      NSDebugLog(@"Removing the task, Op: %d", [currentTask op]);
      [self removeTask: currentTask];
      DESTROY(currentTask);
    }

  // If we have a task in the queue that has its immediate flag set to YES,
  // we fire again our task manager.
  for (i = ([tasks count] - 1); i >= 0; i--)
    {
      if ( [[tasks objectAtIndex: i] immediate] == YES )
	{
	  [(TaskManager *)[connection rootProxy] fire];
	  break;
	}
    }
}


//
//
//
- (void) tick: (id) sender
{
  NSEnumerator *theEnumerator;
  NSDictionary *allValues;
  NSArray *allWindows;
  NSString *aKey;
  id aController;
  
  counter += 1;
 
  aController = [GNUMail lastMailWindowOnTop];

  if ( aController )
    {
      aController = [[GNUMail lastMailWindowOnTop] windowController];

      if ( [aController isKindOfClass: [MessageViewWindowController class]] )
	{
	  aController = [(MessageViewWindowController *)aController mailWindowController];
	}
    }
 
  theEnumerator = [[Utilities allEnabledAccounts] keyEnumerator];

  while ( (aKey = [theEnumerator nextObject]) )
    {     
      allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] 
		     objectForKey: aKey] objectForKey: @"RECEIVE"];

      // If we check for new mails from this account automatically AND
      // If our counter modulo the retreive interval is zero, we check for new mails.
      if ( [[allValues objectForKey: @"RETRIEVEMETHOD"] intValue] == AUTOMATICALLY &&
	   (counter % [[allValues objectForKey: @"RETRIEVEMINUTES"] intValue] ) == 0 )
	{
	  [self checkMailForAccount: aKey
		origin: ORIGIN_TIMER
		owner: aController];
	}
    }

  
  // We now loop in all opened MailWindow:s and all messages of them to verify if we got
  // any messages that are expired, and not selected.   
  allWindows = [GNUMail allMailWindows];
  
  if ( allWindows )
    {
      NSArray *allMessages;
      Folder *aFolder;

      NSCalendarDate *date;
      NSDate *aDate;

      id aWindow, aMessage, aSelectedMessage;
      int i, j, count, seconds;
      
      date = (NSCalendarDate *)[NSCalendarDate date];

      for (i = 0; i < [allWindows count]; i++)
	{
	  aWindow = [allWindows objectAtIndex: i];
	  aFolder = [(MailWindowController *)[aWindow windowController] folder];
	  aSelectedMessage = [(MailWindowController *)[aWindow windowController] selectedMessage];

	  allMessages = [aFolder allMessages];
	  count = [aFolder count];
	  
	  for (j = 0; j < count; j++)
	    {
	      aMessage = [allMessages objectAtIndex: j];
	      
	      // If this message is selected, skip it
	      if ( aMessage == aSelectedMessage )
		{
		  continue;
		}

	      aDate = [aMessage propertyForKey: @"EXPIRE_DATE"];
	      
	      // If this property doesn't exist, that means it was never initialzed, so skip it
	      if ( !aDate ) 
		{
		  continue;
		}
	      
	      [date years: NULL
		    months: NULL
		    days: NULL
		    hours: NULL
		    minutes: NULL
		    seconds: &seconds
		    sinceDate: aDate];
	      
	      // If the message has expired, we release its content and we set it as uninitialzed
	      if ( seconds > RETAIN_PERIOD )
		{
		  [aMessage setInitialized: NO];
		  [aMessage setRawSource: nil];
		  [(Message *)aMessage setProperty: nil  forKey: @"EXPIRE_DATE"];
		}
	    }
	}
    }
}


//
//
//
- (void) updateStatusLabelWithMessage: (NSString *) theString
{
  id aWindow;
  int i;

  for (i = 0; i < [[GNUMail allMailWindows] count]; i++)
    {
      aWindow = [[GNUMail allMailWindows] objectAtIndex: i];

      if ( [[aWindow delegate] isKindOfClass: [MailWindowController class]] )
	{
	  [[aWindow delegate] updateStatusLabelWithMessage: theString];
	}
    }
}

@end
