/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 * Licensed under the GNU GPL.
 * Absolutely no warranty.
 */

#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include <audiofile.h>
#include <gnome.h>
#include <config.h>
#include "mem.h"
#include "marker.h"

#define USE_MCACHE 0

void
marker_list_load(struct marker_list *ml,
                 const char *path,
                 int channel) {
    int i, argc;
    char **argv;
    void *iter;
    char tmp[512], *key, *value;
    snprintf(tmp, 512, "=%s=/Markers for track %d", path, channel);
    iter = gnome_config_init_iterator(tmp);        
    while(iter && gnome_config_iterator_next(iter, &key, &value)) {
        gnome_config_make_vector(value, &argc, &argv);
        if(argc != 4) {
            FAIL("unexpected value count %d (expected 4).\n", argc);
        } else {
            marker_list_insert(ml,
                               atoi(key),
                               atoi(argv[0]),
                               atof(argv[2]),
                               argv[3]);
        }            
        for(i = 0; i < argc; i++) 
            g_free(argv[i]);
        g_free(argv);
        g_free(key);
        g_free(value);
    }
}

void
marker_list_save(struct marker_list *ml,
                 const char *path,
                 int channel) {
    struct marker *m;
    GList *l;
    char *argv[4], args[4][100], key[512];
    argv[0] = args[0]; argv[1] = args[1]; argv[2] = args[2]; argv[3] = args[3];
    for(l = ml->markers; l; l = l->next) {
        m = (struct marker *)l->data;
        snprintf(key, 512, "=%s=/Markers for track %d/%ld", 
                 path, channel, m->frame_offset);
        snprintf(argv[0], 100, "%d", m->type);
        snprintf(argv[1], 100, "%d", m->flags);
        snprintf(argv[2], 100, "%f", m->multiplier);
        snprintf(argv[3], MIN(100, m->label ?
                              (strlen(m->label) + 1) : 2), "%s",
                 m->label ? 
                 m->label : " ");
        gnome_config_set_vector(key, 4, (const char * const *) argv);
    }
}

void
marker_dump(struct marker *m) {
    DEBUG("marker: frame_offset: %ld, type: %d, multiplier: %f, label: %s\n",
          m->frame_offset, m->type, m->multiplier, m->label);
}

struct marker_list **
marker_list_array_new() {
    return mem_calloc(sizeof(struct marker_list *), MAX_TRACKS);
}

void
marker_list_array_destroy(struct marker_list **mla) {
    int i;
    for(i = 0; i < MAX_TRACKS; i++)
        if(mla[i])
            marker_list_destroy(mla[i]);
    mem_free(mla);
}

struct marker *
marker_new(AFframecount frame_offset,
           enum marker_type type, 
           float multiplier,
           const char *label) { 
    struct marker *m = mem_alloc(sizeof(struct marker));
    if(!m) {
        FAIL("could not allocate memory for new marker\n");
        return NULL;
    }

    m->frame_offset = frame_offset;
    m->type = type;
    if(label)
        m->label = strdup(label);
    else
        m->label = NULL;
    m->multiplier = multiplier;
    m->flags = 0;
    return m;
}

struct marker *
marker_clone(struct marker *m) {
    struct marker *m_clone;
    m_clone = marker_new(m->frame_offset,
                         m->type,
                         m->multiplier,
                         m->label);
    m_clone->flags = m->flags;
    return m_clone;
}

void
marker_destroy(struct marker *m) {
    if(m->label)
        free(m->label);
    free(m);
}

double
marker_list_slope_value(struct marker_list *ml,
                        AFframecount frame_offset,
                        enum marker_type type) {
    struct marker *mn = NULL, *mp = NULL;
    double a, r, b, dn, dp, max, min;
    AFframecount x;

    if((ml->marker_types_enabled & type) == 0) 
        return 0;

#if USE_MCACHE
    if(ml->left_cache[type] != ml->right_cache[type] &&
       frame_offset >= ml->left_cache[type] &&
       frame_offset < ml->right_cache[type]) {
        if(ml->last_request[type] + 1 == frame_offset) {
            ml->last_return[type] = ml->last_return[type] + 
                ml->slope[type];
            ml->last_request[type] = frame_offset;
            return ml->last_return[type];
        } else if(ml->last_request[type] - 1 == frame_offset) {
            ml->last_return[type] = ml->last_return[type] - 
                ml->slope[type];
            ml->last_request[type] = frame_offset;
            return ml->last_return[type];
        }
    }
#endif

    mn = marker_list_next(ml,
                          frame_offset,
                          type);
    mp = marker_list_previous(ml,
                              frame_offset,
                              type);
    if(!(mn && mp))
        return 0;

    if(mn->frame_offset - mp->frame_offset <= 0) {
        FAIL("mn->frame_offset <= mp->frame_offset\n");
        return 0;
    }
    
    if(mp->multiplier == mn->multiplier) {
        return mp->multiplier;
    } else if(mp->multiplier > mn->multiplier) {
        max = mp->multiplier;
        min = mn->multiplier;
    } else {
        max = mn->multiplier;
        min = mp->multiplier;
    }
    a = (max - min) / (mn->frame_offset - mp->frame_offset);
    if(mp->multiplier > mn->multiplier) 
        a = -a;
    x = frame_offset - mp->frame_offset;
    b = mp->multiplier;
    
    r = (a * x) + b;

    dp = ABS(mp->multiplier) - ABS(r);
    dn = ABS(mn->multiplier) - ABS(r); 

    if(dp < -0.001 && dn < -0.001) {
        FAIL("rounding error: r: %f, mp->multiplier: %f, mn->multiplier: %f, a: %f, b: %f\n",
             r, mp->multiplier, mn->multiplier, a, b);
        r = CLAMP(r, MIN(mp->multiplier, mn->multiplier), MAX(mp->multiplier, mn->multiplier));
        FAIL("clamped to %f\n", r);
    }

#if USE_MCACHE
    ml->left_cache[type] = mp->frame_offset;
    ml->right_cache[type] = mn->frame_offset;
    ml->last_request[type] = frame_offset;
    ml->slope[type] = a;
#endif

    return r;
}

void
marker_list_invert(struct marker_list *ml,
                   AFframecount frame_offset,
                   AFframecount frame_count,
                   enum marker_type type) {
    
    GList *l;
    for(l = ml->markers; l; l = l->next) {
        if(((struct marker *)(l->data))->frame_offset < frame_offset)
            continue;
        if(((struct marker *)(l->data))->frame_offset > frame_offset + frame_count)
            break;
        if(((struct marker *)(l->data))->type & type)
            ((struct marker *)(l->data))->multiplier *= -1;
    }
}

void
marker_list_marker_delete(struct marker_list *ml,
                          struct marker *m) {
    GList *l;
#if USE_MCACHE
    ml->left_cache[m->type] = ml->right_cache[m->type] = 0;
#endif
    for(l = ml->markers; l; l = l->next) {
        if((struct marker *)(l->data) == m) {
            ml->markers = g_list_remove_link(ml->markers, l);
            marker_destroy(m);
            g_list_free(l);
            break;
        }
    }
}

void
marker_list_marker_position_set(struct marker_list *ml,
                                struct marker *m,
                                AFframecount frame_offset) {
    struct marker *mn = NULL, *mp = NULL;
    mn = marker_list_next(ml,
                          m->frame_offset + 1,
                          m->type);
    mp = marker_list_previous(ml,
                              m->frame_offset,
                              m->type);
    if(mn && frame_offset >= mn->frame_offset)
        frame_offset = mn->frame_offset - 1;
    if(mp && frame_offset <= mp->frame_offset)
        frame_offset = mp->frame_offset + 1;
    m->frame_offset = frame_offset;
}

void
marker_list_dump(struct marker_list *ml) {
    GList *l;
    for(l = ml->markers; l; l = l->next) 
        marker_dump((struct marker *)(l->data));
}

/* FIXME: slow */

struct marker *
marker_list_next(struct marker_list *ml,
                 AFframecount frame_offset,
                 enum marker_type type) {
    GList *l;
    struct marker *m;
    for(l = ml->markers; l; l = l->next) {
        m = (struct marker *)(l->data); 
        if(m->frame_offset >= frame_offset && 
           m->type == type &&
           !(m->flags & MARKER_IS_DISABLED)) 
            return m;
    }
    return NULL;
}

/* FIXME: slow */

struct marker *
marker_list_previous(struct marker_list *ml,
                     AFframecount frame_offset,
                     enum marker_type type) {
    GList *l;
    struct marker *m, *prev = NULL;
    for(l = ml->markers; l; l = l->next) {
        m = (struct marker *)(l->data); 
        if(m->frame_offset < frame_offset &&
           m->type == type &&
           (m->flags & MARKER_IS_DISABLED) == 0)
            prev = m;
        if(m->frame_offset >= frame_offset && 
           m->type == type) 
            break;
    }
    return prev;
}


struct marker_list *
marker_list_delete(struct marker_list *ml,
                   AFframecount frame_offset,
                   AFframecount frame_count,
                   enum marker_type type) {
    GList *l, *l2;
    struct marker *m;
    struct marker_list *ml_deleted = 
        marker_list_clone(ml, frame_offset, frame_count, type);
    if(!ml_deleted)
        return NULL;
#if USE_MCACHE
    ml->left_cache[MARKER_SLOPE] = ml->right_cache[MARKER_SLOPE] = 0;
    ml->left_cache[MARKER_SLOPE_AUX] = ml->right_cache[MARKER_SLOPE_AUX] = 0;
#endif
    for(l = ml->markers; l; ) {
        m = (struct marker *)(l->data);
        DEBUG("looking at marker:\n");
        marker_dump(m);
        if(m->frame_offset > frame_offset + frame_count)
            break;
        if(m->frame_offset >= frame_offset &&
           m->frame_offset <= frame_offset + frame_count &&
           (m->type & type)) {
            DEBUG("destroying marker:\n");
            marker_dump(m);
            marker_destroy(m);
            l2 = l->next;
            ml->markers = g_list_remove_link(ml->markers, l);
            g_list_free(l);
            l = l2;
            continue;
        }
        l = l->next;
    }
    return ml_deleted;
}

struct marker_list *
marker_list_delete_time(struct marker_list *ml,
                        AFframecount frame_offset,
                        AFframecount frame_count,
                        enum marker_type type) {
    GList *l;
    struct marker *m;
    struct marker_list *ml_deleted = 
        marker_list_delete(ml, frame_offset, frame_count, type);
    
    if(!ml_deleted)
        return NULL;

    for(l = ml->markers; l; l = l->next) {
        m = (struct marker *)(l->data);
        if(m->frame_offset < frame_offset)
            continue;
        if(m->type & type)
            m->frame_offset = MAX(0, m->frame_offset - frame_count);
    }

    return ml_deleted;
}

void
marker_list_insert_time(struct marker_list *ml,
                        AFframecount frame_offset,
                        AFframecount frame_count,
                        enum marker_type type) {
    GList *l;
    struct marker *m;

    for(l = ml->markers; l; l = l->next) {
        m = (struct marker *)(l->data);
        if(m->frame_offset < frame_offset)
            continue;
        if(m->type & type)
            m->frame_offset = m->frame_offset + frame_count;
    }
}

gint
marker_list_do_insert(gconstpointer a,
                      gconstpointer b) {
    struct marker *m1, *m2;
    m1 = (struct marker *)a;
    m2 = (struct marker *)b;
    if(m1->frame_offset > m2->frame_offset)
        return 1;
    return 0;
}

void
marker_list_insert_list(struct marker_list *ml,
                        struct marker_list *ml_source,
                        AFframecount frame_offset,
                        AFframecount frame_count,
                        enum marker_type type) {
    struct marker *m;
    GList *l;
    for(l = ml_source->markers; l; l = l->next) {
        m = (struct marker *)l->data;
        if(frame_offset + m->frame_offset > frame_offset + frame_count)
            break;
        if(m->type & type) {
            DEBUG("inserting marker:\n");
            marker_dump(m);
            marker_list_insert(ml,
                               frame_offset + m->frame_offset,
                               m->type,
                               m->multiplier,
                               m->label);
        }
    }
}

/* 
 * @return the marker that was inserted or NULL. 
 */

struct marker *
marker_list_insert(struct marker_list *ml,
                   AFframecount frame_offset,
                   enum marker_type type,
                   float multiplier,
                   const char *label) {
    struct marker *m, *m2;

#if USE_MCACHE
    ml->left_cache[type] = ml->right_cache[type] = 0;
#endif

    m2 = marker_list_next(ml, frame_offset, type);
    if(m2 && m2->frame_offset == frame_offset)
        return NULL;

    m = marker_new(frame_offset, type, multiplier, label);

    if(!m) 
        return NULL;

    ml->markers = g_list_insert_sorted(ml->markers, 
                                       m, 
                                       marker_list_do_insert);
    return m;
}

struct marker_list *
marker_list_new() {
    struct marker_list *ml = mem_calloc(1, sizeof(struct marker_list));
    if(!ml)
        return NULL;
    ml->marker_types_enabled = MARKER_SLOPE | MARKER_SLOPE_AUX;
    ml->markers = NULL;
    return ml;
}

struct marker_list *
marker_list_clone(struct marker_list *ml,
                  AFframecount frame_offset,
                  AFframecount frame_count,
                  enum marker_type type) {
    int fail = 0;
    GList *l, *l_clone = NULL;
    struct marker_list *ml_clone;
    struct marker *m_clone;
    
    ml_clone = marker_list_new();

    if(!ml_clone)
        return NULL;
    
    for(l = ml->markers; l; l = l->next) {
        if(!(type & ((struct marker *)l->data)->type))
            continue;
        if(frame_offset > ((struct marker *)l->data)->frame_offset)
            continue;
        if(frame_offset + frame_count < ((struct marker *)l->data)->frame_offset)
            break;
        m_clone = marker_clone((struct marker *)l->data);
        if(!m_clone) {
            fail = 1;
            break;
        }
        m_clone->frame_offset -= frame_offset;
        l_clone = g_list_append(l_clone, m_clone);
    }
    ml_clone->markers = l_clone;

    if(fail) {
        marker_list_destroy(ml_clone);
        return NULL;
    }
    return ml_clone;
}

void
marker_list_do_destroy(struct marker *m,
                       gpointer user_data) {
    marker_destroy(m);
}

void
marker_list_destroy(struct marker_list *ml) {
    if(!ml) {
        FAIL("ml == NULL\n");
        return;
    }
    if(g_list_length(ml->markers))
        DEBUG("destroying %d markers\n", g_list_length(ml->markers));
    //    marker_list_dump(ml);
    g_list_foreach(ml->markers, (GFunc)marker_list_do_destroy, NULL);
    g_list_free(ml->markers);
    free(ml);
}
