/*
 * Copyright (C) 2002,2003 Pascal Haakmat.
 */

#include <gtk/gtk.h>
#include <glade/glade.h>
#include "../modutils.h"

module modinfo;

typedef struct {
    module_id id;
    shell *shl;
    AFframecount duration;
    int times;
    double decay;
    GtkSpinButton *duration_control;
    GtkSpinButton *times_control;
    GtkRange *decay_control;
    GtkWidget *dialog;
} delay_params;

#ifdef HAVE_GNOME2
#define DELAY_GLADE_FILE "delay-2.glade"
#else
#define DELAY_GLADE_FILE "delay.glade"
#endif

void
delay(shell *shl,
      int track,
      AFframecount start_offset,
      AFframecount end_offset,
      int duration,
      int times,
      double decay) {
    int32_t *delay_buffer = mem_calloc(1, duration * 2 * sizeof(int32_t));
    int i, j = 0, loops = 1;
    double duration_adjust, decay_adjust;
    ITERATOR_INIT(start_offset, end_offset - start_offset);

    if(!delay_buffer) {
        FAIL("not enough memory for delay buffer (%d bytes)\n",
             duration * 2 * sizeof(int32_t));
        ITERATOR_EXIT();
        return;
    }

    ITERATOR(shl, shl->sr->tracks[track], 
             for(i = 0; i < iter_read; i++) {
                 decay_adjust =
                     marker_list_slope_value(shl->sr->markers[track],
                                             iter_frame_offset + i,
                                             MARKER_SLOPE_AUX); 
                 
                 int32_frame_bits[i] = 
                     int32_frame_bits[i] + delay_buffer[j] * 
                     (decay + (decay * decay_adjust));
                 delay_buffer[j] = int32_frame_bits[i];
                 j++;
                 duration_adjust = 
                     marker_list_slope_value(shl->sr->markers[track],
                                             iter_frame_offset + i,
                                             MARKER_SLOPE); 
                 
                 if(j >= duration + (duration * duration_adjust)) {
                     loops++;
                     if(loops > times)
                         ITERATOR_ESCAPE();
                     j = 0;
                 }
             }
             track_int32_frames_replace(shl->sr->tracks[track],
                                        int32_frame_bits,
                                        iter_frame_offset,
                                        iter_read));
    
    free(delay_buffer);
    ITERATOR_EXIT();
    return;
}

module *
module_new() {
    MODULE_INIT(&modinfo, 
		"Delay",
		"Pascal Haakmat",
		"Copyright (C) 2002,2003");
    return &modinfo;
}

void
on_apply_clicked(GtkWidget *w,
                 gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)), 
                          "user_data");
    delay_params *p = (delay_params *)module_state->data;
    p->duration = gtk_spin_button_get_value(p->duration_control) *
        p->shl->grid.gap;
    p->times = gtk_spin_button_get_value_as_int(p->times_control);
    p->decay = gtk_range_get_adjustment(p->decay_control)->value / 100;
    action_do(ACTION_MODULE_EXECUTE_NEW(WITH_UNDO, 
                                        p->shl,
                                        p->id));
}

void
on_ok_clicked(GtkWidget *w,
              gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    delay_params *p = (delay_params *)module_state->data;
    on_apply_clicked(w, NULL);
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_close_clicked(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    delay_params *p = (delay_params *)module_state->data;
    gtk_object_destroy(GTK_OBJECT(p->dialog));
}

void
on_dialog_destroy(GtkWidget *w,
                  gpointer user_data) {
    struct _module_state *module_state = 
        g_object_get_data(G_OBJECT(gtk_widget_get_toplevel(w)),
                          "user_data");
    module_state->is_open = 0;
    free(module_state->data);
}

void
module_open(module_id id,
            shell *shl, 
            int undo) {
    GladeXML *xml;
    GtkWidget *w;
    char path[4096];
    delay_params *p = mem_calloc(1, sizeof(delay_params));
    
    if(!p) {
        gui_alert("Not enough memory.");
        return;
    }

    snprintf(path, 4096, "%s/%s", dirname(modules[id].fname), DELAY_GLADE_FILE);
    xml = glade_xml_new(path, "dialog", NULL);
    
    if(!xml) {
        gui_alert("Delay: could not load interface %s.", path);
        free(p);
        return;
    }

    shl->module_state[id].is_open = 1;
    shl->module_state[id].data = p;
    p->id = id;
    p->shl = shl;
    p->duration_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "duration"));
    p->times_control = GTK_SPIN_BUTTON(glade_xml_get_widget(xml, "times"));
    p->decay_control = GTK_RANGE(glade_xml_get_widget(xml, "decay"));
    p->dialog = GTK_WIDGET(glade_xml_get_widget(xml, "dialog"));
    g_signal_connect(GTK_OBJECT(p->dialog), "destroy", 
                     G_CALLBACK(on_dialog_destroy), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "ok"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_ok_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "close"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_close_clicked), shl);
    w = GTK_WIDGET(glade_xml_get_widget(xml, "apply"));
    g_signal_connect(GTK_OBJECT(w), "clicked", 
                     G_CALLBACK(on_apply_clicked), shl);

    g_object_set_data(G_OBJECT(p->dialog), "user_data",
                      GINT_TO_POINTER(&shl->module_state[id]));
    g_object_unref(G_OBJECT(xml));
}

void
module_close(mod_state *module_state) {
    gtk_object_destroy(GTK_OBJECT(((delay_params *)module_state->data)->dialog));
}

action_group *
module_execute(shell *shl, 
               int undo) {
    AFframecount start = shl->select_start,
        end = shl->select_end;
    int map = shl->select_channel_map;
    int t, id = shl->active_module_id, duration, times;
    double decay;
    delay_params *p = NULL;
    action_group *undo_ag = NULL;

    if(!shl->module_state[id].is_open) { 
        gui_alert("Delay does not have defaults.");
        return NULL;
    }

    p = shl->module_state[id].data;
    DEBUG("delay: duration: %ld, times: %d, decay: %f\n",
          p->duration, p->times, p->decay);
    
    duration = p->duration;
    times = p->times;
    decay = p->decay;

    rwlock_rlock(&shl->sr->rwl);

    if(undo) 
        undo_ag = action_group_undo_create(shl,
                                           map, 
                                           start,
                                           end - start,
                                           start,
                                           end - start);

    for(t = 0; t < snd_track_count(shl->sr); t++) 
        if((1 << t) & map && t < shl->sr->channels) 
            delay(shl,
                  t,
                  start,
                  end,
                  duration,
                  times,
                  decay);
    rwlock_runlock(&shl->sr->rwl);

    return undo_ag;
}
