/*
 * Controller module.
 * Refer to controller.h about details.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <signal.h>

#include <gnome.h>
#include "grotype.h"
#include "controller.h"
#include "gromove.h"
#include "inviswin.h"
#include "visregion.h"
#include "guimisc.h"


/* Interval time(milli-second) is defined as
   (controller->interval_base * interval_delays[controller->delay_i]).
   interval_base can be changed by external modules.
   interval_delays will become larger while nothing happens.*/
#define INTERVAL_BASE_DEFAULT	50
static gint interval_delays[] = {1, 2, 3, 0};


/* Data structure definitions */
/* Controller private data */
struct _GroControllerPrivate {
	guint interval_base;/* milli-second */
	gint timer_id;		/* timer id (signed for -1, which implies stopped) */
	guint delay_i;		/* index of interval_delays array */

	/* These two are handled like union, but it's not union.
	   Because when I use invis, actual_widget should be NULL. */
	InvisWin *invis;/* Invisible window, used for detect push on GroMove. */
	GtkWidget *actual_widget;
};


/* Private function declarations */
static void remove_all_gmoves(GroController *controller);
static void remove_gmove(GroController *controller, GroMove *gmove);
static gint timer_cb(gpointer data);
static int button_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data);


/**
 * gro_controller_new:
 * @window is a window to draw roach.
 * If @actual_widget is NULL, it implies the caller uses root window.
 * In that case, prepare an invisible window to get button press event.
 **/
GroController*
gro_controller_new(GdkWindow *window, GtkWidget *actual_widget)
{
	GroController *controller;
	GroControllerPrivate *privat;

	g_return_val_if_fail(window != NULL, NULL);
	
	controller = g_new(GroController, 1);
	controller->cur_theme = NULL;/* should be attached later */
	controller->gro_win = gro_window_new(window);
	controller->gmove_list = NULL;

	privat = g_new(GroControllerPrivate, 1);
	controller->privat = privat;
	privat->interval_base = INTERVAL_BASE_DEFAULT;
	privat->timer_id = -1;
	privat->delay_i = 0;

	if (actual_widget) {
		privat->invis = NULL;
		privat->actual_widget = actual_widget;
		gtk_widget_add_events(privat->actual_widget, GDK_BUTTON_PRESS_MASK);
		gtk_signal_connect(GTK_OBJECT(privat->actual_widget),
						   "button_press_event",
						   GTK_SIGNAL_FUNC(button_press_cb),
						   controller);
	} else {/* Default, typically root window */
		privat->actual_widget = NULL;
		privat->invis = invis_win_new(controller->gro_win);
		gtk_signal_connect(GTK_OBJECT(privat->invis->invis_w),
						   "button_press_event",
						   GTK_SIGNAL_FUNC(button_press_cb),
						   controller);
	}
	return controller;
}

/**
 * gro_controller_delete:
 * Delete the controller instance.
 **/
void
gro_controller_delete(GroController *controller)
{
	GroControllerPrivate *privat;

	g_return_if_fail(controller != NULL);

	remove_all_gmoves(controller);
	if (controller->cur_theme)
		gro_controller_detach_theme(controller);

	privat = controller->privat;

	if (privat->timer_id != -1) {
		g_warning("the caller should have stopped it.\n");
		gro_controller_stop(controller);
	}

	if (privat->actual_widget) {
		gtk_signal_disconnect_by_func(GTK_OBJECT(privat->actual_widget),
									  GTK_SIGNAL_FUNC(button_press_cb),
									  controller);
	} else {
		gtk_signal_disconnect_by_func(GTK_OBJECT(privat->invis->invis_w),
									  GTK_SIGNAL_FUNC(button_press_cb),
									  controller);
		invis_win_delete(privat->invis);
	}
	
	gro_window_delete(controller->gro_win);

	g_free(privat);
	g_free(controller);
}

/**
 * gro_controller_attach_theme:
 * Attach a theme to the controller. Controller owns the theme.
 **/
void
gro_controller_attach_theme(GroController *controller, GroTheme *theme)
{
	g_return_if_fail(controller != NULL);
	g_return_if_fail(theme != NULL);
	g_return_if_fail(controller->cur_theme == NULL);

	controller->cur_theme = theme;
	gro_theme_ref(controller->cur_theme);/* own it */
	if (theme->theme_init)
		theme->theme_init(controller);
}

/**
 * gro_controller_detach_theme:
 * Implicitly, this might destroy 'theme'.
 * Also remove all gmoves, because they might be dependent on this theme.
 **/
void
gro_controller_detach_theme(GroController *controller)
{
	GroTheme *theme;
	
	g_return_if_fail(controller != NULL);
	g_return_if_fail(controller->cur_theme != NULL);

	theme = controller->cur_theme;
	if (theme->theme_finalize)
		theme->theme_finalize(controller);

	gro_theme_unref(controller->cur_theme);/* leave it */
	controller->cur_theme = NULL;

	remove_all_gmoves(controller);
}

/**
 * gro_controller_run:
 * Timer should be enabled only via this function.
 **/
void
gro_controller_run(GroController *controller)
{
	GroControllerPrivate *privat;

	g_return_if_fail(controller != NULL);
	g_return_if_fail(controller->cur_theme != NULL);
	
	privat = controller->privat;
	g_return_if_fail(privat->timer_id == -1);/* the last timer remains */

	privat->timer_id = gtk_timeout_add(privat->interval_base * interval_delays[privat->delay_i], timer_cb, controller);
}

/**
 * gro_controller_stop:
 * Timer should be disabled only via this function.
 * Timer callback sometimes returns FALSE, and it disables the timer
 * implicitly.
 **/
void
gro_controller_stop(GroController *controller)
{
	GroControllerPrivate *privat;

	g_return_if_fail(controller != NULL);
	/* I accept this, because this is no harm.
	   g_return_if_fail(controller->cur_theme != NULL); */

	privat = controller->privat;
	if (privat->timer_id != -1) {
		gtk_timeout_remove(privat->timer_id);
		privat->timer_id = -1;
	} else {
		g_warning("No timer to stop\n");
	}
}

/**
 * gro_controller_is_running:
 **/
gboolean
gro_controller_is_running(const GroController *controller)
{
	return (controller->privat->timer_id != -1) ? TRUE : FALSE;
}

/**
 * gro_controller_change_interval:
 * It might be safe for the caller to stop timer before calling this.
 * In that case, the caller should call gro_controller_run() to rerun.
 **/
void
gro_controller_change_interval(GroController *controller, guint interval_base)
{
	GroControllerPrivate *privat;

	g_return_if_fail(controller != NULL);

	privat = controller->privat;
	privat->interval_base = interval_base;
}

/**
 * gro_controller_add_gmoves:
 * Before this calling, 'theme' should be attached to this controller.
 **/
void
gro_controller_add_gmoves(GroController *controller, guint num_gmoves)
{
	GroTheme *cur_theme;
	GroControllerPrivate *privat;
	int i, ns, nd;
	
	g_return_if_fail(controller != NULL);
	g_return_if_fail(controller->cur_theme != NULL);

	privat = controller->privat;
	cur_theme = controller->cur_theme;
	for (i = 0; i < num_gmoves; i++) {
		GtkObject *gmove;

		gmove = gro_move_new(cur_theme->num_gstat,
							 cur_theme->num_direct,
							 cur_theme->pix_width, cur_theme->pix_height);
		gtk_object_ref(gmove);
		for (ns = 0; ns < cur_theme->num_gstat; ns++) {
			for (nd = 0; nd < cur_theme->num_direct; nd++) {
				gro_move_set_frames(GRO_MOVE(gmove),
									cur_theme->frames[ns][nd], ns, nd, TRUE);
			}
		}
		if (cur_theme->move_init)
			cur_theme->move_init(controller, GRO_MOVE(gmove));
		controller->gmove_list = g_list_prepend(controller->gmove_list,
												GRO_MOVE(gmove));
	}
}


/* ---The followings are private functions--- */
/**
 * remove_all_gmoves:
 **/
static void
remove_all_gmoves(GroController *controller)
{
	GroTheme *cur_theme = controller->cur_theme;
	GList *node;
	
	for (node = controller->gmove_list; node; node = node->next) {
		GtkObject *gmove = node->data;
		/* This routine is called from gro_controller_delete.
		   Usually, theme has been removed. */
		if (cur_theme && cur_theme->move_finalize)
			cur_theme->move_finalize(controller, GRO_MOVE(gmove));
		gtk_object_unref(gmove);
	}
	g_list_free(controller->gmove_list);
	controller->gmove_list = NULL;
}

/**
 * remove_gmove:
 * Remove one GroMove from contoller.
 **/
static void
remove_gmove(GroController *controller, GroMove *gmove)
{
	GroTheme *cur_theme = controller->cur_theme;

	if (cur_theme && cur_theme->move_finalize)
		cur_theme->move_finalize(controller, gmove);
	gtk_object_unref(GTK_OBJECT(gmove));
	controller->gmove_list = g_list_remove(controller->gmove_list, gmove);
}

/**
 * timer_cb:
 * Do various tasks.
 * At first, remove dead GroMove instances.
 * Next move GroMove instances by calling gro_move_move(), internally emit "move" signal.
 * Finally, take care of invisible window map.
 * Timer interval can be changed, but the controller always has one timer,
 * this routine guarantees it.
 **/
static gint
timer_cb(gpointer data)
{
	GroController *controller = data;
	GroTheme *cur_theme = controller->cur_theme;
	GroControllerPrivate *privat = controller->privat;
	GList *node;
	GdkRegion *vis_region;
	gint num_show = 0;
	gboolean b_changed = FALSE;
	gboolean b_cont_timer = TRUE;/* Return value, TRUE implies not to stop timer */

	g_return_val_if_fail(cur_theme->move_compute, TRUE);

	if (controller->gmove_list == NULL) {
		if (privat->invis)
			invis_win_hide(privat->invis);
		b_cont_timer = FALSE;/* remove timer */
		/* There are no gmoves, send myself SIGHUP signal.
		   Especially for command-line interface,
		   see sighup_handler() in src/cmain.c */
		kill(getpid(), SIGHUP);
		goto DONE;
	}

	/* Remove dead gmove */
	node = controller->gmove_list;
	while (node) {
		GroMove *gmove = node->data;
		node = node->next;
		if (gmove->cur_gstat == GRO_STAT_DEAD) {
			remove_gmove(controller, gmove);
		}
	}

	/* Calculate a new position and move each GroMove */
	vis_region = visible_region_new(controller->gro_win->window);
	for (node = controller->gmove_list; node; node = node->next) {
		GroMove *gmove = node->data;
		GroMoveRet ret;
		GroVector ret_vec;

		num_show++;
		ret = (cur_theme->move_compute)(controller, gmove, vis_region, &ret_vec);
		if (ret == GRO_RET_MOVE) {
			b_changed = TRUE;
			gro_move_move(gmove, controller, &ret_vec);
		} else if (ret == GRO_RET_HIDDEN) {
			num_show--;
		}
	}
	visible_region_delete(vis_region);

	/* Map or unmap invisible window */
	if (privat->invis) {
		if (num_show == 0)
			invis_win_hide(privat->invis);
		else
			invis_win_show(privat->invis);
	}
	
	/* Calculate delay and rerun if necessary */
	if (b_changed == TRUE) {
		if (privat->delay_i != 0) {
			privat->delay_i = 0;
			privat->timer_id = -1;/* set it explicitly */
			b_cont_timer = FALSE;
			gro_controller_run(controller);
		}
	} else {
		if (interval_delays[privat->delay_i + 1] != 0) {
			privat->delay_i++;
			privat->timer_id = -1;/* set it explicitly */
			b_cont_timer = FALSE;
			gro_controller_run(controller);
		}
    }

 DONE:
	return b_cont_timer;
}

/**
 * button_press_cb:
 * If GroMove is pushed, call gro_move_push() for the instance.
 * It internally emits "push" signal, so usually calls "push" event handler,
 * such as default_move_push_cb().
 **/
static int
button_press_cb(GtkWidget *widget, GdkEventButton *event, gpointer data)
{
	GroController *controller = data;
	GList *node;

	/* ignore double click and triple click */
	if (event->type != GDK_BUTTON_PRESS)
		return TRUE;
	
	for (node = controller->gmove_list; node; node = node->next) {
		GroMove *gmove = node->data;
		if (gmove->cur_gstat != GRO_STAT_DEAD
			&& is_rect_point_in(&gmove->cur_rect, event->x, event->y) == TRUE) {
			gro_move_push(gmove, controller, event);
		}
	}

	return TRUE;
}
