/*
 * GroMove widget module
 * Refer to gromove.h about details.
 *
 * Copyright INOUE Seiichiro <inoue@ainet.or.jp>, licensed under the GPL.
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#include <math.h>
#include <gnome.h>
#include "grotype.h"
#include "gromove.h"
#include "theme.h"	/* for GroVector */
#include "gropixmap.h"


/* Data structure definitions */
typedef struct _GroStatFrames GroStatFrames;
typedef struct _GroDirectFrames GroDirectFrames;

/* As I wrote in gromove.h, GroMove has three dimensional states,
   i.e. [stat] x [direction] x [frame].
   The following three data structures, which are contained in array,
   are related to each dimension.
   The last [frame] is contained in GPtrArray. */
struct _GroMovePrivate {
	GroStatFrames *stat_frames;	/* array of GroStatFrames */
};

struct _GroStatFrames {
	GroDirectFrames *direct_frames;	/* array of GroDirectFrames */
};

struct _GroDirectFrames {
	gint cur_frame;	/* current frame number
					   (This is signed value. See gro_move_new() below) */
	/* Array of GroPixmap.
	   For efficiency, the array itself is managed by 'theme' module.
	   This 'frames' just points to it. */
	const GPtrArray *frames;	
};

/* Signals */
enum {
	MOVE,
	PUSH,
	LAST_SIGNAL
};

/* Private function declarations */
static void gro_move_class_init(GroMoveClass *klass);
static void gro_move_init(GroMove *gmove);
static void gro_move_finalize(GtkObject *object);
static void gro_move_real_move(GroMove *gmove, const GroController *controller, const GroVector *vec);

	
static guint move_signals[LAST_SIGNAL] = { 0 };

static GtkObjectClass *parent_class = NULL;

GtkType
gro_move_get_type(void)
{
	static GtkType move_type = 0;

	if (!move_type) {
		static const GtkTypeInfo move_info = {
			"GroMove",
			sizeof(GroMove),
			sizeof(GroMoveClass),
			(GtkClassInitFunc)gro_move_class_init,
			(GtkObjectInitFunc)gro_move_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc)NULL,
		};

		move_type = gtk_type_unique(GTK_TYPE_OBJECT, &move_info);
    }

	return move_type;
}

static void
gro_move_class_init(GroMoveClass *klass)
{
	GtkObjectClass *object_class;

	object_class = (GtkObjectClass*)klass;

	move_signals[MOVE] =
		gtk_signal_new("move",
					   GTK_RUN_LAST,
					   object_class->type,
					   GTK_SIGNAL_OFFSET(GroMoveClass, move),
					   gtk_marshal_NONE__POINTER_POINTER,
					   GTK_TYPE_NONE,
					   2, GTK_TYPE_POINTER, GTK_TYPE_POINTER);

	move_signals[PUSH] =
		gtk_signal_new("push",
					   GTK_RUN_LAST,
					   object_class->type,
					   GTK_SIGNAL_OFFSET(GroMoveClass, push),
					   gtk_marshal_NONE__POINTER_POINTER,
					   GTK_TYPE_NONE,
					   2, GTK_TYPE_POINTER, GTK_TYPE_GDK_EVENT);

	gtk_object_class_add_signals(object_class, move_signals, LAST_SIGNAL);

	parent_class = gtk_type_class(GTK_TYPE_OBJECT);

	object_class->finalize = gro_move_finalize;

	klass->move = gro_move_real_move;
	klass->push = NULL;
}

static void
gro_move_init(GroMove *gmove)
{
	gmove->num_gstat = 0;
	gmove->cur_gstat = GRO_STAT_WAKE;
	gmove->num_direct = 0;
	gmove->cur_direct = 0;

	gmove->cur_rect.x = 0;
	gmove->cur_rect.y = 0;
	gmove->cur_rect.width = 0;
	gmove->cur_rect.height = 0;

	gmove->privat = g_new(GroMovePrivate, 1);
	gmove->privat->stat_frames = NULL;
}

static void
gro_move_finalize(GtkObject *object)
{
	GroMove *gmove = GRO_MOVE(object);
	GroMovePrivate *privat = gmove->privat;
	GroStatFrames *stat_frames;
	GroDirectFrames *direct_frames;
	int ns, nd;

	stat_frames = privat->stat_frames;
	for (ns = 0; ns < gmove->num_gstat; ns++) {
		direct_frames = stat_frames[ns].direct_frames;
		for (nd = 0; nd < gmove->num_direct; nd++) {
			direct_frames[nd].frames = NULL;/* Not g_ptr_array_free() */
		}
		g_free(direct_frames);
	}
	g_free(stat_frames);

	g_free(privat);

	(*GTK_OBJECT_CLASS(parent_class)->finalize)(object);
}

/**
 * gro_move_new:
 * Input:
 * guint num_gstat; Number of stats
 * guint num_direct; Number of directions
 * guint pix_width; Width of pixmap
 * guint pix_height; Height of pixmap
 **/
GtkObject*
gro_move_new(guint num_gstat, guint num_direct, guint pix_width, guint pix_height)
{
	GroMove *gmove;
	GroMovePrivate *privat;
	GroStatFrames *stat_frames;
	GroDirectFrames *direct_frames;
	int ns, nd;

	gmove = gtk_type_new(GRO_TYPE_MOVE);
	gmove->num_gstat = num_gstat;
	gmove->num_direct = num_direct;
	gmove->cur_rect.width = pix_width;
	gmove->cur_rect.height = pix_height;
	gmove->move_env = NULL;
	
	privat = gmove->privat;
	privat->stat_frames = g_new(GroStatFrames, num_gstat);
	stat_frames = privat->stat_frames;
	for (ns = 0; ns < num_gstat; ns++) {
		stat_frames[ns].direct_frames = g_new(GroDirectFrames, num_direct);
		direct_frames = stat_frames[ns].direct_frames;
		for (nd = 0; nd < num_direct; nd++) {
			direct_frames[nd].cur_frame = -1;/* Init kludge */
			/* This will be initialized in gro_move_set_frames() */
			direct_frames[nd].frames = NULL;/* Not g_ptr_array_new() */
											   
		}
	}
	return GTK_OBJECT(gmove);
}


/** Interfaces **/
void
gro_move_set_frames(GroMove *gmove, const GPtrArray *frames, gint gstat, guint direct, gboolean b_static_frames)
{
	GroMovePrivate *privat;
	GroStatFrames *stat_frames;
	GroDirectFrames *direct_frames;

	g_return_if_fail(gmove != NULL);
	g_return_if_fail(GRO_IS_MOVE(gmove));
	
	/* Currently, non-static isn't supported */
	g_return_if_fail(b_static_frames == TRUE);

	g_return_if_fail(gmove->cur_gstat != GRO_STAT_DEAD);
	g_return_if_fail(gstat < gmove->num_gstat);
	g_return_if_fail(direct < gmove->num_direct);

	privat = gmove->privat;
	stat_frames = privat->stat_frames;
	direct_frames = stat_frames[gstat].direct_frames;

	direct_frames[direct].frames = frames;
}

void
gro_move_move(GroMove *gmove, const GroController *controller, const GroVector *vec)
{
	g_return_if_fail(gmove != NULL);
	g_return_if_fail(GRO_IS_MOVE(gmove));
	g_return_if_fail(gmove->cur_gstat != GRO_STAT_DEAD);
	
	gtk_signal_emit(GTK_OBJECT(gmove), move_signals[MOVE], controller, vec);
}

void
gro_move_push(GroMove *gmove, const GroController *controller, GdkEventButton *event)
{
	g_return_if_fail(gmove != NULL);
	g_return_if_fail(GRO_IS_MOVE(gmove));
	g_return_if_fail(gmove->cur_gstat != GRO_STAT_DEAD);
	
	gtk_signal_emit(GTK_OBJECT(gmove), move_signals[PUSH], controller, event);
}

void
gro_move_change_gstat(GroMove *gmove, gint gstat)
{
	g_return_if_fail(gmove != NULL);
	g_return_if_fail(GRO_IS_MOVE(gmove));
	g_return_if_fail(gstat < gmove->num_gstat || gstat == GRO_STAT_DEAD);

	gmove->cur_gstat = gstat;
}

void
gro_move_change_direct(GroMove *gmove, guint direct)
{
	g_return_if_fail(gmove != NULL);
	g_return_if_fail(GRO_IS_MOVE(gmove));
	g_return_if_fail(direct < gmove->num_direct);

	gmove->cur_direct = direct;
}

/**
 * gro_move_is_eof:
 * Return TRUE if end of frame.
 **/
gboolean
gro_move_is_eof(const GroMove *gmove)
{
	GroMovePrivate *privat;
	GroStatFrames *stat_frames;
	GroDirectFrames *direct_frames;

	g_return_val_if_fail(gmove != NULL, FALSE);
	g_return_val_if_fail(GRO_IS_MOVE(gmove), FALSE);

	privat = gmove->privat;
	stat_frames = privat->stat_frames;
	direct_frames = stat_frames[gmove->cur_gstat].direct_frames;

	return (direct_frames[gmove->cur_direct].cur_frame == (direct_frames[gmove->cur_direct].frames->len - 1));
}


/** Internal functions **/
/**
 * gro_move_real_move:
 **/
static void
gro_move_real_move(GroMove *gmove, const GroController *controller, const GroVector *vec)
{
	GdkRectangle *cur_rect = &gmove->cur_rect;
	GroMovePrivate *privat = gmove->privat;
	const GroWindow *gro_win = controller->gro_win;
	GroStatFrames *stat_frames = privat->stat_frames;
	GroDirectFrames *direct_frames = stat_frames[gmove->cur_gstat].direct_frames;
	GroPixmap *frame;

	/* Clear the old position */
	gdk_window_clear_area(gro_win->window, cur_rect->x, cur_rect->y,
						  cur_rect->width, cur_rect->height);

	/* Update the internal data */
	cur_rect->x += vec->ix;
	cur_rect->y += vec->iy;
	direct_frames[gmove->cur_direct].cur_frame++;/* advance a frame */
	if (direct_frames[gmove->cur_direct].cur_frame >= direct_frames[gmove->cur_direct].frames->len) {
		direct_frames[gmove->cur_direct].cur_frame = 0;
	}

	/* Draw the pixmap */
	frame = direct_frames[gmove->cur_direct].frames->pdata[direct_frames[gmove->cur_direct].cur_frame];
	if (frame->pixmap) {
		gdk_gc_set_clip_mask(gro_win->gc, frame->mask);
		gdk_gc_set_clip_origin(gro_win->gc, cur_rect->x, cur_rect->y);
		gdk_draw_pixmap(gro_win->window, gro_win->gc, frame->pixmap,
						0, 0, cur_rect->x, cur_rect->y, cur_rect->width, cur_rect->height);
		gdk_gc_set_clip_mask(gro_win->gc, NULL);
		gdk_gc_set_clip_origin(gro_win->gc, 0, 0);
	} else {/* NULL is legal, which implies just to clear. */
		;/* do nothing */
	}
}
