/* A toolbox window
 *
 * gsumi version 0.5
 *
 * Copyright 1997 Owen Taylor <owt1@cornell.edu>
*/

#include <stdio.h>
#include "gsumi.h"

typedef struct _ToolItem ToolItem;
typedef struct _ToolBox ToolBox;

struct _ToolItem {
  ToolInfo *tool_info;

  gchar *name;
  GtkWidget *vbox;
  GtkWidget *frame;
  GtkWidget *option_menu;
  GtkWidget *properties;
};

struct _ToolBox {
  GList *items;
  ToolItem *current;
  
  GtkWidget *shell;
  GtkWidget *vbox;

  ToolItem *current_prop;
  GtkWidget *prop_label;
  GtkAdjustment *size_data;
  GtkAdjustment *sens_data;
  GtkWidget *brush_widget;
};

static ToolBox *toolBox = NULL;

/* forward declarations */
static void tool_box_set_current_prop(ToolItem *ti);
static ToolItem *tool_box_find_device(guint32 devid);

static int brush_widget_radius;
static int brush_widget_angle;
static int brush_widget_state = FALSE;

static void
tool_item_set_tool(GtkWidget *w,
		   gpointer data)
{
  ToolItem *ti = (ToolItem *)gtk_object_get_user_data(GTK_OBJECT(w));
 
  ti->tool_info->tool = (ToolType)data;
}

static void
tool_item_draw_brush (GtkWidget *w, ToolInfo *tool_info,
		      double xc, double yc, double radius)
{
  int i;
  Blob *b;

  b = blob_ellipse(xc,yc,
		   radius*cos(tool_info->angle),
		   radius*sin(tool_info->angle),
		   -(radius/tool_info->aspect)*sin(tool_info->angle),
		   (radius/tool_info->aspect)*cos(tool_info->angle));

  for (i=0;i<b->height;i++)
    {
      if (b->data[i].left <= b->data[i].right)
	gdk_draw_line (w->window,
		       w->style->fg_gc[w->state],
		       b->data[i].left,i+b->y,
		       b->data[i].right+1,i+b->y);
    }
  
  free(b);
}

static void
tool_item_properties_callback(GtkWidget *widget,
			      ToolItem *ti)
{
  tool_box_set_current_prop(ti);
}

static int
tool_item_properties_expose (GtkWidget      *widget,
			     GdkEventExpose *event,
			     ToolItem *ti)
{
  GdkPoint sens_points[4];
  gint circle_y;
  gdouble size;

  gdk_window_clear_area(widget->window,
			0,0,widget->allocation.width,
			widget->allocation.height);
  
  sens_points[0].x = 0;
  sens_points[0].y = 0;
  sens_points[1].x = 0;
  sens_points[1].y = 4*(1-ti->tool_info->sens);
  sens_points[2].x = 40;
  sens_points[2].y = 4*(1+ti->tool_info->sens);
  sens_points[3].x = 40;
  sens_points[3].y = 0;

  gdk_draw_polygon (widget->window,
		    widget->style->fg_gc[widget->state],
		    TRUE, sens_points, 4);

  size = (ti->tool_info->size/2);
  if (size/ti->tool_info->aspect < 1.0)
    size = 1.0*ti->tool_info->aspect;

  if (size < 16)
    circle_y = 24;
  else
    circle_y = 8+size;

  /*  gdk_draw_arc (widget->window,
		widget->style->fg_gc[widget->state],
		TRUE,
		20-size/2, circle_y-size/2, size, size,
		0,360*64);*/

  tool_item_draw_brush (widget, ti->tool_info, 20, circle_y, size);
  
  return FALSE;
}

static ToolItem *
tool_item_create(ToolInfo *tool_info)
{
  GtkWidget *label;
  GtkWidget *menu;
  GtkWidget *menu_item;
  GdkPixmap *pm;
  GtkWidget *pixmap;
  GtkWidget *inner_box;
  GtkWidget *prop_button;

  ToolItem *ti = g_new(ToolItem, 1);

  ti->tool_info = tool_info;

  ti->vbox = gtk_vbox_new(FALSE,0);
  gtk_widget_ref (ti->vbox);

  /* create the label */

  ti->name = tool_info->name;

  label = gtk_label_new(ti->name);
  gtk_box_pack_start (GTK_BOX (ti->vbox), label, FALSE, FALSE, 0);
  gtk_widget_show(label);

  /* frame and the menu */

  ti->frame = gtk_frame_new (NULL);
  gtk_frame_set_shadow_type (GTK_FRAME(ti->frame),GTK_SHADOW_NONE);
  gtk_box_pack_start (GTK_BOX (ti->vbox), ti->frame, FALSE, FALSE, 0);
  gtk_widget_show (ti->frame);

  inner_box = gtk_hbox_new (FALSE,2);
  gtk_container_border_width (GTK_CONTAINER(inner_box),3);
  gtk_container_add (GTK_CONTAINER(ti->frame), inner_box);
  gtk_widget_show (inner_box);
  
  menu = gtk_menu_new();
  gtk_widget_realize (menu);
  
  menu_item = gtk_menu_item_new();
  gtk_object_set_user_data(GTK_OBJECT(menu_item),ti);

  pm = create_icon_pixmap (menu->window, PEN_TOOL, gray_pixels);
  if (!pm)
    g_error("Could not create pen pixmap");
  pixmap = gtk_pixmap_new(pm,NULL);
  gtk_container_add (GTK_CONTAINER (menu_item), pixmap);
  gtk_widget_show(pixmap);

  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
		      (GtkSignalFunc) tool_item_set_tool,
		      (gpointer)PEN_TOOL);
  gtk_menu_append(GTK_MENU(menu),menu_item);
  gtk_widget_show (menu_item);

  menu_item = gtk_menu_item_new();
  gtk_object_set_user_data(GTK_OBJECT(menu_item),ti);

  pm = create_icon_pixmap (menu->window, ERASER_TOOL, gray_pixels);
  if (!pm)
    g_error("Could not create eraser pixmap");

  pixmap = gtk_pixmap_new(pm,NULL);
  gtk_widget_show(pixmap);

  gtk_container_add (GTK_CONTAINER (menu_item), pixmap);
  
  gtk_signal_connect (GTK_OBJECT (menu_item), "activate",
 		      (GtkSignalFunc) tool_item_set_tool,
		      (gpointer)ERASER_TOOL);
  gtk_menu_append(GTK_MENU(menu),menu_item);
  gtk_widget_show (menu_item);

  
  ti->option_menu = gtk_option_menu_new ();
  gtk_box_pack_start(GTK_BOX(inner_box), ti->option_menu, FALSE,FALSE,0);
  gtk_widget_show (ti->option_menu);

  gtk_option_menu_set_menu (GTK_OPTION_MENU (ti->option_menu), menu);
  gtk_option_menu_set_history (GTK_OPTION_MENU (ti->option_menu), 0); 

  prop_button = gtk_button_new();
  gtk_signal_connect (GTK_OBJECT (prop_button), "clicked",
		      (GtkSignalFunc) tool_item_properties_callback,
		      ti);
  gtk_box_pack_start(GTK_BOX(inner_box), prop_button, FALSE, FALSE, 0);
  gtk_widget_show (prop_button);
  
  ti->properties = gtk_drawing_area_new();
  gtk_drawing_area_size( GTK_DRAWING_AREA(ti->properties), 40, 40);
  gtk_widget_set_events (GTK_WIDGET (ti->properties),
			 GDK_EXPOSURE_MASK);
  gtk_signal_connect (GTK_OBJECT (ti->properties), "expose_event",
		      (GtkSignalFunc) tool_item_properties_expose,
		      ti);
  gtk_container_add(GTK_CONTAINER(prop_button),ti->properties);
  gtk_widget_show (ti->properties);

  gtk_widget_show(ti->vbox);
  
  return ti;
}

static void
tool_item_destroy(ToolItem *ti)
{
  gtk_widget_destroy(ti->vbox);
  gtk_widget_unref (ti->vbox);

  g_free(ti);
}

static void
brush_widget_active_rect (GtkWidget *w, GdkRectangle *rect)
{
  int x,y;
  int r;

  r = MIN(w->allocation.width,w->allocation.height)/2;

  x = w->allocation.width/2 + 0.85*r*toolBox->current_prop->tool_info->aspect/10.
    *cos(toolBox->current_prop->tool_info->angle);
  y = w->allocation.height/2 + 0.85*r*toolBox->current_prop->tool_info->aspect/10.
    *sin(toolBox->current_prop->tool_info->angle);

  rect->x = x-5;
  rect->y = y-5;
  rect->width = 10;
  rect->height = 10;
}

static void
brush_widget_expose (GtkWidget *w, GdkEventExpose *event)
{
  GdkRectangle rect;
  int r0;
  
  r0 = MIN(w->allocation.width,w->allocation.height)/2;

  if (r0 < 2)
    return;

  gdk_window_clear_area(w->window,
			0,0,w->allocation.width,
			w->allocation.height);
  tool_item_draw_brush (w, toolBox->current_prop->tool_info,
			w->allocation.width/2, w->allocation.height/2,
			0.9*r0);

  brush_widget_active_rect (w, &rect);
  gdk_draw_rectangle (w->window, w->style->bg_gc[GTK_STATE_NORMAL],
		      TRUE,	/* filled */
		      rect.x, rect.y, 
		      rect.width, rect.height);
  gtk_draw_shadow (w->style, w->window, w->state, GTK_SHADOW_OUT,
		   rect.x, rect.y,
		   rect.width, rect.height);
}

static void
brush_widget_button_press (GtkWidget *w, GdkEventButton *event)
{
  GdkRectangle rect;

  brush_widget_active_rect (w, &rect);
  
  if ((event->x >= rect.x) && (event->x-rect.x < rect.width) &&
      (event->y >= rect.y) && (event->y-rect.y < rect.height))
    {
      brush_widget_state = TRUE;
      brush_widget_radius = toolBox->current_prop->tool_info->aspect;
      brush_widget_angle = toolBox->current_prop->tool_info->angle;
    }
}

static void
brush_widget_button_release (GtkWidget *w, GdkEventButton *event)
{
  brush_widget_state = FALSE;
}

static void
brush_widget_motion_notify (GtkWidget *w, GdkEventMotion *event)
{
  int x;
  int y;
  int r0;
  int rsquare;

  if (brush_widget_state)
    {
      x = event->x - w->allocation.width/2;
      y = event->y - w->allocation.height/2;
      rsquare = x*x + y*y;

      if (rsquare != 0)
	{
	  toolBox->current_prop->tool_info->angle = atan2(y,x);

	  r0 = MIN(w->allocation.width,w->allocation.height)/2;
	  toolBox->current_prop->tool_info->aspect =
	    10. * sqrt((double)rsquare/(r0*r0)) / 0.85;

	  if (toolBox->current_prop->tool_info->aspect < 1.0)
	    toolBox->current_prop->tool_info->aspect = 1.0;
	  if (toolBox->current_prop->tool_info->aspect > 10.0)
	    toolBox->current_prop->tool_info->aspect = 10.0;

	  gtk_widget_draw(toolBox->current_prop->properties, NULL);
	  gtk_widget_draw(w, NULL);
	}
    }
}

static void
size_scale_update (GtkAdjustment *adjustment,
		   gpointer data)
{
  toolBox->current_prop->tool_info->size = adjustment->value;
  gtk_widget_draw(toolBox->current_prop->properties,NULL);
}

static void
sens_scale_update (GtkAdjustment *adjustment,
		   gpointer data)
{
  toolBox->current_prop->tool_info->sens = adjustment->value;
  gtk_widget_draw(toolBox->current_prop->properties,NULL);
}

#define BUFSIZE 64
static void 
tool_box_set_current_prop(ToolItem *ti)
{
  char buf[BUFSIZE] = "Properties - ";
  if (ti != toolBox->current_prop)
    {
      toolBox->current_prop = ti;
      
      strncat(buf,ti->name,BUFSIZE-strlen(buf)-1);
      buf[BUFSIZE-1] = '\0';
      gtk_label_set(GTK_LABEL(toolBox->prop_label),buf);

      toolBox->size_data->value = ti->tool_info->size;
      gtk_signal_emit_by_name (GTK_OBJECT (toolBox->size_data), "changed");

      toolBox->sens_data->value = ti->tool_info->sens;
      gtk_signal_emit_by_name (GTK_OBJECT (toolBox->sens_data), "changed");

      gtk_widget_draw (toolBox->brush_widget, NULL);

      options_dialog_set_current_device (ti->tool_info->id);
    }
}

static void
tool_box_more_options_cb (void)
{
  options_dialog_set_current_device (toolBox->current_prop->tool_info->id);
  options_dialog_create ();
}

void
tool_box_create()
{
  GtkWidget *button;
  GtkWidget *main_vbox;
  GtkWidget *separator;
  GtkWidget *hbox;
  GtkWidget *slider;
  GtkWidget *aspect_frame;
  GtkWidget *label;
  
  if (!toolBox) 
    {
      toolBox = g_new (ToolBox,1);
      toolBox->items = NULL;
      toolBox->current = NULL;
      toolBox->current_prop = NULL;
      
      /* shell and list vbox */

      toolBox->shell = gtk_window_new(GDK_WINDOW_TOPLEVEL);
      gtk_window_set_title (GTK_WINDOW (toolBox->shell), "Tools");
      gtk_window_set_policy (GTK_WINDOW (toolBox->shell), FALSE, FALSE, TRUE);

      gtk_signal_connect (GTK_OBJECT(toolBox->shell), "destroy",
			  (GtkSignalFunc)  tool_box_destroy,
			  NULL);

      main_vbox = gtk_vbox_new (FALSE, 0);
      gtk_container_border_width(GTK_CONTAINER (main_vbox), 10);
      gtk_container_add (GTK_CONTAINER (toolBox->shell), main_vbox);

      toolBox->vbox = gtk_vbox_new(FALSE, 5);
      gtk_box_pack_start(GTK_BOX(main_vbox),toolBox->vbox,FALSE,FALSE,0);

      gtk_widget_show (toolBox->vbox);

      /* properties section */

      separator = gtk_hseparator_new();
      gtk_box_pack_start(GTK_BOX(main_vbox),separator,TRUE,FALSE,5);
      gtk_widget_show (separator);

      /* label */
      toolBox->prop_label = gtk_label_new("Properties");
      gtk_box_pack_start (GTK_BOX(main_vbox), toolBox->prop_label,
			  FALSE, FALSE, 0);
      gtk_widget_show (toolBox->prop_label);

      /* size slider */
      hbox = gtk_hbox_new (FALSE, 2);
      gtk_box_pack_start (GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
      label = gtk_label_new ("Size:");
      gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 2);

      toolBox->size_data = 
	GTK_ADJUSTMENT (gtk_adjustment_new (8.0, 0.0, 100.0, 1.0, 10.0, 0.0));
      slider = gtk_hscale_new (toolBox->size_data);
      gtk_box_pack_start (GTK_BOX (hbox), slider, TRUE, TRUE, 0);
      gtk_scale_set_value_pos (GTK_SCALE (slider), GTK_POS_TOP);

      gtk_range_set_update_policy (GTK_RANGE (slider), GTK_UPDATE_DELAYED);
      gtk_signal_connect (GTK_OBJECT (toolBox->size_data), "value_changed",
		      (GtkSignalFunc) size_scale_update, NULL);

      gtk_widget_show (label);
      gtk_widget_show (slider);
      gtk_widget_show (hbox);

      /* sens slider */
      hbox = gtk_hbox_new (FALSE, 2);
      gtk_box_pack_start (GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);
      label = gtk_label_new ("Sensitivity:");
      gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 2);

      toolBox->sens_data = 
	GTK_ADJUSTMENT (gtk_adjustment_new (1.0, 0.0, 1.0, 0.01, 0.1, 0.0));
      slider = gtk_hscale_new (toolBox->sens_data);
      gtk_box_pack_start (GTK_BOX (hbox), slider, TRUE, TRUE, 0);
      gtk_scale_set_value_pos (GTK_SCALE (slider), GTK_POS_TOP);

      gtk_range_set_update_policy (GTK_RANGE (slider), GTK_UPDATE_DELAYED);
      gtk_signal_connect (GTK_OBJECT (toolBox->sens_data), "value_changed",
		      (GtkSignalFunc) sens_scale_update, NULL);

      gtk_widget_show (label);
      gtk_widget_show (slider);
      gtk_widget_show (hbox);

      /* Brush shape widget */

      hbox = gtk_hbox_new (FALSE, 2);
      gtk_box_pack_start (GTK_BOX(main_vbox), hbox, FALSE, FALSE, 5);
      label = gtk_label_new ("Shape:");
      gtk_box_pack_start (GTK_BOX (hbox), label, FALSE, FALSE, 0);

      aspect_frame = gtk_aspect_frame_new (NULL, 0.5, 0.5, 1.0, FALSE);
      gtk_frame_set_shadow_type (GTK_FRAME(aspect_frame), GTK_SHADOW_IN);
      gtk_box_pack_start (GTK_BOX (hbox), aspect_frame, TRUE, TRUE, 2);
      gtk_widget_show (aspect_frame);

      toolBox->brush_widget = gtk_drawing_area_new();
      gtk_drawing_area_size (GTK_DRAWING_AREA(toolBox->brush_widget),60,60);
      gtk_container_add (GTK_CONTAINER(aspect_frame), toolBox->brush_widget);
      
      gtk_widget_set_events (toolBox->brush_widget, 
			     GDK_BUTTON_PRESS_MASK | GDK_BUTTON_RELEASE_MASK
			     | GDK_POINTER_MOTION_MASK | GDK_EXPOSURE_MASK);
      gtk_signal_connect (GTK_OBJECT (toolBox->brush_widget), "button_press_event",
			  GTK_SIGNAL_FUNC (brush_widget_button_press), NULL);
      gtk_signal_connect (GTK_OBJECT (toolBox->brush_widget), "button_release_event",
			  GTK_SIGNAL_FUNC (brush_widget_button_release), NULL);
      gtk_signal_connect (GTK_OBJECT (toolBox->brush_widget), "motion_notify_event",
			  GTK_SIGNAL_FUNC (brush_widget_motion_notify), NULL);
      gtk_signal_connect (GTK_OBJECT (toolBox->brush_widget), "expose_event",
			  GTK_SIGNAL_FUNC (brush_widget_expose), NULL);

      gtk_widget_realize (toolBox->brush_widget);
      gtk_style_set_background (toolBox->brush_widget->style, 
				toolBox->brush_widget->window, 
				GTK_STATE_ACTIVE);      

      gtk_widget_show (label);
      gtk_widget_show (toolBox->brush_widget);
      gtk_widget_show (hbox);

      button = gtk_button_new_with_label ("More Options..");
      gtk_box_pack_start (GTK_BOX(main_vbox), button, FALSE, FALSE, 5);
      gtk_widget_show (button);

      gtk_signal_connect (GTK_OBJECT (button), "clicked",
			  GTK_SIGNAL_FUNC (tool_box_more_options_cb),
			  NULL);
      
      gtk_widget_show(main_vbox);
    }
}

void
tool_box_show()
{
  if (!GTK_WIDGET_VISIBLE (toolBox->shell))
    gtk_widget_show (toolBox->shell);
}

void
tool_box_destroy()
{
  GList *tmp_list;
  static int in_destroy = 0;

  if (toolBox && !in_destroy)
    {
      in_destroy = TRUE;
      gtk_widget_destroy (toolBox->shell);
      in_destroy = FALSE;
      
      tmp_list = toolBox->items;
      for (;tmp_list;tmp_list=tmp_list->next) 
	{
	  tool_item_destroy((ToolItem *)tmp_list->data);
	}
      g_list_free (toolBox->items);
      
      g_free(toolBox);
      toolBox = NULL;
    }
}

static ToolItem *
tool_box_find_device(guint32 devid)
{
  GList *tmp_list = toolBox->items;

  for (;tmp_list;tmp_list=tmp_list->next) {
    ToolItem *ti = (ToolItem*)tmp_list->data;
    if (ti->tool_info->id == devid)
      {
	return ti;
      }
  }
  return 0;
}

void
tool_box_add_device(ToolInfo *tool_info)
{
  ToolItem *ti;

  if (!toolBox) return;

  ti = tool_box_find_device(tool_info->id);
  
  if (!ti)
    {
      ti = tool_item_create(tool_info);
      
      gtk_box_pack_start (GTK_BOX (toolBox->vbox), ti->vbox, FALSE, FALSE, 0);
      
      toolBox->items =  g_list_append(toolBox->items, ti);
      
      if (!toolBox->current_prop)
	tool_box_set_current_prop(ti);
    }
}

void
tool_box_remove_device(guint32 devid)
{
  ToolItem *ti;

  if (!toolBox) return;
  
  ti = tool_box_find_device(devid);

  if (ti)
    {
      if (toolBox->current == ti)
	toolBox->current = NULL;

      toolBox->items = g_list_remove(toolBox->items, ti);

      g_return_if_fail (toolBox->items != NULL);
      
      if (toolBox->current_prop == ti)
	{
	  /* just pick the first one in the list */
	  tool_box_set_current_prop((ToolItem *)toolBox->items->data);
	}

      tool_item_destroy(ti);
    }
}

void
tool_box_set_current_device(guint32 devid)
{
  ToolItem *ti;

  if (!toolBox) return;
  
  ti = tool_box_find_device(devid);
  if (ti)
    {
      if (toolBox->current)
	gtk_frame_set_shadow_type(GTK_FRAME(toolBox->current->frame),
				  GTK_SHADOW_NONE);

      gtk_frame_set_shadow_type(GTK_FRAME(ti->frame),GTK_SHADOW_IN);
      toolBox->current = ti;
    }
}

void
tool_box_toggle_current_tool()
{
  ToolItem *ti;

  if (!toolBox) return;

  ti = toolBox->current;
  if (ti)
    {
      if (ti->tool_info->tool == PEN_TOOL)
	{
	  ti->tool_info->tool = ERASER_TOOL;
	}
      else
	{
	  ti->tool_info->tool = PEN_TOOL;
	}
      gtk_option_menu_set_history(GTK_OPTION_MENU(ti->option_menu),
				  ti->tool_info->tool);
    }
}

void
tool_box_set_values(guint32 devid)
{
  ToolItem *ti;

  if (!toolBox) return;

  ti = tool_box_find_device(devid);
  if (ti)
    {
      gtk_option_menu_set_history(GTK_OPTION_MENU(ti->option_menu),
				  ti->tool_info->tool);
      if (ti == toolBox->current_prop)
	{
	  toolBox->size_data->value = ti->tool_info->size;
	  gtk_signal_emit_by_name (GTK_OBJECT (toolBox->size_data), "changed");
	  
	  toolBox->sens_data->value = ti->tool_info->sens;
	  gtk_signal_emit_by_name (GTK_OBJECT (toolBox->sens_data), "changed");

	  gtk_widget_draw (toolBox->brush_widget, NULL);
	}
      gtk_widget_draw(ti->properties,NULL);
    }
}

